/*
  works together with the deviceConfigServer <-> producer Interface
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#include <devices/genericPlayer.h>

static void *writerThread(void *arg){
  ((GenericPlayer*)arg)->writeloop();
  return NULL;
}   


#define _SHUTDOWN_FALSE             0
#define _SHUTDOWN_REQUEST           1
#define _SHUTDOWN_REQUEST_GRANTED   2
#define _SHUTDOWN_TRUE              3



GenericPlayer::GenericPlayer() {
  cThreadPos="Constructor";

  generator=NULL;
  config=new DeviceConfig();
  lExit=false;
  nThreadState=_THREAD_SLEEP;
  lastFileName=new Buffer(40);
  lPlayOnOpen=true;
  lOpenFile=false;
  playerStatus=config->getAudioStream()->getStatusInfo();
  lRunning=true;
  getEventQueue()->setNotifyMode(_NOTIFY_ALL);

  // for init of thread
  shutDownState=_SHUTDOWN_TRUE;
  registerAsStreamProducer();
  pthread_create(&tr,NULL,writerThread,this);


  while (shutDownState == _SHUTDOWN_TRUE) {
    NodeDevice::doSleep(1000);
  }

}


GenericPlayer::~GenericPlayer() {
  void* ret;

  sendSyncMessage(_STOP_WORKING_BECAUSE_OF_SHUTDOWN);
  kill();
  while(lExit==false) {
    writeInLock();
    nThreadState=_THREAD_DELIVER;
    writeInUnlock();
    NodeDevice::doSleep(10000);
  }

  writeInLock();
  nThreadState=_THREAD_SLEEP;
  lExit=true;
  writeInUnlock();
  pthread_join(tr,&ret);
  unregisterAsStreamProducer();
  


  // ok thread destroyed, now remove the streamproducer
  // from the graph.
  

  // The lock of the tree is necessary because we must gurantee that
  // not thread currently works on "our" memory
  // if we lock the tree the threads are all imprisioned in
  // the readloop of the streamproducer.
  // Thus noone works on our memory
  lockDeviceTree();
  if (generator != NULL) {
    delete generator;
    generator=NULL;
  }
  unlockDeviceTree();
  delete lastFileName;
  delete config;
}



char* GenericPlayer::getNodeName() {
  return "GenericPlayer";
}


int GenericPlayer::open(const char* filename ) {
  if (generator != NULL) {
    if (strlen(filename) == 0) {
      return false;
    }
    lOpenFile=true;
    generator->open((char*)filename);
    if (lPlayOnOpen) {
      generator->play();
    } else {
      generator->pause();
      generator->jump(0);
    }
    lastFileName->clear();
    lastFileName->append((char*)filename);
  }
  return true;
}

int GenericPlayer::close() {
  if (generator != NULL) {
    lOpenFile=false;
    generator->close();
  }
  return true;
}


int GenericPlayer::play() {
  if (generator != NULL) {
    // this implements a common interface behaviour.
    // if we are stopped and we click "play"
    // we play the last file we have already played
    if (getPlayerStatus() == _STATUS_STOPPED) {
      if (lOpenFile == false) {
	Buffer* tmp=new Buffer(50);
	tmp->append(lastFileName->getData());
	int ok=open(tmp->getData());
	delete tmp;
	if (ok == false) return true;
      }
    }
    generator->play();
  }
  return true;
}


int GenericPlayer::pause() {
  if (generator != NULL) {
    // This is a common interface behaviour.
    // If we are paused and we hit "pause" again
    // we try a "play"
    if (getPlayerStatus() == _STATUS_PAUSED) {
      generator->play();
    } else {
      if (getPlayerStatus() == _STATUS_PLAYING) {
	generator->pause();
      }
    }
  }
  return true;
}


int GenericPlayer::jump(int sec) {
  if (generator != NULL) {
    generator->jump(sec);
  }
  return true;
}

void GenericPlayer::kill() {
  if (generator != NULL) {
    generator->kill();
  }
}


void GenericPlayer::wakeUpThread(int lLockInProgress) {

  if (generator != NULL) {
     generator->wakeUpThread(lLockInProgress);
  }
  DaisyChain* daisyChain=(DaisyChain*)NodeDevice::getDaisyChain();
  daisyChain->wakeUpThread(this);


}


/**
   The streamState is the value int the statutInfo of the stream.
   Not the status of the thread!
*/
int GenericPlayer::getPlayerStatus() {
  int back;
  
  back=playerStatus->getStatus();
  return back; 
}


int GenericPlayer::getPlayOnOpen() {
  return lPlayOnOpen;
}


void GenericPlayer::setPlayOnOpen(int lPlayOnOpen) {
  this->lPlayOnOpen=lPlayOnOpen;
}


void GenericPlayer::setDataGenerator(DataGenerator* generator) {
  writeInLock();
  this->generator=generator;
  generator->setThreadNotifier(this);
  writeInUnlock();
}


DataGenerator* GenericPlayer::getDataGenerator() {
  return generator;
}


//
// ThreadNotifer Interface [START]
//


void GenericPlayer::sendSyncMessage(int msg) {
  writeInLock();
  switch(msg) {
  case _ALLOW_GENERATOR_ENTER: {
    nThreadState=_THREAD_DELIVER;
    break;
  }
  case _NOT_ALLOW_GENERATOR_ENTER: {
    nThreadState=_THREAD_SLEEP;
    break;
  }
  case _GENERATOR_CRASHED: {
    lRunning=false;
    getEventQueue()->sendEvent(_GS_SIGNAL_GENERATOR_CRASHED); 
    break;
  }
  case _STOP_WORKING_BECAUSE_OF_EOF : {
    lOpenFile=false;
    getEventQueue()->sendEvent(_GS_SIGNAL_PLAYING_READY); 
    lInitListener=false;
    break;
  }
  case _STOP_WORKING_BECAUSE_OF_SHUTDOWN : {
    nThreadState=_THREAD_DELIVER;
    lInitListener=false;
    shutDownState=_SHUTDOWN_REQUEST;
    break;
  }

  default:
    cout << "unknown msg: "<<msg<<" in sendSyncMessage"<<endl;
  }
  writeInUnlock();
}

int GenericPlayer::isRunning() {
  return lRunning;
}


//
// ThreadNotifier Interface [END]
//


/**
   The writeloop method is a single thread. It must be sure that
   the thread never blocks in the function below this.
   This is the only method where the thread can sleep.

   The thread delivers two different informations.
   One information is a statusInformation. If it delivers
   this no other info in the deviceConfig class is new.
   (no new audio pcm sample, no new musictitle etc,...)
   Only the status _change_ is delivered.

   If there is no status change to deliver, the thread
   delivers the things with minor priority. eg. pcm samples,...

   The thread always directly lock the writeIn mut.
   Outer threads must first lock the changemut and then the write
   in mut.
   The thread polls the changemut this gurantees that outer
   threads get the writein mut "in time"
*/

void GenericPlayer::writeloop() {
  static int instanceStatic=0;
  DeviceConfigArray* configArray=new DeviceConfigArray();
  int deliverMode;
  instanceStatic++;
  int instance=instanceStatic;
  pthread_mutex_lock(&writeInMut);



  nThreadState=_THREAD_SLEEP;
  config->setStreamProducer(this);
  lInitListener=false;
  shutDownState=_SHUTDOWN_FALSE;


  while(lExit==false) {
    cThreadPos="trylock changeMut -s";
 
    deliverMode=getDeliverMode();
    if (pthread_mutex_trylock(&changeMut) == EBUSY) {
      cThreadPos="change request -s";
      pthread_cond_wait(&changeCond,&writeInMut);
      cThreadPos="change request -e";
      continue;
    }
    pthread_mutex_unlock(&changeMut);
    cThreadPos="unlock changemut";


    if ( (deliverMode==_STREAMPRODUCER_DELIVERS_SINGLE) && 
         (shutDownState == _SHUTDOWN_FALSE) ) {
      cout << "_STREAMPRODUCER_DELIVERS_SINGLE"<<endl;
      getEventQueue()->sendEvent(_STREAMPRODUCER_DELIVERS_SINGLE);
      cThreadPos="deliverMode == single. Thread goes sleeping";
      setRunCredit(0);

      pthread_cond_wait(&changeCond,&writeInMut);
      cThreadPos="writeloop continues change request ready";
     
      if (getRunCredit() == 0) {
	continue;
      }
    }   
    cThreadPos="after single delivery";


    if (nThreadState == _THREAD_SLEEP){
      cThreadPos="nThreadState = sleep sleeping";
      pthread_cond_wait(&changeCond,&writeInMut);
      cThreadPos="nThreadState = sleep (end)";
      continue;
    }          
    cThreadPos="waitForUnblock -s";
    int locks=waitForUnblock();
    if (locks && (shutDownState == _SHUTDOWN_FALSE)) {
      cThreadPos="we had locks";
      continue;
    }
    if (shutDownState == _SHUTDOWN_FALSE) {
      nThreadState=generator->updateDeviceConfig(config);
    
      if (nThreadState == _THREAD_SLEEP) { 
	continue;
      }
      if (nThreadState == _THREAD_NO_DELIVER) { 
	continue;
      }
      if (nThreadState == _THREAD_DELIVER_THEN_SLEEP) {
	nThreadState=_THREAD_SLEEP;
      }
    }
      // This whole shutdown thing is necessary because
      // there is currently no clean shared memory solution!!!!!!
      
      
    if (shutDownState == _SHUTDOWN_REQUEST) {
      // make sure we do not delet things which do not belong
      // to this object (things from the shared memory)
      // ok this is not clean, but until I have a better solution
      // for this I do this hack
      config->getStatusInfo()->setStatus(_STATUS_STOPPED);
      config->getAudioStream()->getStatusInfo()->setStatus(_STATUS_STOPPED);
      AudioBuffer* buffer=config->getAudioStream()->getAudioBuffer();
      buffer->setMemChunk(NULL);  
      
      shutDownState = _SHUTDOWN_REQUEST_GRANTED;
    } else {
      if (shutDownState == _SHUTDOWN_REQUEST_GRANTED) {
	shutDownState=_SHUTDOWN_TRUE;
	lExit=true;
	continue;
      }
    }

    if ( hasListener() ) {
      configArray->clear();
      configArray->setDeviceConfigAt(0,config);
      configArray->setEntries(1);
      config->setStreamProducer(this);
      if (lInitListener==false) {
	lInitListener=true;
      }
      config->addLock();    // lock streamcontent while delivering

      writeOutLock();
      writeOut(configArray);
      writeOutUnlock();



      config->removeLock();
      config->forward(); 
		  
      
      continue;
    } 
    cThreadPos="no listener -s";
    pthread_cond_wait(&changeCond,&writeInMut);
    cThreadPos="no listener -e";
    if (shutDownState == _SHUTDOWN_REQUEST) {
      shutDownState=_SHUTDOWN_TRUE;
      lExit=true;
    }
  }

  // make sure we do not delet things which do not belong
  // to this object (things from the shared memory)
  // ok this is not clean, but until I have a better solution
  // for this I do this hack
  AudioBuffer* buffer=config->getAudioStream()->getAudioBuffer();
  buffer->setMemChunk(NULL);

  pthread_mutex_unlock(&writeInMut);
  delete configArray;
}
  


 
void GenericPlayer::writeOut(DeviceConfigArray* buf) {
  int i;
  int j;
  NodeDevice* nodeDevice;
  DeviceConfig* config;
  int m=buf->getEntries();
  Edges* edges=getListeners();
  int n=edges->getElements();

  for(j=0;j<m;j++) {
    config=buf->getDeviceConfigAt(j);
    for(i=0;i<n;i++) {
      nodeDevice=edges->getNodeDevice(i);

      nodeDevice->writeInLock();

      nodeDevice->writeIn(this,config);
      nodeDevice->writeInUnlock();
    }
  }
}


void GenericPlayer::dumpThread() {
  cout << "dumpThread:genericPlayer"<<endl;
  cout << cThreadPos<<"instanz:"<<instanz<<endl;
  StreamProducer::dumpThread();
}



