/*
  generic audiodevice
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "nasDevice.h"


#ifdef _HAVE_NAS

static AuBool testEventHandler(AuServer* auServer,
			       AuEvent* auEvent,
			       AuEventHandlerRec* auEventHandlerRec) {

  NASDevice* nasDevice=(NASDevice*)auEventHandlerRec->data;
  cout << "getting nas event"<<endl;
  return (nasDevice->nasEventHandler(auServer,auEvent,auEventHandlerRec));
}



NASDevice::NASDevice() {
  audioInfo=new AudioInfo();
  volumeInfo=new VolumeInfo();
  statusInfo=new StatusInfo();
  lneedInit=true;
  lAutoInit=true;
  lCanWrite=false;

  auServer=NULL;
  auFlowID=AuNone;
  matchDevice=AuNone;
  gainAdjustable=AuFalse;
  
  getEventQueue()->setNotifyMode(_NOTIFY_ALL);
  getEventQueue()->addListener(this);

  qwAuServer=new QwAuServer();

  setGain(80);
}


NASDevice::~NASDevice() {
  delete audioInfo;
  delete volumeInfo;
  delete statusInfo;
  close();
  delete qwAuServer;
}

AuBool NASDevice::nasEventHandler(AuServer* auServer,
				  AuEvent* auEvent,
				  AuEventHandlerRec* auEventHandlerRec) {

  if (auEvent->type==AuEventTypeElementNotify){
    cout << "AuEventTypeElementNotify"<<endl;
    AuElementNotifyEvent *event = (AuElementNotifyEvent *)auEvent;
    switch (event->kind) {
    case AuElementNotifyKindHighWater:{
      cout << "AuElementNotifyKindHighWater"<<endl;
      break;
    }
    case AuElementNotifyKindLowWater:{
      cout << "AuElementNotifyKindLowWater"<<endl;
      sendDataToNASServer(event->num_bytes);
      break;
    }
    case AuElementNotifyKindState:{
      if (event->cur_state==AuStatePause) {
	if (event->reason != AuReasonUser) {
	  cout << "AuStatePause"<<endl;
	  sendDataToNASServer(event->num_bytes);
	} else {
	  cout << "event->reason!=AuStatePause"<<endl;
	}
	return AuTrue;
      }
      cout << "event->cur_state:"<<event->cur_state<<endl;
      return AuTrue;
    }
    default:
      cout << "event->kind:"<<event->kind<<endl;
    }
    return AuTrue;
  } 
  if (auEvent->type==AuEventTypeMonitorNotify) {
    cout << "auEvent->type==AuEventTypeMonitorNotify"<<endl;
    return AuTrue;
  }
  if (auEvent->type==AuEventTypeGrabNotify) {
    cout <<"auEvent->type==AuEventTypeGrabNotify"<<endl;
    return AuTrue;
  }
  cout <<"auEvent->type:(unkown)"<<auEvent->type<<endl;
  return AuTrue;
}

  
char* NASDevice::getNodeName() {
  return "NASDevice";
}

int NASDevice::open(char* displayString) {
  cout << "open NASDevice"<<endl;
  if (isOpen()) {
    close();
  }
  if (qwAuServer->open(displayString)==false) {
     cout << "AuOpenServer failed"<<endl;
     return false;
  }
  cout << "getAuServer"<<endl;
  auServer=qwAuServer->getAuServer();
  cout << "AuRegisterEventHandler"<<endl;
  AuRegisterEventHandler(auServer,0,0,0,testEventHandler,(AuPointer)this);
  cout << "nas is fine"<<endl;
  init(true);
  return true;
}


int NASDevice::isOpen() {
  return (qwAuServer->isOpen());
}


int NASDevice::isCompiledIn() {
  return true;
}

int NASDevice::close() {
  if (isOpen()) {
    destroyFlow();
    qwAuServer->close();
    auServer=NULL;
  }
  return true;
}


int NASDevice::init(int sampleSize,int speed,int stereo) {
  cout << "sampleSize:"<<sampleSize<<endl;
  audioInfo->setStereo(stereo);
  audioInfo->setSampleSize(sampleSize);
  audioInfo->setSpeed(speed);
  return init(true);
}


int NASDevice::init(int lForceSet) {
  int stereo=audioInfo->getStereo();
  int sampleSize=audioInfo->getSampleSize();
  int speed=audioInfo->getSpeed();

  AuElement elements[3];

  if (isOpen() == false) {
    cout << "********** not open"<<endl;
    return false;
  } 

  // now find a device which supports our channels
  int i;
  int n=AuServerNumDevices(auServer);
  matchDevice=AuNone;
  gainAdjustable=AuFalse;
  for (i = 0;i<n;i++) {
    AuDeviceAttributes* currentDevice=AuServerDevice(auServer, i);
    if (AuDeviceKind(currentDevice)==AuComponentKindPhysicalOutput) {
	 if (AuDeviceNumTracks(currentDevice) == (stereo+1)) {
	   matchDevice=AuDeviceIdentifier(currentDevice);
	   int flag=AuDeviceChangableMask(currentDevice)&AuCompDeviceGainMask;
	   if (flag) {
	     gainAdjustable=AuTrue;
	   }
	   break;
	 }
    }
  }
  if (matchDevice == AuNone) {
    cout << "Couldn't find an output device providing"
	 << (stereo+1)
         << "channels" 
         << endl;
    return false;
  }

  if (lForceSet || lAutoInit ) {
    // here we close a existing flow
    destroyFlow();
    if (createFlow() == false) {
      cout << "couldnt init because flow not created"<<endl;
      return false;
    }
    cout << "********* creating components"<<endl;

#ifndef WORDS_BIGENDIAN

    AuMakeElementImportClient(&elements[0],
			      speed, 
			      AuFormatLinearSigned16LSB,stereo+1,
			      AuTrue,RINGBUFFERSIZE,
			      LOW_WATER, 0, NULL);
#else
    AuMakeElementImportClient(&elements[0],
			      speed, 
			      AuFormatLinearSigned16MSB,stereo+1,
			      AuTrue,RINGBUFFERSIZE,
			      LOW_WATER, 0, NULL);
#endif

    AuFixedPoint volume=AuFixedPointFromFraction(50, 100);
    AuMakeElementMultiplyConstant(&elements[1], 0, volume);

    AuMakeElementExportDevice(&elements[2], 1, matchDevice,
			      speed,
			      AuUnlimitedSamples, 0,NULL);
    

    AuSetElements(auServer,auFlowID, AuTrue, 3, elements, NULL);
    AuStartFlow(auServer,auFlowID, NULL);
    setGain(getGain());
    qwAuServer->clear();
    lCanWrite=true;
    return true;
  }
  return false;
}

int NASDevice::getGain(){
  return gain;
}

void NASDevice::setGain(int val){

  this->gain=val;
  if(hasFlow()){
    if (gainAdjustable) {
      AuDeviceAttributes da;
      AuDeviceGain(&da)=AuFixedPointFromSum(gain,100);
      AuSetDeviceAttributes(auServer,matchDevice,
			    AuCompDeviceGainMask,&da,NULL);
    } else {
      cout << "gain not adjustible!"<<endl;
    }
  }

} 

const char* NASDevice::getServerString(){
  if (isOpen()==false) {
    return NULL;
  }
  return AuServerString(auServer);
}


const char* NASDevice::getVendorString(){
  if (isOpen()==false) {
    return NULL;
  }
  return AuServerVendor(auServer);
}


int NASDevice::getMajorProtocolVersion(){
  if (isOpen()==false) {
    return -1;
  }
  return AuServerProtocolMajorVersion(auServer);
}


int NASDevice::getMinorProtocolVersion(){
  if (isOpen()==false) {
    return -1;
  }
  return AuServerProtocolMinorVersion(auServer);
}


void NASDevice::nasSetVolume(float left,float right) {
  AuElementParameters parms;
  if (hasFlow()) {
    AuStatus status=AuSuccess;
    parms.flow = auFlowID;
    parms.element_num = 1;
    parms.num_parameters = AuParmsMultiplyConstant;
    parms.parameters[AuParmsMultiplyConstantConstant] =
      AuFixedPointFromFraction((left+right)/2.0, 100);
    AuSetElementParameters(auServer, 1, &parms, &status);
    if (status != 0) {
      cout << "nasSetVolume status != 0:"<<status<<endl;
      return;
    }
   
  }
} 

void NASDevice::writeIn(NodeDevice* source,DeviceConfig* config) {

  if (getEnabled() == false) {
    return;
  }

  AudioStream* audioStream=config->getAudioStream();
  AudioBuffer* buffer=audioStream->getAudioBuffer();
  MemChunk* memChunk=buffer->getMemChunk();
  StatusInfo* statusInfoStream=audioStream->getStatusInfo();
  AudioInfo* audioInfoStream=audioStream->getAudioInfo();
  VolumeInfo* volumeInfoStream=audioStream->getVolumeInfo();

  // We ignore statusChange messages, because otherwise
  // it is possible that we initialize /dev/dsp with "strange"
  // values. (Linux ignores this FreeBSD hangs)
  if (statusInfoStream->getChange()) {
    statusInfoStream->copyTo(statusInfo);
    if (statusInfoStream->getStatus()==_STATUS_STOPPED) {
      getEventQueue()->sendEvent(_NAS_STREAM_CHANGE);
    }
    return;
  }
  if (!(audioInfoStream->equals(audioInfo)) || lneedInit) {
    audioInfoStream->copyTo(audioInfo);
    getEventQueue()->sendEvent(_NAS_STREAM_CHANGE);
  }
  if (!(volumeInfoStream->equals(volumeInfo)) || lneedInit) {
    volumeInfoStream->copyTo(volumeInfo);
    getEventQueue()->sendEvent(_NAS_STREAM_CHANGE);
  }  
  lneedInit=false;
  if (hasFlow() == false) {
    getEventQueue()->sendEvent(_AUDIODEVICE_CANNOT_OPEN);
    return;
  }
  if (lCanWrite) {
    if (memChunk != NULL) {
      if (memChunk->getLen() > 0) {



	int didWrite=rawWrite(memChunk->getPtr(),memChunk->getLen());

	if (qwAuServer->getFillgrade() > HIGH_WATER) {
	  StreamProducer* producer=
	    (StreamProducer*)config->getStreamProducer();
	  DaisyChain* daisyChain=(DaisyChain*)(producer->getDaisyChain());
	  daisyChain->addLock(this,producer);
	}
	

	if (didWrite == -1) {
	  perror("********* write in NASDevice");
	}
	if (didWrite != memChunk->getLen()){
	  lneedInit=true;
	}
      } else {
	cout << "memChunk len is 0"<<endl;
      }
    } else {
      cout << "memChunk is NULL!"<<endl;
    }
  } else {
    qwAuServer->clear();
    NodeDevice::doSleep(500);
    lneedInit=true;
  }

}
  



int NASDevice::rawWrite(char* buf,int len) {




  getEventQueue()->sendEvent(_NAS_DATA_TO_WRITE);
  qwAuServer->writeToRingBuffer(buf,len);
  return len;
}



/*
  This completly seperation between the thread and the nas
  functions calls is necessary because otherwise nas terminates.
  (seems to have to do with the nas callbacks)
*/ 
void NASDevice::processEvent(char eventID) {

  switch(eventID) {  
  case _NAS_STREAM_CHANGE: {
    if (statusInfo->getChange()) {
      cout << "statusInfo"<<endl;
      statusInfo->setChange(false);
    }
    if (audioInfo->getChange()) {
      cout << "audioInfo"<<endl;

      init(audioInfo->getSampleSize(),
	   audioInfo->getSpeed(),
	   audioInfo->getStereo());
      audioInfo->setChange(false);
      
    }
    if (volumeInfo->getChange()) {
      cout << "volumeInfo"<<endl;
      volumeInfo->setChange(false);      
    }
    break;
  }
  case _NAS_DATA_TO_WRITE: {
    sendDataToNASServer(8192);
    break;
  }

  case _AUDIODEVICE_CLOSE_REQUEST: 
  case _AUDIODEVICE_CANNOT_OPEN: {
    break;
  }
    
 default:
    cout << "unknown msg: "<<eventID<<" in NASDevice::processEvent-1"<<endl;
  }
  
}


void NASDevice::sendDataToNASServer(int bytes){
  int fillgrade=qwAuServer->getFillgrade();


  if (fillgrade < HIGH_WATER) {
    DaisyChain* daisyChain=(DaisyChain*)NodeDevice::getDaisyChain();
    daisyChain->removeAllLocks(this);
  }

  if (fillgrade == 0) {
    return;
  }
  if (hasFlow()) {
    if (qwAuServer->sendDataToNASServer(auFlowID,0,bytes) == 0) {
      return;
    }
  } else {
    qwAuServer->clear();
  }
  
 
}

int NASDevice::createFlow() {
  if (hasFlow()) {
    destroyFlow();
  }
  auFlowID = AuCreateFlow(auServer, NULL);
  cout << "creating flow"<<endl;
  if (auFlowID == AuNone) {
    cout << "AuCreateFlow failed"<<endl;
    return false;
  }
  return true;
}

void NASDevice::destroyFlow(){
  lCanWrite=false;
  if (hasFlow()) {
    cout << "destroying flow"<<endl;
    AuDestroyFlow(auServer, auFlowID, NULL);
    auFlowID=AuNone;
  }

}

int NASDevice::hasFlow() {
  return (auFlowID != AuNone);
}

#endif
