/*
  this device analyses the spectrum and sends notify messages.
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <devices/spectrumAnalyserDevice.h>


SpectrumAnalyserDevice::SpectrumAnalyserDevice(int fftBands=128):
  OutputDevice("SpectrumAnalyserDevice") {

  fftBands_fix=512;
  fftBands_short=256;
  
  realFFTFilter= new RealFFTFilter(fftBands_short);
  fixFFTFilter=new FixFFTFilter(fftBands_fix,75);

  lEnabled=true;
  visAnalyserArray=new float[getBands()];
  fftArray=new int[fftBands_short];
  fftArray2=new float[fftBands_fix];

  lAnalyse=true;
  lAnalyseMode=__ANALYSE_MODE_SHORT;
}
 

SpectrumAnalyserDevice::~SpectrumAnalyserDevice() {
 delete realFFTFilter;
 delete fixFFTFilter;
 delete visAnalyserArray;
 delete fftArray;
 delete fftArray2;
}


char* SpectrumAnalyserDevice::getNodeName() {
  return "SpectrumAnalyserDevice";
}


void SpectrumAnalyserDevice::setAnalyse(int lAnalyse) {
  this->lAnalyse=lAnalyse;
}


int SpectrumAnalyserDevice::getBands() {
  return 75;
}


float* SpectrumAnalyserDevice::getBandPtr() {
  return visAnalyserArray;
}

void SpectrumAnalyserDevice::setAnalyseMode(int mode) {
  lAnalyseMode=mode;
}



/*
  Elements in the spectrumArray describe the percent value
  by a band

*/

void SpectrumAnalyserDevice::writeIn(NodeDevice* source,DeviceConfig* buf) {

  AudioStream* audioStream=buf->getAudioStream();
  StatusInfo* statusInfoStream=audioStream->getStatusInfo();
  if (statusInfoStream->getChange()) {
    getEventQueue()->setNotifyMode(_NOTIFY_ALL);
    if (statusInfoStream->getStatus() != _STATUS_PLAYING) {
      getEventQueue()->sendEvent(__SIGNAL_SPECTRUM_ANALYSER_STOP);
    } else {
      getEventQueue()->sendEvent(__SIGNAL_SPECTRUM_ANALYSER_START);
    }
    getEventQueue()->setNotifyMode(_NOTIFY_IF_CLEAR);
  }
  if (lEnabled == false) {
    return;
  }

  if (lAnalyse == false) {
    return;
  }
  if (lAnalyseMode == __ANALYSE_MODE_SHORT) {
    doRealFFT(buf);
  } else {
    doFloatFFT(buf);
  }
  lEnabled=false;
}


 

void SpectrumAnalyserDevice::doRealFFT(DeviceConfig* config) {
  int i;
  int j;
  int re;
  int im;
  int tmp;
  short* fftPtr;
  int* bitReversed;
  float max=0.0;
  float avg=0.0;

  if (realFFTFilter->fft16(config) == false) {
    return;
  }

  j=0;
  fftPtr=realFFTFilter->getPointPtr();
  bitReversed=realFFTFilter->getBitReversed();

  int pos=0;
  int step=realFFTFilter->getPoints()/getBands();

  for (i=0;i<75;i++) {
    re=(int)fftPtr[bitReversed[pos]];
    im=(int)fftPtr[bitReversed[pos]+1];
    
    tmp=re*re+im*im;  
    // Here I check a new idea. We remove all low values
    // and all values over xyz to xyz.
    fftArray[pos]=(int)(sqrt(sqrt(tmp)));
    
    if (fftArray[pos]<=15) {
      max+=fftArray[pos];
    } else {
      max+=15+fftArray[pos]/2;
    }
    pos=pos+step;
  }
  avg=0.65*max/(float)75;

  // The following term make the spectrum analyser look nice
  // ****dont ask how long I have tested until I found out
  //     that this _might_ be the way winamp does the SPA****
  // Is there any good book about this ????????????? -> email me!
  pos=0;
  for (i=0;i<75;i++) {
    visAnalyserArray[i]=(float)(fftArray[pos]-avg);
    pos=pos+step;
  }

  
}


void SpectrumAnalyserDevice::doFloatFFT(DeviceConfig* config) {
  int i;
  if (fixFFTFilter->fft16(config) == false) {
    return;
  }
  short* loudArray=fixFFTFilter->getPointPtr();
  int pos=0;
  int step=fixFFTFilter->getPoints()/(getBands()*2);
  for(i=0;i<75;i++) {
    if(loudArray[i]>-60)
      visAnalyserArray[i]=(loudArray[pos]+60)/2;
    else 
      visAnalyserArray[i]=0;
    pos+=step;
  }
  

}





int SpectrumAnalyserDevice::getEnabled() {
  return lEnabled;
}


void SpectrumAnalyserDevice::setEnabled(int lEnable) {
  this->lEnabled=lEnable;
}

