/*
  streams data to harddisk.
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include <devices/wavFileDevice.h>


WavFileDevice::WavFileDevice():OutputDevice("WavFileDevice") {
  dirName=new Buffer(10);
  fileName=new Buffer(10);
  currentFile=new Buffer(10);
  file=NULL;
  lEnabled=false;
  lOpen=false;
  bytecounter=0;
  dirName->append("/tmp/");
  musicInfo=new MusicInfo();
  lStopIsEof=true;
}


WavFileDevice::~WavFileDevice() {
  delete dirName;
  delete fileName;
  delete currentFile;
  delete musicInfo;
}

char* WavFileDevice::getNodeName() {
  return "WavFileDevice";
}


int WavFileDevice::getEnabled() {
  return lEnabled;
}


void WavFileDevice::setEnabled(int lEnable) {
  // block the device for a short time
  writeInLock();
  this->lEnabled=lEnable;
  if ((isOpen() == true) && (lEnabled==false)) {
    doClose();
  }
  writeInUnlock();
}


void WavFileDevice::writeIn(NodeDevice* source,DeviceConfig* config) {
  if (lEnabled == false) {
    return;
  }
  AudioStream* audioStream=config->getAudioStream();
  MusicInfo* musicInfoStream=audioStream->getMusicInfo();
  AudioBuffer* audioBufferStream=audioStream->getAudioBuffer();
  StatusInfo* statusInfoStream=audioStream->getStatusInfo();

  if (statusInfoStream->getChange()) {
    if (statusInfoStream->getStatus() == _STATUS_STOPPED) {
      if (lStopIsEof) {
	if (isOpen()) {
	  doClose();
	}
      }
    }
    return;
  }
  if (isOpen() == false) {
    musicInfoStream->copyTo(musicInfo);
    if (doOpen() == false) {
      writerThreadErrno=errno;
      getEventQueue()->sendEvent(_WAVFILEDEVICE_SIGNAL_ERROR);
      lEnabled=false;
      return;
    }
  }
  MemChunk* memChunk=audioBufferStream->getMemChunk();

  if (memChunk != NULL) {
    int wrote;
    char* ptr=memChunk->getPtr();
    int len=memChunk->getLen();
    wrote=fwrite(ptr,sizeof(char),len,file);
    if (wrote !=len) {
      writerThreadErrno=errno;
      strerror(writerThreadErrno);
      getEventQueue()->sendEvent(_WAVFILEDEVICE_SIGNAL_ERROR);
      lEnabled=false;
      writerThreadErrno=errno;
      return;
    }
    bytecounter=bytecounter+wrote;
  }
  
}


void WavFileDevice::createFileName(Buffer* target) {

  target->clear();
  target->append(dirName->getData());
  target->append(musicInfo->getName());
  target->append(".wav");
}
  


int WavFileDevice::doOpen() {
  int back=false;
  createFileName(fileName);
  
  file=fopen(fileName->getData(),"w");
  if (file != NULL) {
    lOpen=true;
    back=true;
    WriteEmptyWavHeader(file);
  }
  bytecounter=0;
  
  return back;
}

int WavFileDevice::isOpen() {
  return lOpen;
}

void WavFileDevice::doClose() {
  if (isOpen()) {
    WriteLengthWavHeader(file,bytecounter);
    fclose(file);
    bytecounter=0;
    lOpen=false;
    file=NULL;
    fileName->clear();
  }
}


void WavFileDevice::WriteEmptyWavHeader(FILE* file){
  /* quick and dirty */

  fwrite("RIFF",sizeof(unsigned char),4,file);     /*  0-3 */
  PutNum(44-8,file,0,4);                  /*  4-7 */
  fwrite("WAVEfmt ",sizeof(unsigned char),8,file); /*  8-15 */
  PutNum(16,file,0,4);                    /* 16-19 */
  PutNum(1,file,0,2);                     /* 20-21 */
  PutNum(2,file,0,2);                     /* 22-23 */
  PutNum(44100,file,0,4);                 /* 24-27 */
  PutNum(44100*2*2,file,0,4);             /* 28-31 */
  PutNum(4,file,0,2);                     /* 32-33 */
  PutNum(16,file,0,2);                    /* 34-35 */
  fwrite("data",sizeof(char),4,file);     /* 36-39 */
  // here we write per default a 0 length
  PutNum(0,file,0,4);                        /* 40-43 */
    
}


void WavFileDevice::WriteLengthWavHeader(FILE* file,long bytes){
  fseek(file,4,SEEK_SET);
  PutNum(bytes+44-8,file,0,4);        /*  4-7 */
  fseek(file,40,SEEK_SET);
  PutNum(bytes,file,0,4);             /* 40-43 */
}
  

void WavFileDevice::PutNum(long num,FILE* file,int endianness,int bytes){
  int i;
  unsigned char c;

  if(!endianness)
    i=0;
  else
    i=bytes-1;
  while(bytes--){
    c=(num>>(i<<3))&0xff;
    if(fwrite(&c,sizeof(unsigned char),1,file)!=1){
      perror("Could not write to output.");
      exit(1);
    }
    if(endianness)
      i--;
    else
      i++;
  }
}
 

int WavFileDevice::getErrno() {
  return writerThreadErrno;
}


char* WavFileDevice::getFileName() {
  // here we must work around a possible race.
  // the thread can change this during access,
  // thus we must make each time a local copy
  writeInLock();
  currentFile->clear();
  currentFile->append(fileName->getData());
  writeInUnlock();
  return currentFile->getData();
}


void WavFileDevice::setTargetDirectory(char* dir) {
  // block the device for a short time
  writeInLock();
  dirName->clear();
  dirName->append(dir);
  writeInUnlock();
}


long WavFileDevice::getByteCounter() {
  return bytecounter;
}


void WavFileDevice::setStopIsEOF(int lStopIsEof) {
  this->lStopIsEof=lStopIsEof;
}
