/***************************************************************************
                          hardware.cpp  -  description
                             -------------------
    begin                : Thu Sep 30 1999
    copyright            : (C) 1999 by Ralf Nolden
    email                : Ralf.Nolden@post.rwth-aachen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
// hardware.cpp  -- This file contains the class definitions for the hardware
// cRadio/ktuner (Carl Radio) v 0.9 : Carl van Schaik 1998
// e-mail carl@leg.uct.ac.za
//
// Some hardware code derived from
// xradiotrack V.1.1  F.W.Brinkman 1995.
//
// Zoltrix card - derived from Frans Brinkman's radiod
//              - better tuning, scanning by Carl van SChaik
//
// Radiotrack II implementation : Leendert Meyer (leendert@usa.net).
//
// RadioTrack II Protocol : Ben Pfaff

#ifndef HARDWARE_CPP
#define HARDWARE_CPP

#include "hardware.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#ifdef FREEBSD
#include <machine/cpufunc.h>
#include <machine/types.h>
#include <sys/ioctl.h>
#else
#include <asm/io.h>
#include <asm/types.h>
  #ifdef USE_KERNEL
  #include <linux/videodev.h>
  #endif //USE_KERNEL
#endif //FREEBSD

#ifdef FREEBSD
    int fpiopl;
#endif


/**************************** Aimslab radiotrack *******************************/

rtrack::rtrack(configFile* cf)
{
  cardFound=0;
  cardBase=0;
  onOff=0;
  muted=0;
  volume=0;
  conf=cf;
}

int rtrack::init(int noVolSet, int noDefSet) //Initialises the card (and returns 1 if error)
{
  //Need to add a lot of stuff here

  cardBase = conf->getBasePort();
  if ((cardBase==0x20f) || (cardBase==0x30f))
    cardFound = 1;

  int p;
  p = getIoPrivs();
  if (p != 0) return 2;
  giveUpRoot();

  if (!noVolSet)
    {
      printf("Setting volume\n");
      setVolume(conf->getVolume());
    }
  else
    volume = conf->getVolume();
  if (!noDefSet)
    tuneFreq(double(conf->getFreq(conf->getDefault()))/20.0 +87.0);

  return 0;  //For now
}

void rtrack:: cardOff()
{
  if (cardFound)
    {
      cardWrite(0);
      cardWrite(0);
    }
}

void rtrack::cardOn()
{
  if (cardFound)
    {
      /* switch sound back on */
      cardWrite(0);
      cardWrite(0xc8);
    }
}

void rtrack::cardWrite(int k)
{
#ifdef FREEBSD
  outb(cardBase,k);
#else
  outb(k,cardBase);
#endif
}

int rtrack::cardToggle()  //Same as mute (if off, sets it on etc)
{
  if (cardFound)
    {
      if ( ! muted )              /* case we weren't muted */
	{
	  muted=1;
	  cardWrite(0xd0);
	  return(1);
	}
      else
	{
	  /* switch sound back on */
	  muted=0;
	  cardOn();
	  return(0);
	}
    }
  else
  	return 0;
}

void rtrack::volDown()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume > 0) volume --;
	  /* vol down a notch */
	  cardWrite(0x48);
	  usleep(10000);
	  cardWrite(0xc8);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }

}


void rtrack::setVolume(int v)
{
  if ((v>=0) && (v<16))
    {
      int i;
      for (i=0;i<16;i++)
	{
	  volDown();
	  usleep(100000);
	}
      for (i=0;i<v;i++)
	{
	  volUp();
	  usleep(100000);
	}
      volume = v;
    }
}

int rtrack::getVolume()
{
  return volume;
}

void rtrack::volUp()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume < 16) volume ++;
	  /* vol up a notch */
	  cardWrite(0x88);
	  usleep(10000);
	  cardWrite(0xc8);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}

void rtrack::tuneFreq(double freq)
{
  if (cardFound)
    {
      /* tunes the radio to the desired frequency */
      unsigned long int bits;
      int i,mask;
      
      bits=(int(freq*40))+10486188;
      /* yes, i know. now where does THAT come from?
       * working out that bits=(freq*multiplier)+base
       * was hard, and working out the base and multiplier
       * was tricky. */
      mask=1;
      for (i=0; i<24; ++i)
	{
	  if ((bits&mask)==0)
	    {
	      /* 0 bit */
	      cardWrite(0x1);
	      cardWrite(0x1);
	      cardWrite(0x3);
	      cardWrite(0x3);
	    }
	  else
	    {
	      /* bit is 1 */
	      cardWrite(0x5);
	      cardWrite(0x5);
	      cardWrite(0x7);
	      cardWrite(0x7);
	    }
	  mask=mask*2;
	}
      /* termination sequence */
      cardWrite(0x0);            /* keeps current volume setting */
      usleep(10000);
      cardWrite(0xc8);
    }
}

int rtrack::isTuned()
{
  if (cardFound)
    {
      int x;
      
      /* test tuning */
      /*
        DO NOT CHANGE THE TIMING !?!?!?
	
        Scanning seems to be pretty unreliable, or at least using
        the one card I've got. A shorter wait will miss lots of valid
        stations, longer ones will make it find lots of noisy signals.
        Should anyone ever find out how it is supposed to work please
        let me know. For absolutely no reason that I can see it also seems
        to work slightly better if the frequencies used are multiples
        of .05, but that might just be coincidence
      */
      cardWrite(0xf8);
      //      usleep(200000);              /* any faster and it won't work? */
      usleep(200000);              /* any faster and it won't work? */
      x = inb(cardBase);
      usleep(10000);
      cardWrite(0xe8);
      
      /* if x == ff not tuned, if x == fd then tuned */
      
      if (( x & 255) != 0x0ff)
      	return(1);
      else
      	return(0);

    }
  else
  	return 0;
}

int radioCard::getIoPrivs()
{
  if (cardFound)
    {
#ifdef FREEBSD
        if ((fpiopl = open( "/dev/io", O_RDONLY ) < 0) )
        {
                return (fpiopl);
        }
        return(0);
#else
      unsigned long basePort = cardBase;
      /* get ioperm for RadioTrack range */
      int err;
  
      err=ioperm(basePort,4,0xFFFF);  /* i could not deduce from
				       * man page what turn_on must
				       * be */
      return(err);
#endif
    }
    else
    	return 0;
}

void radioCard::giveUpRoot()
{
  if (cardFound)
    {
      /* get the real uid and give up root */
#ifdef FREEBSD
      uid_t uid;
#else
      __uid_t uid;
#endif
      int err;

      uid=getuid();
      err=seteuid(uid);
    }
}

/**************************** Aimslab radiotrack II *******************************/

rtrackII::rtrackII(configFile* cf)
{
  cardFound=0;
  cardBase=0;
  onOff=0;
  muted=0;
  volume=0;
  conf=cf;
}

int rtrackII::init(int noVolSet, int noDefSet) //Initialises the card (and returns 1 if error)
{
  //Need to add a lot of stuff here

  cardBase = conf->getBasePort();
  if ((cardBase==0x20f) || (cardBase==0x30f))
    cardFound = 1;

  int p;
  p = getIoPrivs();
  if (p != 0) return 2;
  giveUpRoot();

  if (!noVolSet)
    {
      printf("Setting volume\n");
      setVolume(conf->getVolume());
    }
  else
    volume = conf->getVolume();
  if (!noDefSet)
    tuneFreq(double(conf->getFreq(conf->getDefault()))/20.0 +87.0);

  return 0;  //For now
}

void rtrackII::cardOff()
{
  if (cardFound)
    {
      cardWrite(1);
    }
}

void rtrackII::cardOn()
{
  if (cardFound)
    {
      /* switch sound back on */
      cardWrite(0);
      cardWrite(0xc8);
    }
}

void rtrackII::cardWrite(int k)
{
#ifdef FREEBSD
  outb(cardBase,k);
#else
  outb(k,cardBase);
#endif
}

int rtrackII::cardToggle()  //Same as mute (if off, sets it on etc)
{
  if (cardFound)
    {
      if ( ! muted )              /* case we weren't muted */
	{
	  muted=1;
	  cardWrite(0xd0);
	  return(1);
	}
      else
	{
	  /* switch sound back on */
	  muted=0;
	  cardOn();
	  return(0);
	}
    }
}

void rtrackII::volDown()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume > 0) volume --;
	  /* vol down a notch */
	  cardWrite(0x48);
	  usleep(10000);
	  cardWrite(0xc8);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}


void rtrackII::setVolume(int v)
{
  if ((v>=0) && (v<16))
    {
      int i;
      for (i=0;i<16;i++)
	{
	  volDown();
	  usleep(100000);
	}
      for (i=0;i<v;i++)
	{
	  volUp();
	  usleep(100000);
	}
      volume = v;
    }
}

int rtrackII::getVolume()
{
  return volume;
}

void rtrackII::volUp()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume < 16) volume ++;
	  /* vol up a notch */
	  cardWrite(0x88);
	  usleep(10000);
	  cardWrite(0xc8);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}

void rtrackII::tuneFreq(double freq)
{
	int i;
	double TDFreq = freq * 16000;
	unsigned long TmpFreq = (long) TDFreq;

	if (cardFound)
	{
	   TmpFreq = TmpFreq / 200 + 856;
	   cardWrite(0xc8);
	   cardWrite(0xc9);
	   cardWrite(0xc9);

	   for (i = 0; i < 10; i++)
			   zero ();

	   for (i = 14; i >= 0; i--)
			   if (TmpFreq & (1 << i))
					   one ();
			   else
					   zero ();

	   cardWrite(0xc8);
	   if (muted)
		  cardWrite(0);
	}
}

void rtrackII::zero(void)
{
	cardWrite(1);
	cardWrite(3);
	cardWrite(1);
}

void rtrackII::one(void)
{
	cardWrite(5);
	cardWrite(7);
	cardWrite(5);
}


int rtrackII::isTuned()
{
	if (cardFound)
	{
		int x, x2;

		// cardWrite(0xf8);
		usleep(350000);              /* any faster and it won't work? */
		x = inb(cardBase);
		usleep(10000);
		// cardWrite(0xe8);

		/* if x == fa not tuned, if x == f8 then tuned */

		if (( x & 255) != 0x0fa) return(1);
		else return(0);

	}
}


/********************************* Zoltrix card starts here *******************************/

zoltrix::zoltrix(configFile* cf)
{
  cardFound=0;
  cardBase=0;
  onOff=0;
  muted=0;
  volume=0;
  conf=cf;
}

int zoltrix::init(int noVolSet, int noDefSet) //Initialises the card (and returns 1 if error)
{
  //Need to add a lot of stuff here

  cardBase = conf->getBasePort();
  if ((cardBase==0x20c) || (cardBase==0x30c))
    cardFound = 1;

  int p;
  p = getIoPrivs();
  if (p != 0) return 2;
  giveUpRoot();

  if (!noVolSet)
    {
      printf("Setting volume\n");
      setVolume(conf->getVolume());
    }
  if (!noDefSet)
    tuneFreq(double(conf->getFreq(conf->getDefault()))/20.0 +87.0);

  return 0;  //For now
}

void zoltrix:: cardOff()
{
  if (cardFound)
    {
      cardWrite(0);
      cardWrite(0);
      cardRead(3);
    }
}

void zoltrix::cardOn()
{
  if (cardFound)
    {
      /* switch sound back on */
      cardWrite(volume);
      usleep(10000);
      cardRead(2);
    }
}

void zoltrix::cardWrite(int k)
{
#ifdef FREEBSD
  outb(cardBase,k);
#else
  outb(k,cardBase);
#endif
}

int zoltrix::cardRead(int offset)
{
  return inb(cardBase+offset);
}

int zoltrix::cardToggle()  //Same as mute (if off, sets it on etc)
{
  if (cardFound)
    {
      if ( ! muted )              /* case we weren't muted */
	{
	  muted=1;
	  cardOff();
	  return(1);
	}
      else
	{
	  /* switch sound back on */
	  muted=0;
	  cardOn();
	  return(0);
	}
    }
}

void zoltrix::volDown()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume > 0) volume --;
	  /* vol down a notch */
	  cardWrite(volume);
	  usleep(10000);
	  cardRead(2);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}


void zoltrix::setVolume(int v)
{
  if (cardFound)
    {
      if ((v>=0) && (v<16))
	{
	  int i;
	  for (i=0;i<16;i++)
	    {
	      volDown();
	      usleep(10000);
	    }
	  for (i=0;i<v;i++)
	    {
	      volUp();
	      usleep(10000);
	    }
	  volume = v;
	}
    }
}

int zoltrix::getVolume()
{
  return volume;
}

void zoltrix::volUp()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume < 16) volume ++;
	  /* vol up a notch */
	  cardWrite(volume);
	  usleep(10000);
	  cardRead(2);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}

void zoltrix::tuneFreq(double freq)
{
  if (cardFound)
    {
      /* tunes the radio to the desired frequency */
      unsigned long long bitmask, f;
      int i,m;

      f = (unsigned long long)(((float)(freq)-88.05)*200.0)+0x4d1c;
      bitmask = 0xc480402c10080000ull;
      i = 45;
      cardWrite (0x00); 
      cardWrite (0x00); 
      cardRead(3);

      cardWrite (0x40); 
      cardWrite (0xc0); 
      bitmask = (bitmask ^ ((f & 0xff) << 47) ^ ((f & 0xff00) << 30) ^ (/*stereo*/ 0 << 31));
      while (i--)
        {
          if ((bitmask & 0x8000000000000000ull) != 0)
            {
              cardWrite (0x80);
              cardWrite (0x00);
              cardWrite (0x80);
            }
          else
            {
              cardWrite (0xc0);
              cardWrite (0x40);
              cardWrite (0xc0);
            }
          bitmask *= 2;
        }
      /* termination sequence */
      cardWrite (0x80);
      cardWrite (0xc0);
      cardWrite (0x40);
      usleep (20000);
      cardWrite (volume);
      usleep (10000);
      cardRead(2);
    }
}

int zoltrix::isTuned()
{
  if (cardFound)
    {
      int x1,x2;
      
        outb(0x00, cardBase);         // This stuff I found to do nothing
        outb(volume, cardBase);
        usleep(10000);

        x1 = inb(cardBase);
        usleep(1000);
        x2 = inb(cardBase);

//      if (((a | b) & 255) != 0x0ff) // old method (un-reliable)
        if ((x1 == x2) && (x1 == 0xdf))  // I found this out by playing
           return (1);                // with a binary scanner on the card io

      else return(0);

    }
}

/************************* Kernel video4linux starts here ***************************/

#ifdef USE_KERNEL

struct video_audio va;

kernelCard::kernelCard(configFile* cf)
{
  cardFound=0;
  cardBase=0;
  onOff=0;
  muted=0;
  volume=7;
  save=0;
  conf=cf;
  lastFr=0;
  lastF=87.0;
}

kernelCard::~kernelCard()
{
  if (cardFound)
    close(dev);
}

int kernelCard::init(int noVolSet,int noDefSet) //Initialises the card (and returns 1 if error)
{
  dev = open ("/dev/radio", O_RDONLY);
  if (dev != -1)
    cardFound = 1;
  else
    return 40;

  if (!noVolSet)
    setVolume(conf->getVolume());
  save = conf->getVolume();;

  tuneFreq(double(conf->getFreq(conf->getDefault()))/20.0 +87.0);
  return 0;
}

void kernelCard::cardOff()
{
  if (cardFound)
    {
      save = volume;
      va.flags = VIDEO_AUDIO_MUTE;
      va.audio = 0;
      va.volume = 0;
      ioctl (dev, VIDIOCSAUDIO, &va);
    }
}

void kernelCard::cardOn()
{
  if (cardFound)
    {
      setVolume(save);
    }
}

int kernelCard::cardToggle()  //Same as mute (if off, sets it on etc)
{
  if (cardFound)
    {
        if (!muted) {
	  muted = 1;
	  va.flags = VIDEO_AUDIO_MUTE;
	  va.audio = 0;
	  ioctl (dev, VIDIOCSAUDIO, &va);
        } else {
	  muted = 0;
	  setVolume (volume);
        }
    }
}

void kernelCard::volDown()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume > 0) volume --;
	  /* vol down a notch */
	  setVolume(volume);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}

void kernelCard::volUp()
{
  if (cardFound)
    {
      if ( ! muted )
	{
	  if (volume < 16) volume ++;
	  setVolume(volume);
	}
      else
	{
	  /* just un-mute */
	  muted=0;
	  cardOn();
	}
    }
}

void kernelCard::setVolume(int v)
{
  if (cardFound)
    {
      if ((v>=0) && (v<16))
	{
	  va.flags = VIDEO_AUDIO_VOLUME;
	  va.audio = 0;
	  va.volume = v * (65535/16);
	  ioctl (dev, VIDIOCSAUDIO, &va);
	}
    }
}

int kernelCard::getVolume()
{
  return volume;
}


void kernelCard::cardWrite(int i)
{
}

/* Determine and return the appropriate frequency multiplier for
   the first tuner on the open video device with handle FD. */
static double
get_freq_fact (int fd)
{
       struct video_tuner tuner;
       tuner.tuner = 0;
       if (-1 == ioctl (fd, VIDIOCGTUNER, &tuner)
           || (tuner.flags & VIDEO_TUNER_LOW) == 0)
               return .16;
       return 160.;
}

void kernelCard::tuneFreq(double freq)
{
  if (cardFound)
    {
      lastF = freq;
      struct video_tuner v;
      unsigned long xl_freq = (unsigned long)((freq*100)*get_freq_fact(dev)+0.5);
      lastFr = xl_freq;
      v.tuner = 0;
      ioctl (dev, VIDIOCSFREQ, &xl_freq);
//      ioctl (dev, VIDIOCGTUNER, &v);
//      setVolume(volume);
    }
}

int kernelCard::isTuned()
{
  if (cardFound)
    {
      struct video_tuner v;
      v.tuner = 0;
      ioctl (dev, VIDIOCGTUNER, &v);
      return (v.signal > 32678);
    }
}
#endif //USE_KERNEL

#endif // CONF_CPP
