// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/modules/MWavOutput.cc,v $
// $Revision: 1.2 $
// $Date: 1999/05/13 20:16:42 $
// $State: Exp $
// **************************************************************

#define MODULE_NAME "wav-output"

#include <iostream.h>
#include <fstream.h>
#include <values.h>

#include "ModuleMacros.h"
#include "SamplingConfig.h"

BEGIN_MODULE_DEFINITION(WavOutput);
    Slot *sslot_input;
    PreparedSoundSignal *pssig_input;
    Seconds sampling_interval;
    long end_time;
    long number_of_output_samples;
    ofstream *outputstream;
    streampos position_of_header; // remember where to write wav header
    const SamplingConfig *sampling_config;
public:
    bool isExecutable() const { return true; };
    void prepareForExecution(const SamplingConfig *);
    bool executeBlock(long, long);
    void finishExecution();
private:
    void writeDWord(long);
    void writeWord(short);
END_MODULE_DEFINITION(WavOutput);

// ---------------------------------------------------------------------------
//                            Implementation
// ---------------------------------------------------------------------------

MWavOutput::MWavOutput(string) : pssig_input(0)
{
    addConnector(sslot_input = new Slot(SOUND_TYPE, "input", "Sound signal to generate .wav from",this,3)); 
}


void MWavOutput::prepareForExecution(const SamplingConfig *sc)
{
    sampling_config = sc;
    sampling_interval = 1.0 / (Number)(sc->samplingrate);
    
    // open output file.
    outputstream = new ofstream("testoutput.wav");
    if (!outputstream->good()) {
	delete outputstream;
	outputstream = 0;
	return;
    }

    // I will complete the wav header later, when i know how long the file is!
    position_of_header = outputstream->tellp(); // I know, this is most probably 0, but there is no seek(0) anymore :-(
    number_of_output_samples = 0;
    *outputstream << // Write dummy header
	"RIFF"
	"????" // file length without "RIFF" and this entry i.e. filelength - 8.
	"WAVE"
	"fmt "
	"--16" // Don't know?
	"-1"   // Don't know
	"-1"   // Don't knoe
	"srat" // sampling rate
	"rat2" // bytes per second
	"02"   // bytes per samples
	"16"   // bits per sample
	"data"
	"????";// number of following bytes i.e. filelength - 44

    Parameterset parset;
    parset.setSamplingInterval(sampling_interval);
    Metainfo mi;
    mi.setCutTo(); // means I wan't to know about this parameter.
    // TODO: auch CutFrom() beruecksichtigen.
    
    pssig_input = getPreparedSoundSignal(0, sslot_input, &mi, &parset);
    
    end_time = mi.containsCutTo() ? mi.getCutTo() : MAXLONG; // <values.h>
}


bool MWavOutput::executeBlock(long start_time, long nsamples) 
{
    if (!pssig_input) return false;

    // We write sequentially to the file, ignoring start_time during writing.

    if (end_time < start_time + nsamples) nsamples = end_time - start_time;
    if (nsamples <= 0) return false;

    
    SoundPortion sp = pssig_input->getSoundPortion(start_time, nsamples);

    // For testing I use 16 Bit mono.
    if (SHORTBITS != 16) {
	cerr << "Can't create .wav file since 'short' on this machine is not 16 Bits!" << endl;
	return false;
    }

    short *outputbuffer = new short[nsamples];
    short *out = outputbuffer;
    const Number *in = sp.getSamples();
    for (int i=nsamples; i; i--) *out++ = (short)(*in++ * 32767);
    outputstream->write((char *)outputbuffer, 2 * nsamples); // I hope this write() is transparent!
    number_of_output_samples += nsamples;
    delete outputbuffer;

    return start_time + nsamples < end_time; // not finished. TODO: Test for end.
}


void MWavOutput::finishExecution()
{
    if (outputstream) {
	outputstream->seekp(position_of_header);

	// Write WAV header
	long sampling_rate = (long)(1/sampling_interval);

	*outputstream << "RIFF";
	writeDWord(number_of_output_samples * 2 + 36);
	*outputstream << "WAVE" "fmt ";
	writeDWord(16);
	writeWord(1);
	writeWord(1);
	writeDWord(sampling_rate);
	writeDWord(sampling_rate * 2);
	writeWord(2); // byte per sample
	writeWord(16); // bits per sample
	*outputstream << "data";
	writeDWord(number_of_output_samples * 2);
	
	outputstream->close();
	delete outputstream;
	outputstream = 0;
    }
    if (pssig_input) {
	delete pssig_input;
	pssig_input = 0;
    }
}


/**
  * Writes a 4 byte integer to the outputstream. Note that
  * write(&var, sizeof(var)) will yield different results on
  * different architectures!
  */
void MWavOutput::writeDWord(long dword)
{
    outputstream->put(dword & 0xff);
    outputstream->put((dword >> 8) & 0xff);
    outputstream->put((dword >> 16) & 0xff);
    outputstream->put((dword >> 24) & 0xff);
}   


/**
  * Writes a 2 byte integer to the outputstream.
  */
void MWavOutput::writeWord(short word)
{
    outputstream->put(word & 0xff);
    outputstream->put((word >> 8) & 0xff);
}
