/*
   Motif Drag&Drop Handler

   Copyright 1999 Matt Koss, <koss@napri.sk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <stdio.h>
#include <unistd.h>

#include <qapplication.h>

#include <kdebug.h>
#include <X11/Intrinsic.h>

#include "kmainwidget.h"
#include "motif_dnd.h"
#include "MotifDNDHandler.h"

static QString currentDragUrl;

static bool in_drop_site = false;
static Window cur_window = 0;
static QWidget *drop_widget = 0L;

static Atom Dnd_selection, Dnd_transfer_success, Dnd_transfer_failure;
static Atom incr_atom;
static Atom * src_targets ;
static ushort num_src_targets ;

////////////////////////////////////////////////////////////////////////////


// this stuff is copied from qclipboard_x11.cpp

extern bool qt_xclb_wait_for_event( Display *dpy, Window win, int type,
				    XEvent *event, int timeout );
extern bool qt_xclb_read_property( Display *dpy, Window win, Atom property,
				   bool deleteProperty,
				   QByteArray *buffer, int *size, Atom *type,
				   int *format, bool nullterm );
extern QByteArray qt_xclb_read_incremental_property( Display *dpy, Window win,
						     Atom property,
						     int nbytes, bool nullterm );


QByteArray motif_obtain_data( DndData dnd_data )
{
  QByteArray result;

  Dnd_selection = dnd_data.property;

  if ( XGetSelectionOwner( qt_xdisplay(),
			   Dnd_selection ) == None )
    return result; // should never happen?

  QWidget* tw = drop_widget;
  if ( drop_widget->isDesktop() ) {
    tw = new QWidget;
  }

  // convert selection to a string property
  XConvertSelection (qt_xdisplay(), Dnd_selection, XA_STRING,
 		     Dnd_selection, tw->winId(), dnd_data.time);

  XFlush( qt_xdisplay() );

  XEvent xevent;
  bool got=qt_xclb_wait_for_event( qt_xdisplay(),
				   tw->winId(),
				   SelectionNotify, &xevent, 5000);
  if ( got ) {
    Atom type;

    if ( qt_xclb_read_property( qt_xdisplay(),
				tw->winId(),
				Dnd_selection, TRUE,
				&result, 0, &type, 0, TRUE ) ) {
      if ( type == incr_atom ) {
	int nbytes = result.size() >= 4 ? *((int*)result.data()) : 0;
	result = qt_xclb_read_incremental_property( qt_xdisplay(),
						    tw->winId(),
						    Dnd_selection,
						    nbytes, FALSE );
      }
      
      if ( type != None ) {
	kdebug( KDEBUG_INFO, 5001, "inserting target" );
      }
    }
  }

//   we have to convert selection in order to indicate success to the initiator
  XConvertSelection (qt_xdisplay(), Dnd_selection, Dnd_transfer_success,
 		     Dnd_selection, dnd_data.src_window, dnd_data.time);

  if ( drop_widget->isDesktop() ) {
    delete tw;
  }

  return result;
}


//////////////////////////////////////////////////////
//
// Class methods
//
//////////////////////////////////////////////////////

MotifDNDHandler::MotifDNDHandler() {

  char my_dnd_selection_name[30] ;
  sprintf(my_dnd_selection_name, "_MY_DND_SELECTION_%d", getpid());
  Dnd_selection = XInternAtom( qt_xdisplay(), my_dnd_selection_name, False);

  if ( Dnd_transfer_success )
    return;

  Dnd_transfer_success = XInternAtom( qt_xdisplay(), "XmTRANSFER_SUCCESS", False);
  Dnd_transfer_failure = XInternAtom( qt_xdisplay(), "XmTRANSFER_FAILURE", False);
  incr_atom = XInternAtom( qt_xdisplay(), "INCR", False );
}


void
MotifDNDHandler::setAcceptMotifDrops( QWidget *w )
{
  DndWriteReceiverProperty(qt_xdisplay(), w->winId(),
			   DND_DRAG_DYNAMIC);
}


bool
MotifDNDHandler::handleEvent( XEvent *xevent )
{

  XEvent event = *xevent;
  XClientMessageEvent cm ;
  DndData dnd_data ;
  char receiver ;

  switch(event.type) {

  case ClientMessage:

    if (!(DndParseClientMessage ((XClientMessageEvent*)&event, 
				 &dnd_data, &receiver))) {
//       kdebug( KDEBUG_INFO, 5001,"not a valid Dnd client message\n");
      return false;
//       break ;
    }

    switch ( dnd_data.reason ) {

    case DND_DRAG_MOTION:

      {
	// 	kdebug( KDEBUG_INFO, 5001,"source sending a drag motion\n");
	/* check if in drop site, and depending on the state,
	   send a drop site enter or drop site leave or echo */
	// 	kdebug( KDEBUG_INFO, 5001,"x %d y %d\n",dnd_data.x, dnd_data.y);

	QWidget *widget = QApplication::widgetAt( QPoint( dnd_data.x, dnd_data.y ) );
	if ( widget != 0L && widget->acceptDrops() ) {
	  drop_widget = widget;
	  
	  if (!in_drop_site) {
	    in_drop_site = True ;

	    dnd_data.reason = DND_DROP_SITE_ENTER ;
	    dnd_data.time = CurrentTime ;
	    dnd_data.operation = DND_MOVE|DND_COPY;
	    dnd_data.operations = DND_MOVE|DND_COPY;

	    DndFillClientMessage (event.xclient.display,
				  cur_window,
				  &cm, &dnd_data, 0);

	    XSendEvent(event.xbutton.display,
		       cur_window, False, 0, 
		       (XEvent *)&cm) ;


	    kdebug( KDEBUG_INFO, 5001,"XSendEvent DND_DROP_SITE_ENTER %ld", 
		   cur_window);

	  } else {
	    dnd_data.reason = DND_DRAG_MOTION ;
	    dnd_data.time = CurrentTime ;
	    dnd_data.operation = DND_MOVE|DND_COPY;
	    dnd_data.operations = DND_MOVE|DND_COPY;

	    DndFillClientMessage (event.xclient.display,
				  cur_window,
				  &cm, &dnd_data, 0);

	    XSendEvent(event.xbutton.display,
		       cur_window, False, 0, 
		       (XEvent *)&cm) ;


	    // 	    kdebug( KDEBUG_INFO, 5001,"XSendEvent DND_DRAG_MOTION %ld", 
	    // 		   cur_window);

	  }
	} else {
	  if (in_drop_site) {
	    in_drop_site = False ;

	    dnd_data.reason = DND_DROP_SITE_LEAVE ;
	    dnd_data.time = CurrentTime ;

	    DndFillClientMessage (event.xclient.display,
				  cur_window,
				  &cm, &dnd_data, 0);

	    XSendEvent(event.xbutton.display,
		       cur_window, False, 0, 
		       (XEvent *)&cm) ;


	    kdebug( KDEBUG_INFO, 5001,"XSendEvent DND_DROP_SITE_LEAVE %ld", 
		   cur_window);
	  }
	}
      }
      break;

    case DND_TOP_LEVEL_ENTER:

      /* get the size of our drop site for later use */

      kdebug( KDEBUG_INFO, 5001,"source sending a top level enter %ld",
	     dnd_data.src_window);

      cur_window = dnd_data.src_window ;
		
      /* no answer needed, just read source property */
      DndReadSourceProperty (event.xclient.display,
			     cur_window,
			     dnd_data.property,
			     &src_targets, &num_src_targets);
      kdebug( KDEBUG_INFO, 5001,"src_targets %ld num_src_targets %d",
	     src_targets[0], num_src_targets);
      /* we only support string for now */
      if (num_src_targets && src_targets[0] == XA_STRING) 
	kdebug( KDEBUG_INFO, 5001,"src target ok");

      break;

    case DND_TOP_LEVEL_LEAVE:

      kdebug( KDEBUG_INFO, 5001,"source sending a top level leave");
	  /* no need to do anything */

      break;

    case DND_OPERATION_CHANGED:
      
      kdebug( KDEBUG_INFO, 5001,"source sending an operation changed");
      /* need to echo */

      break;

    case DND_DROP_START:

      {

	kdebug( KDEBUG_INFO, 5001,"source sending a drop start");

	if (!in_drop_site) {
	  // we have to convert selection in order to indicate failure to the initiator
	  XConvertSelection (qt_xdisplay(), dnd_data.property, Dnd_transfer_failure,
			     dnd_data.property, cur_window, dnd_data.time);
	  drop_widget = 0;
	  break;
	}

	/* need to echo and then request a convert */
	dnd_data.reason = DND_DROP_START ;

	// This is commented out, because we need to pass the correct time
	// in order to succesfully convert the selection ( otherwise -> warning )
// 	dnd_data.time = CurrentTime ;

	DndFillClientMessage (event.xclient.display,
			      drop_widget->winId(),
			      &cm, &dnd_data, 0);

	XSendEvent(event.xbutton.display,
		   cur_window, False, 0, 
		   (XEvent *)&cm) ;


	kdebug( KDEBUG_INFO, 5001,"XSendEvent DND_DROP_START %ld", 
	       cur_window);

	/* ask for a convertion - the selection name is
		       the same as the property on which the source
		       stored its target, weird, but that's the way it 
		       was done in Motif... */
	currentDragUrl = motif_obtain_data( dnd_data );
 	kdebug( KDEBUG_INFO, 5001,"Selection = %d  %s", currentDragUrl.length(), currentDragUrl.data() );
	kmain->addTransfer( currentDragUrl );
      }
      
//       kdebug( KDEBUG_INFO, 5001,"drop action %d", dnd_data.completion);
      if (in_drop_site)
	in_drop_site = False ;

      drop_widget = 0;
      cur_window = 0;
      break;

    }   //  end of switch ( dnd_data.reason )

    return true;

    break;
  }   //  end of switch ( event.type )

  return false;
}
