#include "file.h"
#include <config.h>

#include <kio_manager.h>
#include <kio_rename_dlg.h>
#include <kio_skip_dlg.h>

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <dirent.h>
#include <time.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <assert.h>

#include <iostream.h>

#include <k2url.h>

string testLogFile( const char *_filename );
int check( Connection *_con );

void sig_handler( int );
void sig_handler2( int );

int main( int argc, char **argv )
{
  signal(SIGCHLD,sig_handler);
  signal(SIGSEGV,sig_handler2);

  debug( "kio_file : Starting");

  Connection parent( 0, 1 );
  
  FileProtocol file( &parent );
  file.dispatchLoop();

  debug( "kio_file : Done" );
}


void sig_handler2( int )
{
  debug( "kio_file : ###############SEG FILE#############" );
  exit(1);
}


void sig_handler( int )
{
  int pid;
  int status;
    
  while( 1 )
  {
    pid = waitpid( -1, &status, WNOHANG );
    if ( pid <= 0 )
    {
      // Reinstall signal handler, since Linux resets to default after
      // the signal occured ( BSD handles it different, but it should do
      // no harm ).
      signal( SIGCHLD, sig_handler );
      return;
    }
  }
}


FileProtocol::FileProtocol( Connection *_conn ) : IOProtocol( _conn )
{
  m_cmd = CMD_NONE;
  m_bIgnoreJobErrors = false;
}


void FileProtocol::slotMkdir( const char *_url, int _mode )
{
  string url = _url;
  
  K2URL usrc( _url );
  if ( usrc.isMalformed() )
  {
    error( ERR_MALFORMED_URL, url.c_str() );
    m_cmd = CMD_NONE;
    return;
  }

  if ( !usrc.isLocalFile() )
  {
    error( ERR_INTERNAL, "kio_file got non local name in mkdir command" );
    m_cmd = CMD_NONE;
    return;
  }

  struct stat buff;
  if ( stat( usrc.path(), &buff ) == -1 )
  {
    if ( ::mkdir( usrc.path(), S_IRWXU ) != 0 )
    {
      if ( errno == EACCES )
      {  
	error( ERR_ACCESS_DENIED, url.c_str() );
	m_cmd = CMD_NONE;
	return;
      }
      else
      {  
	error( ERR_COULD_NOT_MKDIR, url.c_str() );
	m_cmd = CMD_NONE;
	return;
      }
    }
    else
    {
      if ( _mode != -1 )
	if ( chmod( usrc.path(), _mode ) == -1 )
	{
	  error( ERR_CANNOT_CHMOD, url.c_str() );
	  m_cmd = CMD_NONE;
	  return;
	}

      finished();
      return;
    }
  }

  if ( S_ISDIR( buff.st_mode ) )
  {
    error( ERR_DOES_ALREADY_EXIST, url.c_str() );
    m_cmd = CMD_NONE;
    return;
  }
    
  error( ERR_COULD_NOT_MKDIR, url.c_str() );
  m_cmd = CMD_NONE;
  return;
}


void FileProtocol::slotCopy( list<string>& _source, const char *_dest )
{
  doCopy( _source, _dest, false );
}


void FileProtocol::slotCopy( const char* _source, const char *_dest )
{
  list<string> lst;
  lst.push_back( _source );
  
  doCopy( lst, _dest, true );
}


void FileProtocol::doCopy( list<string>& _source, const char *_dest, bool _rename, bool _move ) {

  if ( _rename )
    assert( _source.size() == 1 );
  
  debug( "kio_file : Making copy to %s", _dest );
  
  // Check wether the URLs are wellformed
  list<string>::iterator soit = _source.begin();
  for( ; soit != _source.end(); ++soit )
  {    
    debug( "kio_file : Checking %s", soit->c_str() );
    K2URL usrc( *soit );
    if ( usrc.isMalformed() )
    {
      error( ERR_MALFORMED_URL, soit->c_str() );
      m_cmd = CMD_NONE;
      return;
    }
    if ( !usrc.isLocalFile() )
    {
      error( ERR_INTERNAL, "kio_file got non local file as source in copy command" );
      m_cmd = CMD_NONE;
      return;
    }
  }

  debug( "kio_file : All URLs ok %s", _dest );

  // Make a copy of the parameter. if we do IPC calls from here, then we overwrite
  // our argument. This is tricky! ( but saves memory and speeds things up )
  string dest = _dest;
  
  // Check wellformedness of the destination
  K2URL udest( dest );
  if ( udest.isMalformed() )
  {
    error( ERR_MALFORMED_URL, dest.c_str() );
    m_cmd = CMD_NONE;
    return;
  }

  debug( "kio_file : Dest ok %s", dest.c_str() );

  // Extract destinations right most protocol
  list<K2URL> lst;
  if ( !K2URL::split( dest.c_str(), lst )  )
  {
    error( ERR_MALFORMED_URL, dest.c_str() );
    m_cmd = CMD_NONE;
    return;
  }

  // Find IO server for destination
  string exec = ProtocolManager::self()->find( lst.back().protocol() );

  if ( exec.empty() )
  {
    error( ERR_UNSUPPORTED_PROTOCOL, lst.back().protocol() );
    m_cmd = CMD_NONE;
    return;
  }

  // Is the right most protocol a filesystem protocol ?
  if ( strcmp( lst.back().protocol(), "file") )
  {
    error( ERR_PROTOCOL_IS_NOT_A_FILESYSTEM, lst.back().protocol() );
    m_cmd = CMD_NONE;
    return;
  }
      
  debug( "kio_file : IO server ok %s", dest.c_str() );

  // Get a list of all source files and directories
  list<Copy> files;
  list<CopyDir> dirs;
  int size = 0;
  debug( "kio_file : Iterating" );

  soit = _source.begin();
  debug( "kio_file : Looping" );
  for( ; soit != _source.end(); ++soit )
  {    
    debug( "kio_file : Executing %s", soit->c_str() );
    K2URL usrc( *soit );
    debug( "kio_file : Parsed URL" );
    // Did an error occur ?
    int s;
    if ( ( s = listRecursive( usrc.path(), files, dirs, _rename ) ) == -1 )
    {
      // Error message is already sent
      m_cmd = CMD_NONE;
      return;
    }
    // Sum up the total amount of bytes we have to copy
    size += s;
  }

  debug( "kio_file : Recursive 1 %s", dest.c_str() );

  // Check wether we do not copy a directory in itself or one of its subdirectories
  struct stat buff2;
  if ( udest.isLocalFile() && stat( udest.path(), &buff2 ) == 0 )
  {
    bool b_error = false;
    for( soit = _source.begin(); soit != _source.end(); ++soit )
    {    
      K2URL usrc( *soit );  

      struct stat buff1;
      // Can we stat both the source, too ? ( Should always be the case )
      if ( stat( usrc.path(), &buff1 ) == 0 )
      {
	bool b_error = false;
	// Are source and dest equal ? => error
	if ( buff1.st_ino == buff2.st_ino )
	  b_error = true;
      }
    }
  
    if ( !b_error )
    {
      // Iterate over all subdirectories
      list<CopyDir>::iterator it = dirs.begin();
      for( ; it != dirs.end() && !b_error; it++ )
	if ( buff2.st_ino == it->m_ino )
	  b_error = true;
    }

    // Do we have a cyclic copy now ? => error
    if ( b_error )
    {
      error( ERR_CYCLIC_COPY, soit->c_str() );
      m_cmd = CMD_NONE;
      return;
    }
  }

  debug( "kio_file : Recursive ok %s", dest.c_str() );

  m_cmd = CMD_GET;
  
  // Start a server for the destination protocol
  Slave slave( exec.c_str() );
  if ( slave.pid() == -1 )
  {
    error( ERR_CANNOT_LAUNCH_PROCESS, exec.c_str() );
    m_cmd = CMD_NONE;
    return;
  }
  
  // Put a protocol on top of the job
  FileIOJob job( &slave, this );

  debug( "kio_file : Job started ok %s", dest.c_str() );

  // Tell our client what we 'r' gonna do
  totalSize( size );
  totalFiles( files.size() );
  totalDirs( dirs.size() );
  
  int processed_files = 0;
  int processed_dirs = 0;
  int processed_size = 0;
  
  // Replace the relative destinations with absolut destinations
  // by prepending the destinations path
  string tmp1 = lst.back().path( 1 );
  // Strip '/'
  string tmp1_stripped = lst.back().path( -1 );

  list<CopyDir>::iterator dit = dirs.begin();
  for( ; dit != dirs.end(); dit++ )
  {
    string tmp2 = dit->m_strRelDest;
    if ( _rename )
      dit->m_strRelDest = tmp1_stripped;
    else
      dit->m_strRelDest = tmp1;
    dit->m_strRelDest += tmp2;
  }
  list<Copy>::iterator fit = files.begin();
  for( ; fit != files.end(); fit++ )
  {
    string tmp2 = fit->m_strRelDest;
    if ( _rename ) // !!! && fit->m_strRelDest == "" )
      fit->m_strRelDest = tmp1_stripped;
    else
      fit->m_strRelDest = tmp1;
    fit->m_strRelDest += tmp2;
  }
  
  debug( "kio_file : Destinations ok %s", dest.c_str() );

  /*****
   * Make directories
   *****/
  
  m_bIgnoreJobErrors = true;
  bool overwrite_all = false;
  bool auto_skip = false;
  list<string> skip_list;
  list<string> overwrite_list;

  // Create all directories
  dit = dirs.begin();
  for( ; dit != dirs.end(); dit++ )
  { 
    // Repeat until we got no error
    do
    {
      job.clearError();

      list<K2URL> l( lst );
      l.back().setPath( dit->m_strRelDest.c_str() );

      string d;
      list<K2URL>::iterator it = l.begin();
      for( ; it != l.end(); it++ )
	d += it->url();

      // Is this URL on the skip list ?
      bool skip = false;
      list<string>::iterator sit = skip_list.begin();
      for( ; sit != skip_list.end() && !skip; sit++ )
	// Is d a subdirectory of *sit ?
	if ( strncmp( sit->c_str(), d.c_str(), sit->size() ) == 0 )
	  skip = true;
      
      if ( skip )
	continue;

      // Is this URL on the overwrite list ?
      bool overwrite = false;
      list<string>::iterator oit = overwrite_list.begin();
      for( ; oit != overwrite_list.end() && !overwrite; oit++ )
	if ( strncmp( oit->c_str(), d.c_str(), oit->size() ) == 0 )
	  overwrite = true;
      
      if ( overwrite )
	continue;
      
      // Tell what we are doing
      makingDir( d.c_str() );
      
      // debug( "kio_file : Making remote dir %s", d );
      // Create the directory
      job.mkdir( d.c_str(), dit->m_mode );
      while( !job.hasFinished() )
	job.dispatch();

      // Did we have an error ?
      if ( job.hasError() )
      {
	// Can we prompt the user and ask for a solution ?
	if ( /* m_bGUI && */ job.errorId() == ERR_DOES_ALREADY_EXIST )
	{    
	  string old_path = l.back().path( 1 );
	  string old_url = l.back().url( 1 );
	  // Should we skip automatically ?
	  if ( auto_skip )
	  {
	    job.clearError();
	    // We dont want to copy files in this directory, so we put it on the skip list.
	    skip_list.push_back( old_url );
	    continue;
	  }
	  else if ( overwrite_all )
	  {    
	    job.clearError();
	    continue;
	  }

	  string n;
	  /* RenameDlg_Mode m = (RenameDlg_Mode)( M_SINGLE | M_OVERWRITE );
	  if ( dirs.size() > 1 )
	    m = (RenameDlg_Mode)(M_MULTI | M_SKIP | M_OVERWRITE ); */
	  RenameDlg_Mode m = (RenameDlg_Mode)( M_MULTI | M_SKIP | M_OVERWRITE );
	  string tmp2 = l.back().url();
	  RenameDlg_Result r = open_RenameDlg( dit->m_strAbsSource.c_str(), tmp2.c_str(), m, n );
	  if ( r == R_CANCEL ) 
	  {
	    error( ERR_USER_CANCELED, "" );
	    m_cmd = CMD_NONE;
	    return;
	  }
	  else if ( r == R_RENAME )
	  {
	    K2URL u( n.c_str() );
	    // The Dialog should have checked this.
	    if ( u.isMalformed() )
	      assert( 0 );
	    // The new path with trailing '/'
	    string tmp3 = u.path( 1 );
	    renamed( tmp3.c_str() );
	    ///////
	    // Replace old path with tmp3 
	    ///////
	    list<CopyDir>::iterator dit2 = dit;
	    // Change the current one and strip the trailing '/'
	    dit2->m_strRelDest = u.path( -1 );
	    // Change the name of all subdirectories
	    dit2++;
	    for( ; dit2 != dirs.end(); dit2++ )
	      if ( strncmp( dit2->m_strRelDest.c_str(), old_path.c_str(), old_path.size() ) == 0 )
		dit2->m_strRelDest.replace( 0, old_path.size(), tmp3 );
	    // Change all filenames
	    list<Copy>::iterator fit2 = files.begin();
	    for( ; fit2 != files.end(); fit2++ )
	      if ( strncmp( fit2->m_strRelDest.c_str(), old_path.c_str(), old_path.size() ) == 0 )
		fit2->m_strRelDest.replace( 0, old_path.size(), tmp3 );
	    // Dont clear error => we will repeat the current command
	  }
	  else if ( r == R_SKIP )
	  {
	    // Skip all files and directories that start with 'old_url'
	    skip_list.push_back( old_url );
	    // Clear the error => The current command is not repeated => skipped
	    job.clearError();
	  }
	  else if ( r == R_AUTO_SKIP )
	  {
	    // Skip all files and directories that start with 'old_url'
	    skip_list.push_back( old_url );
	    // Clear the error => The current command is not repeated => skipped
	    job.clearError();
	    auto_skip = true;
	  }
	  else if ( r == R_OVERWRITE )
	  {
	    // Dont bother for subdirectories
	    overwrite_list.push_back( old_url );
	    // Clear the error => The current command is not repeated => we will
	    // overwrite every file in this directory or any of its subdirectories
	    job.clearError();
	  }
	  else if ( r == R_OVERWRITE_ALL )
	  {
	    job.clearError();
	    overwrite_all = true;
	  }
	  else
	    assert( 0 );
	}
	// No need to ask the user, so raise an error
	else
	{    
	  error( job.errorId(), job.errorText() );
	  m_cmd = CMD_NONE;
	  return;
	}
      }
    }
    while( job.hasError() );
      
    processedDirs( ++processed_dirs );
  }

  debug( "kio_file : Created directories %s", dest.c_str() );
  
  /*****
   * Copy files
   *****/

  time_t t_start = time( 0L );
  time_t t_last = t_start;

  fit = files.begin();
  for( ; fit != files.end(); fit++ ) { 

    bool overwrite = false;
    bool skip_copying = false;

    // Repeat until we got no error
    do { 
      job.clearError();

      list<K2URL> l( lst );
      l.back().setPath( fit->m_strRelDest.c_str() );
    
      // debug( "kio_file : ########### SET Path to '%s'", fit->m_strRelDest.c_str() );
      
      string d;
      list<K2URL>::iterator it = l.begin();
      for( ; it != l.end(); it++ )
	d += it->url();

      // Is this URL on the skip list ?
      bool skip = false;
      list<string>::iterator sit = skip_list.begin();
      for( ; sit != skip_list.end() && !skip; sit++ )
	// Is 'd' a file in directory '*sit' or one of its subdirectories ?
	if ( strncmp( sit->c_str(), d.c_str(), sit->size() ) == 0 )
	  skip = true;
    
      if ( skip )
	continue;
    
      string realpath = "file:"; realpath += fit->m_strAbsSource;
      copyingFile( realpath.c_str(), d.c_str() );
    
      // debug( "kio_file : Writing to %s", d );
       
      // Is this URL on the overwrite list ?
      list<string>::iterator oit = overwrite_list.begin();
      for( ; oit != overwrite_list.end() && !overwrite; oit++ )
	if ( strncmp( oit->c_str(), d.c_str(), oit->size() ) == 0 )
	  overwrite = true;
      
      job.put( d.c_str(), fit->m_mode, overwrite_all || overwrite,
	       false, fit->m_size );

      while( !job.isReady() && !job.hasFinished() )
	job.dispatch();

      // Did we have an error ?
      if ( job.hasError() ) {
	int currentError = job.errorId();

	debug("################# COULD NOT PUT %d", currentError);
	// if ( /* m_bGUI && */ job.errorId() == ERR_WRITE_ACCESS_DENIED )
	if ( /* m_bGUI && */ currentError != ERR_DOES_ALREADY_EXIST &&
			     currentError != ERR_DOES_ALREADY_EXIST_FULL )
	{
	  // Should we skip automatically ?
	  if ( auto_skip )
	  {
	    job.clearError();
	    skip_copying = true;
	    continue;
	  }
	  string tmp2 = l.back().url();
	  SkipDlg_Result r;
	  r = open_SkipDlg( tmp2.c_str(), ( files.size() > 1 ) );
	  if ( r == S_CANCEL )
	  {
	    error( ERR_USER_CANCELED, "" );
	    m_cmd = CMD_NONE;
	    return;
	  }
	  else if ( r == S_SKIP )
	  {
	    // Clear the error => The current command is not repeated => skipped
	    job.clearError();
	    skip_copying = true;
	    continue;
	  }
	  else if ( r == S_AUTO_SKIP )
	  {
	    // Clear the error => The current command is not repeated => skipped
	    job.clearError();
	    skip_copying = true;
	    continue;
	  }
	  else
	    assert( 0 );
	}
	// Can we prompt the user and ask for a solution ?
	else if ( /* m_bGUI && */ currentError == ERR_DOES_ALREADY_EXIST ||
				  currentError == ERR_DOES_ALREADY_EXIST_FULL )
	{    
	  // Should we skip automatically ?
	  if ( auto_skip )
	  {
	    job.clearError();
	    continue;
	  }

	  string n;
	  RenameDlg_Mode m = (RenameDlg_Mode)( M_SINGLE | M_OVERWRITE );
	  if ( files.size() > 1 )
	    m = (RenameDlg_Mode)( M_MULTI | M_SKIP | M_OVERWRITE );

	  string tmp2 = l.back().url();
	  RenameDlg_Result r = open_RenameDlg( fit->m_strAbsSource.c_str(), tmp2.c_str(), m, n );

	  if ( r == R_CANCEL ) 
	  {
	    error( ERR_USER_CANCELED, "" );
	    m_cmd = CMD_NONE;
	    return;
	  }
	  else if ( r == R_RENAME )
	  {
	    K2URL u( n.c_str() );
	    // The Dialog should have checked this.
	    if ( u.isMalformed() )
	      assert( 0 );
	    renamed( u.path( -1 ).c_str() );
	    // Change the destination name of the current file
	    fit->m_strRelDest = u.path( -1 );
	    // Dont clear error => we will repeat the current command
	  }
	  else if ( r == R_SKIP )
	  {
	    // Clear the error => The current command is not repeated => skipped
	    job.clearError();
	  }
	  else if ( r == R_AUTO_SKIP )
	  {
	    // Clear the error => The current command is not repeated => skipped
	    job.clearError();
	    auto_skip = true;
	  }
	  else if ( r == R_OVERWRITE )
	  {
	    overwrite = true;
	    // Dont clear error => we will repeat the current command
	  }
	  else if ( r == R_OVERWRITE_ALL )
	  {
	    overwrite_all = true;
	    // Dont clear error => we will repeat the current command
	  }
	  else
	    assert( 0 );
	}
	// No need to ask the user, so raise an error
	else
	{    
	  error( currentError, job.errorText() );
	  m_cmd = CMD_NONE;
	  return;
	}
      }
    }
    while( job.hasError() );

    if ( skip_copying )
      continue;

    //debug( "kio_file : Opening %s", fit->m_strAbsSource );
    
    FILE *f = fopen( fit->m_strAbsSource.c_str(), "rb" );
    if ( f == 0L )
    {
      error( ERR_CANNOT_OPEN_FOR_READING, fit->m_strAbsSource.c_str() );
      m_cmd = CMD_NONE;
      return;
    }

    char buffer[ 4096 ];
    while( !feof( f ) ) {
      int n = fread( buffer, 1, 2048, f );

      // !!! slow down loop for local testing
//        for ( int tmpi = 0; tmpi < 800000; tmpi++ ) ;

      job.data( buffer, n );
      processed_size += n;
      time_t t = time( 0L );
      if ( t - t_last >= 1 )
      {
	processedSize( processed_size );
	speed( processed_size / ( t - t_start ) );
	t_last = t;
      }
      
      // Check parent
      while ( check( connection() ) )
	dispatch();
      // Check for error messages from slave
      while ( check( &slave ) )
	job.dispatch();

      // An error ?
      if ( job.hasFinished() )
      {
	fclose( f );
	m_cmd = CMD_NONE;
	finished();
	return;
      }
    }

    job.dataEnd();
  
    fclose( f );

    while( !job.hasFinished() )
      job.dispatch();

    time_t t = time( 0L );
    
    processedSize( processed_size );
    if ( t - t_start >= 1 )
    {
      speed( processed_size / ( t - t_start ) );
      t_last = t;
    }
    processedFiles( ++processed_files );
  }

  debug( "kio_file : Copied files %s", dest.c_str() );

  finished();
  m_cmd = CMD_NONE;
}


void FileProtocol::slotGetSize( const char *_url )
{
  string url = _url;
  
//   debug( "kio_file : Getting size" );

  m_cmd = CMD_GET_SIZE;
  
  K2URL usrc( _url );
  if ( usrc.isMalformed() )
  {
    error( ERR_MALFORMED_URL, url.c_str() );
    m_cmd = CMD_NONE;
    return;
  }

  if ( !usrc.isLocalFile() )
  {
    error( ERR_INTERNAL, "kio_file got non local file in get size command" );
    m_cmd = CMD_NONE;
    return;
  }

  struct stat buff;
  if ( stat( usrc.path(), &buff ) == -1 )
  {
    error( ERR_DOES_NOT_EXIST, url.c_str() );
    m_cmd = CMD_NONE;
    return;
  }

  if ( S_ISDIR( buff.st_mode ) )  // !!! needed ?
  {
    error( ERR_IS_DIRECTORY, url.c_str() );
    m_cmd = CMD_NONE;
    return;
  }

  totalSize( buff.st_size );  
  
  finished();
  m_cmd = CMD_NONE;
}


void FileProtocol::slotPut( const char *_url, int _mode, bool _overwrite, bool _resume, int _size )
{
  string url_orig = _url;
  string url_part = url_orig + ".part";

  K2URL udest_orig( url_orig );
  K2URL udest_part( url_part );

  bool m_bMarkPartial = ProtocolManager::self()->getMarkPartial();

  if ( udest_orig.isMalformed() )
  {
    error( ERR_MALFORMED_URL, url_orig.c_str() );
    finished();
    m_cmd = CMD_NONE;
    return;
  }

  if ( !udest_orig.isLocalFile() )
  {
    error( ERR_INTERNAL, "kio_file got non local file as destination in put command" );
    finished();
    m_cmd = CMD_NONE;
    return;
  }
  
  m_cmd = CMD_PUT;

  struct stat buff;

  if ( stat( udest_orig.path(), &buff ) != -1 ) {

    // if original file exists but we are using mark partial -> rename it to XXX.part
    if ( m_bMarkPartial )
      rename ( udest_orig.path(), udest_part.path() );

    if ( !_overwrite && !_resume ) {
      if ( buff.st_size == _size && _size > 0 )
	error( ERR_DOES_ALREADY_EXIST_FULL, udest_orig.path() );
      else
	error( ERR_DOES_ALREADY_EXIST, udest_orig.path() );
      
      finished();
      m_cmd = CMD_NONE;
      return;
    }
  } else if ( stat( udest_part.path(), &buff ) != -1 ) {
    // if file with extension .part exists but we are not using mark partial
    // -> rename XXX.part to original name
    if ( ! m_bMarkPartial )
      rename ( udest_part.path(), udest_orig.path() );

    if ( !_overwrite && !_resume )
      {
	if ( buff.st_size == _size )
	  error( ERR_DOES_ALREADY_EXIST_FULL, udest_orig.path() );
	else
	  error( ERR_DOES_ALREADY_EXIST, udest_orig.path() );

	finished();
	m_cmd = CMD_NONE;
	return;
      }
  }

  K2URL udest;

  // if we are using marking of partial downloads -> add .part extension
  if ( m_bMarkPartial ) {
    debug( "kio_file : Adding .part extension to %s", udest_orig.path() );
    udest = udest_part;
  } else
    udest = udest_orig;


  /* if ( access( udest.path(), W_OK ) == -1 )
  {
    debug("Write Access denied for '%s' %d",udest.path(),errno );
    
    error( ERR_WRITE_ACCESS_DENIED, url.c_str() );
    finished();
    m_cmd = CMD_NONE;
    return;
  } */

  if ( _resume )
    m_fPut = fopen( udest.path(), "ab" );  // append if resuming
  else
    m_fPut = fopen( udest.path(), "wb" );

  if ( m_fPut == 0L )
  {
    debug( "kio_file : ####################### COULD NOT WRITE %s", udest.path() );
    if ( errno == EACCES )
      error( ERR_WRITE_ACCESS_DENIED, udest.path() );
    else
      error( ERR_CANNOT_OPEN_FOR_WRITING, udest.path() );
    m_cmd = CMD_NONE;
    finished();
    return;
  }

  // We are ready for receiving data
  ready();
  
  // Loop until we got 'dataEnd'
  while ( m_cmd == CMD_PUT && dispatch() );

  fclose( m_fPut );

  if ( stat( udest.path(), &buff ) != -1 ) {

    if ( buff.st_size == _size ) {

      if ( m_bMarkPartial ) {  // after full download rename the file back to original name

	if ( rename( udest.path(), udest_orig.path() ) )
	  error( ERR_CANNOT_RENAME, udest_orig.path() );

      } else if ( _mode != -1 ) {  // do chmod only after full download

	if ( chmod( udest_orig.path(), _mode ) == -1 )
	  error( ERR_CANNOT_CHMOD, udest_orig.path() );

      }

    } // if the size is less then minimum -> delete the file
    else if ( buff.st_size < ProtocolManager::self()->getMinimumKeepSize() )
      remove( udest.path() );
    else
      ; // rollback !!!

  }

  // We have done our job => finish
  finished();

  m_cmd = CMD_NONE;
}


void FileProtocol::slotData( void *_p, int _len )
{
  switch( m_cmd )
    {
    case CMD_PUT:
      fwrite( _p, 1, _len, m_fPut );
      break;
    }
}


void FileProtocol::slotDataEnd()
{
  switch( m_cmd )
    {
    case CMD_PUT:  
      m_cmd = CMD_NONE;
    }
}

long FileProtocol::listRecursive( const char *_path, list<Copy>& _files, list<CopyDir>& _dirs, bool _rename )
{
  struct stat buff;

  // Check wether we have to copy the complete directory tree beginning by its root.
  int len = strlen( _path );
  while( len >= 1 && _path[ len -1 ] == '/' )
    len--;
  if ( len == 1 && _path[ 0 ] == '/' )
  {
    if ( stat( "/", &buff ) == -1 )
    {
      error( ERR_DOES_NOT_EXIST, "/" );
      return -1;
    }

    CopyDir c;
    c.m_strAbsSource = _path;
    if ( _rename )
      c.m_strRelDest = "";
    else
      c.m_strRelDest = "Root";
    c.m_ino = buff.st_ino;
    c.m_mode = buff.st_mode;
    _dirs.push_back( c );
    
    return listRecursive2( "/", c.m_strRelDest.c_str(), _files, _dirs );
  }
  
  string p;
  p.assign( _path, len );
  debug( "kio_file : ########## RECURSIVE LISTING %s", p.c_str() );
  
  if ( stat( p.c_str(), &buff ) == -1 )
  {
    error( ERR_DOES_NOT_EXIST, p.c_str() );
    return -1;
  }
  
  K2URL u( p.c_str() );
  // Should be checked before, but who knows
  if ( u.isMalformed() )
    assert( 0 );

  // Is the source not a directory ? => so just copy it and we are done.
  if ( !S_ISDIR( buff.st_mode ) )
  {
    string fname;
    if ( _rename )
      fname = "";
    else
    {
      fname = u.filename();
      // Should be impossible, but who knows ...
      if ( fname.empty() )
	assert( 0 );
    }

    Copy c;
    c.m_strAbsSource = p;
    c.m_strRelDest = fname;
    c.m_mode = buff.st_mode;
    c.m_size = buff.st_size;
    _files.push_back( c );
    return buff.st_size;
  }

  // The source is a directory. So we have to go into recursion here.
  string tmp1;
  if ( _rename )
    tmp1 = u.path( 0 );
  else
  {    
    tmp1 = u.directory( true );
    tmp1 += "/";
  }
  string tmp2;
  if ( _rename )
    tmp2 = "";
  else
    tmp2 = u.filename();
  CopyDir c;
  c.m_strAbsSource = p;
  c.m_strRelDest = tmp2;
  c.m_mode = buff.st_mode;
  c.m_ino = buff.st_ino;
  _dirs.push_back( c );
  debug( "kio_file : ########### STARTING RECURSION with %s and %s",tmp1.c_str(), tmp2.c_str() );
  
  return listRecursive2( tmp1.c_str(), tmp2.c_str(), _files, _dirs );
}

long FileProtocol::listRecursive2( const char *_abs_path, const char *_rel_path,
				   list<Copy>& _files, list<CopyDir>& _dirs )
{
  long size = 0;
  
  string p = _abs_path;
  p += _rel_path;

  DIR *dp = 0L;
  struct dirent *ep;

  scanningDir( p.c_str() );
  
  dp = opendir( p.c_str() );
  if ( dp == 0L )
  {
    error( ERR_CANNOT_ENTER_DIRECTORY, p.c_str() );
    return -1;
  }
    
  while ( ( ep = readdir( dp ) ) != 0L )
  {
    if ( strcmp( ep->d_name, "." ) == 0 || strcmp( ep->d_name, ".." ) == 0 )
      continue;
    
    string p2 = p;
    p2 += "/";
    p2 += ep->d_name;
  
    struct stat buff;
    if ( stat( p2.c_str(), &buff ) == -1 )
    {
      // Should never happen
      error( ERR_DOES_NOT_EXIST, p.c_str() );
      return -1;
    }

    string tmp = _rel_path;
//     if ( tmp != "" )
      tmp += "/";
    tmp += ep->d_name;
  
    if ( !S_ISDIR( buff.st_mode ) )
    {
      Copy c;
      c.m_strAbsSource = p2;
      c.m_strRelDest = tmp;
      c.m_mode = buff.st_mode & ( S_ISUID | S_ISGID | S_ISVTX | S_IRWXU | S_IRWXG | S_IRWXO );
      c.m_size = buff.st_size;
      _files.push_back( c );
      size += buff.st_size;
    }
    else
    {
      // Did we scan this directory already ?
      // This may happen because a link goes backward in the directory tree
      list<CopyDir>::iterator it = _dirs.begin();
      for( ; it != _dirs.end(); it++ )
	if ( it->m_ino == buff.st_ino )
	{
	  error( ERR_CYCLIC_LINK, p2.c_str() );
	  return -1;
	}
      
      CopyDir c;
      c.m_strAbsSource = p2;
      c.m_strRelDest = tmp;
      c.m_mode = buff.st_mode;
      c.m_ino = buff.st_ino;
      _dirs.push_back( c );

      long s;
      if ( ( s = listRecursive2( _abs_path, tmp.c_str(), _files, _dirs ) ) == -1 )
	return -1;
      size += s;
    }
  }
  
  closedir( dp );
  
  return size;
}

void FileProtocol::jobError( int _errid, const char *_txt )
{
  if ( !m_bIgnoreJobErrors )
    error( _errid, _txt );
}

/*************************************
 *
 * FileIOJob
 *
 *************************************/

FileIOJob::FileIOJob( Connection *_conn, FileProtocol *_File ) : IOJob( _conn )
{
  m_pFile = _File;
}
  
void FileIOJob::slotError( int _errid, const char *_txt )
{
  IOJob::slotError( _errid, _txt );
  m_pFile->jobError( _errid, _txt );
}

/*************************************
 *
 * Utilities
 *
 *************************************/

string testLogFile( const char *_filename )
{
  char buffer[ 1024 ];
  struct stat buff;

  string result;
  
  stat( _filename, &buff );
  int size = buff.st_size;
  if ( size == 0 )
  {
    unlink( _filename );
    return result;
  }
  
  FILE * f = fopen( _filename, "rb" );
  if ( f == 0L )
  {
    unlink( _filename );
    result = "Could not read ";
    result += _filename;
    return result;
  }
  
  result = "";  
  char *p = "";
  while ( p != 0L )
  {
    p = fgets( buffer, 1023, f );
    if ( p != 0L )
      result += buffer;
  }

  fclose( f );
    
  unlink( _filename );

  return result;
}


int check( Connection *_con )
{
  int err;
  struct timeval tv;
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  fd_set rfds;
  FD_ZERO( &rfds );
  FD_SET( _con->inFD(), &rfds );
  
again:
  if ( ( err = select( _con->inFD(), &rfds, 0L, 0L, &tv ) ) == -1 && errno == EINTR )
    goto again;

  // No error and something to read ?
  if ( err != -1 && err != 0 )
    return 1;
  
  return 0;
}


void openFileManagerWindow( const char * )
{
  assert( 0 );
}
