#include <iostream.h>
#include "kiconloader.h"
#include "ircDialog.h"
#include "lineDialog.h"
#include "dccFileDialog.h"
#include "kEirc.h"

QList<KEIrc> KEIrc::keircList;
int          KEIrc::nClient=0;

KEIrc::KEIrc(const char *name) : KTopLevelWidget(name)
{
   nClient++;
   name = "kEirc";

   idTbCounter    = 0;
   idTbConnect    = idTbCounter++;
   idTbDisconnect = idTbCounter++;
   idTbCombo      = idTbCounter++;
   idTbDCCChat    = idTbCounter++;
   idTbDCCSend    = idTbCounter++;
   idTbCtcpSound  = idTbCounter++;
   idTbWebServer  = idTbCounter++;
   idSbNick   = 0;
   idSbStatus = 1;
   idSbMode   = 2;
   idSbLagging= 3;
   idSbLive   = 4;

   client = new IrcClient();

   setFocusPolicy(QWidget::StrongFocus);
   qApp->installEventFilter(this);

   createMenu     ();
   createToolbar  ();
   createStatusbar();

   setMenu        (menubar);
   menubar->setFrameStyle(QFrame::WinPanel | QFrame::Raised);
   menubar->setLineWidth(2);
   menubar->setFont(QFont("Times", 13, QFont::Bold, TRUE));

   setStatusBar   (statusbar);
   addToolBar     (toolbar);

   readConfig();

   setFrameBorderWidth(0);

   menubar->show();

   if (showStatusbar)
      enableStatusBar(KStatusBar::Show);
   else
      enableStatusBar(KStatusBar::Hide);

   if (showToolbar) 
      enableToolBar(KToolBar::Show);
   else
      enableToolBar(KToolBar::Hide);

   keircList.setAutoDelete(false);
   keircList.append( this );

   menu["options"]->setItemChecked(idOptToolbar,   showToolbar);
   menu["options"]->setItemChecked(idOptStatusbar, showStatusbar);

   QString geometry = ircapp->readEntry(this->name+"Geometry", "" );
   if (!geometry.isEmpty()){
      int width, height;
      sscanf( geometry, "%dx%d", &width, &height );
      resize( width, height );
   }
   // Connection
   connect(client,
	   SIGNAL(signConnected(IrcServer*)),
	   this,
	   SLOT  (slotConnected(IrcServer*))
	   );
   connect(client,
	   SIGNAL(signDisconnected(IrcServer*)),
	   this,
	   SLOT  (slotDisconnected(IrcServer*))
	   );
   connect(client,
	   SIGNAL(signCurNick(const char*)),
	   this,
	   SLOT  (slotCurNick(const char*)));
   connect(client, 
	   SIGNAL(signAddServer(IrcServer*)), 
	   this,   
	   SLOT  (slotComboAddServer(IrcServer*)));
   connect(client, 
	   SIGNAL(signCurServer(IrcServer*)), 
	   this,   
	   SLOT  (slotCurServer(IrcServer*)));
   connect(client, 
	   SIGNAL(signDelServer(const char*)), 
	   this,   
	   SLOT  (slotComboDelServer(const char*)));
   connect(client, 
	   SIGNAL(signMode(const char*)), 
	   this,   
	   SLOT  (slotMode(const char*)));

   connect(client,
	   SIGNAL(signMinTimeout(IrcCore*)),
	   this,
	   SLOT  (slotLiveStatus(IrcCore*)));
   connect(client,
	   SIGNAL(signLagging(const char*)),
	   this,
	   SLOT  (slotLagging(const char*)));

   connect(client,
	   SIGNAL(signWriteOnActive(int , const char*, bool)),
	   this,
	   SLOT  (slotWriteOnActive(int , const char*, bool)));

   setCaption("kEirc");

   setView(view=new IrcMDI(client, this));
   client->initAll();

   QString entry, value;
   entry = "ShowRawInput";
   value = ircapp->readEntry(entry, "No");
   if (!stricmp(value, "yes"))
      client->addOutputPage("kEirc Raw", MDI_DETACHED|MDI_CLOSED);
   entry = "ShowRawOutput";
   value = ircapp->readEntry(entry, "No");
   if (!stricmp(value, "yes"))
      client->addOutputPage("kEirc Raw", MDI_DETACHED|MDI_CLOSED);

   QString s;
   if ((s=ircapp->readEntry("WebServerAutoStart", "No")) && s.find("Yes", 0, false)==0)
      slotWebServer(true);

   if ((s=ircapp->readEntry("AutoConnect", "No")) && s.find("Yes", 0, false)==0)
      slotFileConnect();

   client->Sleep(2);
   show();
#ifdef EDEBUG
    cout << "Konstruktor KEIrc...Ok" << endl;
#endif 
}


KEIrc::~KEIrc()
{
#ifdef EDEBUG
    cout << "Destruktor KEIrc" << endl;
#endif 
    keircList.removeRef(this);
    nClient--;
    delete menubar;
    delete toolbar;
    delete statusbar;
    delete client;
}

void KEIrc::createMenu()
{
   menu.setAutoDelete (false);
   QPopupMenu* pm;
   menu.insert("file",  pm = new QPopupMenu);
   idFileNew     = pm->insertItem("&New Client",     this, SLOT(slotFileNew()),     CTRL+Key_N);
   pm->insertSeparator();
   idFileClientSetup= pm->insertItem("&Setup Client", this, SLOT(slotFileClientSetup()), CTRL+Key_S);
   pm->insertSeparator();
   idFileConnect    = pm->insertItem("Connect Server",     this, SLOT(slotFileConnect()));
   idFileDisconnect = pm->insertItem("Disconnect Server",  this, SLOT(slotFileDisconnect()));
   pm->insertSeparator();
   idFileClose   = pm->insertItem("&Close Client",   this, SLOT(slotFileClose()),   CTRL+Key_C);
   idFileExit    = pm->insertItem("E&xit",           this, SLOT(slotFileExit()),    CTRL+Key_X);

   menu.insert("options", pm = new QPopupMenu);
   pm->setCheckable( true );

   idOptPref      = pm->insertItem("Pre&ferences...", this, SLOT(slotOptionsPref()), CTRL+Key_F );
   pm->insertSeparator();
   idOptToolbar   = pm->insertItem("Show &Toolbar",   this, SLOT(slotOptionsToolbar  ()));
   idOptStatusbar = pm->insertItem("Show Status&bar", this, SLOT(slotOptionsStatusbar()));
   pm->insertSeparator();
   idOptSave      = pm->insertItem("&Save Options",   this, SLOT(slotOptionsSave()));

   menu.insert("tools", pm = new QPopupMenu);
   pm->setCheckable( true );

   idToolOpActionDlg  = pm->insertItem("Op &Action Config", this, SLOT(slotToolOpActionDlg()),CTRL+Key_A );
   idToolChannelDlg   = pm->insertItem("Pref. &Channel Dlg",this, SLOT(slotToolChannelDlg ()));
   idToolWhoisDlg     = pm->insertItem("&Whois Dlg ",       this, SLOT(slotToolWhoisDlg   ()));

   menu.insert("window", pm=new QPopupMenu);
   pm->setCheckable( true );
   int id;
   id = pm->insertItem("&Tile"); 
   pm->setItemChecked(id,  0);
   id = pm->insertItem("&Cascade");
   pm->setItemChecked(id,  0);
   connect(pm,   SIGNAL(activated(int)),
	   this, SLOT  (slotWindow(int)));

   menu.insert("help", pm=new QPopupMenu);
   pm->insertItem("&Using keirc Help", this, SLOT(slotHelpUsing())); 
   pm->insertSeparator();
   pm->insertItem("&About", this, SLOT(slotHelpAbout()) );

   menubar = new KMenuBar( this );
   CHECK_PTR(menubar);
   menubar->insertItem("&File",    menu["file"]);
   menubar->insertItem("&Options", menu["options"]);
   menubar->insertItem("&Tools",   menu["tools"]);
   menubar->insertItem("&Windows", menu["window"]);
   menubar->insertSeparator();
   menubar->insertItem("&Help",    menu["help"]);
}


void KEIrc::createToolbar()
{
   QPixmap pixmap;

   toolbar = new KToolBar(this, "ircToolBar", 25);
   CHECK_PTR(toolbar);

   // Einstellen
   toolbar->setFullWidth(TRUE);

   pixmap = ircapp->getIconLoader()->loadIcon("keirc-connect-toolbar.xpm");
   toolbar->insertButton(pixmap, idTbConnect, SIGNAL(clicked()), 
			 this, SLOT(slotFileConnect()),
			 TRUE, "Connect to server");

   pixmap = ircapp->getIconLoader()->loadIcon("keirc-disconnect-toolbar.xpm");
   toolbar->insertButton(pixmap, idTbDisconnect, SIGNAL(clicked()), 
			 this, SLOT(slotFileDisconnect()),
			 TRUE, "Disconnect server");

   toolbar->insertSeparator();
   toolbar->insertSeparator();

   pixmap = ircapp->getIconLoader()->loadIcon("keirc-dcc-chat-toolbar.xpm");
   toolbar->insertButton(pixmap, idTbDCCChat, SIGNAL(clicked()), 
			 this, SLOT(slotDCCChat()),
			 TRUE, "DCC Chat");

   pixmap = ircapp->getIconLoader()->loadIcon("keirc-dcc-file-toolbar.xpm");
   toolbar->insertButton(pixmap, idTbDCCSend, SIGNAL(clicked()), 
			 this, SLOT(slotDCCSend()),
			 TRUE, "DCC Send");

   toolbar->insertSeparator();
   toolbar->insertSeparator();

   pixmap = ircapp->getIconLoader()->loadIcon("keirc-sound-toolbar.xpm");
   toolbar->insertButton(pixmap, idTbCtcpSound, SIGNAL(clicked()), 
			 this, SLOT(slotCtcpSound()),
			 TRUE, "CTCP Sound");

   toolbar->insertSeparator();
   toolbar->insertSeparator();

   int idTbCounter=7;
   int id = idTbCounter++;
   pixmap = ircapp->getIconLoader()->loadIcon("keirc-settings-toolbar.xpm");
   toolbar->insertButton(pixmap, id, SIGNAL(clicked()), 
			 this, SLOT(slotOptPref()),
			 TRUE, "Option Preferences");

   toolbar->insertSeparator();
   toolbar->insertSeparator();

   pixmap = ircapp->getIconLoader()->loadIcon("keirc-webserver-toolbar.xpm");
   toolbar->insertButton(pixmap, idTbWebServer, TRUE, "Web Server");
   toolbar->setToggle(idTbWebServer);
   toolbar->addConnection (idTbWebServer, SIGNAL(toggled(bool)), 
			   this,          SLOT  (slotWebServer(bool)));

   toolbar->insertCombo ("", idTbCombo, TRUE, 
			 SIGNAL(activated(const char *)), 
			 this, SLOT(slotComboServer(const char *)), 
			 TRUE, "Server List", 250);

   toolbar->clearCombo(idTbCombo);
   QDictIterator<IrcServer> it(*client->server);
   QString st;
   for(;it.current();++it){
      IrcServer* is=it.current();
      st=is->Host()+":"+is->Port();
      toolbar->insertComboItem(idTbCombo, st, -1);
   }
   toolbar->alignItemRight (idTbCombo);
   toolbar->setBarPos(KToolBar::Top);
}


void KEIrc::createStatusbar()
{
   statusbar = new KStatusBar(this);
   statusbar->insertItem("Nick: 1234567891234567", idSbNick);
   statusbar->insertItem("Status: Not connect", idSbStatus);
   statusbar->insertItem("Mode: 123456", idSbMode);
   statusbar->insertItem("Lagging: 1234567891234", idSbLagging);
   statusbar->insertItem  ("      ", idSbLive);
   statusbar->setAlignment(idSbLive, AlignRight);

   statusbar->changeItem("Nick:                 ", idSbNick);
   statusbar->changeItem("Mode:       ", idSbMode);
   statusbar->changeItem("Lagging:              ", idSbLagging);
   statusbar->changeItem("00:00", idSbLive);
}

void KEIrc::readConfig()
{
   QString  cfgString;;

   cfgString = ircapp->readEntry( "ShowToolBar", "No" );
   if (cfgString=="No")
       showToolbar = false;
   else
       showToolbar = true;

   cfgString = ircapp->readEntry( "ToolBarPos", "Top" );
   if (cfgString.isEmpty() )
      toolBar()->setBarPos(KToolBar::Top);
   else if ("Top" == cfgString) 
      toolBar()->setBarPos(KToolBar::Top);
   else if ("Bottom" == cfgString)
      toolBar()->setBarPos(KToolBar::Bottom);
   else if ("Left" == cfgString)
      toolBar()->setBarPos(KToolBar::Left);
   else if ("Right" == cfgString)
      toolBar()->setBarPos(KToolBar::Right);
   else
      toolBar()->setBarPos(KToolBar::Top);

   cfgString = ircapp->readEntry( "ShowStatusBar", "Yes" );
   if (!cfgString.isEmpty() && cfgString.find( "No", 0, false ) == 0 )
       showStatusbar = false;
   else
       showStatusbar = true;
}

void KEIrc::enableMenuItems()
{
   menu["file"]->setItemEnabled(idFileConnect, 1);
   updateRects();
}


void KEIrc::resizeEvent( QResizeEvent * )
{
   QString Geometry;
   Geometry.sprintf("%dx%d", geometry().width(), geometry().height());
   ircapp->writeEntry(name+"Geometry", Geometry);
   view->setGeometry(0,0,view_right, view_bottom);
   updateRects();
}

void KEIrc::closeEvent (QCloseEvent *)
{
    delete this;
    if (keircList.count()<1)
      //if (nClient<1)
       kapp->quit();
}

void KEIrc::slotFileNew()
{
   KEIrc* newItem;

   newItem = new KEIrc("kEirc");
   CHECK_PTR(newItem);
   newItem->resize(size());
   newItem->show();
}

void KEIrc::slotFileClientSetup()
{ 
  client->clientSetup->show();
}

void KEIrc::slotFileConnect()
{  
  client->slotConnect(NULL);
}

void KEIrc::slotFileDisconnect()
{
  client->slotDisconnected();
}

void KEIrc::slotFileClose()
{
   close();
}

void KEIrc::slotFileExit()
{
  KEIrc* k;
  for (k=keircList.first();(k=keircList.current())!=0L;)
      delete k;
  kapp->quit();
}

void KEIrc::slotOptionsPref()
{
  client->optPrefDlg->show();
}
  

void KEIrc::slotOptionsToolbar()
{
   if (showToolbar){
      enableToolBar(KToolBar::Hide);
      showToolbar = false;
   }
   else{
      enableToolBar(KToolBar::Show);
      showToolbar = true;
   }
  	
   menu["options"]->setItemChecked(idOptToolbar, showToolbar);
   updateRects();
}

void KEIrc::slotToolOpActionDlg()
{
  QWidget* w=client->dlgList->find("OpActionDialog");
  if (w!=0L)
     w->show();
}

void KEIrc::slotToolChannelDlg()
{
  QWidget* w=client->dlgList->find("ChannelDialog");
  if (w!=0L)
     w->show();
}

void KEIrc::slotToolWhoisDlg()
{
  QWidget* w=client->dlgList->find("WhoisDialog");
  if (w!=0L)
     w->show();
}

void KEIrc::slotHelpUsing()
{
   ircapp->invokeHTMLHelp("keirc/keirc.html", "");
}

void KEIrc::slotHelpAbout()
{
   QMessageBox mb;
   mb.setText( "KEIRC \nVersion " + QString( KEIRC_VERSION ) +
	       "\n\nEko Bono <sj039su@uni-duisburg.de>" );
   mb.setCaption( "About keirc" );
   mb.show();
}


void KEIrc::slotOptionsStatusbar()
{
   showStatusbar = !showStatusbar;
   menu["options"]->setItemChecked(idOptStatusbar, showStatusbar); 
   enableStatusBar(KStatusBar::Toggle);
}


void KEIrc::slotOptionsSave()
{
   KConfig *config = kapp->getConfig();

   config->setGroup( "kEirc Option" ); 
   config->writeEntry( "ShowToolBar", showToolbar ? "Yes" : "No" );
   switch (toolbar->barPos()){
	case KToolBar::Top:
	   config->writeEntry( "ToolBarPos", "Top");
	   break;
	case KToolBar::Bottom:
	   config->writeEntry( "ToolBarPos", "Bottom");
	   break;
	case KToolBar::Left:
	   config->writeEntry( "ToolBarPos", "Left");
	   break;
	case KToolBar::Right:
	   config->writeEntry( "ToolBarPos", "Right");
	   break;
	default:
	   warning("KEIrc::slotOptionsSave: illegal default in case reached\n");
	   break;
   }
   config->writeEntry( "ShowStatusBar", showStatusbar ? "Yes" : "No" );
}

void KEIrc::slotWindow(int id)
{
  int idx = menu["window"]->indexOf(id);
  switch(idx){
  case 0:
    view->getMDI()->tile();
    menu["window"]->setItemChecked(0,  1);
    menu["window"]->setItemChecked(1,  0);
    break;
  case 1:
    view->getMDI()->cascade();
    menu["window"]->setItemChecked(1,  1);
    menu["window"]->setItemChecked(0,  0);
    break;
  }
  
}

void KEIrc::slotComboServer(const char* txt)
{
  if (!txt)
     return;
  client->setCurServer(txt);
}

void KEIrc::slotComboAddServer(IrcServer* is)
{
  KCombo* cb=toolbar->getCombo(idTbCombo);
  QString st=is->Host()+":"+is->Port();
  cb->insertItem(st);
}

void KEIrc::slotComboCurServer(IrcServer* is)
{
  KCombo* cb=toolbar->getCombo(idTbCombo);
  for (int i=0;i<cb->count();i++){
      QString st=is->Host()+":"+is->Port();
      if (st==cb->text(i)){
	 cb->setCurrentItem(i);
	 return;
      }
  }
}



void KEIrc::slotComboDelServer(const char* s)
{
  KCombo* cb=toolbar->getCombo(idTbCombo);
  for (int i=0;i<cb->count();i++){
      QString st=s;
      if (st==cb->text(i)){
	 cb->removeItem(i);
	 return;
      }
  }
}

void KEIrc::slotConnected(IrcServer* is)
{
  if (client->state&SOCKET_SOCKS && client->state&SOCKET_CONNECTED)
    ;
  else
    ircapp->restoreOverrideCursor();

  QString str(4096);
  if (client->state & SOCKET_DIRECT){
     str = "Connected to server "+is->Host()+" on port "+is->Port();
     client->slotWriteMsg(TYPE_INFO|TYPE_IMG, str);
  }
  if (client->state & SOCKET_CONNECTED){
     str.sprintf("Your hostaddress is %s , your hostname is %s", 
		 client->getSocket().getLocalInetAddress().getHostAddress().data(),
		 client->getSocket().getLocalInetAddress().getHostName().data());;
     client->slotWriteMsg(TYPE_INFO|TYPE_IMG, str);
  
     menu["file"]->setItemEnabled(idFileConnect,    0);
     toolbar->setItemEnabled(idTbConnect,    0);
     menu["file"]->setItemEnabled(idFileDisconnect, 1);
     toolbar->setItemEnabled(idTbDisconnect, 1);
     statusbar->changeItem("Status: Connected", idSbStatus);
     QString st="";
     QString pr="";
     if (client->state & SOCKET_SOCKS){
        QString p;
	p.setNum(client->socket.getPeerPort());
        pr = " [ via "+client->socket.getInetAddress().getHostAddress()+":"+p+" ]";
     }
     st += "kEirc "+client->curServer->Host()+":"+client->curServer->Port()+pr;
     setCaption(st);
  }
}

void KEIrc::slotDisconnected(IrcServer* is)
{
  QString s="";
  if (is && client->state & SOCKS_CONNECTED){
     QString p;
     p.setNum(client->socket.getPeerPort());
     s = "Connection to ircserver is refused by proxy "+client->socket.getInetAddress().getHostAddress()+
         " on port "+p;
  }
  else if (is && client->state&SOCKET_CONNECTED)
     s="Disconnected from server "+is->Host()+" on port "+is->Port();

  if (is && client->state)
     client->slotWriteMsg(TYPE_INFO|TYPE_IMG, s);

  if (client->state & SOCKET_CONNECTED){
     menu["file"]->setItemEnabled(idFileConnect,    1);
     toolbar->setItemEnabled(idTbConnect,    1);
     if (client->state){
        menu["file"]->setItemEnabled(idFileDisconnect, 0);
	toolbar->setItemEnabled(idTbDisconnect, 0);
     }
     statusbar->changeItem("Status: Not connect", idSbStatus);
     statusbar->changeItem("Mode: ", idSbMode);
     statusbar->changeItem("Lagging: ", idSbLagging);
     statusbar->changeItem("00:00", idSbLive);
  }
  setCaption("kEirc");
}

void KEIrc::slotCurNick(const char* nick)
{
  statusbar->changeItem("Nick: "+QString(nick), idSbNick);
}

void KEIrc::slotMode(const char* txt)
{
  QString st;
  st.sprintf("Mode: %s", txt);
  statusbar->changeItem(st, idSbMode);
}

void KEIrc::slotCurServer(IrcServer* is)
{
  if (client->state>0)
     return;
  slotComboCurServer(is);
  if (client->getSocket().getSocket()>0){
     QString st="";
     st += "kEirc "+is->Host()+":"+is->Port();
     setCaption(st);
  }
}


void KEIrc::slotLiveStatus(IrcCore* ic)
{
  ulong minutes = ic->getMinutes();
  int hour  = minutes/60;
  int min   = minutes%60;
  QString ms;
  ms.sprintf("%02d:%02d", hour, min);
  statusbar->changeItem(ms, idSbLive);
}

void KEIrc::slotLagging(const char* l)
{
  QString ms="Lagging : " +QString(l)+" seconds";
  statusbar->changeItem(ms, idSbLagging);
}

void KEIrc::slotWriteOnActive(int Type, const char* txt, bool Filter)
{
  if (isActiveWindow() && view->getMDI()->SelectedWindow()!=0L)
     return;

  client->slotWriteMsg(Type, txt, Filter);
}

void KEIrc::slotDCCChat()
{
  SingleDialog dlg("To :","","Type nick");
  if (dlg.exec()==QDialog::Accepted){
     QString nick=dlg.getText(TRUE);
     if (!nick.isEmpty()){
        QString s="DCC CHAT "+nick;
	client->slotCommand(nick, "/DCC CHAT "+nick);
     }
  }
}

void KEIrc::slotDCCSend()
{
  client->dccFileDlg->show();
}

void KEIrc::slotCtcpSound()
{
  client->mediaDlg->show();
}

void KEIrc::slotOptPref()
{
  client->optPrefDlg->show();
}


void KEIrc::slotWebServer(bool state)
{
  static int flag=0;

  if (state==true){
     int s=client->webServer->slotOpen();
     if (s<1){
        flag = 1;
        toolbar->setButton(idTbWebServer, false);
     }
     else
        flag = 0;
  }
  else{
     if (flag==0)
        client->webServer->slotClosed();
  }
}


#include "kEirc.moc"

