#include "k2config.h"

#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>
#include <sys/stat.h>
#include <string.h>
#include <fstream.h>
#include <list>
#include <string>

K2ConfigItem::K2ConfigItem( const char *_name, list<string>* _lst, const char *_lang)
{
  if ( _lang == 0L ) _lang = "C";
  m_strName = _name;
  m_iType = TSTRING_LIST;
  m_mapValues[ _lang ].m_stringList = *_lst;
}

K2ConfigItem::K2ConfigItem( const char *_name, const char *_str, ItemType _type, const char *_lang )
{
  if ( _lang == 0L ) _lang = "C";
  m_strName = _name;
  m_iType = _type;
  m_mapValues[ _lang ].m_strText = _str;
}

K2ConfigItem::K2ConfigItem( const char *_name, bool _b, const char *_lang )
{
  if ( _lang == 0L ) _lang = "C";
  m_strName = _name;
  m_iType = TBOOLEAN;
  m_mapValues[ _lang ].m_b = _b;
}

K2ConfigItem::K2ConfigItem( const char *_name, int _i, const char *_lang )
{
  if ( _lang == 0L ) _lang = "C";
  m_strName = _name;
  m_iType = TLONG;
  m_mapValues[ _lang ].m_i = _i;
}

K2ConfigItem::K2ConfigItem( const char *_name, float _f, const char *_lang )
{
  if ( _lang == 0L ) _lang = "C";
  m_strName = _name;
  m_iType = TFLOAT;
  m_mapValues[ _lang ].m_f = _f;
}

const char* K2ConfigItem::text()
{
  map<string,Value>::iterator it = findBestLang();
  return it->second.text();
}

bool K2ConfigItem::fileContent( string& _str )
{
  map<string,Value>::iterator it = findBestLang();
  return it->second.fileContent( _str );
}

int K2ConfigItem::integer()
{
  map<string,Value>::iterator it = findBestLang();
  return it->second.integer();
}

float K2ConfigItem::fp()
{
  map<string,Value>::iterator it = findBestLang();
  return it->second.fp();
}

bool K2ConfigItem::boolean()
{
  map<string,Value>::iterator it = findBestLang();
  return it->second.boolean();
}

list<string>& K2ConfigItem::stringList()
{
  map<string,Value>::iterator it = findBestLang();
  return it->second.stringList();
}

map<string,K2ConfigItem::Value>::iterator K2ConfigItem::findBestLang()
{
  map<string,Value>::iterator it = m_mapValues.find( "C" );
  if ( it != m_mapValues.end() )
    return it;
  it = m_mapValues.begin();
  assert( it != m_mapValues.end() );
  return it;
}

bool K2ConfigItem::Value::fileContent( string& _str )
{
  struct stat buff;
  stat( m_strText.c_str(), &buff );
  int size = buff.st_size;
    
  FILE *f = fopen( m_strText.c_str(), "rb" );
  if ( f )
  {    
    char *p = new char[ size + 1 ];
    int n = fread( p, 1, size, f );
    p[n] = 0;
    fclose( f );
    _str = p;
    delete []p;
  }
  else
    return false;
  
  return true;
}

void K2ConfigItem::save( ostream& out, int _indent )
{
  map<string,Value>::iterator it = m_mapValues.begin();
  for( ; it != m_mapValues.end(); it++ )
  {    
    for( int i = 0; i < _indent; i++ )
      out << ' ';
  
    out << '\'' << name() << '\'';
    if ( it->first != "C" )
      out << " [" << it->first << "] ";

    switch( type() )
    {
    case TBOOLEAN:
      out  << "<bool> = ";
      if ( boolean() )
	out << "true ";
      else
	out << "false ";
      break;
    case TSTRING:
      out << "<string> = '" << it->second.text() << "' ";
      break;
    case TLONG:
      out << "<long> = " << it->second.integer() << ' ';
      break;
    case TFLOAT:
      out << "<float> = " << it->second.fp() << ' ';
      break;
    case TFILE:
      out << "<file> = '" << it->second.fileName() << "' ";
      break;
    case TSTRING_LIST:
      out << "<@string> = { ";
      if ( it->second.stringList().size() > 0 )
      {
	list<string>::iterator it2 = it->second.stringList().begin();
	while( it2 != it->second.stringList().end() )
	{
	  out << '\'' << *it2 << '\'';
	  it2++;
	  if ( it2 != it->second.stringList().end() )
	    out << ", ";
	}
      }
      out << " }";
      break;
    default:
      assert( 0 );
    }
   
    out << ';' << endl; 
  }
}

K2Config::K2Config( const char *_filename )
{
  m_errno = 0;
  m_strFileName = _filename;
  m_strType = "__main__";
  m_strName = "__main__";
  m_pTranslation = 0L;
  m_pEntries = 0L;

  struct stat buff;
  stat( _filename, &buff );
  int size = buff.st_size;
  
  FILE *f = fopen( _filename, "rb" );
  if ( f )
  {    
    char *p = new char[ size + 1 ];
    int n = fread( p, 1, size, f );
    p[n] = 0;
    fclose( f );
    
    parse( p );
    delete [] p;
    makeI18N();
  }
  else
    m_errno = errno;
}

void K2Config::parse( const char* _str )
{
  /**
   * Init
   */
  if ( m_pTranslation )
    delete m_pTranslation;  
  m_pTranslation = 0L;

  if ( m_pEntries )
    delete m_pEntries;
  m_pEntries = 0L;

  m_strName = "";
  m_strType = "";

  /**
   * Parse
   */
  m_pEntries = k2config_parse( _str );
}

void K2Config::insert( iterator _it, K2ConfigBase* _base )
{
  if ( _it == begin() )
  {
    K2ConfigBase *b = m_pEntries;
    m_pEntries = _base;
    _base->setNext( b );
    if ( b )
      b->setPrev( _base );
    return;
  }
  
  if ( _it == end() )
  {
    K2ConfigBase *b = m_pEntries;
    while( b->next() )
      b = b->next();
    b->setNext( _base );
    _base->setPrev( b );
    return;
  }
  
  _base->setPrev( _it->prev() );
  _base->setNext( &(*_it) );
  
  _it->prev()->setNext( _base );
  _it->setPrev( _base );
}

K2Config::iterator K2Config::find( const char *_name, K2Config::iterator it )
{
  for( ; it != end(); ++it )
  {
    if ( it.item() && strcmp( it.item()->name(), _name ) == 0 )
      return it;
    if ( it.group() && strcmp( it.group()->name(), _name ) == 0 )
      return it;
  }
  
  return it;
}

void K2Config::erase( iterator _it )
{
  if ( _it == end() )
    return;
  // else => There are indeed elements

  if ( _it == begin() )
  {
    K2ConfigBase *b = m_pEntries;
    m_pEntries = b->next();
    if ( m_pEntries )
      m_pEntries->setPrev( 0L );
    delete b;
    return;
  }
  
  K2ConfigBase *b = &(*_it);
  b->prev()->setNext( b->next() );
  if ( b->next() )
    b->next()->setPrev( b->prev() );
  // delete b;
}

bool K2Config::readLong( const char *_name, int& _l )
{
  iterator it = find( _name );
  if ( it == end() )
    return false;
  
  if ( it.item() && it.item()->type() == K2ConfigItem::TLONG )
  {    
    _l = it.item()->integer();
    return true;
  }
  
  return false;
}

bool K2Config::readFloat( const char *_name, float& _f )
{
  iterator it = find( _name );
  if ( it == end() )
    return false;
  
  if ( it.item() && it.item()->type() == K2ConfigItem::TFLOAT )
  {    
    _f = it.item()->fp();
    return true;
  }
  
  return false;
}

bool K2Config::readBool( const char *_name, bool& _b )
{
  iterator it = find( _name );
  if ( it == end() )
    return false;
  
  if ( it.item() && it.item()->type() == K2ConfigItem::TBOOLEAN )
  {    
    _b = it.item()->boolean();
    return true;
  }
  
  return false;
}

bool K2Config::readString( const char *_name, string& _s )
{
  iterator it = find( _name );
  if ( it == end() )
    return false;
  
  if ( it.item() && ( it.item()->type() == K2ConfigItem::TSTRING || it.item()->type() == K2ConfigItem::TFILE ) )
  {    
    _s = it.item()->text();
    return true;
  }
  
  return false;
}

bool K2Config::readStringList( const char *_name, list<string>& _lst )
{
  iterator it = find( _name );
  if ( it == end() )
    return false;
  
  if ( it.item() && it.item()->type() == K2ConfigItem::TSTRING_LIST )
  {    
    _lst.clear();
    _lst.insert( _lst.begin(), it.item()->stringList().begin(), it.item()->stringList().end() );
    return true;
  }
  
  return false;
}

void K2Config::clear()
{
  if ( m_pTranslation )
    delete m_pTranslation;  
  m_pTranslation = 0L;

  if ( m_pEntries )
    delete m_pEntries;
  m_pEntries = 0L;
}

K2Config* K2Config::insertGroup( iterator _it, const char *_name, const char *_type )
{
  iterator it = find( _name );
  if ( it != end() && it.item() )
  {
    cerr << "K2Config Error: Item of name " << _name << " does already exist" << endl;
    return 0L;
  }

  if ( it != end() && it.group() )
  {    
    if ( strcmp( _type, it.group()->type() ) != 0L )
    {
      cerr << "K2Config Error: Group of " << _name << " does already exist, but of different type " << it.group()->type() << endl;
      return 0L;
    }

    return it.group();
  }
  
  K2Config *g;
  insert( _it, ( g = new K2Config( _name, _type ) ) );
  return g;
}

void K2Config::writeLong( const char *_name, int _l )
{
  iterator it = find( _name );
  erase( it );
  
  insert( begin(), new K2ConfigItem( _name, _l ) );
}

void K2Config::writeFloat( const char *_name, float _f )
{
  iterator it = find( _name );
  if ( it != end() )
    erase( it );
  
  insert( begin(), new K2ConfigItem( _name, _f ) );
}

void K2Config::writeBool( const char *_name, bool _b )
{
  iterator it = find( _name );
  if ( it != end() )
    erase( it );
  
  insert( begin(), new K2ConfigItem( _name, _b ) );
}

void K2Config::writeString( const char *_name, const char *_s )
{
  iterator it = find( _name );
  if ( it != end() )
    erase( it );
  
  insert( begin(), new K2ConfigItem( _name, _s ) );
}

void K2Config::writeFile( const char *_name, const char *_s )
{
  iterator it = find( _name );
  if ( it != end() )
    erase( it );
  
  insert( begin(), new K2ConfigItem( _name, _s, K2ConfigItem::TFILE ) );
}

void K2Config::writeStringList( const char *_name, list<string>& _lst )
{
  iterator it = find( _name );
  if ( it != end() )
    erase( it );
  
  list<string>* lst = new list<string>( _lst );
  insert( begin(), new K2ConfigItem( _name, lst ) );
}

void K2Config::appendStringList( const char *_name, const char *_value )
{
  iterator it = find( _name );
  if ( it == end() || it.group() || it.item()->type() != K2ConfigItem::TSTRING_LIST )
  {
    list<string> lst;
    lst.push_back( _value );
    writeStringList( _name, lst );
    return;
  }
  
  it.item()->stringList().push_back( _name );
}

bool K2Config::save()
{
  if( m_strFileName.empty() )
    return false;
  
  ofstream out( m_strFileName.c_str() );
  if( !out )
    return false;
  
  save( out );
  
  return true;
}

void K2Config::save( ostream& out, int _indent )
{
  int indent = 0;
  
  if ( !m_strName.empty() )
  {
    indent = 2;
    for( int i = 0; i < _indent; i++ )
      out << ' ';

    out << '\'' << type() << '\'' << " '" << name() << '\'' << endl;
    if ( m_pTranslation )
      m_pTranslation->save( out );

    for( int i = 0; i < _indent; i++ )
      out << ' ';
  
    out << '{' << endl;
  }
  
  iterator it = begin();
  for( ; it != end(); it++ )
  {
    it->save( out, _indent + indent );
  }

  if ( !m_strName.empty() )
  {      
    for( int i = 0; i < _indent; i++ )
      out << ' ';

    out << "};" << endl;
  }
}

/*
K2Config* K2Config::findGroup( const char *_name )
{
  iterator it = find( _name );
  if ( it != end() && it.group() )
    return it.group();
  return 0L;
}
*/

int K2Config::size()
{
  int i = 0;
  K2Config::iterator it = begin();
  for( ; it != end(); it++ )
    i++;

  return i;
}

void K2Config::makeI18N()
{
  K2Config::iterator it = begin();
  for( ; it != end(); it++ )
  {
    if ( it.item() )
    {
      K2Config::iterator it2 = it;
      it2++;
      K2Config::iterator it3;
      while( ( it3 = findItem( it.item()->name(), it2 ) ) != end() )
      {
	it.item()->unite( it3.item() );
	erase( it3 );
	it2 = it;
	it2++;
      }
    }
    else if ( it.group() )
    {
      it.group()->makeI18N();
    }
  }

  intern_i18n_iterator it2 = intern_i18n_begin();
  for( ; it2 != intern_i18n_end(); it2++ )
  {
    m_mapNames[ it2->lang() ] = it2->text();
  }

  if ( m_pTranslation )
    m_pTranslation->deleteAll();
  m_pTranslation = 0L;
}

void K2ConfigItem::unite( K2ConfigItem* _item )
{
  map<string,Value>::iterator it = _item->i18n_begin();
  for( ; it != _item->i18n_end(); it++ )
    m_mapValues[ it->first ] = it->second;
}

ostream& operator<<( ostream& _str, K2ConfigBase& _base )
{
  _base.save( _str );
  return _str;
}
