#include <qdir.h>
#include <qfileinf.h>
#include <qlayout.h>
#include <qframe.h>
#include <qaccel.h>
#include <qstrlist.h>

#include <kmsgbox.h>

#include <stream.h>
#include <stdlib.h>
#include <string.h>

#include "klogin.h"

const char *commandName[] = {"ssh", "rlogin", "telnet", 0};

//=================================================================
//          return 1 if is executable and it is in the path
//
//=================================================================
int isInPath(const char* file)
{
  char *path = getenv("PATH");
  if(path == 0) return 0;

  char *npath = new char[strlen(path)];
  strcpy(npath, path);

  for(char*dir=strtok(npath, ":"); dir!=0; dir=strtok(0, ":")) {
    QDir qdir(dir);
    if(QFileInfo(qdir, file).isExecutable()) {
      delete npath;
      return 1;
    }
  }
  delete npath;
  return 0;
}

//=================================================================
//               Main widget of the program
//
//=================================================================
KLogin::KLogin(KApplication& app):QWidget(),
  hostDict(17, FALSE)
{
  //--- create widgets
  frame = new QFrame(this);

  hostCombo  = new QComboBox(true, frame);
  loginCombo = new QComboBox(true, frame);

  ok     = new QPushButton(klocale->translate("OK"), frame);
  cancel = new QPushButton(klocale->translate("Cancel"), frame);
  help   = new QPushButton(klocale->translate("Help"), frame);

  hostLabel  = new QLabel(klocale->translate("Host"), frame);
  loginLabel = new QLabel(klocale->translate("Login"), frame);

  group = new QButtonGroup("", this);
  for(int i=0; i<4; i++) commandButton[i] = 0;
  for(int i=0, j=0; commandName[i]!=0; i++) {
    if(isInPath(commandName[i])) {
      commandButton[j++] = new QRadioButton(commandName[i], group);
    }
  }

  //--- set styles
  setFocusPolicy(QWidget::StrongFocus);
  frame->setFrameStyle(QFrame::WinPanel | QFrame::Raised);

  hostCombo->setInsertionPolicy(QComboBox::AtTop);

  //--- set input
  QAccel *accel = new QAccel(this);

  hostCombo->setFocusPolicy(QWidget::StrongFocus);
  loginCombo->setFocusPolicy(QWidget::StrongFocus);

  accel->connectItem(accel->insertItem(Key_Escape), this, SIGNAL(quit()));
  accel->connectItem(accel->insertItem(Key_Return), this, SLOT(go()));
  accel->connectItem(accel->insertItem(Key_F1), this, SLOT(showHelp()));

  //--- make connections
  QObject::connect(hostCombo, SIGNAL(activated(const char*)),
		   this, SLOT(hostChanged(const char*)));
  QObject::connect(loginCombo, SIGNAL(activated(const char*)),
		   this, SLOT(loginChanged(const char*)));
  QObject::connect(this, SIGNAL(quit()), qApp, SLOT(quit()));

  QObject::connect(ok, SIGNAL(clicked()), this, SLOT(go()));
  QObject::connect(cancel, SIGNAL(clicked()), qApp, SLOT(quit()));
  QObject::connect(help, SIGNAL(clicked()), this, SLOT(showHelp()));
  QObject::connect(group, SIGNAL(clicked(int)), this, SLOT(commandChanged(int)));

  //--- read configuration
  config = app.getConfig();
}

void KLogin::show()
{
  QWidget::show();

  hostCombo->setFocus();  
}

void KLogin::readConfig()
{
  //--- default settings
  emit commandChanged(0);

  KEntryIterator *it = config->entryIterator("Hosts");
  for(; it!=0 && it->current(); ++(*it)) {
    const char *data  = it->current()->aValue;
    const char *comma = strchr(data, ',');
    QString login(data);
    QString command;
    if(comma == 0) {
      command.setStr("");
    } else {
      command.setStr(comma+1);
      login.resize(comma-data+1);
    }
    hostCombo ->insertItem(it->currentKey(), 0);
    loginCombo->insertItem(login, 0);
    emit commandChanged(command);       // emit command change signal
    emit hostChanged(it->currentKey()); // emit host change signal
    emit loginChanged(login);           // emit login change signal
  }
  loginCombo->insertItem("", 0);
  loginChanged("");
  
  config->setGroup("Options");
  QString last = config->readEntry("last");
  int i;
  for(i=0; i<hostCombo->count()-1; i++)
    if(last == hostCombo->text(i)) break;
  hostCombo->setCurrentItem(i);
  hostChanged(hostCombo->currentText());
}

void KLogin::resizeEvent(QResizeEvent *)
{
  frame->resize(size());

  hostLabel->setGeometry (20, 20, 30, 30);
  loginLabel->setGeometry(hostLabel->x(), hostLabel->y()+hostLabel->height()+20,
			  hostLabel->width(), hostLabel->height());

  hostCombo->setGeometry (hostLabel->x()+hostLabel->width()+20,
			  hostLabel->y(), 150, hostLabel->height());
  loginCombo->setGeometry(loginLabel->x()+loginLabel->width()+20,
			  loginLabel->y(), hostCombo->width(),
			  loginLabel->height());

  ok->setGeometry    (hostCombo->x()+hostCombo->width()+20, hostLabel->y(),
		      60, hostLabel->height());
  cancel->setGeometry(ok->x(), ok->y()+ok->height()+10, ok->width(),
		      ok->height());
  help->setGeometry(cancel->x(), cancel->y()+cancel->height()+10, cancel->width(),
		      cancel->height());

  group->setGeometry(loginLabel->x()+5, loginLabel->y()+44, 200, 44);
  for(int i=0, x=8; commandButton[i]!=0; i++) {
    commandButton[i]->setGeometry(x, 8, (strlen(commandName[i])+1)*10, 30);
    x += commandButton[i]->width();
  }
}

//--------------------------------------------------
//       SLOTS
//
//--------------------------------------------------
void KLogin::hostChanged(const char* host)     // then host combo is changed
{
  static int flag = 0;

  if(flag) return;
  flag = 1;

  if(*host == '\0') {
    hostCombo->removeItem(hostCombo->currentItem());
    flag = 0;
    return;
  }

  hostParam *param;  
  if((param=hostDict.find(host)) == 0) {
    //--- inserrt new item
    param = new hostParam;
    param->login.setStr(loginCombo->currentText());
    param->command.setStr(commandButton[curCommand]->text());
    hostDict.insert(host, param);
  } else {
    //--- set parameters
    int i;
    for(i=0; i<loginCombo->count(); i++)
      if(!strcmp(loginCombo->text(i), param->login)) break;
    loginCombo->setCurrentItem(i);
    emit commandChanged(param->command);
  }
  flag = 0;
}
void KLogin::loginChanged(const char*)         // then login combo is changed
{
  for(int i=0; i<loginCombo->count()-1; i++)
    for(int j=i+1; j<loginCombo->count(); j++)
      if(!strcmp(loginCombo->text(i), loginCombo->text(j))) {
	loginCombo->removeItem(i);
	break;
      }
}
void KLogin::commandChanged(const char * cmd)  // then command button retogled
{
  int i;
  for(i=0; commandButton[i+1]!=0; i++)
    if(!strcmp(commandButton[i]->text(), cmd)) break;
  emit commandChanged(i);
}
void KLogin::commandChanged(int id)            // then command button retogled
{
  static f = 0;
  if(f) return;

  curCommand = id;
  for(int i=0; commandButton[i]!=0; i++)
    commandButton[i]->setChecked(false);
  commandButton[curCommand]->setChecked(true);
  f = 0;
}
void KLogin::go()                              // then go to host connect
{
  //--- get current settings
  const char *host = hostCombo->currentText();
  const char *login= loginCombo->currentText();
  const char *cmd  = commandButton[curCommand]->text();

  if(*host == '\0') {
    qApp->beep();
    KMsgBox::message(0, klocale->translate("Warning"), 
		     klocale->translate("You have to type host name"));
    return;
  }

  //--- prepare command to execute
  QString sys(256);
  if(*login == '\0')
    sys.sprintf("kvt -T \"%s %s %s\" -e %s %s &", cmd, klocale->translate("to"),
		host, cmd, host);
  else
    sys.sprintf("kvt -T \"%s %s %s %s %s\" -e %s -l %s %s &",
		   cmd, klocale->translate("to"), host, klocale->translate("as"),
		login, cmd, login, host);

  //--- save data
  hostDict.remove(host);
  hostChanged(host);

  config->setGroup("Hosts");
  QDictIterator<hostParam> it(hostDict);
  for(int i=0; it.current(); ++it, i++) {
    QString data(it.current()->login);
    data.append(",");
    data.append(it.current()->command);
    config->writeEntry(it.currentKey(), data);
  }
  config->setGroup("Options");
  config->writeEntry("last", host);

  //--- execute command
  system(sys);
  emit quit();
  //qApp->exit();
}

void KLogin::showHelp()
{
  KApplication::getKApplication()->invokeHTMLHelp("klogin/index.html", "");
}

int main(int argc, char** argv)
{
  KApplication app(argc, argv, "klogin");
  KLogin klogin(app);

  app.setMainWidget(&klogin);

  QWidget *desktop = qApp->desktop();
  klogin.setFixedSize(310, 170);
  int width = klogin.width(), height = klogin.height();
  klogin.setGeometry((desktop->width()-width)/2, (desktop->height()-height)/2,
	      width, height);
  klogin.readConfig();
  klogin.show();

  return app.exec();
}
