/*
 * Copyright 1999 Red Hat Software, Inc.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* This probably doesn't work on anything other then Ethernet! It may work
   on PLIP as well, but ARCnet and Token Ring are unlikely at best. */

#include <errno.h>
#include <net/if.h>
#include <net/if_packet.h>
#include <net/ethernet.h>
#include <net/route.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <popt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>
#include <netdb.h>

#include "net.h"

#define DHCP_FLAG_NODAEMON	(1 << 0)
#define DHCP_FLAG_NOCONFIG	(1 << 1)
#define DHCP_FLAG_FORCEHNLOOKUP	(1 << 2)

#define RESULT_OKAY		0
#define RESULT_FAILED		1
#define RESULT_UNKNOWNIFACE	2

#define N_(foo) (foo)

#define PROGNAME "pump"
#define CONTROLSOCKET "/var/run/pump.sock"

#define NUM_RETRIES	50

#define _(foo) ((foo))
#include <stdarg.h>

void logMessage(char * mess, ...) {
    va_list args;

    va_start(args, mess);
    vprintf(mess, args);
    va_end(args);
    puts("");
}

typedef int bp_int32;
typedef short bp_int16;

#define BOOTP_OPTION_NETMASK		1
#define BOOTP_OPTION_GATEWAY		3
#define BOOTP_OPTION_DNS		6
#define BOOTP_OPTION_HOSTNAME		12
#define BOOTP_OPTION_BOOTFILE		13
#define BOOTP_OPTION_DOMAIN		15
#define BOOTP_OPTION_BROADCAST		28

#define DHCP_OPTION_REQADDR		50
#define DHCP_OPTION_LEASE		51
#define DHCP_OPTION_OVERLOAD		52
#define DHCP_OPTION_TYPE		53
#define DHCP_OPTION_SERVER		54
#define DHCP_OPTION_OPTIONREQ		55
#define DHCP_OPTION_MAXSIZE		57
#define DHCP_OPTION_T1			58

#define BOOTP_CLIENT_PORT	68
#define BOOTP_SERVER_PORT	67

#define BOOTP_OPCODE_REQUEST	1
#define BOOTP_OPCODE_REPLY	2

#define NORESPONSE		-10
#define DHCP_TYPE_DISCOVER	1
#define DHCP_TYPE_OFFER		2
#define DHCP_TYPE_REQUEST	3
#define DHCP_TYPE_DECLINE	4
#define DHCP_TYPE_ACK		5
#define DHCP_TYPE_NAK		6
#define DHCP_TYPE_RELEASE	7
#define DHCP_TYPE_INFORM	8

#define BOOTP_VENDOR_LENGTH	64
#define DHCP_VENDOR_LENGTH	340

struct bootpRequest {
    char opcode;
    char hw;
    char hwlength;
    char hopcount;
    bp_int32 id;
    bp_int16 secs;
    bp_int16 flags;
    bp_int32 ciaddr, yiaddr, server_ip, bootp_gw_ip;
    char hwaddr[16];
    char servername[64];
    char bootfile[128];
    char vendor[DHCP_VENDOR_LENGTH];
} ;

struct psuedohUdpHeader {
    bp_int32 source, dest;
    char zero;
    char protocol;
    bp_int16 len;
};

struct command {
    enum { CMD_STARTIFACE, CMD_RESULT, CMD_DIE, CMD_STOPIFACE, 
	   CMD_FORCERENEW, CMD_REQSTATUS, CMD_STATUS } type;
    union {
	struct {
	    int timeout;
	    char device[20];
	    int flags;
	    int reqLease;			/* in hours */
	    char reqHostname[200];
	} start;
	int result;				/* 0 for success */
	struct {
	    char device[20];
	} stop;
	struct {
	    char device[20];
	} renew;
	struct {
	    char device[20];
	} reqstatus;
	struct {
	    struct intfInfo intf;
	    char hostname[1024];
	    char domain[1024];
	    char bootFile[1024];
	} status;
    } u;
};

static char * doDhcp(int timeout, char * device, int flags, int lease,
		     char * reqHostname, struct intfInfo * intf);

static const char vendCookie[] = { 99, 130, 83, 99, 255 };

static char * perrorstr(char * msg);
static char * setupInterface(struct intfInfo * intf, int s);
static char * prepareInterface(struct intfInfo * intf, int s);
static char * getInterfaceInfo(struct intfInfo * intf, int s);
static char * disableInterface(char * device);
static void parseReply(struct bootpRequest * breq, struct intfInfo * intf);
static char * prepareRequest(struct bootpRequest * breq,
			     int sock, char * device, time_t startTime);
static void parseLease(struct bootpRequest * bresp, struct intfInfo * intf);
static void initVendorCodes(struct bootpRequest * breq);
static char * handleTransaction(int s, int timeout, struct bootpRequest * breq,
			        struct bootpRequest * bresp, 
			        struct sockaddr_in * serverAddr,
			        struct sockaddr_in * respondant,
				int useBootPacket, int dhcpResponseType);
static int dhcpMessageType(struct bootpRequest * response);
static void setMissingIpInfo(struct intfInfo * intf);
static int openControlSocket(void);
static int dhcpRenew(struct intfInfo * intf);
static int dhcpRelease(struct intfInfo * intf);

int newKernel(void) {
    struct utsname ubuf;
    int major1, major2;

    uname(&ubuf);
    if (!strcasecmp(ubuf.sysname, "linux")) {
	if (sscanf(ubuf.release, "%d.%d", &major1, &major2) != 2 ||
		(major1 < 2) || (major1 == 2 && major2 == 0)) {
	    return 0;
	}
    }

    return 1;
}

static char * disableInterface(char * device) {
    struct ifreq req;
    int s;

    s = socket(AF_INET, SOCK_DGRAM, 0);

    strcpy(req.ifr_name, device);
    req.ifr_flags &= ~(IFF_UP | IFF_RUNNING);
    if (ioctl(s, SIOCSIFFLAGS, &req)) {
	close(s);
	return perrorstr("SIOCSIFFLAGS");
    }

    close(s);

    return NULL;
}

static char * setupInterface(struct intfInfo * intf, int s) {
    char * rc;
    struct sockaddr_in * addrp;
    struct ifreq req;
    struct rtentry route;

    if ((rc = disableInterface(intf->device))) return rc;

    /* we have to have basic information to get this far */
    addrp = (struct sockaddr_in *) &req.ifr_addr;
    addrp->sin_family = AF_INET;
    strcpy(req.ifr_name, intf->device);
   
    addrp->sin_addr = intf->ip;
    if (ioctl(s, SIOCSIFADDR, &req))
	return perrorstr("SIOCSIFADDR");

    addrp->sin_addr = intf->netmask;
    if (ioctl(s, SIOCSIFNETMASK, &req))
	return perrorstr("SIOCSIFNETMASK");

    addrp->sin_addr = intf->broadcast;
    if (ioctl(s, SIOCSIFBRDADDR, &req))
	return perrorstr("SIOCSIFBRDADDR");

    /* bring up the device, and specifically allow broadcasts through it */
    req.ifr_flags = IFF_UP | IFF_RUNNING | IFF_BROADCAST;
    if (ioctl(s, SIOCSIFFLAGS, &req))
	return perrorstr("SIOCSIFFLAGS");

    /* add a route for this network */
    route.rt_dev = intf->device;
    route.rt_flags = RTF_UP;
    route.rt_metric = 0;

    if (!newKernel()) {
	addrp->sin_family = AF_INET;
	addrp->sin_port = 0;
	addrp->sin_addr = intf->network;
	memcpy(&route.rt_dst, addrp, sizeof(*addrp));
	addrp->sin_addr = intf->netmask;
	memcpy(&route.rt_genmask, addrp, sizeof(*addrp));

	if (ioctl(s, SIOCADDRT, &route)) {
	    /* the route cannot already exist, as we've taken the device down */
	    return perrorstr("SIOCADDRT 1");
	}
    }

    return NULL;
}

static char * setupDefaultGateway(struct in_addr * gw, int s) {
    struct sockaddr_in addr;
    struct rtentry route;

    addr.sin_family = AF_INET;
    addr.sin_port = 0;
    addr.sin_addr.s_addr = INADDR_ANY;
    memcpy(&route.rt_dst, &addr, sizeof(addr));
    memcpy(&route.rt_genmask, &addr, sizeof(addr));
    addr.sin_addr = *gw;
    memcpy(&route.rt_gateway, &addr, sizeof(addr));
    
    route.rt_flags = RTF_UP | RTF_GATEWAY;
    route.rt_metric = 0;
    route.rt_dev = NULL;

    if (ioctl(s, SIOCADDRT, &route)) {
	/* the route cannot already exist, as we've taken the device 
	   down */
	return perrorstr("SIOCADDRT 2");
    }

    return NULL;
}

static char * getInterfaceInfo(struct intfInfo * intf, int s) {
    struct ifreq req;
    struct sockaddr_in * addrp;

    strcpy(req.ifr_name, intf->device);
    if (ioctl(s, SIOCGIFBRDADDR, &req))
	return perrorstr("SIOCGIFBRDADDR");

    addrp = (struct sockaddr_in *) &req.ifr_addr;
    intf->broadcast = addrp->sin_addr;
    intf->set = INTFINFO_HAS_BROADCAST;

    return NULL;
}

static char * prepareInterface(struct intfInfo * intf, int s) {
    struct sockaddr_in * addrp;
    struct ifreq req;
    struct rtentry route;

    addrp = (struct sockaddr_in *) &req.ifr_addr;

    strcpy(req.ifr_name, intf->device);
    addrp->sin_family = AF_INET;
    addrp->sin_port = 0;
    memset(&addrp->sin_addr, 0, sizeof(addrp->sin_addr));

    addrp->sin_family = AF_INET;
    addrp->sin_addr.s_addr = htonl(0);

    if (ioctl(s, SIOCSIFADDR, &req))
	return perrorstr("SIOCSIFADDR");

    if (!newKernel()) {
	if (ioctl(s, SIOCSIFNETMASK, &req))
	    return perrorstr("SIOCSIFNETMASK");

	/* the broadcast address is 255.255.255.255 */
	memset(&addrp->sin_addr, 255, sizeof(addrp->sin_addr));
	if (ioctl(s, SIOCSIFBRDADDR, &req))
	    return perrorstr("SIOCSIFBRDADDR");
    }

    req.ifr_flags = IFF_UP | IFF_BROADCAST | IFF_RUNNING;
    if (ioctl(s, SIOCSIFFLAGS, &req))
	return perrorstr("SIOCSIFFLAGS");

    memset(&route, 0, sizeof(route));
    memcpy(&route.rt_gateway, addrp, sizeof(*addrp));

    addrp->sin_family = AF_INET;
    addrp->sin_port = 0;
    addrp->sin_addr.s_addr = INADDR_ANY;
    memcpy(&route.rt_dst, addrp, sizeof(*addrp));
    memcpy(&route.rt_genmask, addrp, sizeof(*addrp));

    route.rt_dev = intf->device;
    route.rt_flags = RTF_UP;
    route.rt_metric = 0;

    if (ioctl(s, SIOCADDRT, &route)) {
	if (errno != EEXIST) {
	    close(s);
	    return perrorstr("SIOCADDRT 3");
	}
    }

    return NULL;
}

static int dhcpMessageType(struct bootpRequest * response) {
    unsigned char * chptr;
    unsigned char option, length;
   
    chptr = response->vendor;

    chptr += 4;
    while (*chptr != 0xFF) {
	option = *chptr++;
	if (!option) continue;
	length = *chptr++;
	if (option == DHCP_OPTION_TYPE)
	    return *chptr;

	chptr += length;
    }

    return -1;
}

static void setMissingIpInfo(struct intfInfo * intf) {
    bp_int32 ipNum = *((bp_int32 *) &intf->ip);
    bp_int32 nmNum = *((bp_int32 *) &intf->netmask);
    bp_int32 ipRealNum = ntohl(ipNum);

    if (!(intf->set & INTFINFO_HAS_NETMASK)) {
	if (((ipRealNum & 0xFF000000) >> 24) <= 127)
	    nmNum = 0xFF000000;
	else if (((ipRealNum & 0xFF000000) >> 24) <= 191)
	    nmNum = 0xFFFF0000;
	else 
	    nmNum = 0xFFFFFF00;
	*((bp_int32 *) &intf->netmask) = nmNum = htonl(nmNum);
    }

    if (!(intf->set & INTFINFO_HAS_BROADCAST))
	*((bp_int32 *) &intf->broadcast) = (ipNum & nmNum) | ~(nmNum);

    if (!(intf->set & INTFINFO_HAS_NETWORK))
	*((bp_int32 *) &intf->network) = ipNum & nmNum;

    intf->set |= INTFINFO_HAS_BROADCAST | INTFINFO_HAS_NETWORK | 
		 INTFINFO_HAS_NETMASK;
}

static void parseReply(struct bootpRequest * breq, struct intfInfo * intf) {
    unsigned int i;
    unsigned char * chptr;
    unsigned char option, length;

    i = ~(INTFINFO_HAS_IP | INTFINFO_HAS_NETMASK | INTFINFO_HAS_NETWORK |
	  INTFINFO_HAS_BROADCAST);
    intf->set &= i;

    if (strlen(breq->bootfile)) {
	intf->bootFile = strdup(breq->bootfile);
	intf->set |= INTFINFO_HAS_BOOTFILE;
    } else {
	intf->set &= ~INTFINFO_HAS_BOOTFILE;
    }

    memcpy(&intf->ip, &breq->yiaddr, 4);
    intf->set |= INTFINFO_HAS_IP;

    chptr = breq->vendor;
    chptr += 4;
    while (*chptr != 0xFF) {
	option = *chptr++;
	if (!option) continue;
	length = *chptr++;

	switch (option) {
	    case BOOTP_OPTION_DNS:
		intf->numDns = 0;
		for (i = 0; i < length; i += 4)
		    memcpy(&intf->dnsServers[intf->numDns++], chptr + i, 4);
		intf->set |= NETINFO_HAS_DNS;
		break;

	    case BOOTP_OPTION_NETMASK:
		memcpy(&intf->netmask, chptr, 4);
		intf->set |= INTFINFO_HAS_NETMASK;
		break;

	    case BOOTP_OPTION_DOMAIN:
		intf->domain = malloc(length + 1);
		memcpy(intf->domain, chptr, length);
		intf->domain[length] = '\0';
		intf->set |= NETINFO_HAS_DOMAIN;
		break;

	    case BOOTP_OPTION_BROADCAST:
		memcpy(&intf->broadcast, chptr, 4);
		intf->set |= INTFINFO_HAS_BROADCAST;
		break;

	    case BOOTP_OPTION_GATEWAY:
		memcpy(&intf->gateway, chptr, 4);
		intf->set |= NETINFO_HAS_GATEWAY;
		break;

	    case BOOTP_OPTION_HOSTNAME:
		intf->hostname = malloc(length + 1);
		memcpy(intf->hostname, chptr, length);
		intf->hostname[length] = '\0';
		intf->set |= NETINFO_HAS_HOSTNAME;
		break;

	    case BOOTP_OPTION_BOOTFILE:
		/* we ignore this right now */
		break;

	    case DHCP_OPTION_OVERLOAD:
		/* FIXME: we should pay attention to this */
		logMessage("dhcp overload option is currently ignored!");
		break;
	}

	chptr += length;
    }

    setMissingIpInfo(intf);
}

static char * perrorstr(char * msg) {
    static char * err = NULL;
    static int errsize = 0;
    static int newsize;

    newsize = strlen(msg) + strlen(strerror(errno)) + 3;
    if (!errsize) {
	errsize = newsize;
	err = malloc(errsize);
    } else if (errsize < newsize) {
	free(err);
	errsize = newsize;
	err = malloc(errsize);
    } 
 
    sprintf(err, "%s: %s", msg, strerror(errno));

    return err;
}

static void initVendorCodes(struct bootpRequest * breq) {
    memcpy(breq->vendor, vendCookie, sizeof(vendCookie));
}

static char * prepareRequest(struct bootpRequest * breq,
			     int sock, char * device, time_t startTime) {
    struct ifreq req;
    int i;

    memset(breq, 0, sizeof(*breq));

    breq->opcode = BOOTP_OPCODE_REQUEST;

    strcpy(req.ifr_name, device);
    if (ioctl(sock, SIOCGIFHWADDR, &req))
	return perrorstr("SIOCSIFHWADDR");

    breq->hw = 1; 		/* ethernet */
    breq->hwlength = IFHWADDRLEN;	
    memcpy(breq->hwaddr, req.ifr_hwaddr.sa_data, IFHWADDRLEN);

    /* we should use something random here, but I don't want to start using
       stuff from the math library */
    breq->id = time(NULL);
    for (i = 0; i < IFHWADDRLEN; i++)
	breq->id ^= breq->hwaddr[i] << (8 * (i % 4));

    breq->hopcount = 0;
    breq->secs = time(NULL) - startTime;

    initVendorCodes(breq);

    return NULL;
}

static unsigned int verifyChecksum(void * buf, int length, void * buf2,
				   int length2) {
    unsigned int csum;
    unsigned short * sp;

    csum = 0;
    for (sp = (unsigned short *) buf; length > 0; (length -= 2), sp++)
	csum += *sp;

    /* this matches rfc 1071, but not Steven's */
    if (length)
	csum += *((unsigned char *) sp);

    for (sp = (unsigned short *) buf2; length2 > 0; (length2 -= 2), sp++)
	csum += *sp;

    /* this matches rfc 1071, but not Steven's */
    if (length)
	csum += *((unsigned char *) sp);

    while (csum >> 16)
	csum = (csum & 0xffff) + (csum >> 16);

    if (csum != 0xffff) return 0; else return 1;
}


static char * handleTransaction(int s, int timeout, struct bootpRequest * breq,
			        struct bootpRequest * bresp, 
			        struct sockaddr_in * serverAddr,
				struct sockaddr_in * respondant,
				int useBootpPacket, int dhcpResponseType) {
    struct timeval tv;
    fd_set readfs;
    int i, j;
    struct sockaddr_pkt tmpAddress;
    int gotit = 0;
    int tries = NUM_RETRIES;
    int nextTimeout = 2;
    time_t timeoutTime;
    int sin;
    int resend = 1;
    struct ethhdr;
    char ethPacket[ETH_FRAME_LEN];
    struct iphdr * ipHdr;
    struct udphdr * udpHdr;
    struct psuedohUdpHeader pHdr;
    time_t start = time(NULL);

    sin = socket(AF_PACKET, SOCK_DGRAM, ntohs(ETH_P_IP));

    while (!gotit && tries) {
	i = sizeof(*breq);
	if (useBootpPacket)
	    i -= (DHCP_VENDOR_LENGTH - BOOTP_VENDOR_LENGTH);

	if (resend) {
	    if (sendto(s, breq, i, 0, (struct sockaddr *) serverAddr, 
		       sizeof(*serverAddr)) != i) {
		close(sin);
		return perrorstr("sendto");
	    }

	    tries--;
	    nextTimeout *= 2;

	    switch (time(NULL) & 4) {
		case 0:	if (nextTimeout >= 2) nextTimeout--; break;
		case 1:	nextTimeout++; break;
	    }

	    timeoutTime = time(NULL) + nextTimeout;
	    i = timeout + start;
	    if (timeoutTime > i) timeoutTime = i;

	    resend = 0;
	}

	if (dhcpResponseType == NORESPONSE) {
	    close(sin);
	    return NULL;
	}

	tv.tv_usec = 0;
 	tv.tv_sec = timeoutTime - time(NULL);
	if (timeoutTime < time(NULL)) {
	    tries = 0;
	    continue;
	}

	FD_ZERO(&readfs);
	FD_SET(sin, &readfs);
	switch ((select(sin + 1, &readfs, NULL, NULL, &tv))) {
	  case 0:
	    resend = 1;
	    break;

	  case 1:
	    i = sizeof(tmpAddress);
	    if ((j = recvfrom(sin, ethPacket, sizeof(ethPacket), 0, 
		     (struct sockaddr *) &tmpAddress, &i)) < 0)
		return perrorstr("recvfrom");

	    /* We need to do some basic sanity checking of the header */
	    if (j < (sizeof(*ipHdr) + sizeof(*udpHdr))) continue;

	    ipHdr = (void *) ethPacket;
	    if (!verifyChecksum(NULL, 0, ipHdr, sizeof(*ipHdr)))
		continue;

	    if (ntohs(ipHdr->tot_len) > j)
		continue;
	    j = ntohs(ipHdr->tot_len);

	    if (ipHdr->protocol != IPPROTO_UDP) continue;

	    udpHdr = (void *) (ethPacket + sizeof(*ipHdr));
	    pHdr.source = ipHdr->saddr;
	    pHdr.dest = ipHdr->daddr;
	    pHdr.zero = 0;
	    pHdr.protocol = ipHdr->protocol;
	    pHdr.len = udpHdr->len;

/*
	    This doesn't always work!?!

	    if (!verifyChecksum(&pHdr, sizeof(pHdr), 
				udpHdr, j - sizeof(*ipHdr)))
		continue;
*/

	    if (ntohs(udpHdr->source) != BOOTP_SERVER_PORT)
		continue;
	    if (ntohs(udpHdr->dest) != BOOTP_CLIENT_PORT) 
		continue;
	    /* Go on with this packet; it looks sane */

	    memcpy(bresp, (char *) udpHdr + sizeof(*udpHdr), sizeof(*bresp));

	    /* sanity checks */
	    if (bresp->id != breq->id) continue;
	    if (bresp->opcode != BOOTP_OPCODE_REPLY) continue;
	    if (bresp->hwlength != breq->hwlength) continue;
	    if (memcmp(bresp->hwaddr, breq->hwaddr, bresp->hwlength)) continue;
	    i = dhcpMessageType(bresp);
	    if (!(i == -1 && useBootpPacket) && (i != dhcpResponseType))
		continue;
	    if (memcmp(bresp->vendor, vendCookie, 4)) continue;
	    /* if (respondant) *respondant = tmpAddress; */
	    gotit = 1;

	    break;

	  default:
	    close(sin);
	    return perrorstr("select");
	}
    }

    if (!gotit) {
	close(sin);
	return _("No DHCP reply received");
    }

    close(sin);

    return NULL;
}

static void addVendorCode(struct bootpRequest * breq, unsigned char option,
			  unsigned char length, void * data) {
    unsigned char * chptr;
    int theOption, theLength;

    chptr = breq->vendor;
    chptr += 4;
    while (*chptr != 0xFF && *chptr != option) {
	theOption = *chptr++;
	if (!theOption) continue;
	theLength = *chptr++;
	chptr += theLength;
    }

    *chptr++ = option;
    *chptr++ = length;
    memcpy(chptr, data, length);
    chptr[length] = 0xff;
}

static int getVendorCode(struct bootpRequest * bresp, unsigned char option,
			  void * data) {
    unsigned char * chptr;
    unsigned int length, theOption;

    chptr = bresp->vendor;
    chptr += 4;
    while (*chptr != 0xFF && *chptr != option) {
	theOption = *chptr++;
	if (!theOption) continue;
	length = *chptr++;
	chptr += length;
    }

    if (*chptr++ == 0xff) return 1;

    length = *chptr++;
    memcpy(data, chptr, length);

    return 0;
}

static int createSocket(void) {
    struct sockaddr_in clientAddr;
    int s;

    s = socket(AF_INET, SOCK_DGRAM, 0);
    if (s < 0)
	return -1;

    memset(&clientAddr.sin_addr, 0, sizeof(&clientAddr.sin_addr));
    clientAddr.sin_family = AF_INET;
    clientAddr.sin_port = htons(BOOTP_CLIENT_PORT);	/* bootp client */

    if (bind(s, (struct sockaddr *) &clientAddr, sizeof(clientAddr))) {
	close(s); 
	return -1;
    }

    return s;
}

static int dhcpRelease(struct intfInfo * intf) {
    struct bootpRequest breq, bresp;
    unsigned char messageType;
    struct sockaddr_in serverAddr;
    char * chptr;
    int s;

    if (!(intf->set & INTFINFO_HAS_LEASE)) return 0;

    if ((s = createSocket()) < 0) return 1;

    if ((chptr = prepareRequest(&breq, s, intf->device, time(NULL)))) {
	close(s);
	while (1);	/* problem */
    }

    messageType = DHCP_TYPE_RELEASE;
    addVendorCode(&breq, DHCP_OPTION_TYPE, 1, &messageType);
    memcpy(&breq.ciaddr, &intf->ip, sizeof(breq.ciaddr));

    serverAddr.sin_family = AF_INET;
    serverAddr.sin_port = htons(BOOTP_SERVER_PORT);	/* bootp server */
    serverAddr.sin_addr = intf->bootServer;

    if (!handleTransaction(s, 60, &breq, &bresp, &serverAddr, NULL, 0,
			   NORESPONSE)) {
	disableInterface(intf->device);
	close(s);
	return 0;
    }

    disableInterface(intf->device);
    close(s);

    syslog(LOG_INFO, "disabling interface %s", intf->device);

    return 1;
}
    
/* This is somewhat broken. We try only to renew the lease. If we fail,
   we don't try to completely rebind. This doesn't follow the DHCP spec,
   but for the install it should be a reasonable compromise. */
static int dhcpRenew(struct intfInfo * intf) {
    struct bootpRequest breq, bresp;
    unsigned char messageType;
    struct sockaddr_in serverAddr;
    char * chptr;
    int s;
    int i;

    s = createSocket();

    if ((chptr = prepareRequest(&breq, s, intf->device, time(NULL)))) {
	close(s);
	while (1);	/* problem */
    }

    messageType = DHCP_TYPE_REQUEST;
    addVendorCode(&breq, DHCP_OPTION_TYPE, 1, &messageType);
    memcpy(&breq.ciaddr, &intf->ip, sizeof(breq.ciaddr));

    i = htonl(intf->reqLease);
    addVendorCode(&breq, DHCP_OPTION_LEASE, 4, &i);

    serverAddr.sin_family = AF_INET;
    serverAddr.sin_port = htons(BOOTP_SERVER_PORT);	/* bootp server */
    serverAddr.sin_addr = intf->bootServer;

    if (handleTransaction(s, 60, &breq, &bresp, &serverAddr, NULL, 0,
			   DHCP_TYPE_ACK)) {
	close(s);
	return 1;
    }

    parseLease(&bresp, intf);

    syslog(LOG_INFO, "renewed lease for interface %s", intf->device);

    close(s);
    return 0;
}

static void parseLease(struct bootpRequest * bresp, struct intfInfo * intf) {
    int lease;
    time_t now;

    intf->set &= INTFINFO_HAS_LEASE;
    if (getVendorCode(bresp, DHCP_OPTION_LEASE, &lease)) 
	return;

    lease = ntohl(lease);

    if (lease && lease != 0xffffffff) {
	now = time(NULL);
	intf->set |= INTFINFO_HAS_LEASE;
	intf->leaseExpiration = now + lease;
	intf->renewAt = now + (7 * lease / 8);
    }
}

static void createResolvConf(struct intfInfo * intf, char * domain) {
    FILE * f;
    int i;
    char * chptr;

    f = fopen("/etc/resolv.conf", "w");
    if (!f) {
	syslog(LOG_ERR, "cannot create /etc/resolv.conf: %s\n",
	       strerror(errno));
	return;
    }

    fprintf(f, "search");
    chptr = domain;
    do {
	fprintf(f, " %s", chptr);
	chptr = strchr(chptr, '.');
	if (chptr) {
	    chptr++;
	    if (!strchr(chptr, '.'))
		chptr = NULL;
	}
    } while (chptr);

    fprintf(f, "\n");

    for (i = 0; i < intf->numDns; i++)
	fprintf(f, "nameserver %s\n", inet_ntoa(intf->dnsServers[i]));

    fclose(f);

    /* force a reread of /etc/resolv.conf */
    endhostent();
}

void setupDns(struct intfInfo * intf) {
    char * hn, * dn = NULL;
    struct hostent * he;

    if (intf->set & NETINFO_HAS_DNS) {
	if (!(intf->set & NETINFO_HAS_DOMAIN))  {
	    if (intf->set & NETINFO_HAS_HOSTNAME) {
		hn = intf->hostname;
	    } else {
		he = gethostbyaddr((char *) &intf->ip, sizeof(intf->ip),
				   AF_INET);
		if (he) {
		    hn = he->h_name;
		}
	    }

	    if (hn) {
		dn = strchr(hn, '.');
		if (dn)
		    dn++;
	    }
	} else {
	    dn = intf->domain;
	}

	createResolvConf(intf, dn);
    }
}

static char * doDhcp(int timeout, char * device, int flags, int reqLease,
		     char * reqHostname, struct intfInfo * intf) {
    int s, i;
    struct sockaddr_in serverAddr;
    struct sockaddr_in clientAddr;
    struct sockaddr_in broadcastAddr;
    struct bootpRequest breq, bresp;
    unsigned char * chptr;
    unsigned char messageType;
    time_t startTime = time(NULL);
    int true = 1;
    char optionsRequested[50];
    int numOptions;
    short aShort;

    memset(intf, 0, sizeof(*intf));
    strcpy(intf->device, device);
    intf->reqLease = reqLease;
    intf->set |= INTFINFO_HAS_REQLEASE;

    s = socket(AF_INET, SOCK_DGRAM, 0);
    if (s < 0) {
	return perrorstr("socket");
    }

    if (setsockopt(s, SOL_SOCKET, SO_BROADCAST, &true, sizeof(true))) {
	close(s);
	return perrorstr("setsockopt");
    }

    if (flags & DHCP_FLAG_NOCONFIG) {
	if ((chptr = getInterfaceInfo(intf, s))) {
	    close(s);
 	    return chptr;
	}
    } else if ((chptr = prepareInterface(intf, s))) {
	close(s);
	return chptr;
    }

    if ((chptr = prepareRequest(&breq, s, intf->device, startTime))) {
	close(s);
	disableInterface(intf->device);
	return chptr;
    }


    messageType = DHCP_TYPE_DISCOVER;
    addVendorCode(&breq, DHCP_OPTION_TYPE, 1, &messageType);

    memset(&clientAddr.sin_addr, 0, sizeof(&clientAddr.sin_addr));
    clientAddr.sin_family = AF_INET;
    clientAddr.sin_port = htons(BOOTP_CLIENT_PORT);	/* bootp client */

    if (bind(s, (struct sockaddr *) &clientAddr, sizeof(clientAddr))) {
	disableInterface(intf->device);
	close(s);
	return perrorstr("bind");
    }

    serverAddr.sin_family = AF_INET;
    serverAddr.sin_port = htons(BOOTP_SERVER_PORT);	/* bootp server */

#if 0
    /* seems like a good idea?? */
    if (intf->set & INTFINFO_HAS_BOOTSERVER)
	serverAddr.sin_addr = intf->bootServer;
#endif 

    broadcastAddr.sin_family = AF_INET;
    broadcastAddr.sin_port = htons(BOOTP_SERVER_PORT);

#if 0
    /* this too! */
    if (intf->set & INTFINFO_HAS_BROADCAST)
	broadcastAddr.sin_addr = intf->broadcast;
#endif

    memset(&broadcastAddr.sin_addr, 0xff, 
	   sizeof(broadcastAddr.sin_addr));  /* all 1's broadcast */

    if ((chptr = handleTransaction(s, timeout, &breq, &bresp, &broadcastAddr,
				   NULL, 1, DHCP_TYPE_OFFER))) {
	close(s);
	disableInterface(intf->device);
	return chptr;
    }

    /* Otherwise we're in the land of bootp */
    if (dhcpMessageType(&bresp) == DHCP_TYPE_OFFER) {
	if (getVendorCode(&bresp, DHCP_OPTION_SERVER, &serverAddr.sin_addr)) {
	    close(s);
	    disableInterface(intf->device);
	    return "DHCPOFFER didn't include server address";
	}

	initVendorCodes(&breq);
	messageType = DHCP_TYPE_REQUEST;
	addVendorCode(&breq, DHCP_OPTION_TYPE, 1, &messageType);
	addVendorCode(&breq, DHCP_OPTION_SERVER, 4, &serverAddr.sin_addr);
	addVendorCode(&breq, DHCP_OPTION_REQADDR, 4, &bresp.yiaddr);

	aShort = ntohs(sizeof(struct bootpRequest));
	addVendorCode(&breq, DHCP_OPTION_MAXSIZE, 2, &aShort);

	numOptions = 0;
	optionsRequested[numOptions++] = BOOTP_OPTION_NETMASK;
	optionsRequested[numOptions++] = BOOTP_OPTION_GATEWAY;
	optionsRequested[numOptions++] = BOOTP_OPTION_DNS;
	optionsRequested[numOptions++] = BOOTP_OPTION_DOMAIN;
	optionsRequested[numOptions++] = BOOTP_OPTION_BROADCAST;
	addVendorCode(&breq, DHCP_OPTION_OPTIONREQ, numOptions, 
		      optionsRequested);

	if (reqHostname) {
	    addVendorCode(&breq, BOOTP_OPTION_HOSTNAME, strlen(reqHostname), 
			  reqHostname);
	}

	i = htonl(intf->reqLease);
	addVendorCode(&breq, DHCP_OPTION_LEASE, 4, &i);
	
	/* why do we need to use the broadcast address here? better reread the
	   spec! */
	if ((chptr = handleTransaction(s, timeout, &breq, &bresp, 
				       &broadcastAddr, NULL, 0, 
				       DHCP_TYPE_ACK))) {
	    close(s);
	    disableInterface(intf->device);
	    return chptr;
	}

	parseLease(&bresp, intf);

	if (getVendorCode(&bresp, DHCP_OPTION_SERVER, &intf->bootServer)) {
	    close(s);
	    disableInterface(intf->device);
	    return "DHCPACK didn't include server address";
	}

	intf->set |= INTFINFO_HAS_BOOTSERVER;
    }

    parseReply(&bresp, intf);
    if (flags & DHCP_FLAG_FORCEHNLOOKUP)
	intf->set &= ~(NETINFO_HAS_DOMAIN | NETINFO_HAS_HOSTNAME);

    chptr = setupInterface(intf, s);
    if (chptr) {
	close(s);
	disableInterface(intf->device);
	return chptr;
    }

    syslog(LOG_INFO, "configured interface %s", intf->device);

    if (intf->set & NETINFO_HAS_GATEWAY) {
	chptr = setupDefaultGateway(&intf->gateway, s);
    }
    setupDns(intf);

    close(s);

    return NULL;
}

static void runDaemon(int sock) {
    int conn;
    struct sockaddr_un addr;
    size_t addrLength;
    struct command cmd;
    struct intfInfo intf[20];
    int numInterfaces = 0;
    int i;
    int closest;
    struct timeval tv;
    fd_set fds;

    while (1) {
	FD_ZERO(&fds);
	FD_SET(sock, &fds);

	tv.tv_sec = tv.tv_usec = 0;
	closest = -1;
	if (numInterfaces) {
	    for (i = 0; i < numInterfaces; i++)
		if ((intf[i].set & INTFINFO_HAS_LEASE) && 
			(closest == -1 || 
			       (intf[closest].renewAt > intf[i].renewAt)))
		    closest = i;
	    if (closest != -1) {
		tv.tv_sec = intf[closest].renewAt - time(NULL);
		if (tv.tv_sec <= 0) {
		    dhcpRenew(intf + closest);
		    continue;
		}
	    }
	}

	if (select(sock + 1, &fds, NULL, NULL, 
		   closest != -1 ? &tv : NULL) > 0) {
	    conn = accept(sock, &addr, &addrLength);

	    if (read(conn, &cmd, sizeof(cmd)) != sizeof(cmd)) {
		close(conn);
		continue;
	    }

	    switch (cmd.type) {
	      case CMD_DIE:
		syslog(LOG_INFO, "terminating");
		for (i = 0; i < numInterfaces; i++)
		    dhcpRelease(intf + i);

		cmd.type = CMD_RESULT;
		cmd.u.result = 0;
		write(conn, &cmd, sizeof(cmd));
		exit(0);

	      case CMD_STARTIFACE:
		if (doDhcp(cmd.u.start.timeout, cmd.u.start.device,
			   cmd.u.start.flags, cmd.u.start.reqLease, 
			   cmd.u.start.reqHostname ? 
			       cmd.u.start.reqHostname : NULL,
			   intf + numInterfaces)) {
		    cmd.u.result = 1;
		} else {
		    cmd.u.result = 0;
		    numInterfaces++;
		}
		break;

	      case CMD_FORCERENEW:
		for (i = 0; i < numInterfaces; i++)
		    if (!strcmp(intf[i].device, cmd.u.renew.device)) break;
		if (i == numInterfaces)
		    cmd.u.result = RESULT_UNKNOWNIFACE;
		else
		    cmd.u.result = dhcpRenew(intf + i);
		break;

	      case CMD_STOPIFACE:
		for (i = 0; i < numInterfaces; i++)
		    if (!strcmp(intf[i].device, cmd.u.stop.device)) break;
		if (i == numInterfaces)
		    cmd.u.result = RESULT_UNKNOWNIFACE;
		else {
		    cmd.u.result = dhcpRelease(intf + i);
		    if (numInterfaces == 1) {
			cmd.type = CMD_RESULT;
			write(conn, &cmd, sizeof(cmd));
			exit(0);
		    }

		    intf[i] = intf[numInterfaces - 1];
		    numInterfaces--;
		}
		break;

	      case CMD_REQSTATUS:
		for (i = 0; i < numInterfaces; i++)
		    if (!strcmp(intf[i].device, cmd.u.stop.device)) break;
		if (i == numInterfaces) {
		    cmd.u.result = RESULT_UNKNOWNIFACE;
		} else {
		    cmd.type = CMD_STATUS;
		    cmd.u.status.intf = intf[i];
		    if (intf[i].set & NETINFO_HAS_HOSTNAME)
			strncpy(cmd.u.status.hostname,
			    intf->hostname, sizeof(cmd.u.status.hostname));
		    cmd.u.status.hostname[sizeof(cmd.u.status.hostname)] = '\0';

		    if (intf[i].set & NETINFO_HAS_DOMAIN)
			strncpy(cmd.u.status.domain,
			    intf->domain, sizeof(cmd.u.status.domain));
		    cmd.u.status.domain[sizeof(cmd.u.status.domain)] = '\0';

		    if (intf[i].set & INTFINFO_HAS_BOOTFILE)
			strncpy(cmd.u.status.bootFile,
			    intf->bootFile, sizeof(cmd.u.status.bootFile));
		    cmd.u.status.bootFile[sizeof(cmd.u.status.bootFile)] = '\0';
		}

	      case CMD_STATUS:
	      case CMD_RESULT:
		/* can't happen */
		break;
	    }

	    if (cmd.type != CMD_STATUS) cmd.type = CMD_RESULT;
	    write(conn, &cmd, sizeof(cmd));

	    close(conn);
	}
    }

    exit(0);
}

static int openControlSocket(void) {
    struct sockaddr_un addr;
    int sock;
    size_t addrLength;
    pid_t child;
    int status;

    if ((sock = socket(PF_UNIX, SOCK_STREAM, 0)) < 0)
	return -1;

    addr.sun_family = AF_UNIX;
    strcpy(addr.sun_path, CONTROLSOCKET);
    addrLength = sizeof(addr.sun_family) + strlen(addr.sun_path);

    if (!connect(sock, (struct sockaddr *) &addr, addrLength)) 
	return sock;

    if (errno != ENOENT && errno != ECONNREFUSED) {
	fprintf(stderr, "failed to connect to %s: %s\n", CONTROLSOCKET,
		strerror(errno));
	close(sock);
	return -1;
    }

    if (!(child = fork())) {
			close(sock);

			if ((sock = socket(PF_UNIX, SOCK_STREAM, 0)) < 0) {
	    	fprintf(stderr, "failed to create socket: %s\n", strerror(errno));
	    	exit(1);
			}

			unlink(CONTROLSOCKET);
			umask(077);
			if (bind(sock, (struct sockaddr *) &addr, addrLength)) {
	    	fprintf(stderr, "bind to %s failed: %s\n", CONTROLSOCKET,
		    	strerror(errno));
	    	exit(1);
			}
			umask(033);

			listen(sock, 5);

			if (fork()) _exit(0);

			close(0);
			close(1);
			close(2);

			openlog("pumpd", LOG_PID, LOG_DAEMON);
			{
	    	time_t t;

	    	t = time(NULL);
	    	syslog(LOG_INFO, "starting at %s\n", ctime(&t));
			}

			runDaemon(sock);
   }

   waitpid(child, &status, 0);
   if (!WIFEXITED(status) || WEXITSTATUS(status))
	 return -1;

   if (!connect(sock, (struct sockaddr *) &addr, addrLength))
	 return sock;

   fprintf(stderr, "failed to connect to %s: %s\n", CONTROLSOCKET,
	    strerror(errno));

   return 0;
}

void printStatus(struct intfInfo i, char * hostname, char * domain,
		 char * bootFile) {
    int j;

    printf("Device %s\n", i.device);
    printf("\tIP: %s\n", inet_ntoa(i.ip));
    printf("\tNetmask: %s\n", inet_ntoa(i.netmask));
    printf("\tBroadcast: %s\n", inet_ntoa(i.broadcast));
    printf("\tNetwork: %s\n", inet_ntoa(i.network));
    printf("\tBoot server %s\n", inet_ntoa(i.bootServer));

    if (i.set & NETINFO_HAS_GATEWAY)
	printf("\tGateway: %s\n", inet_ntoa(i.gateway));

    if (i.set & INTFINFO_HAS_BOOTFILE)
	printf("\tBoot file: %s\n", bootFile);

    if (i.set & NETINFO_HAS_HOSTNAME)
	printf("\tHostname: %s\n", hostname);

    if (i.set & NETINFO_HAS_DOMAIN)
	printf("\tDomain: %s\n", domain);

    if (i.numDns) {
	printf("\tNameservers:");
	for (j = 0; j < i.numDns; j++)
	    printf(" %s", inet_ntoa(i.dnsServers[j]));
	printf("\n");
    }

    if (i.set & INTFINFO_HAS_LEASE) {
	printf("\tRenewal time: %s", ctime(&i.renewAt)); 
	printf("\tExpiration time: %s", ctime(&i.leaseExpiration)); 
    }
}

int startPump (struct intfInfo* result) {
    char * device = "eth0";
    char * hostname = "hostname";	/*It has been reported that some servers need something here */

    int rc;
    int test = 0;
    int flags = 0;
    int lease = 24;
    int killDaemon = 0;
    int release = 0, renew = 0, status = 0, lookupHostname = 0;
    struct command cmd, response;
    int cont;


    if (geteuid()) {
	fprintf(stderr, _("%s: must be run as root\n"), PROGNAME);
	return 1;
    }

    if (test)
	flags = DHCP_FLAG_NODAEMON | DHCP_FLAG_NOCONFIG;
    if (lookupHostname)
	flags |= DHCP_FLAG_FORCEHNLOOKUP;

    cont = openControlSocket();
    if (cont < 0) 
	return 1;

    if (killDaemon) {
	cmd.type = CMD_DIE;
    } else if (status) {
	cmd.type = CMD_REQSTATUS;
	strcpy(cmd.u.reqstatus.device, device);
    } else if (renew) {
	cmd.type = CMD_FORCERENEW;
	strcpy(cmd.u.renew.device, device);
    } else if (release) {
	cmd.type = CMD_STOPIFACE;
	strcpy(cmd.u.stop.device, device);
    } else {
	cmd.type = CMD_STARTIFACE;
	strcpy(cmd.u.start.device, device);
	cmd.u.start.timeout = 3;
	cmd.u.start.flags = flags;
	cmd.u.start.reqLease = lease * 60 * 60;
	strcpy(cmd.u.start.reqHostname, hostname);
    }

    write(cont, &cmd, sizeof(cmd));
    read(cont, &response, sizeof(response));

    if (response.type == CMD_RESULT && response.u.result &&
	    cmd.type == CMD_STARTIFACE) {
				cont = openControlSocket();
				if (cont < 0)
	    		return 1;
				write(cont, &cmd, sizeof(cmd));
				read(cont, &response, sizeof(response));
    }

    if (response.type == CMD_RESULT) {
			if (response.u.result) {
	    	fprintf(stderr, "Operation failed.\n");
	    	return 1;
			}
    } else if (response.type == CMD_STATUS) {
			printStatus(response.u.status.intf, response.u.status.hostname,
		    response.u.status.domain, response.u.status.bootFile);
    }

    /*Get status info for knetstart */
    memset(&cmd, 0, sizeof(cmd));
    cmd.type = CMD_REQSTATUS;
    strcpy(cmd.u.reqstatus.device, device);

    cont = openControlSocket();
    write(cont, &cmd, sizeof(cmd));
    read(cont, &response, sizeof(response));

    if (response.type == CMD_RESULT) {
			if (response.u.result) {
	    	fprintf(stderr, "Operation failed. Could not get status.\n");
	    	return 1;
			}
    } else if (response.type == CMD_STATUS) {
    	*result = response.u.status.intf;
    }

    return 0;
}
