/**************************************************************************
 * $Id: script.cpp 1.2 Thu, 18 Feb 1999 16:48:37 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "script.h"          /* all prototypes and structures */
#include "scriptdlg.h"
#include <qmsgbox.h>
#include <kapp.h>

#include "script.moc"
//#undef None // !!!!!!!! X11/X.h defined !!!!!!!!!

#define TASK_SWITCH 1000

void SamplinScript::setMsgwin(Edit *ptr)
{
   msgwin=ptr;
   if(ptr!=NULL)ptr->clear();
   
}
void SamplinScript::yyerror(char *msg)
{
   int n;
   char c=' ';
   
   sprintf(string,"%s at %n",msg,&n);
   if (*(scanner->yytext)=='\n'){
      strcat(string,"end of line");
   }
   else
     sprintf(string+n,"'%s' ",scanner->yytext);
   error(ERROR,string);
   
   return;
}

int SamplinScript::yylex()
{
   yylineno=scanner->yylineno;

   return scanner->yylex(&yylval);
}

SamplinScript::SamplinScript(QWidget *parent=0):QObject(parent)
{
   showmsg=FALSE;
   multiflag=TRUE;
   quitflag=FALSE;
   waitMulti=NULL;
   delaylen=0;
   msgwin=NULL;
   scanner=NULL;

//   graphs=NULL;
   program_state=FINISHED;
   infolevel=WARNING;
   timer = new QTimer(this);

   string=(char *)malloc(sizeof(char)*1000);
   symroot=NULL;
   stackroot=NULL;
   cmdroot=NULL;

   pd = new QProgressDialog(NULL);
//   pd->setCaption("S");
   connect(pd,SIGNAL(cancelled()),this, SLOT(progressSlot()));
   
   connect(timer,SIGNAL(timeout()),this,SLOT(stepHandler()));
   //connect(timer,SIGNAL(timeout()),this,SLOT(crashHandler()));
   
   dialogs.setAutoDelete(TRUE);

   devices = new QList<SamplinDevice>;
   devices->setAutoDelete( TRUE );
   graphs.setAutoDelete( FALSE );
   graphs.clear();


}

SamplinScript::~SamplinScript()
{
//   debug("script 1");
   closeDevices();
//   debug("script 2");
   free(string);   
//   debug("script 3");
   freeAlloc();
//   debug("script 4");
   delete devices;
//   debug("script 5");
   graphs.clear();
//   debug("script 6");
   delete pd;
   //   if((quitflag==TRUE) && (graphs!=NULL))delete graphs;
//   debug("script 7");
}

int SamplinScript::compile(const char *str)
{
   FILE *fp;
   int ret;

   script_file=str;
   
   fp=fopen(script_file,"r");
   
   if(fp==NULL){
      QMessageBox::warning(NULL,"Error","Can't open script file "+script_file);
      return(-1);
   }
   
   setProgramState(HATCHED);
    
   interactive=FALSE;
   scanner=new SamplinScanner(this);
   scanner->switch_to_my_file(fp);

      
   initialize();
   setProgramState(INITIALIZED);

   error(DIAGNOSTIC,"Stacks and lists Initialized"); 
   error(NOTE,"Calling parser/compiler");
   setProgramState(COMPILING);

   yyparse();

//   printf("lines=%d\n",yylineno);
   
   create_myend();
   
   sprintf(string,"Read %d line(s)",yylineno);
   error(NOTE,string);
   sprintf(string,"Generated %d command(s)",commandcount);
   error(NOTE,string);

   if (errorlevel>ERROR) {
      error(DIAGNOSTIC,"Program compiled");
      ret=0;
      current=cmdroot;
      setProgramState(COMPILED);
   }   
   else {
      setProgramState(FINISHED);
      error(ERROR,"Program not compiled");
      ret=-1;
      //delete this;
   }

   delete scanner;
   fclose(fp);
   
   return(ret);
}


void SamplinScript::go(int line)
{
   if(program_state==STOPPED || program_state==COMPILED){
      
      goto_line=line;
//      printf("goto line:%i\n",goto_line);
//      printf("current line:%i\n",current->line);
      last_line=current->line;
      step_flag=TRUE;
      timer->start(0,FALSE);
      setProgramState(RUNNING);   
   }

}
/*
void SamplinScript::step()
{
   int status;

   if(program_state==STOPPED){
      setProgramState(RUNNING);
      status=doStep();
      widgetAction();
      timeHandler();   
      setProgramState(STOPPED);
      if(status)
	reset();
   }
}
*/
void SamplinScript::reset()
{
   if(program_state!=FINISHED){
      timer->stop();
      freeAlloc();
      closeDevices();
      sprintf(string,"Program killed by user.");
      error(ERROR,string);
      setProgramState(FINISHED);
   }
}

void SamplinScript::stop()
{
   if(program_state==RUNNING){
      timer->stop();
      sprintf(string,"Program interrupted by user.");
//      error(ERROR,string);
      
      setProgramState(STOPPED);
   }
}

void SamplinScript::stepHandler()
{
   int status=0;

   task_cnt=TASK_SWITCH;

   if(delaylen>0)return;
   
   timer->stop();

   while(status==0 && (task_cnt>0 || goto_line==0) && (goto_line==-1 ||
		(goto_line>0 && (goto_line!=current->line /*|| step_flag*/) ) ||
		(goto_line==0 && step_flag)  )){
//      printf("-> at line:%i, goto_line:%i\n",current->line, goto_line);
      status=doStep();
      widgetAction();
      timeHandler();
      --task_cnt;
      if(step_flag && last_line!=current->line)step_flag=FALSE;
   }
//   printf("at line:%i, taskcnt:%i goto_line:%i\n",current->line,task_cnt, goto_line);
   if(status==0 && (goto_line==-1 || (goto_line>0 && goto_line!=current->line))){
      timer->start(0,FALSE);
   }
   if(status==0 && (goto_line==0 || (goto_line>0 && goto_line==current->line))){
      setProgramState(STOPPED);
//      printf("after line:%i\n",current->line);
   }
     
   if(status!=0){
      freeAlloc();
      closeDevices();
      emit finishedScript();
      if(quitflag==TRUE){
	 kapp->quit();
      }
   }
   //   else delete this;
   
}

int SamplinScript::doStep()
/* execute 1 instruction of the compiled code */
{
   int ret=0;
   
   if(current==cmdhead){/*delete this;*/return(2);}
   
   switch(current->type) {
    case GOTO:case QGOTO:case GOSUB:case QGOSUB:case IGOSUB:
      jump(current); DONE;
    case SKIPPER: 
      skipper(); break;
    case LABEL:case DATA:case NOP: 
      DONE;
    case RETURN:
      myreturn();
      break;
    case PUSHDBLSYM: 
      pushdblsym(current); DONE;
    case PUSHDBL:
      pushdbl(current); DONE;
    case POPDBLSYM:
      popdblsym(current); DONE;
    case POPSTRSYM:
      popstrsym(current); DONE;
    case PUSHSTRSYM: 
      pushstrsym(current); DONE;
    case PUSHSTR:
      pushstr(current); DONE;
    case CONCAT:
      concat(); DONE;
    case PRINT:
      print(current); DONE;
    case MESSAGE:
      message(current); DONE;
    case MYOPEN:
      myopen(current); DONE;
    case MYCLOSE:
      myclose(current); DONE;
    case MYSWITCH:
      myswitch(current); DONE;
    case DEVOPEN:
      devopen(current); DONE;
    case DEVCLOSE:
      devclose(current); DONE;      
    case DEVWR:
      devwr(current); DONE;
    case DEVWRB:
      devwrb(current); DONE;      
    case DEVRD:
      devrd(current); DONE;       
    case DEVRDB:
      devrdb(current); DONE;       
    case DEVCTL:
      devctl(current); DONE;             
    case GRAPHFNC:
      graphfnc(current); DONE;       
    case PLOTFNC:
      plotfnc(current); DONE;             
    case ADDPLOT:
      addplot(current); DONE;       
    case UPDATEPLOT:
      updateplot(current); DONE;       
    case DELPLOT:
      delplot(current); DONE;             
    case ADDGRAPH:
      addgraph(current); DONE;       
    case LOADGRAPH:
      loadgraph(current); DONE;             
    case SAVEGRAPH:
      savegraph(current); DONE;             
    case EXPORTGRAPH:
      exportgraph(current); DONE;             
    case PRINTGRAPH:
      printgraph(current); DONE;             
    case DELGRAPH:
      delgraph(current); DONE;             
    case TITLEGRAPH:
      titlegraph(current); DONE;             
    case ADDDLG:
      adddlg(current); DONE;       
    case PROGRESSDLG:
      progressdlg(current); DONE;       
    case DELDLG:
      deldlg(current); DONE;             
    case HIDEDLG:
      hidedlg(current); DONE;             
    case SHOWDLG:
      showdlg(current); DONE;       
    case ADDWIDGET:
      addwidget(current); DONE;             
    case SLEEP:
      mysleep(); DONE;
    case TIMER:
      mytimer(current); DONE;
    case MULTI:
      multi(current);
      break;
    case CURSOR:
      cursor(current); DONE;
    case MYREAD:
      myread(current); DONE;
    case RESTORE:case QRESTORE:
      restore(current); DONE;
    case READDATA:
      readdata(current); DONE;
    case PROMPT:
      prompt(current); DONE;
    case DBLADD:case DBLMIN:case DBLMUL:case DBLDIV:case DBLPOW:
    case DBLAND:case DBLOR:case DBLXOR:
      dblbin(current); DONE;
    case NEGATE:
      negate(); DONE;
    case EQ:case NE:case GT:case GE:case LT:case LE:
      dblrelop(current); DONE;
    case STREQ:case STRNE:
      strrelop(current); DONE;
    case AND:case OR:case NOT:
      boole(current); DONE;
    case FUNCTION:
      function(current); DONE;
    case DOARRAY:
      doarray(current); DONE;
    case DIM:
      dim(current); DONE;
    case DECIDE:
      decide(); DONE;
    case WAIT:
      mywait(); DONE;
    case BELL:
      bell(); DONE;
    case SETINFOLEVEL:
      setinfolevel(); DONE;
    case END:
      ret=1; myend(); break;
    default:
      sprintf(string,"Unkown Interpreter-Command, Token %d.",current->type);
      error(ERROR,string);
      ret=-2;
      break;
   }

   if(errorlevel<=ERROR)ret=-1;
   
   if(ret!=0){
      setProgramState(FINISHED);
      switch(errorlevel) {
       case NOTE:case DIAGNOSTIC: 
	 error(NOTE,"Program ended normally."); break;
       case WARNING:
	 error(WARNING,"Program ended with a warning"); break;
       case ERROR:
	 error(ERROR,"Program stopped due to an error"); break;
       case FATAL: /* should not come here ... */
	 error(FATAL,"Program terminated due to FATAL error");
	 break;
      }
   }
   
   return(ret);
}

/* ------------- subroutines ---------------- */

struct symbol * SamplinScript::get_sym(const char *name,int type,int add) 
/* gets the value of a symbol, or creates it with initial value type */
{
  struct symbol *curr,*neu;

  curr=symroot;
  while(curr!=symhead) {   /* search 'til head of list */
    if (curr->type==type  /* do the types match ? */
        &&!strcmp(name,curr->name))  /* do the names match ? */
      return curr; /* give back address */
    curr=curr->next; /* try next entry */
  }
  /* run (ppp) through all variables. */
  if (!add) return NULL;  /* dont create, if not found */
  /* end of list, create new element */
  neu=(struct symbol *)my_malloc(sizeof(struct symbol)); /* create new item */
   symhead=neu;  /* advance head */
   curr->name=my_strdup(name);  /* store new name */
   curr->next=neu;
   curr->value=0.0;
   curr->pointer=NULL;
   curr->type=type;
   curr->value=0.0;
   curr->args=NULL;
   if (type==STRING) {   /* create empty string */
    curr->pointer=my_malloc(sizeof(char));
    *(char *)(curr->pointer)='\0';
  }
  return curr;
}
    
    
void SamplinScript::create_pushdbl(double value) /* create command 'pushdbl' */
{
  struct command *cmd;
  
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Creating command 'pushdbl', value=%g",value);
    error(DIAGNOSTIC,string);
  }
  cmd=add_command(PUSHDBL);
  if (cmd->pointer==NULL) cmd->pointer=my_malloc(sizeof(double));
  *(double *)(cmd->pointer)=value;
}


void SamplinScript::pushdbl(struct command *cmd) 
{
  /* push double onto stack */
  struct stackentry *p;

  p=push();
  p->value= *(double *)cmd->pointer;
  p->type=NUMBER;
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Pushing number %g",p->value);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_pushdblsym(char *symbol) /* create command 'pushdblsym' */
{
  struct command *cmd;

  sprintf(string,"Creating command 'pushdblsym' from symbol '%s'",symbol);
  error(DIAGNOSTIC,string);
  cmd=add_command(PUSHDBLSYM);
  /* get room to store specific information */
  cmd->spointer= &(get_sym(symbol,NUMBER,TRUE)->value);
}


void SamplinScript::pushdblsym(struct command *cmd) 
{
  /* push double symbol onto stack */
  struct stackentry *p;
  
  p=push();
  p->value= *(double *)cmd->spointer;
  p->type=NUMBER;
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Pushing symbol %g",*(double *)cmd->spointer);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_popdblsym(char *symbol) /* create command 'popdblsym' */
{
  struct command *cmd;
  struct symbol *s;

  sprintf(string,"Creating command 'popdblsym' to symbol '%s'",symbol);
  error(DIAGNOSTIC,string);
  cmd=add_command(POPDBLSYM);
  /* storing specific information: */
  s=get_sym(symbol,NUMBER,TRUE);
  cmd->spointer= &(s->value);
  
  /* treat internal vars */
  if (!strncmp(symbol,"yab",3)) {
    if (!strcmp(symbol,"yabinfolevel")) {
      error(DIAGNOSTIC,"Creating command 'set infolevel'");
      cmd=add_command(SETINFOLEVEL);
    }
  }
}


void SamplinScript::setinfolevel(void)
{
  /* set infolevel to content of variable infolevel */
  int i;
  static char *levels[]={"FATAL","ERROR","WARNING","NOTE","DIAGNOSTIC"};
  
  i=(int)get_sym("yabinfolevel",NUMBER,FALSE)->value;
  
  if (i!=DIAGNOSTIC && i!=NOTE && i!=WARNING && 
      i!=ERROR && i!=FATAL) return;
  
  if (infolevel<i) infolevel=i;
  if (infolevel<=DIAGNOSTIC) {
    sprintf(string,"setting infolevel to %s",levels[i-FATAL]);
    error(DIAGNOSTIC,string);
  }
  infolevel=i;
}


void SamplinScript::popdblsym(struct command *cmd) 
{
  /* pop double from stack */
  double d;

  d=pop()->value;
  *(double *)(cmd->spointer)=d;
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Popping to symbol; value=%g",d);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_dblbin(char c) /* create command for binary double operation */
{
  sprintf(string,"Creating command 'dblbin %c'",c);
  error(DIAGNOSTIC,string);
  switch(c) {
  case '+':add_command(DBLADD);break;
  case '-':add_command(DBLMIN);break;
  case '*':add_command(DBLMUL);break;
  case '/':add_command(DBLDIV);break;
  case '^':add_command(DBLPOW);break;

  case '&':add_command(DBLAND);break;
  case '|':add_command(DBLOR);break;
  case '~':add_command(DBLXOR);break;     
  
  }
  /* no specific information needed */
}


void SamplinScript::dblbin(struct command *cmd) /* compute with two numbers from stack */
{
  struct stackentry *d;
  double a,b,c;

  b=pop()->value;
  a=pop()->value;
  d=push();
  switch(cmd->type) {
   case(DBLADD):c=a+b; break;
   case(DBLMIN):c=a-b; break;
   case(DBLMUL):c=a*b; break;

   case(DBLAND):
     c=((long)a)&((long)b);
     break;
   case (DBLOR):
     c=((long)a)|((long)b);
     break;
   case (DBLXOR):
     c=((long)a)^((long)b);
     break;
     
   case(DBLDIV): 
    if (fabs(b)<DBL_MIN) {
      sprintf(string,"Division by zero, set to %g",DBL_MAX);
      error(WARNING,string);
      c=DBL_MAX;}
    else
      c=a/b;
    break;
  case(DBLPOW):
    if (b==2) 
      c=a*a;
    else if (a<0) {
      error(ERROR,"Power of negative value. Don't now what to do");
      return;}
    else
      c=exp(b*log(a));
    break;
  }
  d->value=c;
  d->type=NUMBER;
}


void SamplinScript::create_negate() /* creates command negate */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'negate'");
  cmd=add_command(NEGATE);
}


void SamplinScript::negate() /* negates top of stack */
{
  struct stackentry *a,*b;
  double d;

  a=pop();
  d=a->value;
  b=push();
  b->type=NUMBER;
  b->value= -d;
}


void SamplinScript::create_pushstrsym(char *symbol) /* push string-symbol onto stack */
{
  struct command *cmd;

  sprintf(string,"Creating command 'pushstrsym' from symbol '%s'",symbol);
  error(DIAGNOSTIC,string);
  cmd=add_command(PUSHSTRSYM);
  /* get room to store specific information */
  cmd->spointer=&get_sym(symbol,STRING,TRUE)->pointer;
}


void SamplinScript::pushstrsym(struct command *cmd) 
{
  /* push string-symbol onto stack */
  struct stackentry *p;
  
  p=push();
  p->pointer=my_strdup(*(char **)cmd->spointer);
  p->type=STRING;
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Pushing from string-symbol; value='%s'",
            (char *)p->pointer);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_popstrsym(char *symbol) /* create command 'popstrsym' */
{
  struct command *cmd;
  struct symbol *s;

  sprintf(string,"Creating command 'popstrsym' to symbol '%s'",symbol);
  error(DIAGNOSTIC,string);
  cmd=add_command(POPSTRSYM);
  /* storing specific information: */
  s=get_sym(symbol,STRING,TRUE);
  cmd->spointer=(char **)&(s->pointer);
}


void SamplinScript::popstrsym(struct command *cmd)    /* pop string from stack */
{
  struct stackentry *p;

  p=pop();
  if (*(char **)cmd->spointer!=NULL) free(*(char **)cmd->spointer);
  *(char **)cmd->spointer=my_strdup((char *)p->pointer);
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Popping to string-symbol; value='%s'"
            ,(char *)p->pointer);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_concat() /* creates command concat */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'concat'");
  cmd=add_command(CONCAT);
}


void SamplinScript::concat() /* concetenates two strings from stack */
{
  struct stackentry *a,*b,*c;
  char *aa,*bb,*cc;

  a=pop();
  b=pop();
  if (a->type!=STRING || b->type!=STRING) {
    error(FATAL,"Need strings to concat");
    return;
  }
  aa=(char *)a->pointer;
  bb=(char *)b->pointer;
  cc=(char *) my_malloc(sizeof(char)*(strlen(aa)+strlen(bb)+1));
  strcpy(cc,bb);
  strcat(cc,aa);
  c=push();
  c->type=STRING;
  c->pointer=cc;
}  


void SamplinScript::create_pushstr(char *s) /* creates command pushstr */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'pushstr'");
  cmd=add_command(PUSHSTR);
  cmd->pointer=my_strdup(s); /* store string */
}


void SamplinScript::pushstr(struct command *cmd) 
{
  /* push string onto stack */
  struct stackentry *p;
  
  p=push();
  p->pointer=my_strdup((char *)cmd->pointer);
  p->type=STRING;
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Pushing string '%s'",(char *)p->pointer);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_goto(char *label) /* creates command goto */
{
  struct command *cmd;

  sprintf(string,"Creating command 'goto %s'",label);
  error(DIAGNOSTIC,string);
  cmd=add_command(GOTO);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void SamplinScript::create_gosub(char *label) /* creates command gosub */
{
  struct command *cmd;

  sprintf(string,"Creating command 'gosub %s'",label);
  error(DIAGNOSTIC,string);
  cmd=add_command(GOSUB);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void SamplinScript::jump(struct command *cmd) 
/* jump to specific Label; used as goto or gosub */
{
  struct command *curr;
  struct stackentry *ret;
  int type;

  type=cmd->type;
  if (type==QGOSUB || type==GOSUB) {
    ret=push();
    ret->pointer=current;
    ret->type=RETADD;
  }
  if ( type==IGOSUB) {
     ret=push();
     ret->pointer=current;
     ret->wpointer=cmd->spointer;
     ret->type=IRETADD;
  }   
  if (type==QGOSUB || type==QGOTO) {
    current=(struct command *)cmd->spointer;
    error(DIAGNOSTIC,"Performing quick jump");
    return;
  }
  curr=cmdroot;
  sprintf(string,"Searching Label '%s'",(char *)cmd->pointer);
  error(DIAGNOSTIC,string);
  while(curr!=cmdhead) {   /* go through all commands */
    if (curr->type==LABEL && !strcmp((char *)curr->pointer,(char *)cmd->pointer)) {
      /* found right Label ! */
      current=curr; /* jump to new location */
      sprintf(string,"Resolved label '%s', performing jump",
              (char *)cmd->pointer);
       /* use the address instead of the name next time ! */
      cmd->spointer=curr;
      cmd->type=(type==GOTO) ? QGOTO:QGOSUB; /* quick jump from now on */
      error(DIAGNOSTIC,string);
       return;
    }
    curr=curr->next;
  }
  /* label not found */
  sprintf(string,"Cant find label '%s'",(char *)cmd->pointer);
  error(ERROR,string);
}


void SamplinScript::create_return() /* creates command return */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'return'");
  cmd=add_command(RETURN);
}


void SamplinScript::myreturn() /* return from gosub */
{
   struct stackentry *address;
   widgetitem *w;
   
   address=pop();
   if (address->type!=RETADD && address->type!=IRETADD) {
      error(ERROR,"RETURN without GOSUB");
      return;
   }
   current=(struct command *)address->pointer;
   error(DIAGNOSTIC,"Returning from subroutine");
   if(address->type==RETADD){
      current=current->next;

   }
   if(address->type==IRETADD){
      w=(widgetitem *)address->wpointer;
      if(w!=NULL)
	w->active=FALSE;      
   }
   return;
}


void SamplinScript::create_label(char *label) /* creates command label */
{
  struct command *cmd;

  sprintf(string,"Creating command 'label %s'",label);
  error(DIAGNOSTIC,string);
  cmd=add_command(LABEL);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void SamplinScript::create_skipper() /* creating command skipper */
{
  error(DIAGNOSTIC,"Creating command 'skipper'");
  add_command(SKIPPER);
}


void SamplinScript::skipper()
/* used for on_goto/gosub, skip specified number of commands */
{
  int i,n;

  n=(int)pop()->value;
  i=1;
  current=current->next;
  while(i<n && (current->next)->type!=NOP) { /* go through all labels */
    current=current->next; /* next label */
    i++;
  }
  if (infolevel>=DIAGNOSTIC) {
    sprintf(string,"skipped %d labels",i);
    error(DIAGNOSTIC,string);
  }
}


void SamplinScript::create_nop() /* does nothing */
{
  error(DIAGNOSTIC,"Creating command 'nop'");
  add_command(NOP);
  /* thats all folks !*/
}


void SamplinScript::create_myend() /* create command 'end' */
{
  error(DIAGNOSTIC,"Creating command 'end'");
  add_command(END);
}


void SamplinScript::myend() /* is called at the end of program execution */
{
//debug("at end\n");

}


void SamplinScript::create_print(char type) /* create command 'print' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'print'");
  cmd=add_command(PRINT);
  cmd->pointer=my_malloc(sizeof(int));
  /* store type of print  */
  cmd->tag=type;
}


void SamplinScript::print(struct command *cmd) /* print on screen */
{
  int type;
  struct stackentry *p;
   FILE *str;
   char *buf;
   int line,col;


   type=cmd->tag;
   switch(type) {
    case 'n': 
      sprintf(string,"\n");  /* print newline */
      buf=string;
      break;
    case 'd': 
      p=pop();
      if (last_print=='d') sprintf(string," %g",p->value);   /* print double value */
      else sprintf(string,"%g",p->value);
      buf=string;
      break;
    case 's': 
      p=pop();
      buf=(char *)p->pointer;
      break;
   }
   
   if(currentstream!=NULL){
      fprintf(currentstream,"%s",buf);
      fflush(currentstream);
   }
   else {
      if(msgwin!=NULL){
	 if(showmsg==FALSE){
	 //   msgwin->show();
	    showmsg=TRUE;
	 }
//	 msgwin->getCursorPosition(&line,&col);
	 msgwin->insertAtEnd(buf);
      }
      else {
	 fprintf(stdout,"%s",buf);
	 fflush(stdout);
      }
   }
   
   last_print=type;
}

char * SamplinScript::replace(char *string) /* replace \n,\a, etc. */
{
  char *from,*to;

  from=to=string;
  while(*from) {
    if (*from=='\\') {
      from++;
      switch(*from) {
      case 'n':	*to='\n';break;
      case 't':	*to='\t';break;
      case 'v':	*to='\v';break;
      case 'b':	*to='\b';break;
      case 'r':	*to='\r';break;
      case 'f':	*to='\f';break;
      case 'a':	*to='\a';break;
      case '\\': *to='\\';break;
      case '\?': *to='\?';break;
      case '\'': *to='\'';break;
      case '\"': *to='\"';break;
      default:
	*to='\\';
	to++;
	*to=*from;
      }
    }
    else
      *to=*from;
    from++;
    to++;
  }
  *to='\0';
  return string;
}


void SamplinScript::create_myopen(double stream,char *mode) /* create command 'myopen' */
{
  struct command *cmd;

  sprintf(string,"Creating command 'myopen' with mode '%s'",mode);
  error(DIAGNOSTIC,string);
  if (badstream((int)stream)) return;
  cmd=add_command(MYOPEN);
  cmd->args=(int) stream;
  cmd->pointer=my_strdup(mode);
}


void SamplinScript::myopen(struct command *cmd) /* open specified file for given name */
{
  FILE *handle;
  int stream;
  char *name;
  char *mode;
  
  mode=(char *)cmd->pointer;
  name=(char *)(pop()->pointer);
  stream=cmd->args;
  if (streams[stream]!=NULL) {
    error(ERROR,"Stream already in use");
    return;
  }
  handle=fopen(name,mode);
  if (handle==NULL) {
    sprintf(string,"Could not open '%s'",name);
    error(ERROR,string);
    return;
  }
  if (infolevel>=DIAGNOSTIC) {
    sprintf(string,"Opened '%s' with mode '%s'",name,mode);
    error(DIAGNOSTIC,string);
  }
  streams[stream]=handle;
}


void SamplinScript::create_myclose(double stream) /* create command 'myclose' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'myclose'");
  if (badstream((int)stream)) return;
  cmd=add_command(MYCLOSE);
  cmd->args=(int) stream;
  return;
}


void SamplinScript::myclose(struct command *cmd) /* close the specified stream */
{
  int s;
  
  s=cmd->args;
  if (streams[s]==NULL) {
    sprintf(string,"Stream %d already closed",s);
    error(WARNING,string);
    return;
  }
  fclose(streams[s]);
  streams[s]=NULL;
}

void SamplinScript::create_myswitch(double stream) /* create command myswitch */
{
  struct command *cmd;

  if (stream!=0.0 && badstream((int)stream)) return;
  error(DIAGNOSTIC,"Creating command 'myswitch'");
  cmd=add_command(MYSWITCH);
  cmd->args=(int) stream;
}


void SamplinScript::myswitch(struct command *cmd) /* switch to specified stream */
{
  int stream;

  stream=cmd->args;
  if (stream==0) 
    currentstream=NULL;
  else  {
    currentstream=streams[stream]; /* switch to stream */
    if (streams[stream]==NULL) {
      sprintf(string,"Stream %d not opened",stream);
      error(ERROR,string);
      return;
    } 
  }
  if (infolevel>=DIAGNOSTIC) {
    sprintf(string,"Switching to stream %d",stream);
    error(DIAGNOSTIC,string);
  }
  return;
}


int SamplinScript::badstream(int stream) /* test for valid stream id */
{
  int max;

  max=(9>FOPEN_MAX)?(FOPEN_MAX-3):9;
  if (stream>max || stream<1) {
    sprintf(string,"Can handle only streams from 3 to %d",max);
    error(ERROR,string);
    return TRUE;
  }
  return FALSE;
}


void SamplinScript::create_prompt(char *p) /* create command 'prompt' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'prompt'");
  cmd=add_command(PROMPT);
  cmd->pointer=my_strdup(p);
}


void SamplinScript::prompt(struct command *cmd) /* set input prompt */
{
  strncpy(inputprompt,(char *)(cmd->pointer),80);
}


void SamplinScript::create_restore(char *label) /* create command 'restore' */
{
  struct command *c;
  
  error(DIAGNOSTIC,"Creating command 'restore'");
  c=add_command(RESTORE);
  c->pointer=my_strdup(label);
}


void SamplinScript::restore(struct command *cmd) /* reset data pointer to given label */
{
  struct command *curr;

  if (cmd->type==RESTORE) { /* first time; got to search the label */
    if (*((char *)cmd->pointer)=='\0') {
      error(DIAGNOSTIC,"Restore to first data command");
      cmd->spointer=cmdroot;
      cmd->type=QRESTORE;
      goto found; /* restore to first command */
    }
    curr=cmdroot;
    sprintf(string,"Searching Label '%s'",(char *)cmd->pointer);
    error(DIAGNOSTIC,string);
    while(curr!=cmdhead) {   /* go through all commands */
      if (curr->type==LABEL && !strcmp((char *)curr->pointer,(char *)cmd->pointer)) {
        /* found right Label ! */
        sprintf(string,"Restore to label '%s'",(char *)cmd->pointer);
        /* use the address instead of the name next time ! */
        cmd->spointer=curr;
        cmd->type=QRESTORE;
        error(DIAGNOSTIC,string);
        goto found;
      }
      curr=curr->next;
    }
    /* did not found label */
    sprintf(string,"Didn't found label '%s'",(char *)cmd->pointer);
    error(ERROR,string);
    return;
  }
 found:
  datapointer=(command *)(cmd->spointer);
  return;
}


void SamplinScript::create_dbldata(double value)  /* create command dbldata */
{
  struct command *c;

  error(DIAGNOSTIC,"Creating command 'data'");
  c=add_command(DATA);
  c->pointer=my_malloc(sizeof(double));
  *((double *)c->pointer)=value;
  c->tag='d'; /* double value */
}


void SamplinScript::create_strdata(char *value)  /* create command strdata */
{
  struct command *c;

  error(DIAGNOSTIC,"Creating command 'data'");
  c=add_command(DATA);
  c->pointer=my_strdup(value);
  c->tag='s'; /* string value */
}


void SamplinScript::create_readdata(char type) /* create command readdata */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'readdata'");
  cmd=add_command(READDATA);
  cmd->tag=type;
}


void SamplinScript::readdata(struct command *cmd) /* read data items */
{
  struct stackentry *read;
  char type;

  type=cmd->tag;
  while(datapointer->type!=DATA) {
    if (datapointer==cmdhead) {
      error(ERROR,"Run out of data items");
      return;
    }
    datapointer=datapointer->next;
  }
  if (type!=datapointer->tag) {
    error(ERROR,"Type of READ and DATA don't match");
    return;
  }
  read=push();
  if (type=='d') { /* read a double value */
    read->type=NUMBER;
    read->value= *((double *)datapointer->pointer);}
  else {
    read->type=STRING;
    read->pointer=my_strdup((char *)datapointer->pointer);
  }
  datapointer=datapointer->next; /* next item */
}


void SamplinScript::create_dblrelop(char c) /* create command dblrelop */ 
{
  int type;

  switch(c) {
  case '=': type=EQ;break;
  case '!': type=NE;break;
  case '<': type=LT;break;
  case '{': type=LE;break;
  case '>': type=GT;break;
  case '}': type=GE;break;
  }
  sprintf(string,"Creating command 'dblrelop %c'",c);
  error(DIAGNOSTIC,string);
  add_command(type);
}


void SamplinScript::dblrelop(struct command *type)  /* compare topmost double-values */
{
  double a,b,c;
  struct stackentry *result;

  b=pop()->value;
  a=pop()->value;
  switch(current->type) {
  case EQ:c=(a==b);break;
  case NE:c=(a!=b);break;
  case LE:c=(a<=b);break;
  case LT:c=(a<b);break;
  case GE:c=(a>=b);break;
  case GT:c=(a>b);break;
  }
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Comparison resulted in %g",c);
    error(DIAGNOSTIC,string);
  }
  result=push();
  result->value=c;
  result->type=NUMBER;
}    


void SamplinScript::create_strrelop(char c) /* create command strrelop */ 
{
  int type;

  switch(c) {
  case '=': type=STREQ;break;
  case '!': type=STRNE;break;
  }
  sprintf(string,"Creating command 'strrelop %c'",c);
  error(DIAGNOSTIC,string);
  add_command(type);
}


void SamplinScript::strrelop(struct command *type)  /* compare topmost string-values */
{
  char *a,*b;
  double c;
  struct stackentry *result;

  b=(char *)pop()->pointer;
  a=(char *)pop()->pointer;
  switch(current->type) {
  case STREQ:c=(strcmp(a,b)==0);break;
  case STRNE:c=(strcmp(a,b)!=0);break;
  }
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Comparison resulted in %g",c);
    error(DIAGNOSTIC,string);
  }
  result=push();
  result->value=c;
  result->type=NUMBER;
}    


void SamplinScript::create_boole(char c) /* create command boole */ 
{
  int type;

  switch(c) {
  case '|': type=OR;break;
  case '&': type=AND;break;
  case '!': type=NOT;break;
  }
  sprintf(string,"Creating command 'boole %c'",c);
  error(DIAGNOSTIC,string);
  add_command(type);
}


void SamplinScript::boole(struct command *type)  /* perform and/or/not */
{
  int a,b,c;
  struct stackentry *result;

  a=(int)pop()->value;
  if (current->type==NOT) 
    c=!a;
  else {
    b=(int)pop()->value;
    if (current->type==AND)
      c=a&&b;
    else
      c=a||b;
  }
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Boolean operation resulted in %d",c);
    error(DIAGNOSTIC,string);
  }
  result=push();
  result->value=c;
  result->type=NUMBER;
}    


void SamplinScript::create_decide() /* creates command decide */
{
  error(DIAGNOSTIC,"Creating command 'decide'");
  add_command(DECIDE);
}
    

void SamplinScript::decide() /*  skips next command, if 0 on stack */
{
  struct stackentry *a;

  a=pop();
  if (a->type!=NUMBER) {
    error(FATAL,"Dont find number to decide");
    return;
  }
  if (a->value!=0) current=current->next; /* skip one command */
}


void SamplinScript::create_doarray(char *symbol,int command) /* creates array-commands */ 
{
  struct command *cmd;
  struct symbol *a;
  struct array *ar;
  char *c;
  int dimcount;

  a=get_sym(symbol,ARRAY,FALSE);
  if (a==NULL) {
    sprintf(string,
            "array '%s' has not been dimed",symbol);
    error(ERROR,string);
    return;
  }

  dimcount=(int)pop()->value;
  ar=(struct array *)(a->pointer);
  if (dimcount!=ar->dimension) {
    sprintf(string,"improper array dimension (%d) for '%s'",dimcount,symbol);
    error(ERROR,string);
    return;
  }
  
  switch(command) {
  case CALLARRAY:
    cmd=add_command(DOARRAY);
    cmd->args=CALLARRAY;
    c="callarray";
    break;
  case ASSIGNARRAY:
    cmd=add_command(DOARRAY);
    cmd->args=ASSIGNARRAY;
    c="assignarray";
    break;
  case CALLSTRINGARRAY:
    cmd=add_command(DOARRAY);
    cmd->args=CALLSTRINGARRAY;
    c="callstringarray";
    break;
  case CALLPTRARRAY:
    cmd=add_command(DOARRAY);
    cmd->args=CALLPTRARRAY;
    c="callptrarray";
    break;     
  case CALLPTRSTRARRAY:
    cmd=add_command(DOARRAY);
    cmd->args=CALLPTRSTRARRAY;
    c="callptrstrarray";
    break;     
   case ASSIGNSTRINGARRAY:
    cmd=add_command(DOARRAY);
    cmd->args=ASSIGNSTRINGARRAY;
    c="assignstringarray";
    break;
  }

  sprintf(string,"creating command '%s %s'",c,symbol);
  error(DIAGNOSTIC,string);
  cmd->spointer=ar;
  cmd->args=command;

  return;
}


void SamplinScript::doarray(struct command *current) /* call an array */
{
  struct array *ar;
  struct stackentry *stack;
  void *p;
  char **str;
  double *dbl;
  int i,j,bnd,index,call;

  call=(current->args==CALLARRAY || current->args==CALLSTRINGARRAY || current->args==CALLPTRARRAY || current->args==CALLPTRSTRARRAY); 
  if (!call) stack=pop();

  ar=(struct array *)current->spointer;
  index=0;
  for(i=0;i<ar->dimension;i++) {
    bnd=(ar->bounds[i]);
    index*=bnd;
    j=(int)pop()->value;
    if (j<0 || j>=bnd) {
      sprintf(string,"%d. index (=%d) out of range",ar->dimension-i,j);
      error(ERROR,string);
      return;
    }
    index+=j;
  }

  if (call) stack=push();

  p=ar->pointer;
   switch(current->args) {
    case CALLARRAY:
      error(DIAGNOSTIC,"calling numeric array");
      dbl=(double *)p+index;
      stack->value= *dbl;
      stack->type=NUMBER;
      break;
    case ASSIGNARRAY:
      error(DIAGNOSTIC,"assigning numeric array");
      dbl=(double *)p+index;
      *dbl=stack->value;
      break;
   case CALLSTRINGARRAY:
      error(DIAGNOSTIC,"calling string array");
      str=((char **)p+index);
      stack->pointer=my_strdup(*str);
      stack->type=STRING;
      break;
    case CALLPTRARRAY:
      error(DIAGNOSTIC,"calling ptr of double array");
      dbl=(double *)p+index;
      stack->pointer=dbl;
      stack->type=PTR;
      break;     
    case CALLPTRSTRARRAY:
      error(DIAGNOSTIC,"calling ptr of string array");
      str=((char **)p+index);
      stack->pointer=str;
      stack->type=PTR;
      break;       
    case ASSIGNSTRINGARRAY:
      error(DIAGNOSTIC,"assigning string array");
      str=((char **)p+index);
      if (*str!=NULL) free(*str);
      *str=my_strdup((char *)stack->pointer);
      break;
  }
}


void SamplinScript::create_function(int type, char *symbol) /* create command 'function' */
/* type can be sin,cos,mid$ ... */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"creating command 'function'");
  cmd=add_command(FUNCTION);
  cmd->args=type;  
  if(symbol!=NULL)cmd->pointer=my_strdup(symbol);
}


void SamplinScript::function(struct command *current) /* performs a function */
{
   struct stackentry *stack,*a1,*a2,*a3;
   struct tm *timep;
   long atime;
   char *pointer;
   double value;
   int type,result,len,start,i;
   char *str,*ptr,*eptr;
   char *symbol;
   SamplinDevice *dev;  
   dlgitem *dlg;
   
   symbol=(char *)current->pointer;
   
   type=current->args;
   if (type>TWOARGS) a3=pop();
   if (type>ONEARGS) a2=pop();
   if (type>ZEROARGS) a1=pop();
   
  switch (type) {
  case MYSIN:
    error(DIAGNOSTIC,"calling function 'sin'");
    value=sin(a1->value);
    result=NUMBER;
    break;
  case MYASIN:
     error(DIAGNOSTIC,"calling function 'asin'");
     value=asin(a1->value);
     type=NUMBER;
     break;
   case MYCOS:
     error(DIAGNOSTIC,"calling function 'cos'");
     value=cos(a1->value);
     result=NUMBER;
     break;
  case MYACOS:
     error(DIAGNOSTIC,"calling function 'acos'");
     value=acos(a1->value);
     result=NUMBER;
     break;
  case MYTAN:
     error(DIAGNOSTIC,"calling function 'tan'");
     value=tan(a1->value);
     result=NUMBER;
     break;
  case MYATAN:
     error(DIAGNOSTIC,"calling function 'atan'");
     value=atan(a1->value);
     result=NUMBER;
     break;
  case MYEXP:
     error(DIAGNOSTIC,"calling function 'exp'");
     value=exp(a1->value);
     result=NUMBER;
     break;
  case MYLOG:
     error(DIAGNOSTIC,"calling function 'log'");
     value=log(a1->value);
     result=NUMBER;
     break;
  case MYLEN:
     error(DIAGNOSTIC,"calling function 'len'");
     value=(double) strlen((char *)a1->pointer);
     result=NUMBER;
     break;
   case MYASC:
     error(DIAGNOSTIC,"calling function 'asc'");
     value=(double)(*(char *)a1->pointer);
     result=NUMBER;
     break;
   case MYSTR:
     error(DIAGNOSTIC,"calling function 'str$'");
     sprintf(string,"%g",a1->value);
     pointer=my_strdup(string);
     result=STRING;
     break;
   case MYCHR:
     error(DIAGNOSTIC,"calling function 'chr$'");
     string[0]=(char)a1->value;
     string[1]='\0';
     pointer=my_strdup(string);
     result=STRING;
     break;     
   case MYSQRT:
     error(DIAGNOSTIC,"calling function 'sqrt'");
     value=sqrt(a1->value);
     result=NUMBER;
     break;
   case MYINT:
     error(DIAGNOSTIC,"calling function 'int'");
     value=(int) a1->value;
     result=NUMBER;
     break;
   case MYFRAC:
     error(DIAGNOSTIC,"calling function 'frac'");
     value=a1->value-(int) a1->value;
     result=NUMBER;
     break;
   case MYRAN:
     error(DIAGNOSTIC,"calling function 'ran'");
     value=a1->value*(float)rand()/RAND_MAX;
     result=NUMBER;
     break;
   case MYRAN2:
     error(DIAGNOSTIC,"calling function 'ran'");
     value=(float)rand()/RAND_MAX;
     result=NUMBER;
     break;
   case MYVAL:
     error(DIAGNOSTIC,"calling function 'val'");
     ptr=(char *)a1->pointer;
     if((eptr=strstr(ptr," E"))!=NULL){
	memmove(eptr, eptr+1, strlen(eptr+1));
     }
     else if((eptr=strstr(ptr," e"))!=NULL){
	memmove(eptr, eptr+1, strlen(eptr+1));
     }
     
     i=sscanf(ptr,"%lf",&value);
     if (i!=1) value=0;
     result=NUMBER;
     break;
   case MYATAN2:
     error(DIAGNOSTIC,"calling function 'atan2'");
     value=atan2(a1->value,a2->value);
     result=NUMBER;
     break;
   case MYLEFT:
     error(DIAGNOSTIC,"calling function 'left$'");
     str=(char *)a1->pointer;
     len=(int)a2->value;
     pointer=fromto(str,1,len);
     result=STRING;
     break;
  case MYRIGHT:
     error(DIAGNOSTIC,"calling function 'right$'");
     str=(char *)a1->pointer;
     len=(int)a2->value;
     pointer=fromto(str,-len,-1);
     result=STRING;
     break;
   case MYMID:
     error(DIAGNOSTIC,"calling function 'mid$'");
     str=(char *)a1->pointer;
     start=(int)a2->value;
     len=(int)a3->value;
     pointer=fromto(str,start,start+len-1);
     result=STRING;
     break;
   case MYINKEY:
     error(DIAGNOSTIC,"calling function 'inkey$'");
     pointer=inkey();
     result=STRING;
     break;
   case MYDATE:
     error(DIAGNOSTIC,"calling function 'date$'");
     pointer=(char*)malloc(32);
     time(&atime);
     timep=localtime(&atime);
     sprintf(pointer,"%02i.%02i.%04i",timep->tm_mday,timep->tm_mon+1,timep->tm_year+1900);
     result=STRING;
     break;
   case MYTIME:
     error(DIAGNOSTIC,"calling function 'time$'");
     pointer=(char*)malloc(32);
     time(&atime);
     timep=localtime(&atime);
     sprintf(pointer,"%02i:%02i:%02i",timep->tm_hour,timep->tm_min,timep->tm_sec);
     result=STRING;
     break;     
   case MYREADY:
     dev = getDevice(symbol);
     if(dev==NULL){
	sprintf(string,"Device %s not available.",symbol);
	error(WARNING,string);      
	value=-1;
     }
     else{
	if(dev->getType()==DEVICE_GPIB){
	   sprintf(string,"DEVREADY called on GPIB device %s.",symbol);
	   error(WARNING,string);
	   value=-1;
	}
	else{
	   value=dev->Ready();
	   if(value==-3){
	      error(WARNING,dev->getErrorLog());
	   }      
	   if(value==-1){
	      sprintf(string,"Device %s not open.",symbol);
	      error(WARNING,string);
	   }
	}
     }
     result=NUMBER;
     break;
   case MYSPOLL:
     dev = getDevice(symbol);
     if(dev==NULL){
	sprintf(string,"Device %s not available.",symbol);
	error(WARNING,string);      
	value=-1;
     }
     else{
	if(dev->getType()!=DEVICE_GPIB){
	   sprintf(string,"DEVSPOLL called on non-GPIB device %s.",symbol);
	   error(WARNING,string);
	   value=-1;
	}
	else{
	   value=dev->GpibFunc(GPIB_DVRSP);
	   if(value==-3){
	      error(WARNING,dev->getErrorLog());
	   }
	   if(value==-1){
	      sprintf(string,"Device %s not open.",symbol);
	      error(WARNING,string);
	   }
	}      
     }
     result=NUMBER;
     break;
   case MYACTIVE:
     dlg=findDlg(symbol);
     if(dlg==NULL){
	sprintf(string,"Cannot find dialog %s.",symbol);
	error(WARNING,string);      
	value=0;
     }
     else{
	if(dlg->dlg!=NULL)value=1;
	else value=0;
     }
     result=NUMBER;
     break;
   default:
    error(ERROR,"function called but not implemented");
    return;
  }
  
  stack=push();
  /* copy result */
  stack->type=result;
  if (result==STRING)
    stack->pointer=pointer;
  else
    stack->value=value;
}


void SamplinScript::create_dim(char *name,char type) /* create command 'dim' */
/* type can be 's'=string or 'd'=double Array */
{ 
  struct command *cmd;
  struct symbol *s;
  struct array *ar;
  int dimcount;

  dimcount=(int)pop()->value;
  if (infolevel>=DIAGNOSTIC) {
    sprintf(string,"Creating command 'dim %s' with dimension %d",
            name,dimcount);
    error(DIAGNOSTIC,string);
  }
  cmd=add_command(DIM);
  s=get_sym(name,ARRAY,FALSE); /* search for array */
  if (s!=NULL) {
   sprintf(string,"array '%s' has been dimed already",name);
   error(ERROR,string);
   return;
  }
  s=get_sym(name,ARRAY,TRUE); /* create array */
  ar=(struct array *)my_malloc(sizeof(struct array));
  cmd->spointer=ar;
  s->pointer=ar;
  ar->type=type;
  ar->dimed=FALSE;
  ar->dimension=dimcount;
  if (dimcount>10) {
    error(ERROR,"Dimension larger than 10");
    return;
  }
}


void SamplinScript::dim(struct command *cmd) /* get room for array */
{
  struct array *ar;
  struct stackentry *s;
  char *nul,**str;
  double *dbl;
  int total,size,i;
  
  ar=(struct array *)cmd->spointer;
  if (ar->dimed) {
    error(ERROR,"Array has been dimed already");
    return;
  }
  total=1; /* count total amount of memory */
  for(i=0;i<ar->dimension;i++) {
    s=pop();
    if (s->type!=NUMBER) {
      error(ERROR,"Improper index in dim statement");
      return;
    }
    size=(int) s->value;
    if (size<=0) {
      error(ERROR,"One bound is less or equal zero");
      return;
    }
    size++; /* allow for zero-index-element */
    (ar->bounds)[i]=size;
    total*=size;
  }
  if (infolevel>=DIAGNOSTIC) {
    sprintf(string,"Getting memory for %d elements",total);
    error(DIAGNOSTIC,string);
  }
  ar->total=total;
  if (ar->type=='s')         /* it is a string array */
    ar->pointer=my_malloc(total*sizeof(char *));
  else
    ar->pointer=my_malloc(total*sizeof(double));
  if (ar->pointer==NULL) {
    error(ERROR,"Could not get enough memory for dim");
    return;
  }
  /* initialize Array */
  if (ar->type=='s') { 
    str=(char **)(ar->pointer);
    for(i=0;i<total;i++) {
      nul=(char *)my_malloc(sizeof(char));
      *nul='\0';
      *(str+i)=nul;}}
  else {
    dbl=(double *)ar->pointer;
    for(i=0;i<total;i++) *(dbl+i)=0.0;
  }
  ar->dimed=TRUE;
}


char * SamplinScript::fromto(char *str,int from,int to) /* gives back portion of string */
/* from and to can be in the range 1...strlen(str) */
{
  int len,i;
  char *part;
  
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"fromto('%s',%d,%d)",str,from,to);
    error(DIAGNOSTIC,string);
  }
  len=strlen(str);
  /* negative values count from the right */
  if (from<0) from=len+1+from;
  if (to<0) to=len+1+to;
  if (to<from) to=from;  /* from must be less than to */
  if (from>len) from=len; /* to and from must be less than len */
  if (to>len) to=len;
  part=(char *)my_malloc(sizeof(char)*(to-from+2)); /* characters and '/0' */
  for(i=from;i<=to;i++) part[i-from]=str[i-1]; /* copy */
  part[i-from]='\0';
  return part;
}


char * SamplinScript::inkey(void) /* gets char from keyboard, blocks and doesnt print */
{
  char *ret; /* string to be returned */

  ret=(char *)malloc(2);

   do {
    ret[0]=getchar();
  } while(!isprint(ret[0]));

   ret[1]='\0';
  
  return ret;
}

void SamplinScript::create_mywait() /* create Command 'wait' */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'wait'");
  cmd=add_command(WAIT);
}


void SamplinScript::mywait() /* wait given number of seconds */
{
  int delay;

  time_t start,now;
  
  delay=abs((int)(pop()->value));
  if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Wait %g second(s)",delay);
    error(DIAGNOSTIC,string);
  }

   delaylen=delay;
   task_cnt=0;
   QTimer::singleShot(delay*1000,this, SLOT(pauseSlot()));
   
}

void SamplinScript::pauseSlot()
{
 delaylen=0;  
}

void SamplinScript::create_multi(int f)
{
   struct command *cmd;
  
   error(DIAGNOSTIC,"Creating command 'multi '");
   cmd=add_command(MULTI);
   cmd->tag=f;
}


void SamplinScript::multi(struct command *cmd)
{
   multiflag=cmd->tag;
   current=current->next;

   if(multiflag==1)task_cnt=0;
   
}

void SamplinScript::create_timer(int f)
{
   struct command *cmd;
   
   if(f==1)
     error(DIAGNOSTIC,"Creating command 'timer enable'");
   else
     error(DIAGNOSTIC,"Creating command 'timer disable'");
   
   cmd=add_command(TIMER);
   cmd->tag=f;
}


void SamplinScript::mytimer(struct command *cmd)
{
   timerflag=cmd->tag;
   if(timerflag==1)
     error(DIAGNOSTIC,"Enabling timer");
   else
     error(DIAGNOSTIC,"Disabling timer");
}


void SamplinScript::create_mysleep() /* create Command 'wait' */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'wait'");
  cmd=add_command(SLEEP);
}


void SamplinScript::mysleep() /* wait given number of seconds */
{
   double delay;
   unsigned long udelay;
   
   time_t start,now;
  
   delay=fabs(pop()->value);
   udelay=(unsigned long)delay*1000;
   
   if (DIAGNOSTIC<=infolevel) {
    sprintf(string,"Wait %g milisecond(ms)",delay);
    error(DIAGNOSTIC,string);
  }

   usleep(udelay);

}


void SamplinScript::create_bell() /* create Command 'bell' */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'bell'");
  cmd=add_command(BELL);
}


void SamplinScript::bell() /* ring ascii bell */
{
  error(DIAGNOSTIC,"ringing bell");
  printf("\007");
  fflush(stdout);
}


void SamplinScript::pushname(char *name) /* push a name on stack */
{
  struct stackentry *s;

  s=push();
  s->pointer=my_strdup(name);
  s->type=STRING;
}


void SamplinScript::pushlabel() /* generate goto and push label on stack */
{
  char *st;
  struct stackentry *en;
  
  st=(char *) my_malloc(sizeof(char)*20);
  sprintf(st,"***%d",labelcount);
  sprintf(string,"Generating 'goto %s', pushing the label",st);
  error(DIAGNOSTIC,string);
  labelcount++;
  create_goto(st);
  en=push();
  en->type=LBL;
  en->pointer=st;
}
  

void SamplinScript::poplabel() /* pops a label and generates the matching command */
{
  struct stackentry *en;
  
  en=pop();   /* get name of label */
  if (en->type!=LBL) {
    error(FATAL,"Not a goto on stack");
    return;
  }
  create_label((char *)en->pointer);  /* and create it */
}


void SamplinScript::pushgoto() /* generate label and push goto on stack */
{
  char *st;
  struct stackentry *en;
  
  st=(char *) my_malloc(sizeof(char)*20);
  sprintf(st,"***%d",labelcount);
  sprintf(string,"Generating 'label %s', pushing the goto",st);
  error(DIAGNOSTIC,string);
  labelcount++;
  create_label(st);
  en=push();
  en->type=GTO;
  en->pointer=st;
}
  

void SamplinScript::popgoto() /* pops a goto and generates the matching command */
{
  struct stackentry *en;
  
  en=pop();   /* get name of goto */
  if (en->type!=GTO) {
    error(FATAL,"Not a goto on stack");
    return;
  }
  create_goto((char *)en->pointer);  /* and create it */
}


void SamplinScript::pushcounter(void) 
{
  /* push number '0' on stack, will be used as a counter */

  struct stackentry *p;
  
  p=push();
  p->type=NUMBER;
  p->value=0;
  error(DIAGNOSTIC,"pushing counter on stack");
}

void SamplinScript::pushcopy(void) 
{
  /* push copy of the last number on stack */

   struct stackentry *p;
   double value;
   void *ptr;
   int type;
   
   value = stackhead->prev->value;
   type = stackhead->prev->type;   
   ptr = stackhead->prev->pointer;
   
   p=push();
   p->type=type;
   p->value=value;
   p->pointer=ptr;
   error(DIAGNOSTIC,"pushing copy on stack");
}


void SamplinScript::inccounter(void) 
{
  /* increment topmost stack element */

  (stackhead->prev->value)++;
  sprintf(string,"incrementing counter to %g",stackhead->prev->value);
  error(DIAGNOSTIC,string);
}


void SamplinScript::swap() /*swap topmost elements on stack */
{
  struct stackentry *a,*b;
  
  error(DIAGNOSTIC,"Swapping on stack");
  if ((a=stackhead->prev)==NULL || (b=a->prev)==NULL) {
    error(ERROR,"Nothing to swap on stack !");
    return;
  }
  a->prev=b->prev;b->next=a->next;   /* just swap the pointers */
  a->next=b;b->prev=a;
  stackhead->prev=b;
  if (b==stackroot) stackroot=a;  /* treat root special */
  else (a->prev)->next=a;
}


struct stackentry * SamplinScript::push() 
/* push element on stack and enlarge it*/
{
  struct stackentry *neu;
  
  if (stackhead->next==NULL) { /* no next element */
    /* create new element */
    neu=(struct stackentry *)my_malloc(sizeof(struct stackentry)); 
    /* and initialize it */
    neu->next=NULL;  
    neu->value=0.0;
    neu->type=FREE;
    neu->prev=stackhead;
    neu->pointer=NULL;
     neu->wpointer=NULL;
     stackhead->next=neu;
  }
  stackhead=stackhead->next; /* advance head */
  /* any content is set free */
  if ((stackhead->prev)->pointer!=NULL && (stackhead->prev)->type==STRING) 
    free((stackhead->prev)->pointer);
  (stackhead->prev)->pointer=NULL;
  return stackhead->prev;
}
    

struct stackentry * SamplinScript::pop()
/* pops element to memory and looks for pop-error */
{
  /* test if there is something on the stack */
  if (stackhead==stackroot) {
    error(FATAL,"Popped too much.");
    return stackhead;
  }
  stackhead=stackhead->prev; /* move down in stack */
  if (stackhead->type==FREE) 
    error(WARNING,"Popped element without content.");
  return stackhead;  /* this is your value; use it quickly ! */
}

    
struct command * SamplinScript::add_command(int type) 
/* get room for new command, and make a link from old one */
{
  struct command *neu,*old;

  cmdhead->type=type;  /* store command */
  cmdhead->line=yylineno;
  commandcount++;
  cmdhead->pointer=NULL;  /* no data yet */ 
  cmdhead->spointer=NULL;
  cmdhead->pointer2=NULL;
   /* no next element, so, create it: */
  neu=(struct command *)my_malloc(sizeof(struct command)); 
  /* and initialize */
  neu->next=NULL;
  neu->pointer=NULL;
   neu->spointer=NULL;
   neu->pointer2=NULL;
   cmdhead->next=neu;
  old=cmdhead;
  cmdhead=cmdhead->next;
  return old;
}

void SamplinScript::initialize() 
     /* give correct values to pointers etc ... */
{
  int i;

   yylineno=1;
   scanner->yylineno=1;
   timerflag=FALSE;
   multiflag=TRUE;
   waitMulti=NULL;
   delaylen=0;
   last_print=0;
   
   if(msgwin!=NULL)msgwin->clear();
   
   /* install exception handler */
   
   signal(SIGFPE,signal_handler);
   signal(SIGSEGV,signal_handler);
 
  /* initialize error handling: no errors seen 'til now */
  errorlevel=DIAGNOSTIC;  
  diagnostic_count=0;
  note_count=0;
  warning_count=0;
  error_count=0;

   freeAlloc();

   /* initialize symbol table */
  symroot=(struct symbol *)my_malloc(sizeof(struct symbol)); /* create first */
  symroot->type=FREE;
  symroot->pointer=NULL;
  symroot->next=NULL;
  symroot->name=NULL;
  symroot->value=0.0;
  symroot->args=NULL;
   
  /* initialize numeric stack */
  /* create first : */
  stackroot=(struct stackentry *)my_malloc(sizeof(struct stackentry)); 
  stackroot->next=NULL;
  stackroot->prev=NULL;
  stackroot->value=0.0;
  stackroot->wpointer=NULL;
  /* initialize command stack */
  /* create first : */
  cmdroot=(struct command *)my_malloc(sizeof(struct command)); 
  cmdroot->next=NULL;
  cmdroot->pointer=NULL;
   cmdroot->spointer=NULL;
   cmdroot->pointer2=NULL;
  /* initialize random number generator */
  srand((unsigned int)time(NULL));

   /* file stuff */
   for(i=1;i<=9;i++) streams[i]=NULL;
   printerfile=NULL; /* no ps-file yet */
   
  resetptr();

  datapointer=cmdroot; /* restore for read data */


}


void signal_handler(int sig)   /* handle signals */
{
  switch (sig) {
  case SIGFPE:
    printf("Floating point exception, cannot proceed.\n");
    exit(TRUE);
  case SIGSEGV:
    printf("Segmentation violation, cannot proceed.\n");
    exit(TRUE);
#ifdef UNIX
  case SIGALRM: /* ignore */
    break;
#endif
  default:
    break;
  }
}

void SamplinScript::resetptr() 
/*
   reset pointers to their initial values, 
   initialize variables and functions 
*/
{
  struct symbol *s;
  struct stackentry *base;
  int i;

  symhead=symroot; /* list of double symbols */
  stackhead=stackroot; /* stack of double values */
  base=push();
  base->type=NIL; /* push nil, so that pop will not crash */
  cmdhead=cmdroot; /* list of commands */;
  commandcount=0;

  /* create useful variables */
  s=get_sym("PI",NUMBER,TRUE);
  s->value=3.14159265359;
  s=get_sym("EULER",NUMBER,TRUE);
  s->value=2.71828182864;

   /* add internal variables */
   get_sym("yabinfolevel",NUMBER,TRUE)->value=infolevel;
   get_sym("yabdiagnostic",NUMBER,TRUE)->value=DIAGNOSTIC;
   get_sym("yabnote",NUMBER,TRUE)->value=NOTE;
   get_sym("yabwarning",NUMBER,TRUE)->value=WARNING;
   get_sym("yaberror",NUMBER,TRUE)->value=ERROR;
   get_sym("yabfatal",NUMBER,TRUE)->value=FATAL;
   
   get_sym("progress_cancel",NUMBER,TRUE)->value=FALSE;
   

   get_sym("timer_msec",NUMBER,TRUE)->value=0;
   get_sym("timer_sec",NUMBER,TRUE)->value=0;
   get_sym("timer_min",NUMBER,TRUE)->value=0;
   get_sym("timer_hour",NUMBER,TRUE)->value=0;
   
  /* file stuff */
  for(i=1;i<=9;i++) 
    if (streams[i]!=NULL) {
      sprintf(string,"Stream %d not closed; closing it now",i);
      error(NOTE,string);
      fclose(streams[i]);
    }
}

void SamplinScript::error(int severe, const char *message) 
/* reports an basic error to the user and possibly exits */
{
   char tmp[128];
   char buff[1024];
   
   if (severe<=infolevel) {
      sprintf(buff,"-> ");
      switch(severe) {
       case(DIAGNOSTIC): 
	 strcat(buff,"Diagnostic"); 
	 diagnostic_count++;
	 break;
       case(NOTE): 
	 strcat(buff,"Note"); 
	 note_count++;
	 break;
       case(WARNING): 
	 strcat(buff,"Warning"); 
	 warning_count++;
	 break;
       case(ERROR): 
	 strcat(buff,"Error"); 
	 error_count++;
	 break;
       case(FATAL): 
	 strcat(buff,"Fatal"); 
	 break;
      }

      if (program_state==COMPILING) {
	 if (yylineno<0) 
	   strcat(buff," at end of file");
	 else {
	    sprintf(tmp," in line %d",yylineno);
	    strcat(buff,tmp);
	 }
      }
      else if (program_state==RUNNING){
	 sprintf(tmp," in line %d",current->line);
	 strcat(buff, tmp);
      }

      strcat(buff,": ");
      strcat(buff, message);
      strcat(buff, "\n");
      
      if(msgwin==NULL)fprintf(stderr,"%s\n",buff);
      else {
	if(showmsg==FALSE){
	   //msgwin->show();
	   showmsg=TRUE;
	}
	msgwin->insertAtEnd(buff);
     }
  }
   if (severe<errorlevel) errorlevel=severe;
/*   if (severe<=FATAL) {
      if(msgwin==NULL)
	fprintf(stderr,"-> Immediate exit to system, due to a fatal error.\n");
      else msgwin->insertAtEnd("-> Immediate exit due to a fatal error.\n");

      exit(TRUE);
   }
*/
}


void SamplinScript::uif(int id) /* issues warning on screen */
{
  sprintf(string,
          "Feature at line %d not yet implemented, id %d (sorry !)",
          yylineno,id);
  error(WARNING,string);
}


char * SamplinScript::my_strdup(const char *arg)  /* my own version of strdup, checks for failure */
{
  char *ret;
  int l;

  l=strlen(arg);

  ret=(char *)my_malloc(l+1);
  strncpy(ret,arg,l);
  *(ret+l)='\0';
  
  return ret;
}


void * SamplinScript::my_malloc(unsigned num) /* Alloc memory and issue warning on failure */
{
  void *room;
  
  room=malloc(num);
  if (room==NULL) {
    sprintf(string,"Can't malloc %d bytes of memory",num);
    error(FATAL,string);
  }
  return room;
}

void SamplinScript::setProgramState(int state)
{
   program_state=state;
   emit changedState(state);
   
}

int SamplinScript::programState(void)
{
   return program_state;
   
}

void SamplinScript::freeAlloc(void)
{
   struct stackentry *stck, *tmp_stck;   
   struct symbol *sym, *tmp_sym;
   struct command *cmd, *tmp_cmd;
   dlgitem *d;

   error(NOTE,"Starting freeAlloc");

   pd->hide();
   
   sym=symroot;
   while(sym!=NULL ){
      tmp_sym=sym;
      if(sym!=symhead){
	 if(sym->pointer!=NULL)free(sym->pointer);
	 if(sym->name!=NULL)free(sym->name);      
	 if(sym->args!=NULL)free(sym->args);            
	 sym=sym->next;
      }
      else sym=NULL;

      free(tmp_sym);
   }
   symroot=NULL;

   error(NOTE,"Symbols deleted");
   
   cmd=cmdroot;
   while(cmd!=NULL ){
      tmp_cmd=cmd;
      if(cmd!=cmdhead){
	 if(cmd->pointer!=NULL)free(cmd->pointer);
	 if(cmd->pointer2!=NULL)free(cmd->pointer2);      
	 cmd=cmd->next;
      }
      else cmd=NULL;

      free(tmp_cmd);
   }   
   cmdroot=NULL;

   error(NOTE,"Commands deleted");
   
   stck=stackroot;
   while(stck!=NULL){
      tmp_stck=stck;
      if(stck!=stackhead){
	 stck=stck->next;
      }
      else stck=NULL;
      
      free(tmp_stck);
   }      
   stackroot=NULL;

   error(NOTE,"Stack deleted");
   
   while(!dialogs.isEmpty()){
      d=dialogs.last();
      if(d!=NULL){
	 while(!d->widgets->isEmpty()){
	    d->widgets->remove(d->widgets->last());
	 }
//	 printf("Widgets removed\n");
//	 printf("removing title %s\n",d->title);
	 if(d->title!=NULL)free(d->title);
//	 printf("dlg title removed\n");
	 if(d->dlg!=NULL){
//	    printf("closing dialog %s\n",d->dlg->name());
	    d->dlg->close();
	    delete d->dlg;
	 }
//	 printf("dlg closed\n");
	 dialogs.remove(d);
      }
   }

   error(NOTE,"Finishing freeAlloc");
}

void SamplinScript::timeHandler(void)
{
   static long sec=0,usec=0,msec=0;
   static long diff=0,sdiff,mdiff;
   struct timeval tv;
   struct timezone tz;
   static int init_flag=TRUE;
   
   if(timerflag==FALSE){
      init_flag=TRUE;
      return;
   }
   
   gettimeofday(&tv,&tz);
   
   if(init_flag==FALSE){
      
      if(tv.tv_usec>usec)
	diff+=tv.tv_usec-usec;
      if(tv.tv_usec<usec)
	diff+=tv.tv_usec+(1000000-usec)+1;
      
      mdiff=diff/1000;
      diff-=mdiff*1000;
      sdiff=tv.tv_sec-sec-1;
      if(sdiff>0)mdiff+=sdiff*1000;
      msec+=mdiff;
      
      if(mdiff>0){
	 (get_sym("timer_msec",NUMBER,TRUE)->value)+=mdiff;
      }
      if(msec>=1000){
	 sdiff=msec/1000;
	 (get_sym("timer_sec",NUMBER,TRUE)->value)+=sdiff;
	 msec-=sdiff*1000;
      }
   }
   else {
      init_flag=FALSE;
      diff=0;
      msec=0;      
   }
   usec=tv.tv_usec;
   sec=tv.tv_sec;
}

int SamplinScript::currentLine(void)
{
   int ret;
   
   if(program_state==RUNNING || program_state==STOPPED || program_state==COMPILED)
     ret=current->line;
   else if(program_state==FINISHED && errorlevel==ERROR){
      ret=yylineno;
   }
     else ret=-1;

   return ret;
}
