/**************************************************************************
 * $Id: script_dlg.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "script.h"          /* all prototypes and structures */
#include "scriptdlg.h"
#include <qmsgbox.h>
#include <kapp.h>
#include <fcntl.h>

#undef None // !!!!!!!! X11/X.h defined !!!!!!!!!

struct array* SamplinScript::plotArrayOk(const char *astr)
{
   struct array *ar;
   struct symbol *a;
   
   if(astr==0)return 0;
   
   a=get_sym(astr,ARRAY,FALSE);
   
   if (a==NULL) {
      error(ERROR,"array '%s' has not been dimed",astr);
      return 0;
   }
   ar=(struct array *)(a->pointer);
   if (ar->dimension!=1) {
      error(ERROR,"array dimension must be 1 for '%s'",astr);
      return 0;
   }
   if(ar->type!='d'){
      error(ERROR,"Invalid array type '%s' (double required)",astr);
      return 0;
   }
   return ar;
   
}


void SamplinScript::create_plotfnc(int fnc, char *symbol1, char *symbol2)
{
   struct command *cmd;
   struct symbol *a;
   struct array *ar1,*ar2;
   int a1=0,a2=0;
   
   if(fnc==PLOTSETDATA || fnc==PLOTSETDATAL || fnc==PLOTGETDATA){
      if((ar1=plotArrayOk(symbol1))==0)return;
      if((ar2=plotArrayOk(symbol2))==0)return;
   }
   if(fnc==PLOTSETDATAX || fnc==PLOTSETDATAY || fnc==PLOTSETDATAXL || fnc==PLOTSETDATAYL){
      if((ar1=plotArrayOk(symbol1))==0)return;
   }
   if(fnc==PLOTAXES){
      a1=SamplinPlotData::axis(symbol1);
      if(a1==-1){
	 error(ERROR,"Invalid axis %s",symbol1);
	 return;
      }
      a2=SamplinPlotData::axis(symbol2);
      if(a2==-1){
	 error(ERROR,"Invalid axis %s",symbol2);
	 return;
      }
   }
   
   if(fnc==PLOTSTYLE){
      a1=SamplinCurve::curveStyle(symbol1);
      if(a1==-1){
	 error(ERROR,"Invalid plot style %s",symbol1);
	 return;
      }
   }
   
   if(fnc==PLOTPEN){
      a1=SamplinCurve::lineStyle(symbol1);
      if(a1==-1){
	 error(ERROR,"Invalid pen style %s",symbol1);
	 return;
      }
   }
   if(fnc==PLOTSYMBOL){
      a1=SamplinCurve::symbolStyle(symbol1);
      if(a1==-1){
	 error(ERROR,"Invalid symbol style %s",symbol1);
	 return;
      }
   }

   error(DIAGNOSTIC,"Creating command 'plotfnc', fnc=%i",fnc);
   cmd=add_command(PLOTFNC);
   cmd->args=fnc;
   cmd->spointer=ar1;
   cmd->spointer2=ar2;
   cmd->arg1=a1;
   cmd->arg2=a2;
   
}

void SamplinScript::plotfnc(struct command *cmd)
{
   GraphPanel *s;
   SamplinCurve *crv;
   char *name,*gname,*color1,*color2,*title;
   struct stackentry *p;
   struct array *ar1,*ar2;
//   float begin, step,*tmp;
   int i,type;
   uint ckey;
   double from,step;
   int len=MAXINT,width,size;
   
   ar1=(struct array *)cmd->spointer;
   ar2=(struct array *)cmd->spointer2;
   type=cmd->args;

   switch(type){
    case PLOTADD:
      break;
    case PLOTDEL:
      break;
    case PLOTSETDATAL:
      p=pop();
      len=int(p->value);
      break;
    case PLOTSETDATA:
    case PLOTGETDATA:
      break;
    case PLOTSETDATAX:
    case PLOTSETDATAY:
      p=pop();
      step=double(p->value);
      p=pop();
      from=double(p->value);
      break;
    case PLOTSETDATAXL:
    case PLOTSETDATAYL:
      p=pop();
      len=int(p->value);
      p=pop();
      step=double(p->value);
      p=pop();
      from=double(p->value);
      break;
    case PLOTAXES:
      break;
    case PLOTSTYLE:
      break;
    case PLOTPEN:
      p=pop();
      color1=(char*)p->pointer;
      p=pop();
      width=int(p->value);
      break;
    case PLOTSYMBOL:
      p=pop();
      color2=(char*)p->pointer;
      p=pop();
      color1=(char*)p->pointer;
      p=pop();
      size=int(p->value);
      p=pop();
      width=int(p->value);
      break;
    default:
      break;
   }

   p=pop();
   if(p->type!=STRING){
      error(ERROR,"STRING required for argument #2 'plot'.");
   }
   name=(char *)p->pointer;
   p=pop();
   if(p->type!=STRING){
      error(ERROR,"STRING required for argument #1 'graph'");
   }
   gname=(char *)p->pointer;   

   s=findGraph(gname);
   if(s==NULL){
      error(WARNING,"Cannot find graph %s.",gname);
      return;
   }
   if(type!=PLOTADD){
      ckey=s->curveKey(name);
      if(!ckey){
	 error(WARNING,"Cannot find plot %s in graph %s.",name,gname);	 
	 return;
      }
   }
   
   switch(type){
    case PLOTADD:
      ERROR(NOTE,(NOTE,"Adding plot %s, graph %s",name,gname));
      s->insertCurve(name);
      s->replot();
      break;
    case PLOTDEL:
      ERROR(NOTE,(NOTE,"Deleting plot %s, graph %s",name,gname));
      s->removeCurve(ckey);
      s->replot();
      break;
    case PLOTSETDATA:
    case PLOTSETDATAL:
      ERROR(NOTE,(NOTE,"Setting plot %s data, graph %s",name,gname));
      s->setCurveData(ckey,(double *)ar1->pointer,(double *)ar2->pointer,MIN(len,MIN(ar1->total,ar2->total)));	 
      s->replot();
      break;
    case PLOTSETDATAX:
    case PLOTSETDATAXL:
      ERROR(NOTE,(NOTE,"Setting plot %s X data, graph %s",name,gname));
      s->setCurveData(ckey,(double *)ar1->pointer,from,step,MAX(0,MIN(ar1->total,len)));	 
      s->replot();
      break;
    case PLOTSETDATAY:
    case PLOTSETDATAYL:
      ERROR(NOTE,(NOTE,"Setting plot %s Y data, graph %s",name,gname));
      s->setCurveData(ckey,from,step,(double *)ar1->pointer,MAX(0,MIN(ar1->total,len)));	 
      s->replot();
      break;
    case PLOTGETDATA:
      ERROR(NOTE,(NOTE,"Getting plot %s data, graph %s",name,gname));
      crv=s->findCurve(ckey);
      len=MIN(crv->size(),MIN(ar1->total,ar2->total));
      for(i=0;i<len;++i){
	 *((double *)ar1->pointer+i)=crv->x(i);
	 *((double *)ar2->pointer+i)=crv->y(i);	 
      }
      break;
    case PLOTAXES:
      ERROR(NOTE,(NOTE,"Setting plot %s axis, graph %s",name,gname));
      s->mapCurve(ckey,cmd->arg1);
      s->mapCurve(ckey,cmd->arg2);
      s->replot();
      break;
    case PLOTSTYLE:
      ERROR(NOTE,(NOTE,"Setting plot %s style, graph %s",name,gname));
      s->setCurveStyle(ckey,(SamplinCurveData::CurveStyle)cmd->arg1);
      s->replot();
      break;
    case PLOTPEN:
      ERROR(NOTE,(NOTE,"Setting plot %s pen, graph %s",name,gname));
      s->setCurvePen(ckey,QPen(QColor(color1),width,(PenStyle)cmd->arg1));
      s->replot();
      break;
    case PLOTSYMBOL:
      ERROR(NOTE,(NOTE,"Setting plot %s symbol, graph %s",name,gname));
      s->setCurveSymbol(ckey,QwtSymbol(cmd->arg1,QColor(color1),QPen(QColor(color2),width),QSize(size,size)));
      s->replot();
      break;
    default:
      break;
   }
   
}

void SamplinScript::create_graphfnc(int fnc, int subfnc, char *symbol1, char *symbol2)
{
   struct command *cmd;
   struct symbol *a;
   int a1=0,a2=0;
   
   if(fnc==GRAPHAXISTITLE || fnc==GRAPHAXISSCALE || fnc==GRAPHAXISFLAGS || fnc==GRAPHAXISSTYLE){
      a1=SamplinPlotData::axis(symbol1);
      if(a1==-1){
	 error(ERROR,"Invalid axis %s",symbol1);
	 return;
      }
   }
   
   if(fnc==GRAPHAXISSTYLE){
      a2=SamplinScaleBar::style(symbol2);
      if(a2==-1){
	 error(ERROR,"Invalid style %s",symbol2);
	 return;
      }
   }

   if(fnc==GRAPHCOMMENT || fnc==GRAPHLEGEND){
      a2=SamplinPlotData::lposition(symbol2);
      if(a2==-1){
	 error(ERROR,"Invalid legend/comment position %s",symbol2);
	 return;
      }
   }
   
   error(DIAGNOSTIC,"Creating command 'graphfnc', fnc=%i",fnc);
   cmd=add_command(GRAPHFNC);
   cmd->args=fnc;
   cmd->arg1=a1;
   cmd->arg2=a2;
   cmd->arg3=subfnc;
   
}

void SamplinScript::graphfnc(struct command *cmd)
{
   GraphPanel *s;
   char *gname,*color,*color1,*title;
   struct stackentry *p;
   int i,type;
   double min,max,step,ref;
   int len;
   int subfnc,axis,style,prec,minor;
   
   type=cmd->args;
   axis=cmd->arg1;
   subfnc=cmd->arg3;
   style=cmd->arg2;
   
   switch(type){
    case GRAPHAXISFLAGS:
      break;
    case GRAPHAXISSCALE:
      switch(subfnc){
       case 0:
	 break;
       case 1:
	 p=pop();
	 ref=(int)p->value;
	 break;
       case 2:
	 p=pop();
	 minor=(int)(p->value);
	 p=pop();
	 ref=(int)p->value;
	 break;
       case 3:
	 p=pop();
	 step=p->value;
	 p=pop();
	 max=p->value;
	 p=pop();
	 min=p->value;
	 break;
       case 4:
	 p=pop();
	 ref=p->value;
	 p=pop();
	 step=p->value;
	 p=pop();
	 max=p->value;
	 p=pop();
	 min=p->value;
	 break;
       case 5:
	 p=pop();
	 minor=(int)p->value;
	 p=pop();
	 ref=p->value;
	 p=pop();
	 step=p->value;
	 p=pop();
	 max=p->value;
	 p=pop();
	 min=p->value;
	 break;
      }
      break;
    case GRAPHAXISSTYLE:
      if(subfnc==0){
	 p=pop();
	 prec=(int)(p->value);
      }
      break;
    case GRAPHAXISTITLE:
    case GRAPHTITLE:
      p=pop();
      title=(char *)(p->pointer);      
      break;
    case GRAPHCOMMENT:
      if(subfnc==1){
	 p=pop();
	 title=(char *)(p->pointer);      
      }
      break;
    case GRAPHCOLORS:
      p=pop();
      color=(char *)(p->pointer);      
      p=pop();
      color1=(char *)(p->pointer);      
      break;
    default:
      break;
   }
   p=pop();
   if(p->type!=STRING){
      error(ERROR,"STRING required for argument #1 'graph'");
   }
   gname=(char *)p->pointer;   

   s=findGraph(gname);
   if(s==NULL){
      error(WARNING,"Cannot find graph %s.",gname);
      return;
   }
   
   switch(type){
    case GRAPHAXISTITLE:
      error(NOTE,"Setting axis %i title %s, graph %s",axis,title,gname);
//      s->setTitle(title);
//      s->setAxisTitle(xBottom,xtitle);
      s->setAxisTitle(axis,title);
      s->replot();
      break;
    case GRAPHTITLE:
      error(NOTE,"Setting title %s, graph %s",title,gname);      
      s->setTitle(title);
      s->replot();
      break;
    case GRAPHCOMMENT:
      error(NOTE,"Setting comment, graph %s",gname);      
      if(subfnc==1)s->setComment(title);
      s->setCommentPosition(style);
      s->replot();
      break;
    case GRAPHLEGEND:
      error(NOTE,"Setting legend, graph %s",gname);      
      s->setLegendPosition(style);
      s->replot();
      break;
    case GRAPHCOLORS:
      error(NOTE,"Setting colors, graph %s",gname);      
      s->setPanelColor(QColor(color));
      s->setPlotColor(QColor(color1));
      s->replot();
      break;
    case GRAPHAXISSTYLE:
	{
	   SamplinScaleBox *axisp=s->axis(axis);
	   if(subfnc==1)prec=axisp->labelPrec();
	   error(NOTE,"Setting axis %i style %i, prec %i",style,title,prec);      
	   axisp->setLabelFormat(SamplinScaleBar::Format(style),prec);
	   s->replot();
	}
      break;
    case GRAPHAXISFLAGS:
	{
	   SamplinScaleBox *axisp=s->axis(axis);
	   axisp->setOptions(subfnc);
	   s->replot();
	}
      break;
    case GRAPHAXISSCALE:
	{
	   SamplinScaleBox *axisp=s->axis(axis);
	   switch(subfnc){
	    case 2:
	      axisp->setMaxMinor(minor);
	    case 1:
	      axisp->setReference(ref);
	    case 0:
	      axisp->setAutoScale(TRUE);
	      break;
	    case 5:
	      axisp->setMaxMinor(minor);
	    case 4:
	      axisp->setReference(ref);
	    case 3:
	      axisp->setScale(min,max,step);
	      break;
	   }
	}
      s->replot();
      break;
      
    default:
      break;
   }
   
}

//===========================================
/*
void SamplinScript::create_addplot(int type, char *symbol1, char *symbol2)
{
  struct command *cmd;
  struct symbol *a;
  struct array *ar1,*ar2;

   if(symbol1!=0){
      a=get_sym(symbol1,ARRAY,FALSE);
      
      if (a==NULL) {
	 error(ERROR,"array '%s' has not been dimed",symbol1);
	 return;
      }
      ar1=(struct array *)(a->pointer);
      if (ar1->dimension!=1) {
	 error(ERROR,"array dimension must be 1 for '%s'",symbol1);
	 return;
      }
      if(ar1->type!='d'){
	 error(ERROR,"Invalid array type '%s' (double required)",symbol1);
	 return;
      }
   }
   
   if(symbol2!=0){
      a=get_sym(symbol2,ARRAY,FALSE);
      
      if (a==NULL) {
	 error(ERROR,"array '%s' has not been dimed",symbol2);
	 return;
      }
      
      ar2=(struct array *)(a->pointer);
      if (ar2->dimension!=1) {
	 error(ERROR,"array dimension must be 1 for '%s'",symbol2);
	 return;
      }
      if(ar2->type!='d'){
	 error(ERROR,"Invalid array type '%s' (double required)",symbol2);
	 return;
      }
   }
   
   error(DIAGNOSTIC,"Creating command 'addplot'");
   cmd=add_command(ADDPLOT);
   cmd->args=type;
   cmd->spointer=ar1;
   cmd->spointer2=ar2;
}

void SamplinScript::addplot(struct command *cmd)
{
   GraphPanel *s;
   char *name,*gname,*color;
   struct stackentry *p;
   struct array *ar1,*ar2;
   float begin, step,*tmp;
   int i,type;
   
   ar1=(struct array *)cmd->spointer;
   ar2=(struct array *)cmd->spointer2;
   type=cmd->args;
   
   p=pop();
   if(p->type!=STRING){
      error(ERROR,"STRING required for argument 'color'.");
   }
   color=(char *)p->pointer;   
   
   if(type==1 || type==2){
      p=pop();
      if(p->type!=NUMBER){
	 error(ERROR,"NUMBER required for argument 'step'.");
      }
      step=(double)p->value;
      p=pop();
      if(p->type!=NUMBER){
	 error(ERROR,"NUMBER required for argument 'begin'.");
      }
      begin=(double)p->value;
   }
   
   p=pop();
   if(p->type!=STRING){
      error(ERROR,"STRING required for argument #2 'plot'.");
   }
   name=(char *)p->pointer;
   p=pop();
   if(p->type!=STRING){
      error(ERROR,"STRING required for argument #1 'graph'");
   }
   gname=(char *)p->pointer;   
   
   
   ERROR(NOTE,(NOTE,"Adding plot %s in %s",name,gname));

   s=findGraph(gname);
   
   if(s==NULL){
      error(WARNING,"Cannot find graph %s.",gname);
      return;
   }
   else{
      uint ckey;
      
      ckey=s->insertCurve(name);
      s->setCurvePen(ckey,QPen(QColor(color),1,SolidLine));

      switch(type){
       case 1:
	 s->setCurveData(ckey,begin,step,(double *)ar1->pointer,ar1->total);
	 break;
       case 2:
	 s->setCurveData(ckey,(double *)ar1->pointer,begin,step,ar1->total);	 
	 break;
       case 3:
	 s->setCurveData(ckey,(double *)ar1->pointer,(double *)ar2->pointer,MIN(ar1->total,ar2->total));	 
	 break;
      }

      s->replot();
   }

   return;
}
*/





dlgitem *SamplinScript::findDlg(const char *name)
{
   dlgitem *dlg;
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next())
     if(!strcmp(name,dlg->name) && strlen(name)==strlen(dlg->name))
       return(dlg);
   
   return NULL;
}



uint SamplinScript::newGraphId(void){

   uint newkey = graphs.count();

//   printf("start key= %u\n",newkey);
   
   if (newkey > 1)
     {
	if (graphs.find(newkey))
	  {
	     newkey = 1;
	     while (newkey <= graphs.size())
	       {
		  if (graphs.find(newkey))
		    newkey++;
		  else
		    break;
	       }
	     if (newkey > graphs.size())
	       {
		  while (!graphs.find(newkey))
		    {
		       newkey++;
		       if (newkey > 10000)
			 {
			    newkey = 0;
			    break;
			 }
		    }
	       }
	  }
     }
//   printf("generate key: %u\n",newkey);
   return newkey;
}

void SamplinScript::closeGraphSlot(uint id){

   if(graphs.find(id)){
      graphs.remove(id);
      emit closeGraphSignal(id);
   }
}

bool SamplinScript::closeGraph(const char *name){
   
   GraphWindow *gwin;

   gwin=findGraphWindow(name);
   return(gwin->close());
}

bool SamplinScript::closeGraph(uint id){
   
   GraphWindow *gwin;

   gwin=findGraphWindow(id);
   return(gwin->close());
}

bool SamplinScript::closeAllGraphs(void){
   QIntDict<GraphWindow> tmp_graphs;
   QIntDictIterator<GraphWindow> graphsI(tmp_graphs);
   GraphWindow *s=NULL;
   
   tmp_graphs=graphs;
   
   graphsI.toFirst();
   while(graphsI.current()){
      s=graphsI.current();
      if(s->close()){
	 
      }
      else return(FALSE);
      ++graphsI;
   }

   return(TRUE);
}

GraphWindow * SamplinScript::openGraph(const char *name){
   
   GraphWindow *gwin;
   uint id;
   
   id=newGraphId();
   gwin = new GraphWindow(NULL,name,id);
   gwin->setGeometry(10,40,590,330);
   gwin->setRootDir(cnf.root_dir+cnf.data_dir);
   graphs.insert(id,gwin);
   connect(gwin, SIGNAL(closeSignal(uint)),this,SLOT(closeGraphSlot(uint)));
   gwin->show();
   emit openGraphSignal(id,gwin);
   return(gwin);
}

GraphPanel * SamplinScript::findGraph(const char *name){
   GraphWindow *w=NULL;
   GraphPanel *s=NULL;
   widgetitem *wdt;
   dlgitem *dlg;
   bool found=FALSE;

   w=findGraphWindow(name);
   if(w!=0)s=w->graph();
   
   if(s==NULL){
      dlg=dialogs.first();
      if(dlg!=0)do{
	 if(dlg->dlg!=NULL){
	    wdt=dlg->widgets->first();
	    if(wdt!=0)do{
	       if(wdt->type==WIDGET_GRAPH){
		  if(!strcmp(wdt->text,name) && strlen(wdt->text)==strlen(name)){
		     s=(GraphPanel *)wdt->pointer;
		     found=TRUE;
		  }
	       }
	       wdt=dlg->widgets->next();	    
	    }while(wdt!=0 && found==FALSE);
	 }
	 dlg=dialogs.next();
      }while(dlg!=0 && found==FALSE);
   }
   
   return(s);
}

GraphWindow * SamplinScript::findGraphWindow(const char *name){
   QIntDictIterator<GraphWindow> graphsI(graphs);
   GraphWindow *s=NULL;
   const char *gname;
   
   graphsI.toFirst();
   while(graphsI.current()){
      gname=graphsI.current()->name();
      if(!strcmp(gname,name) && strlen(gname)==strlen(name))      
	{s=graphsI.current();break;}
      ++graphsI;
   }

   return(s);
}

GraphWindow * SamplinScript::findGraphWindow(uint id){

   return(graphs.find(id));
}

void SamplinScript::create_message(char type) /* create command 'message' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'message'");
  cmd=add_command(MESSAGE);
//  cmd->pointer=my_malloc(sizeof(int));
  /* store type of print  */
  cmd->tag=type;
}


void SamplinScript::message(struct command *cmd) /* open message dlg */
{
  struct stackentry *p;

   p=pop();

   switch(cmd->tag){
    case 'i':
      QMessageBox::information(NULL,"",(char *)p->pointer);
      break;
    case 'w':
      QMessageBox::warning(NULL,"",(char *)p->pointer);
      break;      
    case 'e':
      QMessageBox::critical(NULL,"",(char *)p->pointer);
      break;
   }
}

void SamplinScript::create_addgraph()
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'addgraph'");
  cmd=add_command(ADDGRAPH);

}

void SamplinScript::create_loadgraph(int type)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'loadgraph'");
  cmd=add_command(LOADGRAPH);
  cmd->args=type;
}

void SamplinScript::create_savegraph(int type)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'savegraph'");
  cmd=add_command(SAVEGRAPH);
  cmd->args=type;
}

void SamplinScript::create_exportgraph(int type)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'exportgraph'");
  cmd=add_command(EXPORTGRAPH);
  cmd->args=type;
}

void SamplinScript::create_printgraph()
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'printgraph'");
  cmd=add_command(PRINTGRAPH);
}

void SamplinScript::create_delgraph()
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'delgraph'");
  cmd=add_command(DELGRAPH);
//  cmd->pointer=my_strdup(dev);

}

/*
void SamplinScript::create_titlegraph()
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'titlegraph'");
  cmd=add_command(TITLEGRAPH);

}
*/

void SamplinScript::create_adddlg(char *dev)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'adddlg'");
  cmd=add_command(ADDDLG);
  cmd->pointer=my_strdup(dev);
}

void SamplinScript::create_progressdlg(int type,int)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'progressdlg'");
  cmd=add_command(PROGRESSDLG);
  cmd->args=type;
}

void SamplinScript::create_deldlg(char *dev)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'deldlg'");
  cmd=add_command(DELDLG);
  cmd->pointer=my_strdup(dev);
}

void SamplinScript::create_hidedlg(char *dev)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'hidedlg'");
  cmd=add_command(HIDEDLG);
  cmd->pointer=my_strdup(dev);
}

void SamplinScript::create_showdlg(char *dev)
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'showdlg'");
  cmd=add_command(SHOWDLG);
  cmd->pointer=my_strdup(dev);
}

void SamplinScript::create_addwidget(char *dlg, int type, char *action, char *value) 
{
   struct command *cmd;
   struct symbol *a;
   struct array *ar;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Creating command 'addwidget'"));
   cmd=add_command(ADDWIDGET);
   cmd->pointer=my_strdup(dlg);
   cmd->args=type;

   if(type==WIDGET_SELECT){
      cmd->spointer=(double*)&get_sym(value, NUMBER, TRUE)->value;      
      a=get_sym(action,ARRAY,FALSE);
      
      if (a==NULL) {
	 error(ERROR,"array '%s' has not been dimed",action);
	 return;
      }
      ar=(struct array *)(a->pointer);
      if (ar->dimension!=1) {
	 error(ERROR,"array dimension must be 1 for '%s'",action);
	 return;
      }
      cmd->spointer2=ar;
   }
   
//   if(type==WIDGET_TEXT)
//     cmd->spointer=(char**)&get_sym(action, STRING, TRUE)->pointer;
   
   if(type==WIDGET_BUTTON )cmd->pointer2=my_strdup(action);
   if(type==WIDGET_INPUT){
      cmd->spointer=(char**)&get_sym(action, STRING, TRUE)->pointer;
   }
   if(type==WIDGET_LED || type==WIDGET_LCD){
      cmd->spointer=(double*)&get_sym(action, NUMBER, TRUE)->value;
   }
   if(type==WIDGET_SWITCH){
      cmd->pointer2=my_strdup(action);
      cmd->spointer=(double*)&get_sym(value, NUMBER, TRUE)->value;      
   }
   if(type==WIDGET_WHEEL){
      cmd->spointer=(double*)&get_sym(value, NUMBER, TRUE)->value;         
   }
   if(type==WIDGET_KNOB){
      cmd->spointer=(double*)&get_sym(value, NUMBER, TRUE)->value;         
   }
   if(type==WIDGET_LEVEL){
      cmd->spointer=(double*)&get_sym(value, NUMBER, TRUE)->value;         
   }
   if(type==WIDGET_SLIDER){
      cmd->spointer=(double*)&get_sym(value, NUMBER, TRUE)->value;         
   }
}

void SamplinScript::loadgraph(struct command *cmd)
{
   char *name,*file;
   struct stackentry *p;
   int status;
   GraphPanel *s;   
   
   if(cmd->args==1)
     {
	p=pop();
	file=(char *)(p->pointer);
	error(NOTE,"Loading graph '%s' from file '%s'",name,file);
     }
   else {
      file=NULL;
      error(NOTE,"Loading graph '%s'",name);
   }
   p=pop();
   name=(char *)(p->pointer);

   s=findGraph(name);
   if(s==NULL){
      error(WARNING,"Cannot find graph '%s'.",name);      
   }
   else if(s->load(file)==FALSE)
     {
	error(WARNING,"Cannot open file '%s' for reading.",file);      	
     }
   
   return;
}

void SamplinScript::savegraph(struct command *cmd)
{
   char *name,*file;
   struct stackentry *p;
   int status;
   GraphPanel *s;   
   
   if(cmd->args==1)
     {
	p=pop();
	file=(char *)(p->pointer);
	error(NOTE,"Saving graph '%s' to file '%s'",name,file);
     }
   else
     {
	file=NULL; 
	error(NOTE,"Saving graph '%s'",name);
     }
   
   p=pop();
   name=(char *)(p->pointer);
   
   s=findGraph(name);
   if(s==NULL){
      error(WARNING,"Cannot find graph '%s'.",name);      
   }
   else if(s->save(file)==FALSE)
     {
	error(WARNING,"Cannot open file '%s' for writing.",file);      	
     }
   
   return;
}

void SamplinScript::exportgraph(struct command *cmd)
{
   char *name,*file;
   struct stackentry *p;
   int status;
   GraphPanel *s;   
   
   if(cmd->args==1)
     {
	p=pop();
	file=(char *)(p->pointer);
	error(NOTE,"Exporting graph '%s' to file '%s'",name,file);
     }
   else
     {
	file=NULL; 
	error(NOTE,"Exporting graph '%s'",name);
     }
   
   p=pop();
   name=(char *)(p->pointer);
   
   s=findGraph(name);
   if(s==NULL){
      error(WARNING,"Cannot find graph '%s'.",name);      
   }
   else if(s->export(file)==FALSE)
     {
	error(WARNING,"Cannot open file '%s' for writing.",file);      	
     }
   
   return;
}

void SamplinScript::printgraph(struct command *cmd)
{
   char *name;
   struct stackentry *p;
   int status;
   GraphPanel *s;   
   
   p=pop();
   name=(char *)(p->pointer);
   
   error(NOTE,"Printing graph '%s'",name);

   s=findGraph(name);
   if(s==NULL){
      error(WARNING,"Cannot find graph '%s'.",name);      
   }

   s->print();
   return;
}


void SamplinScript::addgraph(struct command *cmd)
{
   char *name;
   struct stackentry *p;
   int status;
   GraphWindow *gwin;
   
   p=pop();
   
   name=(char *)(p->pointer);
   
   error(NOTE,"Adding graph window '%s'",name);

   if((gwin=findGraphWindow(name))!=NULL){
      error(WARNING,"Graph window '%s' already open.",name);
   }
   else  openGraph(name);
   
   return;
}

void SamplinScript::delgraph(struct command *cmd)
{
   char *name;
   struct stackentry *p;
   int status;
   GraphWindow *s;
   
   p=pop();
   
   name=(char *)(p->pointer);
   
   error(NOTE,"Deleting graph '%s'",name);
   
   s=findGraphWindow(name);
   
   if(s==NULL){
      error(WARNING,"Cannot find graph '%s'.",name);      
   }
   else closeGraph(name);
   
   return;
}

void SamplinScript::progressdlg(struct command *cmd) 
{
   struct stackentry *p;
   char *title;
   int steps;
   int i;
   QProgressDialog *ppp;
   static int tt=0;
   if(tt>2)tt=0;
   
   
   p=pop();
   steps=(int)p->value;
   
   if(cmd->args==0)
     {
	p=pop();
	title=my_strdup((char *)p->pointer);
	pd->reset();
	pd->setLabelText(title);
	pd->setTotalSteps(steps);
	pd->setMinimumDuration(0);
	pd -> setProgress(0);
	get_sym("progress_cancel",NUMBER,TRUE)->value=FALSE;   
     }
   if(cmd->args==1){
	pd -> setProgress(steps);
   }
   
}

void SamplinScript::adddlg(struct command *cmd)
{
   dlgitem *newdlg;
   dlgitem *dlg;
   char *name;
   char *title;
   bool found;
   struct stackentry *p;

   p=pop();

   title=my_strdup((char *)p->pointer);
   name=(char *)cmd->pointer;

   ERROR(NOTE,(NOTE,"Adding dialog: %s",name));
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next()){
      if(!strcmp(name,dlg->name) && strlen(name)==strlen(dlg->name)){
	 error(WARNING,"Dialog already exists");
	 return;
      }
   }
   
   newdlg = new dlgitem;
   newdlg->name=name;
   newdlg->title=title;
   newdlg->widgets = new QList<widgetitem>;
   newdlg->widgets->setAutoDelete(TRUE);
   newdlg->dlg=NULL;
   dialogs.append(newdlg);
   return;
}

void SamplinScript::hidedlg(struct command *cmd)
{
   dlgitem *dlg;
   char *name;
   bool found;
   
   found=FALSE;
   name=(char *)cmd->pointer;

   ERROR(NOTE,(NOTE,"Hiding dialog: %s",name));   
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next()){
      if(!strcmp(name,dlg->name) && strlen(name)==strlen(dlg->name)){
	 if(dlg->dlg!=NULL){
	    dlg->dlg->close();
	    delete dlg->dlg;
	    dlg->dlg=NULL;
	 }
	 else error(WARNING,"Dialog not active");
	   
	 found=TRUE;
	 return;
      }
   }
   if(found==FALSE){
      error(WARNING,"Cannot find dialog %s.",name);   
   }
   
   return;
}

void SamplinScript::deldlg(struct command *cmd)
{
   dlgitem *dlg;
   char *name;
   bool found;
   
   found=FALSE;
   name=(char *)cmd->pointer;

   ERROR(NOTE,(NOTE,"Deleting dialog: %s",name));   
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next()){
      if(!strcmp(name,dlg->name) && strlen(name)==strlen(dlg->name)){
	 if(dlg->dlg!=NULL){
	    dlg->dlg->close();
	    delete dlg->dlg;
	 }
	 delete dlg->widgets;
	 free(dlg->title);
	 dialogs.remove(dlg);
	 found=TRUE;
	 return;
      }
   }
   if(found==FALSE){
      error(WARNING,"Cannot find dialog %s.",name);   
   }
   
   return;
}


void SamplinScript::showdlg(struct command *cmd)
{
   dlgitem *dlg;
   widgetitem *wdt;
   char *name;
   bool found;
   int i;

   found=FALSE;
   name=(char *)cmd->pointer;

   ERROR(NOTE,(NOTE,"Show dialog: %s",name));      
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next()){
      if(!strcmp(name,dlg->name) && strlen(name)==strlen(dlg->name)){

	 if(dlg->dlg!=NULL){
	    error(WARNING,"Dialog already active");
	    return;
	 }
	 
	 dlg->dlg = new ScriptDlg(NULL, name);
	 dlg->dlg->setDlgItem(dlg);
	 dlg->dlg->setCaption(dlg->title);

	 for(wdt=dlg->widgets->first();wdt!=0;wdt=dlg->widgets->next()){
	    ScriptLabel *tmpLabel;
	    ScriptPushButton *tmpButton;
	    
	    switch(wdt->type){
	     case WIDGET_TEXT:
	       tmpLabel = new ScriptLabel(dlg->dlg,"text");
	       tmpLabel->setText(wdt->text);
	    //   tmpLabel->setPointer(wdt->transfer);
	       tmpLabel->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);
	       wdt->pointer=tmpLabel;
	       break;
	     case WIDGET_INPUT:case WIDGET_INPUTA:
	       ScriptLineEdit *tmpPtr;
	       char *tmpPtr2;
	       tmpPtr = new ScriptLineEdit(dlg->dlg,"edit");
	       tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
	       tmpPtr->setWidget(wdt);
	       tmpPtr->setPointer(wdt->transfer);
	       //	       tmpPtr->setText("xxx");
//	       break;
	       //tmpPtr2=(char*)&get_sym(wdt->text,STRING,FALSE)->pointer;
	       if(wdt->transfer!=NULL)tmpPtr->setText(*(wdt->transfer));
	       wdt->pointer=tmpPtr;
	       break;
	     case WIDGET_BUTTON:
	       tmpButton = new ScriptPushButton(dlg->dlg,"button");
	       tmpButton->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
	       tmpButton->setText(wdt->text);
	       tmpButton->setWidget(wdt);
	       connect(tmpButton, SIGNAL(widgetSignal(widgetitem *)), this, SLOT(widgetSlot(widgetitem *)));
	       wdt->pointer=tmpButton;
	       break;
	     case WIDGET_SELECT:
		 {
		    ScriptComboBox *tmpPtr;
		    tmpPtr = new ScriptComboBox(FALSE,dlg->dlg,"select");
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setWidget(wdt);
		    tmpPtr->setValue(wdt->value);
		    if(wdt->transfer!=NULL)
		       for(i=0;i<wdt->count;++i){
			  tmpPtr->insertItem(*(wdt->transfer+i));
		       }
		    wdt->pointer=tmpPtr;
		    break;
		 }
	     
	     case WIDGET_SWITCH:
		 {
		    ScriptCheckBox *tmpPtr;
		    tmpPtr = new ScriptCheckBox(dlg->dlg,"switch");
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setText(wdt->text);
		    tmpPtr->setValue(wdt->value);
		    tmpPtr->setWidget(wdt);
		    connect(tmpPtr, SIGNAL(widgetSignal(widgetitem *)), this, SLOT(widgetSlot(widgetitem *)));
		    wdt->pointer=tmpPtr;
		    break;	       
		 }

	     case WIDGET_WHEEL:
		 {
//		    QwtWheel *tmpPtr;
//		    tmpPtr = new QwtWheel(dlg->dlg);
//		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
//		    tmpPtr->setValue(wdt->value);
//		    tmpPtr->setWidget(wdt);
//		    tmpPtr->setRange(wdt->from, wdt->to, wdt->step);
//		    tmpPtr->setTickCnt(6);
//		    tmpPtr->setTotalAngle(wdt->angle);
//		    connect(tmpPtr, SIGNAL(widgetSignal(widgetitem *)), this, SLOT(widgetSlot(widgetitem *)));
		    break;	       
		 }	       
	     case WIDGET_GRAPH:
		 {
		    GraphPanel *tmpPtr;
		    tmpPtr = new GraphPanel(dlg->dlg,"graph");
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setRootDir(cnf.root_dir+cnf.data_dir);
		    wdt->pointer=tmpPtr;
		    break;	       
		 }	       
	     case WIDGET_KNOB:
		 {
		    ScriptKnob *tmpPtr;
		    tmpPtr = new ScriptKnob(dlg->dlg,"knob");
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setRange(wdt->from, wdt->to, wdt->step);
//		    tmpPtr->setFont(QFont("Helvetica", 10, QFont::Bold));
		    tmpPtr->setVariable(wdt->value);
//		    tmpPtr->setWidget(wdt);
//		    connect(tmpPtr, SIGNAL(widgetSignal(widgetitem *)), this, SLOT(widgetSlot(widgetitem *)));
		    wdt->pointer=tmpPtr;
		    break;	       
		 }	       	       
	     case WIDGET_SLIDER:
		 {
		    ScriptSlider::Orientation or;
		    ScriptSlider::ScalePos sc;
		    ScriptSlider *tmpPtr;

		    if(wdt->w>wdt->h){
		       or = ScriptSlider::Horizontal;
		       switch(wdt->style){
			case 0: sc=ScriptSlider::None;break;
			case 1: sc=ScriptSlider::Bottom;break;
			case 2: sc=ScriptSlider::Top;break;			  
			default: sc=ScriptSlider::Bottom;break;			  
		       }
		    }
		    else{
		      or = ScriptSlider::Vertical;
		       switch(wdt->style){
			case 0:sc=ScriptSlider::None;break;
			case 1:sc=ScriptSlider::Left;break;
			case 2:sc=ScriptSlider::Right;break;			  
			default:sc=ScriptSlider::Left;break;			  
		       }
		    }

		    tmpPtr = new ScriptSlider(dlg->dlg,"slider",or,sc);
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setRange(wdt->from, wdt->to, wdt->step);
//		    tmpPtr->setFont(QFont("Helvetica", 10, QFont::Bold));
		    tmpPtr->setVariable(wdt->value);
//		    tmpPtr->setWidget(wdt);
//		    connect(tmpPtr, SIGNAL(widgetSignal(widgetitem *)), this, SLOT(widgetSlot(widgetitem *)));
		    wdt->pointer=tmpPtr;
		    break;	       
		 }	       	       	       
	     case WIDGET_LEVEL:
		 {
		    ScriptLevel::Orientation or;
		    ScriptLevel::ScalePos sc;
		    ScriptLevel *tmpPtr;
		    
		    if(wdt->w > wdt->h){
		       or = ScriptLevel::Horizontal;
		       sc = ScriptLevel::Bottom;
		       
		       if(wdt->style==0)sc = QwtThermo::None;
		       if(wdt->style==1)sc= ScriptLevel::Bottom;
		       if(wdt->style==2)sc= ScriptLevel::Top;		       
		    }
		    else{
		       or = ScriptLevel::Vertical;
		       sc = ScriptLevel::Left;

		       if(wdt->style==0)sc=ScriptLevel::None;
		       if(wdt->style==1)sc=ScriptLevel::Left;
		       if(wdt->style==2)sc=ScriptLevel::Right;		       
		    }
		    
		    tmpPtr = new ScriptLevel(dlg->dlg,"level");
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setOrientation(or,sc);
		    tmpPtr->setPipeWidth(wdt->linewidth);
		    tmpPtr->setRange(wdt->from, wdt->to);
		    tmpPtr->setFillColor(QColor(wdt->color));
		    tmpPtr->setVariable(wdt->value);
		    wdt->pointer=tmpPtr;
		    break;	       
		 }	       
	     
	     case WIDGET_LCD:
		 {
		    ScriptLCD *tmpPtr;
		    tmpPtr = new ScriptLCD(dlg->dlg,"LCD");
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setSmallDecimalPoint(TRUE);
		    tmpPtr->setNumDigits(2*wdt->w/wdt->h);
		    tmpPtr->setNumDecDigits(wdt->prec);
		    switch(wdt->modif){
		     case 0:
		       tmpPtr->setMode(LCDisplay::DEC);
		       break;
		     case 1:
		       tmpPtr->setMode(LCDisplay::SCI);
		       break;
		     case 2:
		       tmpPtr->setMode(LCDisplay::OCT);
		       break;
		     case 3:
		       tmpPtr->setMode(LCDisplay::BIN);
		       break;
		     case 4:
		       tmpPtr->setMode(LCDisplay::HEX);
		       break;
		     default:
		       tmpPtr->setMode(LCDisplay::DEC);
		       break;
		    }
		    tmpPtr->setValue(wdt->value);
		    QColorGroup ocg=tmpPtr->colorGroup();
		    QColorGroup g(ocg.foreground(),ocg.background(),ocg.light(),ocg.dark(),ocg.mid(),QColor(wdt->color),ocg.base());
		    QPalette p(g,g,g);
		    tmpPtr->setFrameStyle(QFrame::Panel|QFrame::Sunken);
		    tmpPtr->setPalette(p);
		    tmpPtr->setSegmentStyle(LCDisplay::Filled);
		    wdt->pointer=tmpPtr;
		    break;	       	       
		 }
	     case WIDGET_LED: case WIDGET_LEDA:
		 {
		    ScriptLED *tmpPtr;
		    tmpPtr = new ScriptLED(dlg->dlg);
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
//		    tmpPtr->setState((*wdt->value==0) ? KLedLamp::On:KLedLamp::Off);
		    tmpPtr->setColor(wdt->color);
		    tmpPtr->setText(wdt->text);
		    tmpPtr->setValue(wdt->value);
		    wdt->pointer=tmpPtr;
		    break;	       	       	       
		 }
	     case WIDGET_FRAME:
		 {
		    QFrame *tmpPtr;
		    tmpPtr = new QFrame(dlg->dlg);
		    tmpPtr->setGeometry(wdt->x,wdt->y,wdt->w, wdt->h);	       
		    tmpPtr->setFrameStyle(wdt->style);
		    tmpPtr->setLineWidth(wdt->linewidth);
		    wdt->pointer=tmpPtr;
		    break;	       	       	       
		 }	       
	    
	    }
	    
	 }
	 
	 dlg->dlg->show();
	 found=TRUE;
	 return;
      }
   }
   if(found==FALSE){
      error(WARNING,"Cannot find dialog %s.",name);   
   }

   return;
}

void SamplinScript::addwidget(struct command *cmd)
{
   dlgitem *dlg;
   widgetitem *wdt;
   struct stackentry *p,*s;
   char *name;
   bool found;
   struct array *ar;
   
   found=FALSE;
   name=(char *)cmd->pointer;
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next()){
      if(!strcmp(name,dlg->name) && strlen(name)==strlen(dlg->name)){
	 wdt = new widgetitem;
	 wdt->type=cmd->args;
	 wdt->action=(char *)cmd->pointer2;
	 wdt->active=FALSE;
	 
	 switch(wdt->type){
	  case WIDGET_INPUT:
//	    if(cmd->pointer2!=NULL)
//	      wdt->transfer=get_sym((char *)cmd->pointer2, STRING, FALSE);
//	    else wdt->transfer=NULL;
	    wdt->transfer=(char **)cmd->spointer;
//	    wdt->text=(char *)cmd->pointer2;//my_strdup((char *)p->pointer);
	    break;
	    case WIDGET_INPUTA:
	    p=pop();
	    wdt->transfer=(char **)p->pointer;
	    break;
	  case WIDGET_LED:
	    p=pop();
	    wdt->color=my_strdup((char *)p->pointer);
	    p=pop();
	    wdt->text=my_strdup((char *)p->pointer);
	    wdt->value=(double *)cmd->spointer;
	    break;
	  case WIDGET_LEDA:
	    p=pop();
	    wdt->color=my_strdup((char *)p->pointer);
	    p=pop();
	    wdt->text=my_strdup((char *)p->pointer);
	    p=pop();
	    wdt->value=(double *)p->pointer;	    	    
	    break;	  
	  case WIDGET_LCD:
	    p=pop();
	    wdt->color=my_strdup((char *)p->pointer);
	    p=pop();
	    wdt->modif=(int)p->value;
	    p=pop();
	    wdt->prec=(int)p->value;
	    
	    wdt->value=(double *)cmd->spointer;
	    break;	    
	  case WIDGET_SELECT:
	    wdt->value=(double *)cmd->spointer;

	    ar=(struct array *)current->spointer2;
	    if(ar->type!='s'){
	       error(ERROR,"Invalid array type (string required)");
	       return;
	    }
	    wdt->count=ar->total;
	    wdt->transfer=(char **)(ar->pointer);	    
	    break;
	  case WIDGET_TEXT:
	    p=pop();
	    wdt->text=my_strdup((char *)p->pointer); //!!!
//	    wdt->transfer=(char **)cmd->spointer;
	    
	    break;
	  case WIDGET_BUTTON:
	    p=pop();
	    wdt->text=my_strdup((char *)p->pointer); //!!!
	    break;
	  case WIDGET_SWITCH:
	    p=pop();
	    wdt->text=my_strdup((char *)p->pointer); //!!!
	    wdt->value=(double *)cmd->spointer;
	    break;
	  case WIDGET_FRAME:
	    p=pop();
	    wdt->linewidth=(int)p->value;
	    p=pop();
	    wdt->style=(int)p->value;
	    break;
	  case WIDGET_WHEEL:
	    p=pop();
	    wdt->angle=p->value;
	    p=pop();
	    wdt->step=p->value;
	    p=pop();
	    wdt->to=p->value;
	    p=pop();
	    wdt->from=p->value;
	    
	    wdt->value=(double *)cmd->spointer;
	    break;
	  case WIDGET_GRAPH:
	    p=pop();
	    wdt->text=my_strdup((char *)p->pointer);
	    break;
	  case WIDGET_KNOB:
	    p=pop();
	    wdt->step=p->value;
	    p=pop();
	    wdt->to=p->value;
	    p=pop();
	    wdt->from=p->value;
	    wdt->value=(double *)cmd->spointer;
	    break;
	  case WIDGET_SLIDER:
	    p=pop();
	    wdt->style=(int)p->value;
	    p=pop();
	    wdt->step=p->value;
	    p=pop();
	    wdt->to=p->value;
	    p=pop();
	    wdt->from=p->value;
	    wdt->value=(double *)cmd->spointer;
	    break;	    
	  case WIDGET_LEVEL:
	    p=pop();
	    wdt->style=(int)p->value;
	    p=pop();
	    wdt->color=my_strdup((char *)p->pointer);
	    p=pop();
	    wdt->to=p->value;
	    p=pop();
	    wdt->from=p->value;
	    p=pop();
	    wdt->linewidth=(int)p->value;
	    wdt->value=(double *)cmd->spointer;
	    break;
	 }

	 p=pop();
	 wdt->h=MAX(1,(int)p->value);
	 p=pop();
	 wdt->w=MAX(1,(int)p->value);	 
	 p=pop();
	 wdt->y=(int)p->value;
	 p=pop();
	 wdt->x=(int)p->value;	 

	 dlg->widgets->append(wdt);	 
	 found=TRUE;
	 return;
      } 
   }
   if(found==FALSE){
      error(WARNING,"Cannot find dialog %s.",name);   
   }
   
   return;
}

void SamplinScript::create_myreaddlg(char type) /* create command 'read' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'readdlg'");
  cmd=add_command(MYREADDLG);
  lastcommand=cmd;
  cmd->tag=type;
  cmd->args=FALSE;
}

void SamplinScript::myreaddlg(struct command *cmd) /* read string or double */
{
   Script_input *dlg;
   double d;
   int i,c;
   static char buffer[1000];
   struct stackentry *s,*p;
   FILE *str;
   int tileol; /* read til end of line */

   ERROR(NOTE,(NOTE,"begin myreaddlg"));
   
   p=pop();
   //   if(p->type!=PTR)error(ERROR,"Input expected pointer on stack");
   
   tileol=cmd->args;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Waiting for input til %s",tileol ? "end of line":"whitespace"));
   
   dlg = new Script_input(NULL,"Input dialog");
   dlg->setTitle(inputprompt);
   strcpy(buffer,"");
   if(cmd->tag=='d')
     sprintf(buffer,"%g",*(double*)(p->pointer));
   if(cmd->tag=='s')
     strncpy(buffer,(char *)p->pointer,999);
   
   dlg->setText(buffer);
   dlg->show();
   strncpy(buffer,dlg->getText(),999);
   delete dlg;	
   
   if (cmd->tag=='s') { /* read string */
      s=push();
      s->type=STRING;
      s->pointer=my_strdup(buffer);
      //     if(p->pointer)free(p->pointer);
//     p->pointer=my_strdup(buffer);
   }
   else { /* read double */
      //    s=push();
      //    s->type=NUMBER;
      if (/*c==EOF ||*/ (sscanf(buffer,"%lf",&d)==0))
	//      s->value=0.0;
	*(double*)(p->pointer)=0.0;
      else 
	//      s->value=d;
	*(double*)(p->pointer)=d;
   }
   
   ERROR(NOTE,(NOTE, "end myreaddlg"));
}

void SamplinScript::create_cursor(int f)
{
   struct command *cmd;
  
   error(DIAGNOSTIC,"Creating command 'cursor '");
   cmd=add_command(CURSOR);
   cmd->tag=f;
}

void SamplinScript::cursor(struct command *cmd)
{
   int type;
   dlgitem *dlg;
   
   type=cmd->tag;
   
   for(dlg=dialogs.first();dlg!=0;dlg=dialogs.next()){
      if(dlg->dlg!=NULL)
	switch(type){
	 case 0:
	   dlg->dlg->setCursor(arrowCursor);
	   break;
	 case 1:
	   dlg->dlg->setCursor(waitCursor);
	   break;
	}
      
   }
   
}



void SamplinScript::widgetSlot(widgetitem *w)
{
   ERROR(NOTE,(NOTE,"Message got: type:%i, active=%i,action=%s\n",w->type,w->active,w->action));
   
   if(waitMulti==NULL)waitMulti=w;
   
}

void SamplinScript::widgetAction(void)
{
   struct command jmpcmd;
   
   if(multiflag==FALSE)return;
   
   if(waitMulti!=NULL){
      ERROR(NOTE,(NOTE,"Message exec: type:%i, active=%i,action=%s\n",waitMulti->type,waitMulti->active,waitMulti->action));
      if(waitMulti->active==FALSE){
	 waitMulti->active=TRUE;
	 jmpcmd.type=IGOSUB;
	 jmpcmd.pointer=waitMulti->action;
	 jmpcmd.spointer=waitMulti;
	 jump(&jmpcmd);
	 current=current->next;
      }
      waitMulti=NULL;
   }

}

void SamplinScript::progressSlot()
{
   get_sym("progress_cancel",NUMBER,TRUE)->value=TRUE;   
}

