#include <qwt_drange.h>
#include <qwt_math.h>

const double QwtDblRange::MinRelStep = 1.0e-10;
const double QwtDblRange::DefaultRelStep = 1.0e-2;
const double QwtDblRange::MinEps = 1.0e-10;

//--------------------------------------------------
//
//.S	QwtDblRange
//
//  Constructor 
//
//.s Description
//
//--------------------------------------------------
QwtDblRange::QwtDblRange()
{
    //
    // set some reasonable values
    //
    d_minValue = 0;
    d_maxValue = 100.0;
    d_value = 0.0;
    d_step = 1.0;

}


//------------------------------------------------------------
//
//.S QwtDblRange::setNewValue
//
//  
//
//.s Parameters
//	double x
//
//
//------------------------------------------------------------
void QwtDblRange::setNewValue(double x, int align = 0)
{
    double vmin,vmax;
    
    d_prevValue = d_value;

    vmin = qwtMin(d_minValue, d_maxValue);
    vmax = qwtMax(d_minValue, d_maxValue);

    // 
    // Range check
    //
    if (x < vmin)
    {
	if ((d_periodic) && (vmin != vmax))
	   d_value = vmax + floor( (x - vmax ) / (vmin - vmax ) ) 
	      * (vmin - vmax);
	else
	   d_value = vmin;
    }
    else if (x > vmax)
    {
	if ((d_periodic) && (vmin != vmax))
	   d_value = vmin + floor( ( x - vmin) / (vmax - vmin )) 
	      * (vmax - vmin);
	else
	   d_value = vmax;
    }
    else
       d_value = x;

    d_exactPrevValue = d_exactValue;
    d_exactValue = d_value;
    
    // align to grid
    if (align)
    {
	if (d_step != 0.0)
	   d_value = d_minValue + rint((d_value - d_minValue) / d_step ) * d_step;
	else
	   d_value = d_minValue;
	
	// correct rounding error at the border
	if (fabs(d_value - d_maxValue) < MinEps * fabs(d_step))
	   d_value = d_maxValue;

	// correct rounding error if value = 0
	if (fabs(d_value) < MinEps * fabs(d_step))
	   d_value = 0.0;
    }

    if (d_prevValue != d_value)
       valueChange();

}


//--------------------------------------------------
//
//.S fitValue
//
//	
//
//.s Description
//.s Parameters
//		double x
//
//--------------------------------------------------
void QwtDblRange::fitValue(double x)
{
    setNewValue(x,1);
}


//--------------------------------------------------
//
//.S setValue
//
//	
//
//.s Parameters
//		double x
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::setValue(double x)
{
    setNewValue(x,0);
}

//--------------------------------------------------
//
//.S setRange
//
// 
//
//.s Parameters
//	double vmin:
//      double vmax:
//      double vstep:
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::setRange(double vmin, double vmax, double vstep = 0.0)
{
    
    int rchg;
    
    rchg = ((d_maxValue != vmax) || (d_minValue != vmin));
    
    if (rchg) {
	    d_minValue = vmin;
	    d_maxValue = vmax;
	}
    
    //
    // look if the step width has an acceptable 
    // value or otherwise change it.
    //
    if (vstep == 0.0) 
       setStep(d_step);
    else
    {
	rchg = 1;  
	setStep(vstep);
    }
    
    
    // 
    // If the value lies out of the range, it 
    // will be changed. Note that it will not be adjusted to 
    // the new step width if the step width has 
    // been changed.
    //
    setNewValue(d_value,0);
    
    // call notifier after the step width has been 
    // adjusted.
    if (rchg)
       rangeChange();
    
}

//--------------------------------------------------
//
//.S setStep
//
// 
//
//.s Parameters
//	double vstep
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::setStep(double vstep)
{
    double newStep,intv;

    intv = d_maxValue - d_minValue;
    
    if (vstep == 0.0)
       newStep = intv * DefaultRelStep;
    else
    {
	if ((intv > 0) && (vstep < 0) || (intv < 0) && (vstep > 0))
	   newStep = -vstep;
	else
	   newStep = vstep;
	
	if ( fabs(newStep) < fabs(MinRelStep * intv) )
	   newStep = MinRelStep * intv;
    }
    
    if (newStep != d_step) {
	    d_step = newStep;
	    stepChange();
	}

}


//--------------------------------------------------
//
//.S incValue
//
//  increment value by n steps
//
//.s Parameters
//	int nSteps:
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::incValue(int nSteps)
{
    setNewValue(d_value + double(nSteps) * d_step,1);
}



//--------------------------------------------------
//
//.S valueChange
//
//  notify change of value 
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::valueChange()
{
}


//--------------------------------------------------
//
//.S rangeChange
//
//  notify change of range
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::rangeChange()
{
}


//--------------------------------------------------
//
//.S stepChange
//
//  notify change of step width
//
//.s Description
//
//--------------------------------------------------
void QwtDblRange::stepChange()
{
}



