#include <iostream.h>

#include "tree.h"
#include "tree.moc"

#include "debug.h"
#include "page.h"
#include "pageobject.h"
#include "plot.h"
#include "worksheet.h"


/* KOMainTreeItem - items in the main tree. These are KTreeListIcons
   with an QObject* field to store the page/plot/worksheet.
*/


static bool TreeDestruction = false;


TreeItem::TreeItem (const char* theText, 
					const QPixmap* thePixmap)
  : KTreeListItem(theText,thePixmap)
{
  the_object = 0;
}


TreeItem::~TreeItem ()
{
}



/* KOMainTree - the main widget tree view. This is a KTreeList with
   some special right mouse mutton functions.
*/

Tree::Tree (QWidget* parent, const char* name, WFlags f)
  : KTreeList (parent,name,f)
{
  tree_menu = new QPopupMenu();
  tree_menu->insertItem("Delete", this, SLOT(deleteCurrentItem()));
}


Tree::~Tree ()
{
  TreeDestruction = true;
  delete tree_menu;
}


void Tree::pageChanged (Page* p)
{
  int i;
  IFDEBUG("korigin.tree",3)
	cout << "Tree::slotPageChanged(): " << p->caption() << endl;
  ENDDEBUG;
  for (i=0; i<(int)count(); i++)
	if (itemAt(i) == p->treeitem()) {
	  if (i == currentItem()) return;
	  setCurrentItem(i);
	  break;
	}
  repaint();
}


void Tree::plotChanged (Plot* p)
{
  int i;
  IFDEBUG("korigin.tree",3)
	cout << "Tree::plotChanged(): " << p->description() << endl;
  ENDDEBUG;
  for (i=0; i<(int)count(); i++)
	if (itemAt(i) == p->treeitem()) {
	  if (i == currentItem()) return;
	  setCurrentItem(i);
	  break;
	}
  repaint();
}


void Tree::worksheetChanged (Worksheet* w)
{
  IFDEBUG("korigin.tree",3)
	cout << "Tree::slotWorksheetChanged(): " << w->caption() << endl;
  ENDDEBUG;
}


void Tree::mousePressEvent (QMouseEvent* e)
{
  int itemClicked;
  KTreeListItem* item;
  QPoint mouseCoord;

  KTreeList::mousePressEvent(e);

  if (e->button() != RightButton) return;

  mouseCoord = e->pos();
  itemClicked = findRow(mouseCoord.y());
  if(itemClicked == -1) return;
  item = itemAt(itemClicked);
  if(!item) return;

  tree_menu->popup(mapToGlobal(mouseCoord));
}


void Tree::deleteCurrentItem ()
{
  int dlg;
  char msg [1024];
  Page* page;
  PageObject* obj;
  Worksheet* ws;
  TreeItem* item;

  item = (TreeItem*)itemAt(currentItem());
  if (item == 0 || item->object() == 0) return;

  IFDEBUG("korigin.tree",2)
	cout << "Tree::slotDeleteItem()" << endl
		 << "   This would delete item #" << currentItem() << " (a "
		 << item->object()->className() << ") from the tree." << endl;
  ENDDEBUG;

  if (strcmp(item->object()->className(),"Page")==0)
	{
	  page = (Page*)(item->object());
	  sprintf(msg,"Delete the following page and all objects on it:\n %s",
			  page->caption());
	  dlg = QMessageBox::warning(0,"Delete Page?",msg,"OK","Abort",0,1,1);
	  if (dlg != 0) return;
	  takeItem(currentItem());
	  emit removePage(page);
	}
  else if (strcmp(item->object()->className(),"Worksheet")==0)
	{
	  ws = (Worksheet*)(item->object());
	  sprintf(msg,"Delete the following worksheet and all its data:\n %s",
			  ws->caption());
	  dlg = QMessageBox::warning(0,"Delete Worksheet?",msg,"OK","Abort",0,1,1);
	  if (dlg != 0) return;
	  takeItem(currentItem());
	  emit removeWorksheet(ws);
	}
  else if (item->object()->inherits("PageObject"))
	{
	  obj = (PageObject*)(item->object());
	  sprintf(msg,"Delete the following object on\npage %s:\n %s",
			  obj->page()->caption(),obj->description());
	  dlg = QMessageBox::warning(0,"Delete Object?",msg,"OK","Abort",0,1,1);
	  if (dlg != 0) return;
	  takeItem(currentItem());
	  emit removePageObject(obj);
	}
  else
	{
	  IFDEBUG("korigin.tree",3)
		cout << "Tree::slotDeleteItem() found a Thing That Should Not Be: " 
			 << item->object()->className() << endl;
	  ENDDEBUG;
	}
}
