// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with the package.



#ifndef KROT_ARNIROT_H
#define KROT_ARNIROT_H



#include "calculationParameter.h"
#include "transition.h"

#include <qobject.h>

#include <kprocess.h>

#include <map>
#include <string>
#include <vector>



class QMessageBox;

typedef map< QNum, double > mapAssignment;



// TODO:
//
// arnirot shall write out progress status to stderr.
// Here we need to install a periodically called function that reads/parses it
// and displays/updates an progress indicator.



/**
 * Interface function to external arnirot.
 *
 * @param param  On input takes the start parameter for the calculation, on
 *               output contains the final parameters.
 * @param lines  Provide an empty vector to fill up with calculated lines.
 * @param asn    Assignments to use for fits. For simulations this might be empty.
 * @return       Success status
 */
class Arnirot : public QObject
{
    Q_OBJECT

public:
    
    /**
     * Constructor
     *
     * Initialize all class data, start arnirot process, put data to its stdin,
     * setup reading of stdout
     *
     * @param param Calculation parameter
     * @param asn   Assignments
     */
    Arnirot( const CalculationParameter& param, const mapAssignment& asn = mapAssignment() );

    /**
     * Destructor
     *
     * Stop arnirot, if it running.
     */
    ~Arnirot();

    /**
     * Actually start the program.
     */
    void start();
    
protected slots:

    /**
     * arnirot is finished.
     *
     * Get all output and send emit it.
     */
    void finished( KProcess * );

    /**
     * Send signal to process.
     *
     * @parent signal Signal to send.
     * @return success report.
     */
    bool kill( int signal=SIGTERM );
    
    /**
     * read arnirots stdout and put it into string output
     *
     * @param buffer Additional arnirot output.
     * @param len    Length ot this output buffer
     */
    void readStdout( KProcess *, char *buffer, int len );
    
    /**
     * This sends all input data to arnirot. Since that needs to happen
     * asynchronously, we register ourself to get the wroteStdin signal and
     * proceed with input then.
     */
    void send( KProcess * );
    
signals:

    /**
     * Send the results and the calculated lines to the world.
     *
     * @param param Final calculation parameter. On a simulation these are the
     *              input parameter, on a fit the resulting parameter of the
     *              fit.
     * @param lines Calculated transitions.
     */
    void done( const CalculationParameter param, vector< Transition > lines );

    void message( const QString& msg );
    

protected:

    int idx;
    
    string input, output;

    QMessageBox *msgbox;
    
    CalculationParameter pars;

    KProcess process;
};



#include "arnirot_inline.h"



#endif
