// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "dataset.h"
#include "functor.h"
#include "exception.h"
#include "krot.h"
#include "krotData.h"
#include "spectrum.h"

#include <qstring.h>

#include <kapp.h>

#include <sys/types.h>

#include <algorithm>
#include <cstdio>
#include <fstream>
#include <numeric>
#include <vector>


DataSet& DataSet::operator=( const DataSet& set )
{
    KROT_LAUNCH( "Launching DataSet::operator=" );
    // check for object identity
    if( this == &set )
	return *this;
    // copy easy data
    data_valid = set.data_valid;
    nam = set.nam;
    realstrt = set.realstrt;
    realstp = set.realstp;
    scalg = set.scalg;
    siz = set.siz;
    strt = set.strt;
    stp = set.stp;
    xy_shifts = set.xy_shifts;
    // memory reallocation
    if( 0 != data )
	delete [] data;
    if( 0 != realdata )
	delete [] realdata;
    data = new double[ siz ];
    realdata = new double[ siz ];
    // copy data
    copy( set.data, &( set.data[ siz ] ), data );
    copy( set.realdata, &( set.realdata[ siz ] ), realdata );
    // determine addresses of min/max
    mn = std::min_element( data, &( data[ siz ] ) );
    mx = std::max_element( data, &( data[ siz ] ) );
    maxmin_valid = true;
    return *this;
}



double DataSet::chi2( const DataSet& other ) const
{
    KROT_LAUNCH( "Launching DataSet::chi2" );
    int64_t start = std::max( strt, other.strt ),
	stop = std::min( stp, other.stp );
    if( start > stop )
	throw NoDataSetOverlap( *this, other );
    double chisq( 0.0 );
    for( int64_t i=start; i<= stop; i++ )
	chisq += sqr( ( *this )[ i ] - other[ i ] );
    return chisq / siz;
}



Spectrum& DataSet::difference( const DataSet& other ) const
{
    KROT_LAUNCH( "Launching DataSet::difference" );
    int64_t start = std::max( strt, other.strt ),
	stop = std::min( stp, other.stp );
    if( start > stop )
	throw NoDataSetOverlap( *this, other );
    Spectrum *diff = new Spectrum( *this );
    diff->siz = stop - start + 1;
    // memory reallocation
    if( 0 != diff->data )
	delete [] diff->data;
    if( 0 != diff->realdata )
	delete [] diff->realdata;
    diff->data = new double[ diff->siz ];
    diff->realdata = new double[ diff->siz ];
    std::transform( diff->realdata, &( diff->realdata[ diff->siz ] ),
		    &( data[ start - strt ] ), &( other.data[ other.strt - start ] ), minus<double>() );
    std::copy( diff->realdata, &( diff->realdata[ diff->siz ] ), diff->data );
    diff->data_valid = true;
    diff->mx = std::max_element( diff->data, &( diff->data[ diff->siz ] ) );
    diff->mn = std::min_element( diff->data, &( diff->data[ diff->siz ] ) );
    diff->maxmin_valid = true;
    diff->setName( QString().sprintf( i18n( "Difference of %s and %s" ),
				      shortName().data(), other.shortName().data() ) );
    diff->scalg = 1.0;
    diff->stp = stop;
    diff->realstp = stop;
    diff->strt = start;
    diff->realstrt = start;
    diff->xy_shifts = make_pair( 0, 0 );
    return *diff;
}



double DataSet::integral() const
{
    KROT_LAUNCH( "Launching DataSet::integral" );
    if( siz < 2 )
	throw ShortDataSet( *this );
    double sum = 0.5 * data[ 0 ] + 0.5 * data[ siz - 1 ];
    std::accumulate( &( data[ 1 ] ), &( data[ siz - 1 ] ), sum );
    return sum / dlta;
}



void DataSet::load( const QString& name, const int type )
{
    KROT_LAUNCH( "Launching DataSet::load" );
    switch(type) {
    case LINES_DAT_DATA:
	loadLinesDat( name );
	break;
    case XY_DATA:
	loadXY( name );
	break;
    case UNKNOWN_DATA:
    default:
	throw UnknownFileType();
	break;
    }
    return;
}



void DataSet::recalculateData()
{
    KROT_LAUNCH( "Launching DataSet::recalculate" );
    if( ! data_valid ) {
	for( u_int64_t i=0; i<siz; i++ )
	    data[ i ] = realdata[ i ] * scalg + xy_shifts.second;
	data_valid = true;
    }
    if( ! maxmin_valid ) {	
	mx = std::max_element( data, &( data[ siz ] ) );
	mn = std::min_element( data, &( data[ siz ] ) );
	maxmin_valid = true;
    }
    krotData->checkLimits();
}



void DataSet::setData( const int64_t start, const int64_t stop, double *newdata )
{
    KROT_LAUNCH( "Launching DataSet::setData" );
    if( stop <= start )
	throw ShortDataSet( *this );
    data_valid = false;
    maxmin_valid = false;
    realstp = stop;
    realstrt = start;
    stp = stop;
    strt = start;
    siz = stop - start + 1;
    xy_shifts = make_pair( 0, 0 );
    if( 0 != data )
	delete [] data;
    if( 0 != realdata )
	delete [] realdata;
    data = new double[ siz ];
    realdata = newdata;
    scalg = 500 / ( *std::max_element( newdata, &( newdata[ siz ] ) )
		    - *std::min_element( newdata, &( newdata[ siz ] ) ) );
    recalculateData();
}



void DataSet::save( const QString& name, const int type ) const
{
    KROT_LAUNCH( "Launching DataSet::save" );
    switch( type ) {
    case XY_DATA:
	saveXY( name );
	break;
    case UNKNOWN_DATA:
    default:
	throw UnknownFileType();
	break;
    }
}



void DataSet::saveXY( const QString& name ) const
{
    KROT_LAUNCH( "Launching DataSet::saveXY" );
	ofstream file( name.data() );
    if( ! file.is_open() )
		throw FileNotFound( name.data() );
	for( double i = start(); i <= stop(); i += delta() ) {
		file.width( 22 );
		file << i;
		file.width( 20 );
		file << operator[]( i ) << endl;
    }
}



void DataSet::setName( const QString& name )
{
    KROT_LAUNCH( "Launching DataSet::setName" );
    nam = name;
    // check wether this name already exists
    mapSimulation::const_iterator sim  = krotData->simulations().begin();
    mapSpectrum::const_iterator   spec = krotData->spectra().begin();
    int i = 0;
    do {
	char buf[8];
	while( krotData->spectra().end() != spec )
	    if( nam != spec->first )
		spec++;
	    else {
		i++;
		sprintf(buf,"<%d>",i);
		nam = name + buf;
		sim  = krotData->simulations().begin();
		spec = krotData->spectra().begin();
	    }
	while( krotData->simulations().end() != sim )
	    if( nam != sim->first )
		sim++;
	    else {
		i++;
		sprintf(buf,"<%d>",i);
		nam = name + buf;
		sim  = krotData->simulations().begin();
		spec = krotData->spectra().begin();
	    }
    } while( ( krotData->spectra().end() != spec )
	     || ( krotData->simulations().end() != sim ) );
    KROT_DEBUG1( "Name: %s", name.data() );
    return;
}



//--------------------------------------------------------------------------------------------------
// These are declared, but shouldn't be defined here !
// These should be abstract functions, but we cannot make DataSet an abstract
// class. Also compiler complain on missing references when they are not
// defined, although they are not used, since they are overloaded in the
// approbiate derived classes (Jochen).

bool DataSet::loadLinesDat( const QString& )
{
    KROT_ERROR( KDEBUG_KROT_GENERAL, "DataSet::loadLinesDat should be abstract!" );
    return false;
}



void DataSet::loadXY( const QString& )
{
    KROT_ERROR( KDEBUG_KROT_GENERAL, "DataSet::loadXY should be abstract!" );
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
