// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "deleteSet.h"
#include "krot.h"
#include "krotData.h"

#include <qdialog.h>
#include <qlayout.h> 
#include <qlistbox.h> 

#include <kapp.h>
#include <kbuttonbox.h>
#include <klocale.h>
#include <kmsgbox.h> 

#include <iostream.h>
#include <string>
#include <vector>



DialogDeleteSet::DialogDeleteSet( KRotData *krotdata, QWidget *parent, const char *name )
    : QDialog(parent, name, true),
      krotData( krotdata )
{
    KROT_LAUNCH( "Launching DialogDeleteSet constructor" );
    setCaption( i18n("KRot - Delete sets ...") );
    QBoxLayout *layout = new QVBoxLayout( this, KROT_DIALOGMARGIN );

    // *** listbox ***
    list = new QListBox(this, "Listbox",
			WStyle_Customize | WStyle_DialogBorder | WStyle_SysMenu | WStyle_Title );
    // insert items for all sets
    list->setAutoUpdate( false );
    mapSpectrum::const_iterator spec = krotData->spectra().begin();
    while( spec != krotData->spectra().end() ) {
	list->inSort( spec->second.shortName().data() );
	spec++;
    }
    mapSimulation::const_iterator sims = krotData->simulations().begin();
    while( sims != krotData->simulations().end() ) {
	list->inSort( sims->first.data() );
	sims++;
    }
    list->setAutoUpdate( true );
    layout->addWidget( list );

    // *** buttons ***
    KButtonBox *bbox = new KButtonBox(this);
    bbox->addStretch( 5 );
    QPushButton *kill = bbox->addButton( i18n( "Delete" ) );
    if( list->count() == 0 )
	kill->setEnabled( false );
    bbox->addStretch( 10 );
    QPushButton *killall = bbox->addButton( i18n( "Delete All" ) );
    if( list->count() == 0 )
	kill->setEnabled( false );
    bbox->addStretch( 10 );
    QPushButton *cancel = bbox->addButton( i18n( "Close" ) );
    bbox->addStretch( 10 );
    QPushButton *help   = bbox->addButton( i18n( "Help" ) );
    bbox->addStretch( 5 );
    bbox->layout();
    bbox->setFixedHeight( bbox->sizeHint().height() + 10 );
    
    layout->addWidget( bbox );
    layout->activate();
    
    // connect signals
    connect( list, SIGNAL( selected(const char *) ), this, SLOT( doDeletion(const char *) ) );
    connect( kill, SIGNAL( clicked() ), this, SLOT( doDeletion() ) );
    connect( killall, SIGNAL( clicked() ), this, SLOT( deleteAll() ) );
    connect( cancel, SIGNAL( clicked() ), this, SLOT( reject() ) );
    connect( help, SIGNAL( clicked() ), this, SLOT( help() ) );

    setGeometry( x(), y(), bbox->sizeHint().width(), height() );
    show();
    return;
}



void DialogDeleteSet::deleteAll()
{
    if( 1 == KMsgBox::yesNo( this, i18n( "Kill All Datasets" ),
			     i18n( "Are you sure you want to kill all datasets ?" ),
			     KMsgBox::QUESTION | KMsgBox::DB_SECOND,
			     i18n( "Yes" ), i18n( "No" ) ) ) {
	krotData->eraseAll();
	accept();
    }
    return;
}



void DialogDeleteSet::doDeletion()
{
    KROT_LAUNCH( "Launching DialogDeleteSet::doDeletion()" );
    QString msg = ( i18n( "Are you sure to kill " )
		    + QString( list->text( list->currentItem() ) )
		    + QString( i18n( " ?" ) ) );
    int res = KMsgBox::yesNo( this, i18n("Delete set"), msg,
			      KMsgBox::QUESTION | KMsgBox::DB_SECOND,
			      i18n( "Yes" ), i18n( "No" ) );
    if( 1 == res ) {
	QString msg( list->text( list->currentItem() ) );
	krotData->erase( msg );
	msg += " deleted.";
	emit message( msg );
	list->removeItem( list->currentItem() );
	if( 0 == list->count() ) {
	    emit message( "Last set deleted." );
	    accept();
	}
    }
    return;
}



void DialogDeleteSet::doDeletion( const char * )
{
    KROT_LAUNCH( "Launching DialogDeleteSet::doDeletion(const char *)" );
    doDeletion();
    return;
}



void DialogDeleteSet::help()
{
    KROT_LAUNCH( "Launching DialogDeleteSet::help" );
    kapp->invokeHTMLHelp("","");
    return;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
