// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.
//



#include "lineSelection.h"

#include "krotData.h"
#include "transition.h"

#include <qpushbutton.h>


DialogLineSelection::DialogLineSelection( KRotData *krotdata, QWidget* parent, const char* name )
    : DialogLineSelectionData( parent, name ),
      krotData( krotdata )
{
    KROT_LAUNCH( "Launching DialogLineSelection constructor" );
    setCaption( i18n( "Line selection" ) );
    buttons->addStretch( 5 );
    QPushButton *close = buttons->addButton( i18n( "Close" ) );
    buttons->addStretch( 10 );
    QPushButton *help = buttons->addButton( i18n( "Help" ) );
    buttons->addStretch( 5 );
    buttons->layout();
    QFont f( "Lucidatypewriter", 14 );
    lines->setFont( f );
    connect( close, SIGNAL(released()), this, SLOT(hide()) );
    connect( help, SIGNAL(released()), this, SLOT(help()) );
    connect( combo, SIGNAL(highlighted(const char *)), this, SLOT(newSet(const char *)) );

    return;
}



QNum DialogLineSelection::selection()
{
    int idx = lines->currentItem();
    if( idx >= 0 ) {
	const char * buffer = lines->text( idx );
	const unsigned short asn[] = { atoi(&buffer[22]), atoi(&buffer[26]), atoi(&buffer[30]),
				       atoi(&buffer[39]), atoi(&buffer[43]), atoi(&buffer[47]) };
	return QNum( asn );
    }
    return QNum();
}



void DialogLineSelection::setRange( const int64_t start, const int64_t stop )
{
    KROT_DEBUG( "Launching DialogLineSelection::setRange" );
    combo->clear();
    mapSimulation::const_iterator sims = krotData->simulations().begin();
    while( sims != krotData->simulations().end() ) {
	combo->insertItem( sims->first.data() );
	sims++;
    }
    lines->clear();
    if( 0 < combo->count() ) {
	const Simulation& sim = krotData->simulations().find(combo->text( combo->currentItem()))->second;
	lines->setAutoUpdate( false );
	vector<Transition>::const_iterator iter = sim.lines().begin();
	while( iter->frequency() < start )
	    iter++;
	while( iter->frequency() <= stop ) {
	    char buffer[ 64 ];
	    sprintf( buffer, "%10.0f  %8f   %3d %3d %3d  <-- %3d %3d %3d",
		     iter->frequency(), iter->intensity(),
		     iter->operator[]( QNum::eJ ),
		     iter->operator[]( QNum::eKa ), iter->operator[]( QNum::eKc ),
		     iter->operator[]( QNum::gJ ),
		     iter->operator[]( QNum::gKa ), iter->operator[]( QNum::gKc )
		);
	    lines->inSort( buffer );
	    iter++;
	}
	lines->setAutoUpdate( true );
	if( isVisible() ) {
	    lines->repaint();
	} else {
	    show();  
	}
	raise();
    }
    return;
}
