/* -*- C++ -*-
 * Dialog widget using the addressbook,
 * provided for inclusion in other programs.
 *  
 * the Configuration Database library
 * copyright:  (C) Mirko Sucker, 1998
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@hamburg.netsurf.de>
 *                          <mirko.sucker@unibw-hamburg.de>
 * requires:   C++-compiler, STL, string class, NANA
 * $Revision: 1.1 $
 */

#ifndef KABAPI_H
#define KABAPI_H

#include <qdialog.h>
#include "widget.h"

class QPushButton;
class QFrame;

/** The class KabAPI provides a public interface to access the 
  * users address database created using kab. The complete 
  * functionality regarding database operations provided by kab is
  * available using an object of this class.
  *
  * The class is derived from the class QDialog, thus objects
  * can be used as a dialog where the user may select an address
  * similar to the interface of kab. 
  * The following code may be used to let the user select an address:
  * <pre>
  * KabAPI kabapi(this);
  * if(dialog.init()!=KabAPI::NoError)
  *   {
  *     ... error handling
  *   }
  * AddressBook::Entry entry;
  * if(kabapi.exec())
  *   {
  *     if(!kabapi.getEntry(entry))
  *       {
  *         // ... the database is empty
  *       } else {
  *         // ... use the entry
  *       }
  *   }
  * ...
  * </pre>
  * Please note that the kab code assumes that a QImage-IO-handler for
  * jpeg pictures has been installed. You will get a message that the 
  * background image could not been loaded if you forget it. 
  * See kab/main.cc for an example.
  * @short The class KabAPI defines the API to access user address databases.
  * @author Mirko Sucker <mirko.sucker@hamburg.netsurf.de>
  * @version $Id: kabapi.h,v 1.1 1998/05/02 13:31:04 mirko Exp $
  * @see AddressBook AddressWidget QDialog
  */

class KabAPI : public QDialog
{
  Q_OBJECT
public:
  /** ErrorCode is used in all methods that may return more status 
    * codes than true or false.<br>
    * <tt> NoError </tt> The operations has been finished successful.<br>
    * <tt> NotImplemented </tt> The feature you requested is not 
    * implemented by KabAPI.<br>
    * <tt> PermDenied </tt> Not all permissions needed for the operation 
    * were available. Also returned if a database is locked.<br>
    * <tt> NoSuchFile </tt> No database file exists for this user.<br>
    * <tt> NoEntry </tt> You tried to retrieve an entry but there is none.<br>
    * <tt> InternError </tt> There was a program error.
    */
  enum ErrorCode {
    NoError,
    NotImplemented,
    PermDenied,
    NoSuchFile,
    NoEntry,
    InternError };
  /** The constructor creates a KabAPI object, but it does not load the
    * database itselfes, as you could not query if this could be done 
    * without failures. Thus you have to call @ref #init before you can 
    * use the database.
    * @param parent The QWidget pointer to the parent widget.
    */
  KabAPI(QWidget* parent=0);
  /** You must call init before accessing the dialog. init opens the
    * database file (usually $HOME/.kde/share/apps/kab/addressbook.database)
    * and loads its contents. If the database is already opened by another 
    * instance of kab and readonly is NOT true, init returns 
    * PermDenied. 
    * You may try to open a database for reading and writing first,
    * and if it fails with PermDenied try again to open it readonly.
    * The dialog will not provide the capabilities to store the database or 
    * to edit an entry then.
    * @return NoError if all succeeded or a valid ErrorCode.
    * @see #ErrorCode
    */
  ErrorCode init(bool readonly=true);

  /** The method getEntry returns the selected entry. 
    * @return NoError if all succeeded or a valid ErrorCode.
    * @see #ErrorCode
    * @param entry Reference to an AddressBook::Entry -object.
    */
  ErrorCode getEntry(AddressBook::Entry& entry);
  /** The method requires that the database is not opened readonly.
    * @short Adds an entry to the users database.
    * @return NoError if all succeeded or a valid ErrorCode, especially PermDenied.
    * @param entry Reference to the entry to be added.
    */
  ErrorCode add(const AddressBook::Entry& entry);
protected:
  AddressWidget* widget;
  QPushButton* buttonOK;
  QPushButton* buttonCancel;
  QFrame* frameLine;
public slots:
  void initializeGeometry();
};

#endif // KABAPI_H
