#include "kbiff.h"
#include "kbiff.moc"

KBiff::KBiff(QWidget* parent)
	: QLabel(parent)
{
	// Init the audio server.  serverStatus return 0 when it is ok
	m_hasAudio = (m_audioServer.serverStatus() == 0) ? true : false;

	// enable the session management stuff
	connect(kapp, SIGNAL(saveYourself()), this, SLOT(m_saveYourself()));

	// by default, we don't want to dock
	m_isDocked = false;

	// init the pop3 size even if we aren't using it
	m_lastSize = 0;
	m_storedSize = 0;
	m_popPort = 110;

	// we don't want a margin
	setMargin(0);
	setAlignment(AlignLeft | AlignTop);

	// setup a default checker
	connect(this, SIGNAL(checkNewMail()), SLOT(m_checkLocalMail()));
}

KBiff::~KBiff()
{
}

void KBiff::m_displayPixmap(bool force_pixmap)
{
	// we will try to deduce the pixmap (or gif) name now.  it will
	// vary depending on dock state and MAIL_STATE
	QString pixmap_name, mini_pixmap_name;
	switch (m_mailState)
	{
		case NO_MAIL:
			pixmap_name = m_noMailPixmapPath;
			break;
		case OLD_MAIL:
			pixmap_name = m_oldMailPixmapPath;
			break;
		case NEW_MAIL:
			pixmap_name = m_newMailPixmapPath;
			break;
	}
	mini_pixmap_name = "mini-" + pixmap_name;

	// Get a list of all the pixmap paths.  This is needed since the
	// icon loader only returns the name of the pixmap -- NOT it's path!
	QStrList *dir_list = kapp->getIconLoader()->getDirList();
	QFileInfo file, mini_file;
	for (unsigned int i = 0; i < dir_list->count(); i++)
	{
		QString here = dir_list->at(i);

		// check if this file exists.  If so, we have our path
		file.setFile(here + '/' + pixmap_name);
		mini_file.setFile(here + '/' + mini_pixmap_name);

		// if we are docked, check if the mini pixmap exists FIRST.
		// if it does, we go with it.  if not, we look for the
		// the regular size one
		if (m_isDocked && mini_file.exists())
		{
			file = mini_file;
			break;
		}

		if (file.exists())
			break;
	}

	QPixmap pixmap(file.absFilePath());

	// this kludge is needed since QPixmap doesn't support
	// animated gifs and QMovie doesn't support pixmaps!
	if (force_pixmap)
		setPixmap(pixmap);
	else
	{
		m_movie = QMovie((const char*)file.absFilePath());
		m_movie.connectStatus(this, SLOT(m_movieStatus(int)));
		setMovie(m_movie);
	}

	// and set the size
	setFixedSize(pixmap.width(), pixmap.height());
	resize(pixmap.width(), pixmap.height()); 
}

void KBiff::m_movieStatus(int status)
{
	// check the status of this movie.  if it cannot be
	// loaded, then it is not a gif.  if so, force m_displayPixmap
	// to use a pixmap
	if (status == QMovie::UnrecognizedFormat)
	{
		m_displayPixmap(true);
	}
}

void KBiff::m_resetState()
{
	// load in the pixmaps
	if (m_isDocked)
	{
		QString mini_nomail, mini_oldmail, mini_newmail;
		mini_nomail.sprintf("mini-%s", (const char*)m_noMailPixmapPath);
		mini_newmail.sprintf("mini-%s", (const char*)m_newMailPixmapPath);
		mini_oldmail.sprintf("mini-%s", (const char*)m_oldMailPixmapPath);
		m_noMailPixmap = kapp->getIconLoader()->loadMiniIcon(mini_nomail);
		m_newMailPixmap = kapp->getIconLoader()->loadMiniIcon(mini_newmail);
		m_oldMailPixmap = kapp->getIconLoader()->loadMiniIcon(mini_oldmail);
	}
	else
	{
		m_noMailPixmap = kapp->getIconLoader()->loadIcon(m_noMailPixmapPath);
		m_newMailPixmap = kapp->getIconLoader()->loadIcon(m_newMailPixmapPath);
		m_oldMailPixmap = kapp->getIconLoader()->loadIcon(m_oldMailPixmapPath);
	}

	// start the timer (kill the old one if it exists)
	if (m_timerID > 0)
		killTimer(m_timerID);

	m_timerID = startTimer(m_poll * 1000);

	// initialize the state
	if (m_readLocalMailbox)
	{
		QFileInfo mailbox(m_mailboxPath);
		m_lastRead = mailbox.lastRead();
		m_mailState = (mailbox.size() > 0) ? OLD_MAIL : NO_MAIL;
	}
	else
		m_mailState = NO_MAIL;

	// repaint this thing
	m_displayPixmap();

	// put the name on this thing (the first setCaption() is _needed_
	// in some strange cases.  no clue why.
	setCaption("");
	setCaption((const char*)m_name);

	// disconnect the current signals
	disconnect(this);

	// connect the appropriate slot
	if (m_readLocalMailbox)
		connect(this, SIGNAL(checkNewMail()), SLOT(m_checkLocalMail()));
	else
		connect(this, SIGNAL(checkNewMail()), SLOT(m_checkPOPMail()));

	// do one check initially
	m_checkMailNow();
}

bool KBiff::readConfig(const char* profile)
{
	if (profile)
	{
		KBiffSetupDlg setup_dlg;

		if (setup_dlg.readMailboxConfig(profile))
		{
			m_command = setup_dlg.getCommand();
			m_poll = setup_dlg.getPoll();
			m_noMailPixmapPath = setup_dlg.getNoMailPixmapPath();
			m_newMailPixmapPath = setup_dlg.getNewMailPixmapPath();
			m_oldMailPixmapPath = setup_dlg.getOldMailPixmapPath();
			m_name = setup_dlg.getName();

			// get the autodock option
			m_isDocked = setup_dlg.getDock();
			m_sessionManagement = setup_dlg.getSessionManagement();

			m_readLocalMailbox = setup_dlg.getReadLocalMailbox();
			if (m_readLocalMailbox)
				m_mailboxPath = setup_dlg.getMailboxPath();
			else
			{
				m_popServer = setup_dlg.getPopServer();
				m_username  = setup_dlg.getUser();
				m_password  = setup_dlg.getPassword();
				m_popPort   = setup_dlg.getPort();
			}

			m_runCommand = setup_dlg.getRunCommand();
			m_runCommandPath = setup_dlg.getRunCommandPath();
			m_playSound = setup_dlg.getPlaySound();
			m_playSoundPath = setup_dlg.getPlaySoundPath();
			m_beep = setup_dlg.getBeep();
		}

		if (!m_mailboxPath.isEmpty() || !m_popServer.isEmpty())
		{
			m_resetState();
			QToolTip::add(this, m_name);

			return true;
		}
		else
			return false;
	}
	else
	// read in the config spec only if restored
	if (kapp->isRestored())
	{
		KConfig *config = kapp->getSessionConfig();

		config->setGroup("KBiff");
	
		m_noMailPixmapPath = config->readEntry("NoMailPixmapPath");
		m_newMailPixmapPath = config->readEntry("NewMailPixmapPath");
		m_oldMailPixmapPath = config->readEntry("OldMailPixmapPath");

		m_poll = config->readNumEntry("Poll");
		m_command = config->readEntry("Command");
		m_isDocked = config->readBoolEntry("IsDocked");
		m_name = config->readEntry("MailboxName");
		m_beep = config->readBoolEntry("Beep");
		m_runCommand = config->readBoolEntry("RunCommand");
		m_playSound = config->readBoolEntry("PlaySound");
		m_runCommandPath = config->readEntry("RunCommandPath");
		m_playSoundPath = config->readEntry("PlaySoundPath");

		m_readLocalMailbox = config->readBoolEntry("ReadLocalMailbox");

		if (m_readLocalMailbox)
		{
			m_mailboxPath = config->readEntry("MailboxPath");
		}
		else
		{
			m_popServer = config->readEntry("PopServer");
			m_username = config->readEntry("Username");
			m_password = m_scramble.unscramble(config->readEntry("Password"));
		}

		m_sessionManagement = true;

		// reset the current state
		m_resetState();

		return true;
	}
	else
	{
		return m_setup();
	}
}

void KBiff::m_dock()
{
	// destroy the old window
	if (this->isVisible())
	{
		this->hide();
		this->destroy(true, true);
		this->create(0, true, false);
		kapp->setMainWidget(this);

		// we don't want a "real" top widget if we are _going_ to
		// be docked.
		if (m_isDocked)
			kapp->setTopWidget(this);
		else
			kapp->setTopWidget(new QWidget);
	}

	if (m_isDocked == false)
	{
		m_isDocked = true;

		// enable docking
		KWM::setDockWindow(this->winId());
	}
	else
		m_isDocked = false;

	// (un)dock it!
	this->show();

	// reset the state 
	m_resetState();
}

void KBiff::m_saveYourself()
{
	// save ourself ONLY if we have m_sessionManagement 
	if (m_sessionManagement)
	{
		KConfig *config = kapp->getSessionConfig();

		config->setGroup("KBiff");
		config->writeEntry("IsDocked", m_isDocked);
		config->writeEntry("NoMailPixmapPath", m_noMailPixmapPath);
		config->writeEntry("NewMailPixmapPath", m_newMailPixmapPath);
		config->writeEntry("OldMailPixmapPath", m_oldMailPixmapPath);
		config->writeEntry("Poll", m_poll);
		config->writeEntry("Command", m_command);
		config->writeEntry("MailboxName", m_name);
		config->writeEntry("Beep", m_beep);
		config->writeEntry("RunCommand", m_runCommand);
		config->writeEntry("RunCommandPath", m_runCommandPath);
		config->writeEntry("PlaySound", m_playSound);
		config->writeEntry("PlaySoundPath", m_playSoundPath);
		config->writeEntry("ReadLocalMailbox", m_readLocalMailbox);

		if (m_readLocalMailbox)
		{
			config->writeEntry("MailboxPath", m_mailboxPath);
		}
		else
		{
			config->writeEntry("PopServer", m_popServer);
			config->writeEntry("Username", m_username);
			config->writeEntry("Password", m_scramble.scramble(m_password));
		}
		config->sync();
	}
}

void KBiff::m_checkLocalMail()
{
	QFileInfo mailbox(m_mailboxPath);

	// first, see if we need to adjust our state a little
	// for no mail
	if ((mailbox.size() == 0) && (m_mailState != NO_MAIL))
	{
		// the user has just nuked his entire mailbox
		m_mailState = NO_MAIL;
		m_lastRead = mailbox.lastRead();
		m_lastSize = 0;
		m_displayPixmap();
	}

	// then, see if we have received new mail
	if ((mailbox.size() > 0) && (mailbox.lastModified() > m_lastRead) &&
		 ((int)mailbox.size() > m_lastSize))
	{
		// we did!
		m_mailState = NEW_MAIL;
		m_lastRead = mailbox.lastModified();
		m_lastSize = mailbox.size();
		m_displayPixmap();

		// now beep if we are allowed to
		if (m_beep)
		{
			kapp->beep();
		}

		// run a command if we have to
		if (m_runCommand)
		{
			// make sure the command exists
			if (!m_runCommandPath.isEmpty())
			{
				// make sure we run this in the background
				if (m_runCommandPath[m_runCommandPath.length() - 1] == '&')
					system(m_runCommandPath);
				else
					system(m_runCommandPath + '&');
			}
		}

		// play a sound if we have to
		if (m_playSound && m_hasAudio)
		{
			// make sure something is specified
			if (!m_playSoundPath.isEmpty())
			{
				m_audioServer.play(m_playSoundPath);
				m_audioServer.sync();
			}
		}
	}

	// finally, see if we have already read the mail
	if ((mailbox.size() > 0) && (mailbox.lastRead() > m_lastRead))
	{
		// we have..
		m_mailState = OLD_MAIL;
		m_lastRead = mailbox.lastRead();
		m_lastSize = mailbox.size();
		m_displayPixmap();
	}
}

void KBiff::m_checkPOPMail()
{
	// get the current size of the mailbox
	m_lastSize = m_getPopSize();

	// first, see if we need to adjust our state a little
	// for no mail
	if ((m_lastSize == 0) && (m_mailState != NO_MAIL))
	{
		// the user has just nuked his entire mailbox
		m_mailState = NO_MAIL;
		m_storedSize = 0;
		m_displayPixmap();
	}

	// then, see if we have received new mail
	if ((m_lastSize > m_storedSize) && (m_lastSize > 0) &&
		 (m_mailState != NEW_MAIL))
	{
		// we did!
		m_mailState = NEW_MAIL;
		m_displayPixmap();

		// now beep if we are allowed to
		if (m_beep)
		{
			kapp->beep();
		}

		// run a command if we have to
		if (m_runCommand)
		{
			// make sure this isn't empty
			if (!m_runCommandPath.isEmpty())
			{
				// make sure we run this in the background
				if (m_runCommandPath[m_runCommandPath.length() - 1] == '&')
					system(m_runCommandPath);
				else
					system(m_runCommandPath + '&');
			}
		}

		// play a sound if we have to
		if (m_playSound && m_hasAudio)
		{
			// make sure something is specified
			if (!m_playSoundPath.isEmpty())
			{
				m_audioServer.play(m_playSoundPath);
				m_audioServer.sync();
			}
		}
	}

	if ((m_lastSize <= m_storedSize) && (m_lastSize > 0) &&
		 (m_mailState != OLD_MAIL))
	{
		// we have old mail
		m_mailState = OLD_MAIL;
		m_displayPixmap();

		m_storedSize = m_lastSize;
	}
}

void KBiff::m_checkMailNow()
{
	emit(checkNewMail());
}

void KBiff::timerEvent(QTimerEvent *)
{
	emit(checkNewMail());
}

void KBiff::mousePressEvent(QMouseEvent *event)
{
	// check if this is a right click
	if(event->button() == RightButton)
	{
		// popup the context menu
		m_popupMenu(event->x(), event->y());
	}
	else
	{
		// reset the size of this is a pop3 mailbox
		if (m_readLocalMailbox == false)
		{
			m_storedSize = m_lastSize;
			if (m_lastSize > 0)
				m_mailState = OLD_MAIL;
			else
				m_mailState = NO_MAIL;
			m_displayPixmap();
		}

		// execute the command
		// we want it to execute in the background, so append
		// a '&' is there is not one already
		if (!m_command.isEmpty())
		{
			if (m_command[m_command.length() - 1] == '&')
				system(m_command);
			else
				system(m_command + '&');
		}
	}
}

void KBiff::m_help()
{
	kapp->invokeHTMLHelp("kbiff/kbiff.html", "");
}

void KBiff::m_popupMenu(int, int)
{
	QPopupMenu *popup = new QPopupMenu(0, "popup");

	if (m_isDocked == false)
		popup->insertItem(klocale->translate("&Dock"), this, SLOT(m_dock()));
	else
		popup->insertItem(klocale->translate("&UnDock"), this, SLOT(m_dock()));
	popup->insertItem(klocale->translate("&Setup..."), this, SLOT(m_setup()));
	popup->insertSeparator();
	popup->insertItem(klocale->translate("&Help..."), this, SLOT(m_help()));
	popup->insertSeparator();
	popup->insertItem(klocale->translate("&Check mail now"), this, SLOT(m_checkMailNow()));
	popup->insertSeparator();
	popup->insertItem(klocale->translate("E&xit"), kapp, SLOT(quit()));

	popup->popup(QCursor::pos());
}

bool KBiff::m_setup()
{
	KBiffSetupDlg setup_dlg(m_name);

	if (setup_dlg.exec())
	{
		m_command = setup_dlg.getCommand();
		m_poll = setup_dlg.getPoll();
		m_noMailPixmapPath = setup_dlg.getNoMailPixmapPath();
		m_newMailPixmapPath = setup_dlg.getNewMailPixmapPath();
		m_oldMailPixmapPath = setup_dlg.getOldMailPixmapPath();
		m_name = setup_dlg.getName();

		// save the old docked state
		bool was_docked = m_isDocked;

		// get the autodock option
		m_isDocked = setup_dlg.getDock();

		m_sessionManagement = setup_dlg.getSessionManagement();

		m_runCommand = setup_dlg.getRunCommand();
		m_runCommandPath = setup_dlg.getRunCommandPath();
		m_playSound = setup_dlg.getPlaySound();
		m_playSoundPath = setup_dlg.getPlaySoundPath();
		m_beep = setup_dlg.getBeep();

		m_readLocalMailbox = setup_dlg.getReadLocalMailbox();
		if (m_readLocalMailbox)
		{
			m_mailboxPath = setup_dlg.getMailboxPath();
			m_popServer   = "";
			m_username    = "";
			m_password    = "";
		}
		else
		{
			m_mailboxPath = "";
			m_popServer   = setup_dlg.getPopServer();
			m_username    = setup_dlg.getUser();
			m_password    = setup_dlg.getPassword();
			m_popPort     = setup_dlg.getPort();
		}

		// we want to change our dock state ONLY if it really is changed!
		if (was_docked != m_isDocked)
		{
			// m_dock expects the reverse of m_isDocked!  so do it
			m_isDocked = m_isDocked ? false : true;
			m_dock();
		}

		if ((m_readLocalMailbox && m_mailboxPath.isEmpty()) ||
			 (!m_readLocalMailbox && m_popServer.isEmpty()))
		{
			return false;
		}
		else
		{
			m_resetState();
			QToolTip::add(this, m_name);

			return true;
		}
	}
	else
	{
		if ((m_readLocalMailbox && m_mailboxPath.isEmpty()) ||
			 (!m_readLocalMailbox && m_popServer.isEmpty()))
		{
			return false;
		}
		else
			return true;
	}
}

int KBiff::m_getPopSize()
{
	void (*oldHandler)(int) = signal(SIGALRM, SIG_IGN);
	void (*pipeHandler)(int) = signal(SIGPIPE, SIG_IGN);
	int num, size;          // number of all msgs / size of all msgs

	DwPopClient client;
	client.SetReceiveTimeout(20);

	if (client.Open((const char*)m_popServer, m_popPort) != '+')
	{
		return 0;
	}
	else if (client.User((const char*)m_username) != '+')
	{
		client.Quit();
		return 0;
	}
	else if (client.Pass((const char*)m_password) != '+')
	{
		client.Quit();
		return 0;
	}
	else if (client.Stat() != '+')
	{
		client.Quit();
		return 0;
	}
	else
	{
		QString response = client.SingleLineResponse().c_str();
		client.Quit();

		sscanf(response.data(), "+OK %d %d", &num, &size);
	}

	signal(SIGALRM, oldHandler);
	signal(SIGPIPE, pipeHandler);

	return size;
}
