/*
 * kbootsel.cpp
 *
 * Copyright (C) 1999 Stefan van den Oord <s.m.vandenoord@student.utwente.nl>
 */
#include "kbootsel.h"

#include <qkeycode.h>

#include <kfm.h>
#include <kiconloader.h>

KBootSel::KBootSel ( bool enableDebug )
{
	setCaption ( CAPTION );

	// Create the configuration object
	config = new KConfig ( CONFIGFILE, LOCALCONFIGFILE );

	// Set the right group
	config->setGroup ( CONFIGGROUP_LILO );

	// Create the LiloInfo object using the configuration information
	lilo = new LiloInfo ( config->readEntry ( CONFIGKEY_LILOLOC, DEFAULT_LILOLOCATION ),
	                      config->readEntry ( CONFIGKEY_BOOTMAPLOC, DEFAULT_BOOTMAPLOCATION),
	                      config->readBoolEntry ( CONFIGKEY_LILOHACK, true ),
	                      enableDebug );

	// Create the object that is the view of the KTMainWindow
	view = new KBootSelWidget ( lilo, this );

	// tell the KTMainWindow that this is indeed the main widget
	setView(view);

	// Even now a Lilo error may have occured! (The constructor of LiloInfo may do that)
	if ( view->getLiloResult() < 0 )
	{
		if ( showErrorDialog() == 2 )
		{
			//kapp->quit(); // Why does this not work?!
			exit ( 1 );
		}
		// If there was an error, pretend that the configuration button was clicked
		configureClicked();
	}

	debug = enableDebug;

	// Create and add the toolbar buttons
	toolBar()->insertButton(Icon("tick.xpm"), 0, SIGNAL(clicked()), this, SLOT(okClicked()), true, i18n("Activate"));
	toolBar()->insertButton ( Icon("filedel.xpm"), 1, SIGNAL(clicked()), this, SLOT(cancelClicked()), true, i18n("Cancel") );
	toolBar()->insertSeparator();
	toolBar()->insertButton ( Icon("eraser.xpm"), 2, SIGNAL(clicked()), this, SLOT(clearClicked()), true, i18n("Clear next boot option") );
	toolBar()->insertSeparator();
	toolBar()->insertButton ( Icon("configure.xpm"), 3, SIGNAL(clicked()), this, SLOT(configureClicked()), true, i18n("Settings") );
	toolBar()->insertSeparator();
	toolBar()->insertButton ( Icon("contents.xpm"), 4, SIGNAL(clicked()), this, SLOT(helpClicked()), true, i18n("Help") );
	toolBar()->insertButton ( Icon("help.xpm"), 5, SIGNAL(clicked()), this, SLOT(aboutClicked()), true, i18n("About") );
	toolBarFloating = false;

	// When the toolbar is moved, the size of the main window must be recomputed
	connect ( toolBar(), SIGNAL(moved(BarPosition)), this, SLOT(toolBarMoved()) );

	// Compute the minimum size of the window
	recomputeMinimumSize();

	// Resize the window to the minimum size
	resize ( minimumSize().width(), minHeight );
}

KBootSel::~KBootSel()
{
	delete lilo;
	delete config;
}

void KBootSel::configureClicked()
{
	int retval;
	int oldWidth = width();
	int oldHeight = height();

	// Check whether the current size is the minimum size
	bool wasMinimum = ( oldHeight == minimumSize().height() );

	// Run the configuration application
	KProcess control;
	control << KApplication::kde_bindir() + "/kcmbootsel";
	control.start ( KProcess::Block );

	// Reparse the configuration file, because it was changed by kcmbootsel
	config->reparseConfiguration();

	// Modify the parameters of the LiloInfo object
	lilo->changeLiloLocation ( config->readEntry ( CONFIGKEY_LILOLOC,
	                                               DEFAULT_LILOLOCATION ) );
	lilo->changeBootmapLocation ( config->readEntry ( CONFIGKEY_BOOTMAPLOC,
	                                                  DEFAULT_BOOTMAPLOCATION) );
	lilo->changeHackState ( config->readBoolEntry ( CONFIGKEY_LILOHACK, true ) );

	// Hide the current window, because it is going to be rebuilt
	hide();

	// Recreate the view
	delete view;
	view = new KBootSelWidget ( lilo, this );

	retval = view->getLiloResult();
	while ( retval < 0 )
	{
		if ( showErrorDialog() == 2 )
		{
			//kapp->quit();
			exit ( 1 );
		}

		// When there was a LiloInfo error, run the configuration again
		control.start ( KProcess::Block );

		config->reparseConfiguration();

		lilo->changeLiloLocation ( config->readEntry ( CONFIGKEY_LILOLOC,
													   DEFAULT_LILOLOCATION ) );
		lilo->changeBootmapLocation ( config->readEntry ( CONFIGKEY_BOOTMAPLOC,
														  DEFAULT_BOOTMAPLOCATION) );
		lilo->changeHackState ( config->readBoolEntry ( CONFIGKEY_LILOHACK, true ) );

		delete view;
		view = new KBootSelWidget ( lilo, this );
		retval = view->getLiloResult();
	}
	
	// Set the view to the created KBootSelWidget
	setView ( view );

	// Recompute the minimum size
	recomputeMinimumSize();

	// If the size of the window was the minimum size, make it the minimum size again.
	// This way, when the user resized the window, the size will NOT be changed (unless
	// the minimum is larger than the current size, of course).
	if ( wasMinimum ) resize ( oldWidth, minimumSize().height() );

	// Show the window again
	show();
}

void KBootSel::okClicked()
{
	// Write the next option
	view->writeNextOption ( lilo );

	// When the user wants that, logout KDE
	if ( view->chk_logout->isChecked() ) KWM::logout();

	kapp->quit();
}

void KBootSel::cancelClicked()
{
	kapp->quit();
}

void KBootSel::clearClicked()
{
	view->clearOptions();
}

void KBootSel::helpClicked()
{
	kapp->invokeHTMLHelp ( "kbootsel/kbootsel.html", "" );
}

void KBootSel::aboutClicked()
{
	QMessageBox::about(this,
		i18n("About") + QString(" ") + QString(CAPTION) + QString(" ") + QString(VERSION),
		QString(CAPTION) + QString(" ") + i18n("version") + QString(" ") + QString(VERSION) +
		QString("\n(c) 1999 ") + QString(AUTHOR) + QString(" <") + QString(EMAIL) + QString(">\n") +
		QString(CAPTION) + " " + i18n("is free software and comes with ABSOLUTELY NO WARRANTY.\n\n")+
		i18n("Homepage") + QString(": ") + QString(HOMEPAGE));
}

void KBootSel::toolBarMoved()
{
	KToolBar::BarPosition barPosition = toolBar()->barPos();
	if ( barPosition == KToolBar::Floating && !toolBarFloating )
	{
		// If the toolbar is floating, and was not floating before, decrease the
		// height of the window
		minHeight -= toolBarHeight;
		setMinimumHeight ( minHeight );
		resize ( width(), height() - toolBar()->height() );
		toolBarFloating = true;
	}
	else if ( barPosition == KToolBar::Top && toolBarFloating )
	{
		// If the toolbar is not floating and it was floating before, increase the
		// height of the window
		minHeight += toolBarHeight;
		setMinimumHeight ( minHeight );
		resize ( width(), height() + toolBar()->height() );
		toolBarFloating = false;
	}
	else
		// The toolbar may only be floating or on top
		toolBar()->setBarPos ( KToolBar::Top );
}

void KBootSel::recomputeMinimumSize()
{
	toolBarHeight = toolBar()->height();
	minHeight = view->minimumSize().height() + toolBarHeight;

	setMinimumHeight ( minHeight );
	setMinimumWidth ( max ( toolBar()->sizeHint().width(),
	                        view->minimumSize().width() ) );
}

int KBootSel::showErrorDialog()
{
	int retval;

	KMsgBox *msg = new KMsgBox (
		this,
		CAPTION,
		i18n("An error occured. Verify the configuration."),
		KMsgBox::DB_FIRST | KMsgBox::STOP,
		i18n("Configure..."),
		i18n("Abort"),
		i18n("Details...")
	);
	retval = msg->exec();
	while ( retval == 3 )
	{
		// Details button was pressed...
		KMsgBox::message ( this, CAPTION, lilo->getErrorDescription(),
						   KMsgBox::DB_FIRST | KMsgBox::STOP,
						   i18n("Close") );
		retval = msg->exec();
	}
	return retval;
}
