/*
 * pgp2key.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <iostream.h>

#include <ksimpleconfig.h>

#include "pgp2key.h"

#include "kpgp.h"
#include "md5.h"
#include "pgpmpi.h"
#include "pgpsignature.h"
#include "pgp2signature.h"
#include "pgp2userid.h"
#include "pgpkeyring.h"
#include "pgp2trust.h"

#include <qdatetm.h>
#include <qstring.h>

Pgp2Key::Pgp2Key(QDataStream &source, UINT8 &ctb) : 
  PgpKey(source, ctb), 
  CTB(ctb)
{
  UINT8  len8;
  UINT16 len16;
  UINT32 len32;
  
  if (((ctb >> 2) & 15) != PGP2_KEY)
    debug("Pgp2Key::Constructor: no key packet!");
  
  switch (ctb & 3)
    {
      case 0: source >> len8;   // 1 byte packet len
	      len = len8;
	      break;
      case 1: source >> len16;  // 2 byte packet len
	      len = len16;
	      break;
      case 2: source >> len32;  // 4 byte packet len
	      len = len32;
	      break;
      case 3: len = 0;      // unknown packet len
	      break;
    }

  source >> version;
  source >> timestamp;
  source >> validity;
  source >> algo;

  n << source;
  e << source;
  
  source >> ctb;
  
  if (((ctb >> 2) & 15) != PGP2_TRUST)
    debug("ERROR: Pgp2Key::Constructor - missing trust packet!");
  
  trust = new Pgp2Trust(source, ctb);
  
  is_private = (trust->getTrustByte() &  7) ==  7;
  is_paused  = (trust->getTrustByte() & 32) == 32;

  // load revocation
  source >> ctb;

  if (((ctb >> 2) & 15) == PGP2_SIGNATURE)
    {
      addRevocation(new Pgp2Signature(source, ctb));
      source >> ctb;
    }

  // load user-IDs
  while (((ctb >> 2) & 15) == PGP2_USERID)
    insertUserId(new Pgp2UserId(source, ctb));

  debug("Key loaded: " + getUserId()->userId());
};

Pgp2Key::~Pgp2Key()
{};

void 
Pgp2Key::store(QDataStream &dest)
{
  PgpUserId    *uid;
  PgpSignature *sig;
  
  dest << CTB;
  
  switch (CTB & 3)
    {
      case 0: dest << (UINT8)  len;  // 1 byte packet len
	      break;
      case 1: dest << (UINT16) len;  // 2 byte packet len
	      break;
      case 2: dest << (UINT32) len;  // 4 byte packet len
	      break;
      case 3: len = 0;               // unknown packet len
	      break;
    }
  
  dest << version;
  dest << timestamp;
  dest << validity;
  dest << algo;
  
  n >> dest;
  e >> dest;
  
  trust->store(dest);
  
  for (sig = revocation_list->first(); sig != 0; sig = revocation_list->next())
    {
      sig->store(dest);
    }
  
  for (uid = userId_list->first(); uid != 0; uid = userId_list->next())
    {
      uid->store(dest);
    }
};

PgpUserId*  
Pgp2Key::newUserId(QString uid)
{
  PgpUserId    *new_uid = new Pgp2UserId(uid);
  PgpSignature *new_sig;

  insertUserId(new_uid, 0);
  new_sig = sign(this);

  if (new_sig)
    getUserId()->addSignature(new_sig, TRUE);
  else
    {
      userId_list->remove((UINT32) 0);     // remove new user id 
      return 0;
    }

  return new_uid;
};

void        
Pgp2Key::extractKey(QString filename)
{
  Kpgp::getKpgp()->extractKey(getUserId()->userId(), filename);
};


void     
Pgp2Key::setDefault(bool _default)
{
  if ((_default != is_default) && is_private)
    {
      is_default = _default;

      if (_default)
	{
	  /*
	 
	  */
	}
      else
	{
	  // PgpKey::defaultKey = 0;
	  Kpgp::getKpgp()->clear(TRUE);
	}
    }
};
 
void     
Pgp2Key::setPaused(bool paused)
{
  if (paused != is_paused)
    {
      is_paused = paused;

      if (paused)
	trust->setTrustByte(trust->getTrustByte() | 32);
      else
	trust->setTrustByte(trust->getTrustByte() & 223);
    }
};
 
void     
Pgp2Key::setRevoked()
{
  if (!is_revoked)
    {
      UINT8        CTB;
      PgpKey       *tmpKey;
      PgpSignature *tmpSignature;
      QString      tmp_name("/tmp/kcmpgp_tmp");
      QDataStream  output;
      QFile        tmp_file(tmp_name);

      // store key
      tmp_file.open(IO_WriteOnly);
      output.setDevice(&tmp_file);
      store(output);
      tmp_file.close();

      Kpgp::getKpgp()->revokeKey(getUserId()->userId(), "/tmp/kcmpgp_tmp");
  
      // find new REVOKATION
      tmp_file.open(IO_ReadOnly);
      output.setDevice(&tmp_file);

      output >> CTB;

      if ((!output.eof()) && (CTB != 0xff))
	{
	  if (((CTB >> 2) & 15) == PGP2_KEY)
	    {
	      debug("pgp packet found");
	      tmpKey = new Pgp2Key(output, CTB);
	      output >> CTB;
	      
	      if (((CTB >> 2) & 15) == PGP2_SIGNATURE)
		{
		  tmpSignature = new Pgp2Signature(output, CTB);
		  addRevocation(tmpSignature);
		  debug("FIXME: revoked key = default key???");
		}
	      else
		debug("error: no revocation found");
	    }
	  else
	    debug("no key id found");
	}
      else
	debug("error while revoking key");
    }
};
  
// returns key-id
QString          
Pgp2Key::keyId() const
{
  return n.keyId();
};

QString 
Pgp2Key::fingerprint()
{
  int      i;
  QString  fp, help;
  MD5      md5;
  UINT8    *DIGEST = new UINT8[16];
  
  UINT32 len = n.byteLen() + e.byteLen();
  UINT8 *BYTES = new UINT8[len];
 
  n.store(BYTES);
  e.store(BYTES + n.byteLen());
  
  md5.hash(BYTES, len, DIGEST);
  
  for (i=0; i<8; i++)
    fp += help.sprintf("%2.2X ", DIGEST[i]);
  fp += "  ";
  for (i=0; i<8; i++)
    fp += help.sprintf("%2.2X ", DIGEST[8+i]);
  
  delete [] DIGEST;
  delete [] BYTES;
  
  return fp;
};

PgpSignature*
Pgp2Key::sign(PgpKey *key)
{
  UINT8        CTB;
  PgpKey       *tmpKey;
  PgpSignature *tmpSignature;
  QString      tmp_name("/tmp/kcmpgp_tmp");
  QDataStream  output;
  QFile        tmp_file(tmp_name);
  
  tmp_file.open(IO_WriteOnly);
  output.setDevice(&tmp_file);
  
  /*
   * To sign a key, it is neccesary, to store the key to be signed and the
   * signers public key in the temporary keyring.
   */
  key->store(output);
  store(output);
  
  tmp_file.close();
  
  debug("FIXME: workaround because the new user-id isnt stored into the secret keyring");

  // use the next userId because the default useId isnt stored into the secret keyring
  if (this == key)
    Kpgp::getKpgp()->signKey(key->getUserId()->userId(), getUserId(1)->userId(), "/tmp/kcmpgp_tmp");
  else
    Kpgp::getKpgp()->signKey(key->getUserId()->userId(), getUserId()->userId(), "/tmp/kcmpgp_tmp");
  
  // find new signature
  tmp_file.open(IO_ReadOnly);
  output.setDevice(&tmp_file);

  output >> CTB;

  while ((!output.eof()) && (CTB != 0xff))
    {
      // PGP 2.6.x
      switch (((CTB >> 2) & 15))
	{
	case PGP2_KEY:
	  tmpKey = new Pgp2Key(output, CTB);
	  
	  // first check, if a signature is available...
	  if (tmpKey->getUserId(0)->count() == 0)
		{
		   delete tmpKey;
		   Kpgp::getKpgp()->clear(TRUE); // remove old password 
	  	   return 0;
		}

	  tmpSignature = tmpKey->getUserId(0)->signature(0);
	  // Ok, now we have read the first signature of the default userId.
	  // If it has the same key-ID like our key, everything is ok...
	  // else an error has happend.
	  if (tmpSignature->keyId() == keyId())
	    {
	      tmp_file.close();
		  delete tmpKey;
	      debug("signature found");
	      return tmpSignature;
	    }
	  else
	    {
	      delete tmpSignature;
	      delete tmpKey;
	      tmp_file.close();
		  Kpgp::getKpgp()->clear(TRUE); // remove password (wrong?) 
	      return 0;
	    }
	  break;

	default:
	  debug("ERROR: unknown key packet!");
	}
      
      output >> CTB;
    }

  

  return 0;
};

int         
Pgp2Key::bitlen()
{
  return n.bitLen(); 
};

QDateTime   
Pgp2Key::created()
{
  QDateTime dt;

  dt.setTime_t(timestamp);

  return dt;
};

int      
Pgp2Key::getTrust() const
{
  if (!is_revoked)
    return trust->getTrustByte() & 7;
  else
    return 2;
};

void        
Pgp2Key::setTrust(int new_trust)
{
  trust->setTrustByte((trust->getTrustByte() & 248) | (new_trust & 7));
};

/* returns a descrption of the key algorithm */
QString     
Pgp2Key::algorithm() const
{
  return "RSA"; 
};

/* returns version information of this key */
QString     
Pgp2Key::format() const
{
  return "PGP 2.6.x";
};
