/*
 * pgp2signature.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "pgpkey.h"
#include "pgpsignature.h"

#include "pgpkeyring.h"
#include "pgp2trust.h"
#include "pgp2key.h"
#include "pgp2signature.h"

Pgp2Signature::Pgp2Signature(QDataStream &source, UINT8 &ctb) :
  PgpSignature(),
  CTB(ctb)
{
  UINT8  len8;
  UINT16 len16;
  UINT32 len32;
  
  
  if (((ctb >> 2) & 15) != PGP2_SIGNATURE)
    debug("Pgp2Signature::Constructor: no signature packet!");
  
  
  switch (CTB & 3)
    {
      case 0: source >> len8;   // 1 byte packet len
	      len = len8;
	      break;
      case 1: source >> len16;  // 2 byte packet len
	      len = len16;
	      break;
      case 2: source >> len32;  // 4 byte packet len
	      len = len32;
	      break;
      case 3: len = 0;      // unknown packet len
	      break;
    }
  
  source >> version;
  source >> length;
  source >> sigType;
  source >> timestamp;
  source.readRawBytes((char*)id, 8);
  source >> concAlgo;
  source >> hashAlgo;
  source >> testBytes;
  
  digest << source;
  
  if (sigType == PGP2_SIGNATURE_KEY)
    {
      source >> ctb;
  
      if (((ctb >> 2) & 15) != PGP2_TRUST)
	debug("Pgp2Signature::Constructor: missing trust packet");
  
      trust = new Pgp2Trust(source, ctb);
    }
};

Pgp2Signature::~Pgp2Signature()
{};

void 
Pgp2Signature::store(QDataStream &dest)
{
  dest << CTB;
  
  switch (CTB & 3)
    {
      case 0: dest << (UINT8)  len;  // 1 byte packet len
	      break;
      case 1: dest << (UINT16) len;  // 2 byte packet len
	      break;
      case 2: dest << (UINT32) len;  // 4 byte packet len
	      break;
      case 3: len = 0;               // unknown packet len
	      break;
    }

  dest << version;
  dest << length;
  dest << sigType;
  dest << timestamp;
  dest.writeRawBytes((char*)id, 8);
  dest << concAlgo;
  dest << hashAlgo;
  dest << testBytes;
  
  digest >> dest;
  
  if (sigType == PGP2_SIGNATURE_KEY)
    trust->store(dest);
};

// returns key-is of key who made this signature
QString
Pgp2Signature::keyId()
{
  QString id_str;
  
  id_str.sprintf(KEYID_FORMAT, id[4], id[5], id[6], id[7]);
  
  return id_str;
};

int     
Pgp2Signature::getTrust()
{
  return trust->getTrustByte() & 7;
};

void    
Pgp2Signature::setTrust(int new_trust)
{
  trust->setTrustByte((trust->getTrustByte() & 248) | (new_trust & 7));
}

QString 
Pgp2Signature::userId()
{
  PgpKey *myKey = PgpKeyring::getPgpKeyring()->findKeyId(keyId());

  if (myKey)
    return myKey->getUserId()->userId();
  else
    return keyId() + " (Unknown signator, can't be checked)";
};
