/*
 * pgp2userid.cpp
 *
 * Copyright (c) 1997 Christian Stueble  stueble@ls6.cs.uni-dortmund.de
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "pgp2key.h"
#include "pgp2signature.h"
#include "pgp2trust.h"
#include "pgp2userid.h"


int Pgp2UserId::MARGINALS_NEEDED = 0;
int Pgp2UserId::COMPLETES_NEEDED = 0;

Pgp2UserId::Pgp2UserId(QDataStream &source, UINT8 &ctb) : 
  PgpUserId(),
  CTB(ctb)
{
  UINT8  len8;
  UINT16 len16;
  UINT32 len32;
  
  if (Pgp2UserId::MARGINALS_NEEDED == 0)
    Pgp2UserId::MARGINALS_NEEDED = 2;

  if (Pgp2UserId::COMPLETES_NEEDED == 0)
    Pgp2UserId::COMPLETES_NEEDED = 1;

  if (((ctb >> 2) & 15) != PGP2_USERID)
    debug("Pgp2UserId::Constructor: no userId packet!");
  
  
  switch (CTB & 3)
    {
      case 0: source >> len8;   // 1 byte packet len
	      len = len8;
	      break;
      case 1: source >> len16;  // 2 byte packet len
	      len = len16;
	      break;
      case 2: source >> len32;  // 4 byte packet len
	      len = len32;
	      break;
      case 3: len = 0;      // unknown packet len
	      break;
    }
  
  name = new char[len+1];
  source.readRawBytes(name, len);
  name[len] = 0;
      
  source >> ctb;
  
  if (((ctb >> 2) & 15) != PGP2_TRUST)
    debug("Pgp2UserId::Constructor: missing trust packet!");
  
  trust = new Pgp2Trust(source, ctb);    

  source >> ctb;

  while (((ctb >> 2) & 15) == PGP2_SIGNATURE)
    {
      addSignature(new Pgp2Signature(source, ctb));
      // debug("key signature loaded " + signature(i++)->keyId());
      source >> ctb;
    }
};

Pgp2UserId::Pgp2UserId(const char* new_name) :
  CTB((PGP2_USERID << 2) | 128)
{
  len = strlen(new_name);
  
  name = new char[len+1];
  strcpy(name, new_name);
  
  trust = new Pgp2Trust(7);
};
 
Pgp2UserId::~Pgp2UserId()
{};

void Pgp2UserId::store(QDataStream &dest)
{
  PgpSignature *sig;
  
  dest << CTB;
  
  switch (CTB & 3)
    {
      case 0: dest << (UINT8)  len;  // 1 byte packet len
	      break;
      case 1: dest << (UINT16) len;  // 2 byte packet len
	      break;
      case 2: dest << (UINT32) len;  // 4 byte packet len
	      break;
      case 3: len = 0;               // unknown packet len
	      break;
    }
  
  dest.writeRawBytes((char*) name, len);
      
  trust->store(dest);
  
  for (sig = signatureList->first(); sig != 0; sig = signatureList->next())
    {
      sig->store(dest);
    }
};

int           
Pgp2UserId::getTrust()
{
  return trust->getTrustByte() & 3;
};

void       
Pgp2UserId::updateTrust()
{
  int           trusts[8] = {0, 0, 0, 0, 0, 0, 0, 0};
  UINT8         WARNONLY  = trust->getTrustByte() & 128;
  PgpSignature  *sig;
  
  for (sig = signatureList->first(); sig != 0; sig = signatureList->next())
    trusts[sig->getTrust() & 7]++;

  /*  
  int i;

  for (i=0; i<8; i++)
    cout << trusts[i] << " ";
  cout << endl;
  */

  if ((trusts[7]) || 
      (trusts[6] > Pgp2UserId::COMPLETES_NEEDED) || 
      (trusts[5] > Pgp2UserId::MARGINALS_NEEDED))
    trust->setTrustByte(3 | WARNONLY);
  else
    if (trusts[6] || trusts[5])
      trust->setTrustByte(2 | WARNONLY);
  else
    if (trusts[2])
      trust->setTrustByte(1 | WARNONLY);
  else
    trust->setTrustByte(0);
};
