//
// syntaxscanner.h
//
// written by Michael Riedel <Michael.Riedel@gmx.de>
//

#ifndef Colorizer_h
#define Colorizer_h

#include <assert.h>
#include <qstring.h>
#include <qcolor.h> 
#include <qlist.h>
#include <qdict.h>
#include <qfont.h>
#include <qregexp.h>
#include <qvector.h>

#include "SyntaxState.h"



////////////////////////////////////////////////////////////////
/**
	This class is used by KSyntaxMultiLineEdit to perform the syntax highlighting.
*/
class Colorizer
{
public:
		/**
			Constructs a Colorizer; loads a rule file with a given filename.
			@param filename full name of rule file to load. this parameter can
			be null or empty indicating that no rule file is to be loaded. Such
			a Colorizer can be used for displaying text as well.		
		*/
    Colorizer(const char* filename);
    
    	/**
    		Destructs the Colorizer.
    	*/
	~Colorizer();
    
		/**
			Does a minimum of required initialization to make a Colorizer
			work. This includes the creation of a normal state (index 0)
			in the StateArray.
			@param createNormalState. this indicates wether the normal state is to be generated or not!
		*/
	void init(bool createNormalState);
	
		/**
			loads info for this Colorizer from a file names filename.
			@param filename name of file to load settings from.
		*/ 
	bool load(const QString& filename);
	
		/**
			saves Colorizer info to a file named filename; existing file will be
			overwritten. Currently this function is not implemented.
			@param filename name of file to receive the contents of this.
			@result returns true, in case of success, false as failure indication.
					Currently always false is returned.
		*/
	bool save(const QString& filename);

    
		/**
			This method can be called to get a new part of text to be colored.
			string is the string to be searched; the search will start at 
			startAt. The method returns a non-negative number that indicates
			the starting index of a found text part; in this case the parameters
			len, info and state will return the length of the 
			portion of text, the state for the matched text and the state
			that follows after the matched text. 
				On entry, state can contain a valid SyntaxState index so that
			only SyntaxUnits with the specified validPrevState will have chance to
			match. if there is no match, state will not be altered.
		*/    
    int match(const QString& string, int startAt, int &len, int& innerState, int& state);

		/**
			Used to retrieve the state information for the specified state @ st.
			This method must not be called with an invalid s. It returns 
			a reference to the desired state information.
		*/
	SyntaxState& getStateInfo(unsigned int s)
	{
		if(s >= StateArraySize)
		{
	        	warning("invalid state s=%d", s);
			s = 0;
		}
//		assert(s < StateArraySize);
		return *StateArray[s];
	}

	void setStateInfo(const SyntaxState& ss, unsigned int s)
	{
        if(s >= StateArraySize)
            warning("invalid state s =%d", s);
 		assert(s < StateArraySize);
		StateArray.insert(s, new SyntaxState(ss));
	}
	
	int stateArraySize()
	{ return StateArraySize; }


protected:
    QList<SyntaxUnit> RuleList;
    QDict<bool>       KeywordDict;

	StateList		  StateArray;
	unsigned int	  StateArraySize;	
	int 			  KeywordCheckIx;
	int				  KeywordStateIx;

	void insertState(const QString& s);
	void insertRule(const QString& s);
	void insertKeyword(const QString& k);
	void readSetting(const QString& s);

    inline bool testForKeyword(const QString& test) const;
};
////////////////////////////////////////////////////////////////


#endif // Colorizer_h
