#ifndef _KDIRWATCH_H
#define _KDIRWATCH_H

#include <sys/stat.h>
#include <unistd.h>

#include <qtimer.h>
#include <qstrlist.h>

#include <kfm.h>
 /**
  * Class for watching directory changes. It uses stat (2) and
  * compares stored and actual changed time of directories. If
  * there is a difference notifies KFM for change. Directories can be
  * aded, removed from the list and scanning of particular directories
  * can be stoped and restarted. The whole class can be stoped and
  * restarted. Directories can be added/removed from list in
  * any state.
  * There may be only one KDirWatch object in application. There is a
  * static function that returns a pointer that object.
  * @short Class for watching directory changes.
  * @author Sven Radej <sven@lisa.exp.univie.ac.at>
  */
class KDirWatch : public QObject
{
  Q_OBJECT
    
  public:
   /**
    * Constructor. Does not begin with scanning until @ref #startScan
    * is called. Default frequency is 500 ms. The created list of
    * directories has deep copies.
    */
   KDirWatch (int freq = 500);

   /**
    * Destructor. Stops scanning and cleans up.
    */
   ~KDirWatch();

   /**
    * Adds directory to list of directories to be watched. (The list
    * makes deep copies).
    */
   void addDirListEntry(const char *_url);

   /**
    * Removes directory from list of scanned directories. If specified
    * url is not in the list, does nothing.
    */
   void removeDirListEntry(const char *_url);

   /**
    * Stops scanning for specified url. Does not delete dir from list,
    * just skips it. Call this function when you make an huge operation
    * on this directory (copy/move big files or lot of files). When finished,
    * call @ref #restartDirScan (url).
    * Returns 'false' if specified url is not in list, 'true' otherwise.
    */
   bool stopDirScan(const char *_url);

   /**
    * Restarts scanning for specified url. Resets ctime. It doesn't notify
    * the change, since ctime value is reset. Call it when you are finished
    * with big operations on that url, *and* when *you* have refreshed that
    * url.
    * Returns 'false' if specified url is not in list, 'true' otherwise.
    */
   bool restartDirScan(const char *_url);

   /**
    * Starts scanning of all dirs in list. If notify is true, all changed
    * dirs (since @ref #stopScan call) will be notified for refresh. If
    * notify is false, all ctimes will be reset (except those who are stopped,
    * but only if skippedToo is false) and changed dirs won't be
    * notified. You can start scanning even if the list is empty. First call
    * should be called with 'false' or else all dirs in list will be notified.
    * Note that direcories that were.
    * If 'skippedToo' is true, the skipped dirs, (scanning of which was
    * stopped with  @ref #stopDirScan ) will be reset and notified for change.
    * Otherwise, stopped dirs will continue to be unnotified.
    */
   void startScan(bool notify=false, bool skippedToo=false);

   /**
    * Stops scanning of all dirs in list. List is not cleared, just the
    * timer is stopped.
    */
   void stopScan();

   /**
    * Returns list of directories.
    */
   QStrList *getDirlist();

   /**
    * Static function that returns a pointer to dirwatcher.
    */
   static KDirWatch *getKDirWatch() {return mySelf;}

     
 protected:
   void notifyKFM (const char *a);
   void resetList (bool reallyall);
   static KDirWatch *mySelf;
   
 protected slots:
   void slotRescan();
   
 private:
   KFM *kfm;
   QTimer *timer;
   QStrList *dirList;
   int freq;
   time_t ctime[128]; // anyone has more than 128 kfm's open at a time?
   struct stat statbuff;
};

#endif