/*
	cathandler.cpp - part of KDiskCat the KDE Disk Catalog.

	Copyright (c) 1998,1999 Balzs Ternyi <terenyi@freemail.c3.hu>

	KDiskCat is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
			
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
					
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <kapp.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qstrlist.h>
#include <strstream.h>

#include "cathandler.h"

extern void micro(QString* the_text);
extern CatHandler* cat;
extern const char* KDISKCAT_VERSION_INFO;

QString CatHandler::Path2Char(KPath path)
{
   QString ret;
   QString item;

   ret="";

   while (!path.isEmpty())
   {
      item=(const char*) *path.pop();
      item="/" + item;
      ret.insert(0,(const char*) item);
   }
      
   return ret;
}

QString CatHandler::Path2Char(KPath path,uint depth)
{
   QString ret;
   QString item;

   ret="";

   while (path.count() > depth)
   {
      item=(const char*) *path.pop();
      item="/" + item;
      ret.insert(0,(const char*) item);
   }

   return ret;
}

KPath CatHandler::Char2Path(const QString* str)
{
   KPath ret;
   QString *pstr;
   int poz;
   int veg;
   
   poz=str->find("/",0,true);
   
   while ((uint)poz < str->length())
   {
      veg=str->find("/",poz+1,true);
      if (veg== -1) veg=str->length();
      pstr=new QString(str->mid(poz+1,veg - poz - 1));
      ret.push(pstr);
      poz=veg;
   }
   
   return ret;
}

CatHandler::CatHandler(QObject* parent=0,const char* name=0)
   :QObject(parent,name)
{
   catindex.setAutoDelete(true);
   catalog.setAutoDelete(true);
   category_file="";
   is_catalog_modified=false;
	is_catindex_modified=false;
	is_catalog_loading_on=true;
//   current_branch.setAutoDelete(true);
};

CatHandler::~CatHandler()
{
};

void CatHandler::init(QString filename)
{
   if (category_file != "")
   {
      synchronizeAll();
   }
   is_catalog_loading_on=true;
   category_file=filename;
   clearCatalog();
   clearIndex();
   loadIndex();
};

void CatHandler::setCurrentBranch(KPath what)
{
   IndexEntry fie;
   IndexEntry *ie;

   synchronize();
   if (Path2Char(what) != Path2Char(current_branch))
	{
	   fie.key=Path2Char(what);
	   if (catindex.find(&fie) != -1)
	   {
	      ie=catindex.current();
	      current_branch=Char2Path(&ie->key);
	      current_branch_position=ie->position;
	      current_branch_index=catindex.at();
	      loadCatalog();
	   }
	}
};

KPath CatHandler::getCurrentBranch()
{
   return current_branch;
};

int CatHandler::firstBranch()
{
   IndexEntry *ie;
   int ret=1;

   synchronize();

   ie=catindex.first();

   current_branch.clear();
   if (ie != 0)
   {
      current_branch=Char2Path(&ie->key);
      current_branch_position=ie->position;
      current_branch_index=catindex.at();
      loadCatalog();
   }
   else
   {
      ret= -1;
   }

   return ret;
};

KPath CatHandler::lastBranch()
{
   IndexEntry *ie;
   KPath ret;

   synchronize();

   ie=catindex.last();

   current_branch.clear();
   if (ie != 0)
   {
      current_branch=Char2Path(&ie->key);
      current_branch_position=ie->position;
      current_branch_index=catindex.at();
      loadCatalog();
      ret=current_branch;
   }

   return ret;
};

KPath CatHandler::nextBranch()
{
   IndexEntry *ie;
   KPath ret;

   if (is_catalog_loading_on)
   {
      synchronize();
   }

   catindex.at(current_branch_index);
  	ie=catindex.next();
	
	current_branch.setAutoDelete(true);
  	current_branch.clear();
   current_branch.setAutoDelete(false);
   if (ie != 0)
   {
      current_branch=Char2Path(&ie->key);
      current_branch_index=catindex.at();
      current_branch_position=ie->position;
      if (is_catalog_loading_on)
      {
         loadCatalog();
      }
      ret=current_branch;
   }

   return ret;
};

KPath CatHandler::prevBranch()
{
   IndexEntry *ie;
   KPath ret;

   synchronize();

   catindex.at(current_branch_index);
   ie=catindex.prev();

   current_branch.clear();
   if (ie != 0)
   {
      current_branch=Char2Path(&ie->key);
      current_branch_position=ie->position;
      current_branch_index=catindex.at();
      loadCatalog();
      ret=current_branch;
   }

   return ret;
};

CatEntry* CatHandler::first()
{
   return catalog.first();
};

CatEntry* CatHandler::last()
{
   return catalog.last();
};

CatEntry* CatHandler::next()
{
   return catalog.next();
};

CatEntry* CatHandler::prev()
{
   return catalog.prev();
};

int CatHandler::findEntry(KPath what)
{
   QString* item_name;
   CatEntry* ce;
   int ret;

   item_name=what.pop();
   setCurrentBranch(what);

   ce=catalog.first();
   while (ce != 0)
   {
      if (ce->filename==*item_name)
      {
         break;
      }
      ce=catalog.next();
   }
   if (ce != 0)
   {
      ret= 1;
   }
   else
   {
      ret= -1;
   }

   return ret;
};

CatEntry* CatHandler::getCurrentEntry()
{
   return catalog.current();
};

void CatHandler::setCurrentEntry(CatEntry* ce)
{
   CatEntry* curr_item;

   curr_item=catalog.current();
   curr_item->entry_type=ce->entry_type;
   curr_item->filename=(const char*)ce->filename;
   curr_item->mod_date.setDate(ce->mod_date.date());
   curr_item->mod_date.setTime(ce->mod_date.time());
   curr_item->f_size=ce->f_size;
   curr_item->description=(const char*)ce->description;
   curr_item->mount_point=(const char*)ce->mount_point;
   is_catalog_modified=true;
};

void CatHandler::synchronizeAll()
{
   synchronize();
   if (is_catindex_modified)
   {
      saveIndex();
   }
};

bool CatHandler::scanNewCatalog(QString catalog_name,QString mount_point)
{
   KPath kp;
   QString *str;
   CatEntry *ce;
   uint total_size;
   int fbranch_ret;
   bool ret=true;

   fbranch_ret=firstBranch();
   if (fbranch_ret == -1)
   {
      newBranch(kp);
   }

   str= new QString((const char*)catalog_name);
   kp.push(str);

  	postproc= new PostProcess();
   	
   total_size=scanDir((const char*)mount_point,kp);

   if (total_size == 0) // The mount point was empty or invalid
   {
   	ret=false;
   }
   else
   {
   	firstBranch();
		ce=new CatEntry(CatEntry::E_CATALOG,QString(catalog_name),QDateTime::currentDateTime(),0,QString(""),mount_point);
		insertInSortEntry(ce);
   }

   delete postproc;
   kp.pop();
   delete str;

   return ret;
};

void CatHandler::newBranch(KPath branch)
{
   IndexEntry *ie;

//   cerr<<"New branch: ";
//   cerr<<Path2Char(branch);
//   cerr<<"\n";

   synchronize();
   ie=new IndexEntry;
   ie->key=(const char*)Path2Char(branch);
   ie->position=0;
   catindex.inSort(ie);
   current_branch=branch;
   current_branch_position=0;
   current_branch_index=catindex.at();
   clearCatalog();
   is_catindex_modified=true;
   is_catalog_modified=true;
};

void CatHandler::newBranch(QString key,uint pos)
{
   IndexEntry *ie;

//   cerr<<"New branch: ";
//   cerr<<Path2Char(branch);
//   cerr<<"\n";

   synchronize();
   ie=new IndexEntry;
   ie->key=(const char*)key;
   ie->position=pos;
   catindex.inSort(ie);
   clearCatalog();
   is_catindex_modified=true;
};

void CatHandler::deleteBranch()
{
	IndexEntry* ie;
	
   clearCatalog();
   catindex.remove();
   ie=catindex.current();
   current_branch=Char2Path(&ie->key);
   current_branch_position=ie->position;
   current_branch_index=catindex.at();

   is_catindex_modified=true;
};

void CatHandler::addEntry(CatEntry* ce)
{
   catalog.insert(catalog.at()+1,ce);
   is_catindex_modified=true;
};

void CatHandler::insertInSortEntry(CatEntry* ce)
{
   catalog.inSort(ce);
   is_catalog_modified=true;
};

void CatHandler::synchronize()
{
   uint poz;
   IndexEntry *ie;

   if (is_catalog_modified)
   {
      poz=saveCatalog();
      ie=catindex.at(current_branch_index);
      ie->position=poz;
      is_catindex_modified=true;
   }
};

int CatHandler::loadCatalog()
{
   CatEntry *ce;
   int ret= 1;
   QFile f(category_file+".kdc");
   int et=1;
   char* fname;
   char* desc;
   char* mp;
   QDateTime dt;
   uint size;
   QString mount_point;

   clearCatalog();

   f.open(IO_ReadOnly);
   f.at(current_branch_position);
   QDataStream s(&f);

   while (et!=0)
   {
      s >> et;
      if (et!=0)
      {
         if (et == CatEntry::E_CATALOG)
         {
            s >> mp;
            mount_point=QString(mp);
            delete mp;
         }
         else
         {
            mount_point="";
         }
         s >> fname;
         s >> dt;
         s >> size;
         s >> desc;

         ce=new CatEntry(et,QString(fname),dt,size,QString(desc),mount_point);
         catalog.append(ce);
         delete fname;
         delete desc;
      }
   }

   f.close();

   is_catalog_modified=false;

   return ret;
};

uint CatHandler::saveCatalog()
{
   CatEntry *ce;
   uint ret= 0;
   QFile f(category_file+".kdc");

   f.open(IO_ReadWrite);
   ret=f.size();
   f.at(ret);
   QDataStream s(&f);

   ce=catalog.first();
   while (ce != 0)
   {
      s << ce->entry_type;
      if (ce->entry_type == CatEntry::E_CATALOG)
      {
         s << (const char *) ce->mount_point;
      }
      s << (const char *) ce->filename;
      s << ce->mod_date;
      s << ce->f_size;
      s << (const char *) ce->description;

      ce=catalog.next();
   }

   s << (int) 0;

   f.close();

   is_catalog_modified=false;

   return ret;
};

int CatHandler::loadIndex()
{
   int ret=1;
   QFile f(category_file+".kdi");
   char* key;
   uint position;
   IndexEntry *ie;
   QString info;

   f.open(IO_ReadOnly);
	
   QDataStream s(&f);

   s >> info;

   if (info == QString(KDISKCAT_VERSION_INFO))
   {
	   while (!s.eof())
   	{
	      s >> key;
   	   s >> position;
	      ie= new IndexEntry;
	      ie->key=key;
	      ie->position=position;
	      catindex.append(ie);
	      delete key;
   	}   	
   }
   else
   {
   	cerr << "This file ("+category_file+".kdi) is not a real KDiskCat data file!";
   }

   f.close();

   is_catindex_modified=false;

   return ret;
};

int CatHandler::saveIndex()
{
   int ret=1;
   QFile f(category_file+".kdi");
   IndexEntry *ie;

   f.open(IO_WriteOnly);

   QDataStream s(&f);

	s << KDISKCAT_VERSION_INFO;

   ie=catindex.first();

   while (ie != 0)
   {
      s << (const char*) ie->key;
      s << (uint) ie->position;
      ie=catindex.next();
   }

   f.close();

   is_catindex_modified=false;

   return ret;
};

void CatHandler::clearIndex()
{
   catindex.clear();
};

void CatHandler::clearCatalog()
{
   catalog.clear();
   is_catalog_modified=false;
};

uint CatHandler::scanDir(const char *cur_dir, KPath cur_path)
{
   QDir current_dir(cur_dir,0,QDir::DirsFirst+QDir::Name);

   if (!current_dir.isReadable())
   {
      return 0;
   }

   const QFileInfoList *files= current_dir.entryInfoList();
   QFileInfoListIterator files_it(*files);
   QFileInfo *f;
   CatEntry *ce;
   int et;
   QString *str;
   QString new_dir;
   uint total_size=0;
   uint subdir_size;
   QString mstr;
   QString desc;

   mstr=i18n("Scanning directory: ");
   mstr+=cur_dir;
   mstr+="...";
   micro(&mstr);

   newBranch(cur_path);

   while ((f=files_it.current()) != 0)
   {
      ++files_it;
      if (f->fileName() != "." && f->fileName() != "..")
      {
         if (f->isDir()) et=CatEntry::E_DIR;
            else         et=CatEntry::E_FILE;
         if (f->isSymLink()) et+=CatEntry::E_LINK;

			desc=postproc->getDescription(f,&current_dir);
         ce=new CatEntry(et,f->fileName(),f->lastModified(),f->size(),desc,QString(""));
         catalog.insert(catalog.at()+1,ce);

         if (et != CatEntry::E_DIR)
         {
            total_size+=f->size();
         }
      }
   }

   files_it.toFirst();

   while ((f=files_it.current()) != 0)
   {
      ++files_it;
      if (f->fileName() != "." && f->fileName() != ".." && f->isDir() && !f->isSymLink())
      {
         str=new QString((const char*)f->fileName());
         cur_path.push(str);

         new_dir=cur_dir;
         new_dir+="/";
         new_dir+=f->fileName();

         subdir_size=scanDir((const char *)new_dir,cur_path);
         total_size+=subdir_size;
         cur_path.pop();
         delete str;
      }
   }

   return total_size;
};

void CatHandler::setCatalogLoading(bool onoff)
{
   synchronize();
   is_catalog_loading_on=onoff;
}

uint CatHandler::branchCount()
{
	return catindex.count();
}

void CatHandler::clearAll()
{
	clearCatalog();
	clearIndex();
   is_catalog_loading_on=true;
   category_file="";
}

void CatHandler::deleteEntry()
{
	catalog.remove();
	is_catalog_modified=true;
}

CatEntry::CatEntry(int p_entry_type,QString p_filename,QDateTime p_mod_date,uint p_f_size,QString p_description,QString p_mp)
{
   entry_type=p_entry_type;
   filename=p_filename;
   mod_date=p_mod_date;
   f_size=p_f_size;
   description=p_description;
   mount_point=p_mp;
};

CatEntry::CatEntry()
{
};

CatEntry::~CatEntry()
{
};

int IndexList::compareItems(GCI s1,GCI s2)
{
   return strcmp((const char*)((IndexEntry*)s1)->key,(const char*)((IndexEntry*)s2)->key);
};

uint CatHandler::getCurrentBranchPosition()
{
	return current_branch_position;
}

//*******************************************************************************************

CategoryHandler::CategoryHandler()
{
   cat=new CatHandler(0,"cat");
   categories.setAutoDelete(true);
};

CategoryHandler::~CategoryHandler()
{
   delete cat;
};

void CategoryHandler::init(QString dir)
{
   QDir d;
   QString str;

   str=d.homeDirPath();
   str+="/";
   str+=dir;
   d.setPath((const char*)str);
   if (!d.exists())
   {
      if (!d.mkdir(str,TRUE))
      {
         cerr << "Unable to create the data directory in "+str;
      }
   }

   db_dir=str;
   refreshCategoryList();
};

int CategoryHandler::newCatalog(QString name,QString mp)
{
	int ret=1;
	
   if (!cat->scanNewCatalog(name,mp))
   {
   	ret= -1;
   }

   return ret;
};

int CategoryHandler::copyCatalog(QString source,QString dest_category_name)
{
};

int CategoryHandler::deleteCatalog(QString name)
{
	KPath kp;
	
	kp.push(&name);
	cat->findEntry(kp);
	cat->deleteEntry();
	cat->setCurrentBranch(kp);
	cat->setCatalogLoading(false);
	cat->deleteBranch();
	kp=cat->getCurrentBranch();
	while ((kp.count() != 0) && (getCatalogName(kp) == name))
	{
		cat->deleteBranch();
		kp=cat->getCurrentBranch();
	}
	cat->setCatalogLoading(true);
};

int CategoryHandler::renameCatalog(QString old_name,QString new_name)
{
	int ret=1;
	IndexList newindexes;
	KPath kp;
	CatEntry* ce;
	IndexEntry* ie;
	QString str;
	
	newindexes.setAutoDelete(true);
	
	kp.push(&old_name);
	cat->findEntry(kp);
	ce=cat->getCurrentEntry();
	ce->filename=new_name;
	cat->setCurrentEntry(ce);
	cat->setCurrentBranch(kp);
	cat->setCatalogLoading(false);
	
	kp=cat->getCurrentBranch();
	while ((kp.count() != 0) && (getCatalogName(kp) == old_name))
	{
		ie=new IndexEntry();
		if (kp.count() > 1)
		{
			str="/"+new_name+cat->Path2Char(kp,1);
		}
		else
		{
			str="/"+new_name;
		}
		ie->key=(const char*) str;
		ie->position=cat->getCurrentBranchPosition();
		newindexes.append(ie);
		kp=cat->nextBranch();
	}

	ie=newindexes.first();
	while (ie != 0)
	{
//		cerr << ie->key;
//		cerr << ":";
//		cerr << ie->position;
//		cerr << "\n";
		cat->newBranch(ie->key,ie->position);
		ie=newindexes.next();
	}
	
	cat->setCatalogLoading(true);
	
	newindexes.clear();
	
	deleteCatalog(old_name);
	
	cat->synchronizeAll();
		
	return ret;
};

int CategoryHandler::newCategory(QString name)
{
   QFile f;
   int ret;

	name=db_dir+"/"+name+".kdc";
	f.setName(name);

   if (f.open(IO_WriteOnly))
   {
   	f.close();
   	name=name.left(name.length() - 4)+".kdi";
   	f.setName(name);
	   if (f.open(IO_WriteOnly))
	   {
		   QDataStream s(&f);
		   s << KDISKCAT_VERSION_INFO;
	   	f.close();
	   	ret=1;
	   }
	   else
	   {
	   	ret= -1;
	   }
   }
   else
   {
   	ret= -1;
   }

   if (ret == 1)
   {
   	refreshCategoryList();
   }

   return ret;
};

int CategoryHandler::deleteCategory(QString name)
{
	int ret= -1;
	QFile f;
	
	name=db_dir+"/"+name+".kdc";
	f.setName(name);

   if (f.remove())
   {
   	name=name.left(name.length() - 4)+".kdi";
   	f.setName(name);
   	if (f.remove())
   	{
   		cat->clearAll();
   		refreshCategoryList();
			ret=1;		
   	}
	}
	
	return ret;
};

int CategoryHandler::renameCategory(QString old_name,QString new_name)
{
	int ret= -1;
	QString str;
	QDir d;
	
	old_name=db_dir+"/"+old_name+".kdc";
	new_name=db_dir+"/"+new_name+".kdc";
	
	if (d.exists(new_name,true))
	{
		return -1;
	}
		
	if (d.rename(old_name,new_name,true))
	{
		old_name=old_name.left(old_name.length() - 4)+".kdi";
		new_name=new_name.left(new_name.length() - 4)+".kdi";
		if (d.rename(old_name,new_name,true))
		{
   		cat->clearAll();
   		refreshCategoryList();
			ret=1;
		}
	}
	return ret;
};

void CategoryHandler::compactAll()
{
};

QStrList CategoryHandler::getCategoryNames()
{
   CategoryEntry *ent;
   QStrList ret;

   ent=categories.first();
   while (ent != 0)
   {
      ret.append((const char*) ent->filename);
      ent=categories.next();
   }

   return ret;
};

void CategoryHandler::setActiveCategory(QString name)
{
   QString str;

   str=db_dir;
   str+="/";
   str+=name;

   cat->init(str);
   current_category=name;
};

QString CategoryHandler::getCurrentCategory()
{
   return current_category;
}

void CategoryHandler::refreshCategoryList()
{
   QDir data_dir(db_dir,0,QDir::Name);
   const QFileInfoList *files= data_dir.entryInfoList();
   QFileInfoListIterator files_it(*files);
   QFileInfo *f;
   QString s;
   CategoryEntry *ent;

   categories.clear();

   while ((f=files_it.current()) != 0)
   {
      ++files_it;
      if (f->fileName() != "." && f->fileName() != ".." &&
         f->isFile() && QString(f->fileName()).right(4)==".kdi")
      {
         s=f->fileName();
         s=s.left(s.length() - 4);

         ent=new CategoryEntry();
         ent->filename=(const char*) s;
         categories.append(ent);
      }
   }
   current_category="";
};

QString CategoryHandler::getCatalogName(KPath kp)
{
	QString str;
	
	while (kp.count() > 0)
	{
		str=*kp.pop();
	}
	
	return str;
}
















