/***********************************************************************
**
**   basemapelement.h
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht, Florian Ehinger
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#ifndef BASEMAPELEMENT_H
#define BASEMAPELEMENT_H

#include <mapcalc.h>

#include <qpainter.h>

/**
  * Baseclass for all mapelements. The class will be inherited by all
  * classes implementing the map-elements. The clas provides several
  * virtual function for writing and drawing the elements. Additionaly,
  * the cass provides two enums for the element-type and the type of
  * elevation-values.
  *
  * @author Heiner Lamprecht, Florian Ehinger
  * @version 0.1
  */
class BaseMapElement
{
 public:
  /**
    * Creates a new (virtual) mapelement.
    */
  BaseMapElement();
  /**
    *
    */
  virtual ~BaseMapElement();

  /**
    * Virtual function for drawing the element into the given painter. The
    * QRegion-pointer is used in those mapelements, which will create a
    * QRegion when drawn (e.g. AreaElement).
    * The function must be implemented in the child-classes.
    */
  virtual void drawMapElement(QPainter* targetPainter,
          double dX, double dY, long mapCenterLon) = 0;
  virtual QRegion* drawRegion(QPainter* targetPainter,
          double dX, double dY, long mapCenterLon);
  /**
    * Virtual function for writing the element-data into the given
    * binary-map-file. The function must be implemented in the
    * child-classes.
    */
  virtual bool writeElementBin(const char* fileName) = 0;
  /**
    * Virtual function for writing the element-data into the given
    * ascii-map-file. The function must be implemented in the
    * child-classes.
    */
  virtual bool writeElementAscii(const char* fileName) = 0;
  /**
    * Virtual function which returns the name of the object.
    * The name is no member of BaseMapElement, but is needed in several
    * child-classes. Therefor, the memberfunction "getName" is declared
    * here.
    */
  virtual QString getName();
  /**
    * Returns the typeID of the element.
    */
  virtual unsigned int getTypeID();
  /**
    * List of all accessable element-types.
    */
  enum objectType {NotSelected, IntAirport, Airport, MilAirport,
        CivMilAirport, Airfield, ClosedAirfield, CivHeliport, MilHeliport,
        AmbHeliport, Glidersite, UltraLight, HangGlider, Parachute, Ballon,
        Outlanding, VOR, VORDME, VORTAC, NDB, AirC, AirCtemp, AirD, AirDtemp,
        ControlD, AirElow, AirEhigh, AirF, Restricted, Danger, LowFlight,
        Obstacle, LightObstacle, ObstacleGroup, LightObstacleGroup, CompPoint,
        HugeCity, BigCity, MidCity, SmallCity, Village, Oiltank, Factory,
        Castle, Church, Tower, Highway, HighwayEntry, MidRoad, SmallRoad,
        RoadBridge, RoadTunnel, Railway, RailwayBridge, Station,
        AerialRailway, Coast, BigLake, MidLake, SmallLake, BigRiver,
        MidRiver, SmallRiver, Dam, Lock, Spot, Pass, Glacier, WayPoint, Flight};
  /**
    * The three types of elevation-data used in the maps.
    */
  enum elevationType {NotSet, MSL, GND, FL};

 protected:
  /**
    * The type-id of the element.
    */
  unsigned int typeID;
};

#endif
