/***********************************************************************
**
**   kflog.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  1999 by Heiner Lamprecht ( heiner@kijumfo.de )
**   begin        :  Sun Jun  6 1999
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <iostream>
#include <unistd.h>

#include <kfiledialog.h>
#include <kfm.h>
#include <klocale.h>
#include <ktmainwindow.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kflog.h>
#include <map.h>
#include <mapcalc.h>
#include <mapfile.h>
#include <resource.h>
#include <tablet.h>

KFLogApp::KFLogApp()
{
  setCaption("KFLog " VERSION);

  setMinimumSize(750,520);

  readOptions();

  initMenuBar();
  initToolBar();
  initStatusBar();


  if (!bViewToolbar)
    enableToolBar(KToolBar::Hide,0);
  if (!bViewStatusbar)
    enableStatusBar(KStatusBar::Hide);

  menuBar()->setMenuBarPos(menu_bar_pos);
  toolBar()->setBarPos(tool_bar_pos);

  initView();
}

KFLogApp::~KFLogApp()
{

}

void KFLogApp::enableCommand(int id_)
{
  menuBar()->setItemEnabled(id_,true);
  toolBar()->setItemEnabled(id_,true);
}

void KFLogApp::disableCommand(int id_)
{
  menuBar()->setItemEnabled(id_,false);
  toolBar()->setItemEnabled(id_,false);
}


void KFLogApp::initMenuBar()
{
  fileMenu = new QPopupMenu();
  fileMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/open.xpm"), i18n("&Open flight"), ID_FILE_OPEN );
  fileMenu->insertItem(i18n("&Close flight"),
                       ID_FILE_CLOSE );
//  fileMenu->insertItem(i18n("Close &all"),
//                       ID_FILE_CLOSE_ALL );
  fileMenu->insertSeparator();
//  fileMenu->insertItem(i18n("Import &Waypoints"),
//                       ID_FILE_WAYPOINT );
  fileMenu->insertItem(i18n("Import &Mapfile"),
                       ID_FILE_MAPIMPORT );
  fileMenu->insertSeparator();
  fileMenu->insertItem(i18n("&View map"), ID_FILE_MAP);
  fileMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-tablet.xpm"), i18n("&Create map"),
                       ID_FILE_CREATE);
  fileMenu->insertSeparator();
  fileMenu->insertItem(i18n("Save map"), ID_FILE_SAVE);
//  fileMenu->insertItem(i18n("Konvertieren"), ID_FILE_KONVERT);
//  fileMenu->insertSeparator();
//  fileMenu->insertItem(Icon(KApplication::kde_datadir() +
//            "/kflog/pics/print.xpm"), i18n("&Print"), ID_FILE_PRINT );
  fileMenu->insertSeparator();
  fileMenu->insertItem(i18n("E&xit"), ID_FILE_QUIT );

  // file_menu key accelerators
  fileMenu->setAccel(CTRL+Key_O, ID_FILE_OPEN);
  fileMenu->setAccel(CTRL+Key_W, ID_FILE_CLOSE);
  fileMenu->setAccel(CTRL+Key_I, ID_FILE_MAPIMPORT);
  fileMenu->setAccel(CTRL+Key_M, ID_FILE_MAP);
  fileMenu->setAccel(CTRL+Key_D, ID_FILE_CREATE);
  fileMenu->setAccel(CTRL+Key_S, ID_FILE_SAVE);
  fileMenu->setAccel(CTRL+Key_Q, ID_FILE_QUIT);

  mapMenu = new QPopupMenu();
  mapMenu->setCheckable(true);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/hydrography.xpm"),
            i18n("H&ydrography"), ID_LAYER_HYDRO);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-k.xpm"),
            i18n("&Topography"), ID_LAYER_TOPO);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/strassen.xpm"),
            i18n("&Highways and roads"), ID_LAYER_ROAD);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/eisenbahn.xpm"),
            i18n("&Railways and stations"), ID_LAYER_RAIL);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/stadt1.xpm"),
            i18n("&Cities and villages"), ID_LAYER_CITY);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/landmarken.xpm"),
            i18n("&Landmarks"), ID_LAYER_LAND);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/stadt2.xpm"),
            i18n("&Navigation facilities"), ID_LAYER_NAV);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/jump.xpm"),
            i18n("Additional &sites"), ID_LAYER_ADDSITES);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/civmilair.xpm"),
            i18n("&Airports"), ID_LAYER_AIRPORT);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/glider.xpm"),
            i18n("&Glider-sites"), ID_LAYER_GLIDER);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-k.xpm"),
            i18n("A&irspace-structure"), ID_LAYER_AIRSPACE);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/outlanding.xpm"),
            i18n("&Outlandingfields"), ID_LAYER_OUT);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/waypoint.xpm"),
            i18n("&Waypoints"), ID_LAYER_WAYPOINT);
  mapMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-k.xpm"),
            i18n("&Flights"), ID_LAYER_FLIGHT);

  QObject::connect(mapMenu, SIGNAL(activated(int)), SLOT(slotMenuLayer(int)));

  viewMenu = new QPopupMenu();
  viewMenu->setCheckable(true);
  viewMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-turn.xpm"), i18n("Redraw Map"), ID_REDRAW);
  viewMenu->insertItem(i18n("&Layer"), mapMenu);
  viewMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/zoomin.xpm"), i18n("Zoom in"), ID_ZOOM_IN);
  viewMenu->insertItem(Icon(KApplication::kde_datadir() +
            "/kflog/pics/zoomout.xpm"), i18n("Zoom out"), ID_ZOOM_OUT);
  viewMenu->insertSeparator();
  viewMenu->insertItem(i18n("Tool&bar"), ID_VIEW_TOOLBAR);
  viewMenu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR );

  viewMenu->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
  viewMenu->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);

  viewMenu->setAccel(CTRL+Key_R, ID_REDRAW);
  viewMenu->setAccel(Key_Plus, ID_ZOOM_IN);
  viewMenu->setAccel(Key_Minus, ID_ZOOM_OUT);

  confMenu = new QPopupMenu();
  confMenu->insertItem(i18n("Configure &map"), ID_CONF_MAP);
  confMenu->insertItem(i18n("KFLog-&setup"), ID_CONF_KFLOG);

  helpMenu = new QPopupMenu();
  helpMenu->insertItem(Icon("mini/kdehelp.xpm"),
        i18n("User &Manual"), this, SLOT(slotHelpContents()),0,
            ID_HELP_CONTENTS);
  helpMenu->insertSeparator();
  helpMenu->insertItem(Icon("mini/kflog.xpm"),
        i18n("&About KFLog..."), this, SLOT(slotHelpAbout()),0,
            ID_HELP_ABOUT);

  menuBar()->insertItem(i18n("&File"), fileMenu);
  menuBar()->insertItem(i18n("&View"), viewMenu);
  menuBar()->insertItem(i18n("&Options"), confMenu);
  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), helpMenu);

	KAccel* key_accel = new KAccel(this);
	key_accel->connectItem(KAccel::Help, kapp, SLOT(appHelpActivated()));

  CONNECT_CMD(fileMenu);
  CONNECT_CMD(viewMenu);
  CONNECT_CMD(confMenu);
}

void KFLogApp::initToolBar()
{
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/open.xpm"), ID_FILE_OPEN,
            true, i18n("Open Flight"));
//  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
//            "/kflog/pics/print.xpm"), ID_FILE_PRINT,
//            true, i18n("Print"));

  toolBar()->insertSeparator();

  toolBar()->insertButton(Icon("mini/mini-turn.xpm"),
            ID_REDRAW, true, i18n("Redraw Map"));

  toolBar()->insertSeparator();

  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/hydrography.xpm"), ID_LAYER_HYDRO,
            true, i18n("Show the hydrography"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-k.xpm"), ID_LAYER_TOPO,
            true, i18n("Show the topography"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/strassen.xpm"), ID_LAYER_ROAD,
            true, i18n("Show highways and roads"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/eisenbahn.xpm"), ID_LAYER_RAIL,
            true, i18n("Show railways and stations"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/stadt1.xpm"), ID_LAYER_CITY,
            true, i18n("Show cities and villages"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/landmarken.xpm"), ID_LAYER_LAND,
            true, i18n("Show landmarks"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/stadt2.xpm"), ID_LAYER_NAV,
            true, i18n("Show navigation facilities"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/jump.xpm"), ID_LAYER_ADDSITES,
            true, i18n("Show additional sites"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/civmilair.xpm"), ID_LAYER_AIRPORT,
            true, i18n("Show airports"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/glider.xpm"), ID_LAYER_GLIDER,
            true, i18n("Show glider-sites"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-k.xpm"), ID_LAYER_AIRSPACE,
            true, i18n("Show airspace-structure"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/outlanding.xpm"), ID_LAYER_OUT,
            true, i18n("Show outlandingfields"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/map/small/waypoint.xpm"), ID_LAYER_WAYPOINT,
            true, i18n("Show waypoints"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/mini-k.xpm"), ID_LAYER_FLIGHT,
            true, i18n("Show flights"));

  toolBar()->setToggle(ID_LAYER_ADDSITES, true);
  toolBar()->setToggle(ID_LAYER_AIRPORT, true);
  toolBar()->setToggle(ID_LAYER_AIRSPACE, true);
  toolBar()->setToggle(ID_LAYER_CITY, true);
  toolBar()->setToggle(ID_LAYER_FLIGHT, true);
  toolBar()->setToggle(ID_LAYER_GLIDER, true);
  toolBar()->setToggle(ID_LAYER_HYDRO, true);
  toolBar()->setToggle(ID_LAYER_LAND, true);
  toolBar()->setToggle(ID_LAYER_NAV, true);
  toolBar()->setToggle(ID_LAYER_OUT, true);
  toolBar()->setToggle(ID_LAYER_RAIL, true);
  toolBar()->setToggle(ID_LAYER_ROAD, true);
  toolBar()->setToggle(ID_LAYER_TOPO, true);
  toolBar()->setToggle(ID_LAYER_WAYPOINT, true);

  toolBar()->insertSeparator();

  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/zoomin.xpm"), ID_ZOOM_IN,
            true, i18n("Zoom in"));
  toolBar()->insertButton(Icon(KApplication::kde_datadir() +
            "/kflog/pics/zoomout.xpm"), ID_ZOOM_OUT,
            true, i18n("Zoom out"));

  toolBar()->insertSeparator();

  QWhatsThis* whats_this = new QWhatsThis;
  QToolButton* helpButton = whats_this->whatsThisButton(toolBar());
  QToolTip::add(helpButton, i18n("What's this...?"));
  toolBar()->insertWidget(ID_HELP_WHATS_THIS,
          helpButton->sizeHint().width(), helpButton);

  QWhatsThis::add(toolBar()->getButton(ID_FILE_OPEN),
    i18n("Open flight\n\n"
         "Shows the Open file dialog to select a file to be opened."));

  this->connect(toolBar()->getButton(ID_FILE_OPEN), SIGNAL(clicked()),
                this, SLOT(slotFileOpen()));
  this->connect(toolBar()->getButton(ID_REDRAW), SIGNAL(clicked()),
                this, SLOT(slotRedrawMap()));
  this->connect(toolBar()->getButton(ID_LAYER_ADDSITES), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_AIRPORT), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_AIRSPACE), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_CITY), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_FLIGHT), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_GLIDER), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_HYDRO), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_LAND), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_NAV), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_OUT), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_RAIL), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_ROAD), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_TOPO), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_LAYER_WAYPOINT), SIGNAL(clicked()),
                this, SLOT(slotShowLayer()));
  this->connect(toolBar()->getButton(ID_ZOOM_IN), SIGNAL(clicked()),
                this, SLOT(slotScaleAdd()));
  this->connect(toolBar()->getButton(ID_ZOOM_OUT), SIGNAL(clicked()),
                this, SLOT(slotScaleSub()));
}

void KFLogApp::initStatusBar()
{
  statusProgress = new KProgress(0, 100, 0,
    KProgress::Horizontal, statusBar());
  statusBar()->insertItem(IDS_COORD_DEFAULT_H, ID_STATUS_COORD_H);
  statusBar()->insertItem(IDS_COORD_DEFAULT_R, ID_STATUS_COORD_R);
  statusBar()->insertWidget(statusProgress, 150, ID_STATUS_PROGRESS);
  statusBar()->insertItem(IDS_DEFAULT, ID_STATUS_MSG );
  statusBar()->setInsertOrder(KStatusBar::RightToLeft);
}

void KFLogApp::initView()
{
  enableCommand(ID_ZOOM_IN);
  enableCommand(ID_ZOOM_OUT);

  mapContents = new MapContents();
  readBinaryFile(mapFile, mapContents);

  KApplication::getKApplication();
  view = new KFLogView(this, mapContents);
  setFrameBorderWidth(5);
  setView(view);

  // inserting the map:
  map = new Map(view->getMapFrame(), this, view);
  QVBoxLayout* mapLayout = new QVBoxLayout(view->getMapFrame(),2,1);
  mapLayout->addWidget(map);
  mapLayout->activate();

  KDNDDropZone* dropZone = new KDNDDropZone(map, DndURL);
  connect(dropZone, SIGNAL(dropAction(KDNDDropZone*)),
    SLOT(slotDropEvent(KDNDDropZone*)));
}

bool KFLogApp::queryExit()
{
  return true;
}

void KFLogApp::saveOptions()
{
  KConfig* config = kapp->getConfig();

  config->setGroup("General");
	config->writeEntry("Geometry", size() );
  config->writeEntry("MapFile", mapFile);

  config->setGroup("Appearance");
  config->writeEntry("ShowToolbar",toolBar()->isVisible());
  config->writeEntry("ShowStatusbar",statusBar()->isVisible());
  config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
  config->writeEntry("ToolBar_Pos", (int)toolBar()->barPos());
}

void KFLogApp::readOptions()
{
  KConfig *config = kapp->getConfig();

  config->setGroup("General");
  QSize size=config->readSizeEntry("Geometry");
	if(!size.isEmpty())
		resize(size);
  if(config->hasKey("MapFile")) {
    mapFile = config->readEntry("MapFile");
  } else {
    mapFile = kapp->kde_datadir();
    mapFile += "/kflog/map.wld";
  }

  if(config->hasKey("FlightDir")) {
    flightDir = config->readEntry("FlightDir");
  } else {
    flightDir = ".";
  }

  config->setGroup("Appearance");
  bViewToolbar = config->readBoolEntry("ShowToolbar", true);
  bViewStatusbar = config->readBoolEntry("ShowStatusbar", true);
  menu_bar_pos = (KMenuBar::menuPosition)config->readNumEntry("MenuBarPos",
                                                         KMenuBar::Top);
  tool_bar_pos = (KToolBar::BarPosition)config->readNumEntry("ToolBar_Pos",
                                                         KToolBar::Top);
}

void KFLogApp::showCoords(long lat, long lon)
{
  QString X, Y;

  X = printPos(lat);
  Y = printPos(lon, false);

  statusBar()->clear();
  statusBar()->changeItem(X, ID_STATUS_COORD_R);
  statusBar()->changeItem(Y, ID_STATUS_COORD_H);
}

MapContents* KFLogApp::getMapContents()
{
  return mapContents;
}

void KFLogApp::slotMenuLayer(int id)
{
  toolBar()->getButton(id)->toggle();
  slotShowLayer();
}

void KFLogApp::slotShowMap()
{
  view->showMapFrame();
  enableCommand(ID_ZOOM_IN);
  enableCommand(ID_ZOOM_OUT);
  enableCommand(ID_REDRAW);

  enableCommand(ID_LAYER_ADDSITES);
  enableCommand(ID_LAYER_AIRPORT);
  enableCommand(ID_LAYER_AIRSPACE);
  enableCommand(ID_LAYER_CITY);
  enableCommand(ID_LAYER_FLIGHT);
  enableCommand(ID_LAYER_GLIDER);
  enableCommand(ID_LAYER_HYDRO);
  enableCommand(ID_LAYER_LAND);
  enableCommand(ID_LAYER_NAV);
  enableCommand(ID_LAYER_OUT);
  enableCommand(ID_LAYER_RAIL);
  enableCommand(ID_LAYER_ROAD);
  enableCommand(ID_LAYER_TOPO);
  enableCommand(ID_LAYER_WAYPOINT);
}

void KFLogApp::slotShowDig()
{
  disableCommand(ID_ZOOM_IN);
  disableCommand(ID_ZOOM_OUT);
  disableCommand(ID_REDRAW);
  disableCommand(ID_LAYER_ADDSITES);
  disableCommand(ID_LAYER_AIRPORT);
  disableCommand(ID_LAYER_AIRSPACE);
  disableCommand(ID_LAYER_CITY);
  disableCommand(ID_LAYER_FLIGHT);
  disableCommand(ID_LAYER_GLIDER);
  disableCommand(ID_LAYER_HYDRO);
  disableCommand(ID_LAYER_LAND);
  disableCommand(ID_LAYER_NAV);
  disableCommand(ID_LAYER_OUT);
  disableCommand(ID_LAYER_RAIL);
  disableCommand(ID_LAYER_ROAD);
  disableCommand(ID_LAYER_TOPO);
  disableCommand(ID_LAYER_WAYPOINT);

  view->showTabFrame();
}

void KFLogApp::slotOpenFlight(QString fileName)
{
  if(fileName.right(4) == ".igc") {
    // "Formatprfung" noch nicht ausreichend ...
    map->loadFlight(mapContents, fileName);
  }
}

void KFLogApp::slotDropEvent(KDNDDropZone* dropZone)
{
  QStrList urlList = dropZone->getURLList();
  QStrListIterator it(urlList);
  const char* current;
  while((current = it.current()) != 0) {
    ++it;
    QString tempFile;
    if(KFM::download(current, tempFile)) {
      slotOpenFlight(tempFile);
    }
  }
}

void KFLogApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));

  KFileDialog fD(flightDir);

  QString fileName = fD.getOpenFileName("/home/heiner/C++/kflog",
          "*.igc", this);

  if(fileName != NULL) {
    map->loadFlight(mapContents, fileName);
  }

  slotStatusMsg(IDS_DEFAULT);
}

void KFLogApp::slotFlightClose()
{
  map->slotDeleteFlight();
}

void KFLogApp::slotLoadWaypoint()
{
  slotStatusMsg(i18n("Opening file..."));

  KFileDialog fD(flightDir);

  QString fileName = fD.getOpenFileName("/home/heiner/C++/kflog",
          "*iw", this);

  if(fileName != NULL) {
    map->loadWaypoint(mapContents, fileName);
  }
}

void KFLogApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));

  QPrinter printer;
  if (printer.setup(this)) {
    QPainter painter;
    painter.begin( &printer );
    painter.end();
  };

  slotStatusMsg(IDS_DEFAULT);
}

void KFLogApp::slotFileQuit()
{
  if(this->queryExit()) {
    saveOptions();
    KTMainWindow::deleteAll();
    kapp->quit();
  } else {
    slotStatusMsg(IDS_DEFAULT);
  }
  return;
}

void KFLogApp::slotRedrawMap()
{
  map->slotEnableMap(true);
  map->slotRedrawMap();
}

void KFLogApp::slotShowLayer()
{
  map->slotShowLayer();
}

void KFLogApp::slotScaleAdd()
{
  map->slotScaleAdd();
}

void KFLogApp::slotScaleSub()
{
  map->slotScaleSub();
}

void KFLogApp::slotShowObjectData(unsigned int objectID)
{

}

void KFLogApp::slotHelpContents(){
  kapp->invokeHTMLHelp("kflog/index.html","");
}

void KFLogApp::slotHelpAbout()
{
  QMessageBox aboutmsg(this, "About KFLog");
  aboutmsg.setCaption(i18n("About KFLog..."));
  aboutmsg.setButtonText(1, i18n("OK"));
  aboutmsg.setText(i18n("KFLog < " VERSION " > \n\n"
											"(c) 1999, 2000"
											"\n\t\tHeiner Lamprecht <heiner@kflog.de>\n"
                      "\t\tFlorian Ehinger <florian@kflog.de>\n\n"
                      "This program is free software; you can\n"
                      "redistribute it and/or modify it under\n"
                      "the terms of the GNU General Public License\n"
                      "as published by the Free Software Foundation;\n"
                      "either version 2 of the License, or (at your\n"
                      "option) any later version.\n\n"
                      "This program comes in the hope that it will\n"
											"be useful, but WITHOUT ANY WARRANTY; \n"
											"without even the implied warranty of\n"
											"MERCHANTABILITY or FITNESS FOR A\n"
											"PARTICULAR PURPOSE.\n"));

  QPixmap logo;
  logo.load(KApplication::kde_datadir() +
              "/kflog/pics/aboutlogo.xpm");
  aboutmsg.setIconPixmap(logo);
  aboutmsg.show();
}

void KFLogApp::slotImportMap()
{
  slotStatusMsg(i18n("Import mapfile..."));

  KFileDialog fD(flightDir);

  // Endung von Kartendateien ???
  QString fileName = fD.getOpenFileName("/home/heiner/C++/kflog",
          "*out", this);

  if(fileName != NULL) {
    readAsciiFile(fileName, mapContents);
    map->slotRedrawMap();
  }
}

void KFLogApp::slotConfigureMap()
{
  map->slotConfigureMap();
}

void KFLogApp::slotConfigureKFLog()
{

}

void KFLogApp::slotConvertFile()
{
  ascii2bin();
}

void KFLogApp::slotSaveFile()
{
  slotStatusMsg(i18n("Saving map-file..."));

  KFileDialog fD(mapDir);

  QString fileName = fD.getOpenFileName(mapDir, "*.wld", this);

  if (fileName != NULL) {
    writeBinaryFile(fileName, mapContents);
  }
  slotStatusMsg(IDS_DEFAULT);
}

void KFLogApp::slotSetProgress(int value)
{
  statusProgress->setValue(value);
}

void KFLogApp::slotViewToolBar()
{
  bViewToolbar=!bViewToolbar;
  menuBar()->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
  enableToolBar(KToolBar::Toggle,0);
  enableToolBar(KToolBar::Toggle,ID_PRIV_TOOLBAR);
  slotStatusMsg(IDS_DEFAULT);
}

void KFLogApp::slotViewStatusBar()
{
  bViewStatusbar=!bViewStatusbar;
  menuBar()->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
  enableStatusBar();
  slotStatusMsg(IDS_DEFAULT);
}

void KFLogApp::slotStatusMsg(const char* text)
{
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG );
}

void KFLogApp::slotStatusHelpMsg(const char* text)
{
  statusBar()->message(text, 2000);
}

void KFLogApp::commandCallback(int id_){
  switch(id_) {
    ON_CMD(ID_FILE_OPEN,                slotFileOpen())
    ON_CMD(ID_FILE_CLOSE,               slotFlightClose())
    ON_CMD(ID_FILE_WAYPOINT,            slotLoadWaypoint())
    ON_CMD(ID_FILE_MAPIMPORT,           slotImportMap())
    ON_CMD(ID_FILE_MAP,                 slotShowMap())
    ON_CMD(ID_FILE_CREATE,              slotShowDig())
    ON_CMD(ID_FILE_PRINT,               slotFilePrint())
    ON_CMD(ID_FILE_QUIT,                slotFileQuit())

    ON_CMD(ID_FILE_KONVERT,             slotConvertFile());
    ON_CMD(ID_FILE_SAVE,                slotSaveFile());

    ON_CMD(ID_VIEW_TOOLBAR,             slotViewToolBar())
    ON_CMD(ID_VIEW_STATUSBAR,           slotViewStatusBar())
    ON_CMD(ID_LAYER_ADDSITES,           slotShowLayer())
    ON_CMD(ID_LAYER_AIRPORT,            slotShowLayer())
    ON_CMD(ID_LAYER_AIRSPACE,           slotShowLayer())
    ON_CMD(ID_LAYER_CITY,               slotShowLayer())
    ON_CMD(ID_LAYER_FLIGHT,             slotShowLayer())
    ON_CMD(ID_LAYER_GLIDER,             slotShowLayer())
    ON_CMD(ID_LAYER_HYDRO,              slotShowLayer())
    ON_CMD(ID_LAYER_LAND,               slotShowLayer())
    ON_CMD(ID_LAYER_NAV,                slotShowLayer())
    ON_CMD(ID_LAYER_OUT,                slotShowLayer())
    ON_CMD(ID_LAYER_RAIL,               slotShowLayer())
    ON_CMD(ID_LAYER_ROAD,               slotShowLayer())
    ON_CMD(ID_LAYER_TOPO,               slotShowLayer())
    ON_CMD(ID_LAYER_WAYPOINT,           slotShowLayer())
    ON_CMD(ID_REDRAW,                   slotRedrawMap())
    ON_CMD(ID_ZOOM_IN,                  slotScaleAdd())
    ON_CMD(ID_ZOOM_OUT,                 slotScaleSub())

    ON_CMD(ID_CONF_MAP,                 slotConfigureMap())
    ON_CMD(ID_CONF_KFLOG,               slotConfigureKFLog())
  }
}

void KFLogApp::statusCallback(int id_){
  switch(id_) {
    ON_STATUS_MSG(ID_FILE_OPEN,
                  i18n("Opens an existing flight"))
    ON_STATUS_MSG(ID_FILE_PRINT,
                  i18n("Prints the current map"))
    ON_STATUS_MSG(ID_FILE_MAPIMPORT,
                  i18n("Imports an existing mapfile"));
    ON_STATUS_MSG(ID_FILE_QUIT,
                  i18n("Exits the program"))
    ON_STATUS_MSG(ID_VIEW_TOOLBAR,
                  i18n("Enables / disables the actual Toolbar"))
    ON_STATUS_MSG(ID_VIEW_STATUSBAR,
                  i18n("Enables / disables the Statusbar"))
  }
}
