/***********************************************************************
**
**   kflogview.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  1999 by Heiner Lamprecht ( heiner@kijumfo.de )
**   begin        :  Sun Jun  6 1999
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <kbutton.h>
#include <kcombo.h>
#include <kintegerline.h>
#include <klocale.h>
#include <kmsgbox.h>
#include <krestrictedline.h>
#include <qcheckbox.h>
#include <qfileinfo.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qtextstream.h>

#include <flight.h>
#include <map.h>
#include <kflog.h>
#include <kflogview.h>
#include <mapcalc.h>
#include <mapcontents.h>
#include <tablet.h>

#include <intairport.h>

#define ENTRY(parent, text) new QListViewItem(parent,"",i18n(text));

void KFLogView::delWPListView()
{
  wpList->clear();
  text1->setText("");
  text2->setText("");
  text3->setText("");
}

void KFLogView::showFlightData(Flight* flight)
{
  text1->setText(flight->getDate());
  text2->setText(flight->getPilot());
  text3->setText(flight->getType() + " / " + flight->getID());

  QList<struct wayPoint>* wayPointList = flight->getWPList();

  QListViewItem* line = 0;
  QListViewItem* pre = 0;

  if(wayPointList->count() == 0) {
    warning("KFLog: Flight contains no waypoints!");
    return;
  }

  wpList->clear();

  double ddist;
  int dist1, dist2;

  wayPoint* prePoint = wayPointList->at(wayPointList->count() - 2);
  wayPoint* current = wayPointList->at(wayPointList->count() - 1);

  ddist = dist(prePoint->latitude, prePoint->longitude,
                     current->latitude, current->longitude);

  dist1 = (int) ddist;
  dist2 = (int) ( (ddist - dist1) * 10 );

  QString distString;
  distString.sprintf("%3d,%d KM ", dist1, dist2);

  QString X, Y;
  X = printPos(current->latitude);
  Y = printPos(current->longitude, false);

  line = new QListViewItem(wpList);
  line->setText(0, "L");
  line->setText(1, current->name);
  line->setText(2, flight->getLandTime());
  line->setText(3, X + " / " + Y);
  line->setText(4, distString);

  pre = line;

  for(unsigned int loop = (wayPointList->count() - 2); loop > 0; loop--) {
    prePoint = wayPointList->at(loop - 1);
    current = wayPointList->at(loop);

    X = printPos(current->latitude);
    Y = printPos(current->longitude, false);

    ddist = dist(prePoint->latitude, prePoint->longitude,
                       current->latitude, current->longitude);

    dist1 = (int) ddist;
    dist2 = (int) ( (ddist - dist1) * 10 );

    line = new QListViewItem(wpList);
    QString number;
    number.sprintf("%d", loop);
    line->setText(0, number);
    line->setText(1, current->name);

    if(current->time != 0) {
      line->setText(2, printTime(current->time));
    } else {
      line->setText(2,"0:00:00");
    }

    line->setText(3, X + " / " + Y);

    QString distString;
    distString.sprintf("%3d,%d KM ", dist1, dist2);
    line->setText(4, distString);

    pre = line;
  }

  current = wayPointList->at(0);

  X = printPos(current->latitude);
  Y = printPos(current->longitude, false);

  line = new QListViewItem(wpList);
  line->setText(0, "S");
  line->setText(1, current->name);
  line->setText(2, flight->getStartTime());
  line->setText(3, X + " / " + Y);
}

void KFLogView::showMapFrame()
{
  tabView->hide();
  mapView->show();
}

void KFLogView::showTabFrame()
{
  mapView->hide();
  tabView->show();
}

QFrame* KFLogView::getMapFrame()
{
  return mapFrame;
}

QFrame* KFLogView::getMapView()
{
  return mapView;
}

QFrame* KFLogView::getTabView()
{
  return mapView;
}

void KFLogView::selectMapElement(QListViewItem* item)
{
  bool aliasState  = false;
  bool elevState   = false;
  bool lowerTState = false;
  bool uppTState   = false;
  bool freqState   = false;
  bool gliderState = false;
  bool runwayState = false;
  bool uppState    = false;
  bool vdfState    = false;
  pointList   = false;
  // NOTE: nameState is used so often, that we enable it here
  bool nameState   = true;

  if(item->depth() == 1) {
    QString textString = item->text(1);
    typLabel->setText(item->text(1));
    if(textString == i18n("International Airport")) {
      aliasState  = true;
      elevState   = true;
      freqState   = true;
      runwayState = true;
      vdfState    = true;
      type        = BaseMapElement::IntAirport;
    } else if(textString == i18n("Airport, Airfield IFR")) {
      aliasState  = true;
      elevState   = true;
      freqState   = true;
      runwayState = true;
      vdfState    = true;
      type        = BaseMapElement::Airport;
    } else if(textString == i18n("Military Aerodrome")) {
      aliasState  = true;
      elevState   = true;
      freqState   = true;
      runwayState = true;
      vdfState    = true;
      type        = BaseMapElement::MilAirport;
    } else if(textString == i18n("Airfield (civil/military)")) {
      aliasState  = true;
      elevState   = true;
      freqState   = true;
      runwayState = true;
      vdfState    = true;
      type        = BaseMapElement::CivMilAirport;
    } else if(textString == i18n("Airfield")) {
      aliasState  = true;
      elevState   = true;
      freqState   = true;
      runwayState = true;
      vdfState    = true;
      type        = BaseMapElement::Airfield;
    } else if(textString == i18n("Airfield (closed)")) {
      type        = BaseMapElement::ClosedAirfield;
    } else if(textString == i18n("civ. Heliport")) {
      elevState   = true;
      freqState   = true;
      type        = BaseMapElement::CivHeliport;
    } else if(textString == i18n("mil. Heliport")) {
      elevState   = true;
      freqState   = true;
      type        = BaseMapElement::MilHeliport;
    } else if(textString == i18n("amb. Heliport")) {
      elevState   = true;
      freqState   = true;
      type        = BaseMapElement::AmbHeliport;
    } else if(textString == i18n("Glider site")) {
      elevState   = true;
      freqState   = true;
      gliderState = true;
      runwayState = true;
      type        = BaseMapElement::Glidersite;
    } else if(textString == i18n("Ultra light flying")) {
      type        = BaseMapElement::UltraLight;
    } else if(textString == i18n("Hang glider site")) {
      type        = BaseMapElement::HangGlider;
    } else if(textString == i18n("Parachute jumping site")) {
      type        = BaseMapElement::Parachute;
    } else if(textString == i18n("Free ballon site")) {
      type        = BaseMapElement::Ballon;
    } else if(textString == i18n("Qutlanding Field")) {
      elevState   = true;
      type        = BaseMapElement::Outlanding;
    } else if(textString == i18n("VOR")) {
      aliasState  = true;
      freqState   = true;
      type        = BaseMapElement::VOR;
    } else if(textString == i18n("VOR / DME")) {
      aliasState  = true;
      freqState   = true;
      type        = BaseMapElement::VORDME;
    } else if(textString == i18n("VOR-TACAN")) {
      aliasState  = true;
      freqState   = true;
      type        = BaseMapElement::VORTAC;
    } else if(textString == i18n("NDB")) {
      aliasState  = true;
      freqState   = true;
      type        = BaseMapElement::NDB;
    } else if(textString == i18n("Airspace C")) {
      uppTState   = true;
      lowerTState = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::AirC;
    } else if(textString == i18n("Airspace C (temp)")) {
      uppTState   = true;
      lowerTState = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::AirCtemp;
    } else if(textString == i18n("Airspace D")) {
      uppTState   = true;
      lowerTState = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::AirD;
    } else if(textString == i18n("Airspace D (temp)")) {
      uppTState   = true;
      lowerTState = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::AirDtemp;
    } else if(textString == i18n("Control Zone Airspace D")) {
      uppTState   = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::ControlD;
    } else if(textString == i18n("Airspace E (1700ft)")) {
      uppTState   = true;
      pointList   = true;
      type        = BaseMapElement::AirEhigh;
    } else if(textString == i18n("Airspace E (1000ft)")) {
      uppTState   = true;
      pointList   = true;
      type        = BaseMapElement::AirElow;
    } else if(textString == i18n("Airspace F (HX)")) {
      uppTState   = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::AirF;
    } else if(textString == i18n("Restricted Area")) {
      uppTState   = true;
      lowerTState = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::Restricted;
    } else if(textString == i18n("Danger Area")) {
      uppTState   = true;
      lowerTState = true;
      pointList   = true;
      uppState    = true;
      type        = BaseMapElement::Danger;
    } else if(textString == i18n("250ft low flight area")) {
      pointList   = true;
      type        = BaseMapElement::LowFlight;
    } else if(textString == i18n("Obstacle")) {
      type        = BaseMapElement::Obstacle;
      elevState   = true;
      nameState   = false;
    } else if(textString == i18n("Obstacle (lighted)")) {
      type        = BaseMapElement::LightObstacle;
      elevState   = true;
      nameState   = false;
    } else if(textString == i18n("Obstaclegroup")) {
      type        = BaseMapElement::ObstacleGroup;
      elevState   = true;
      nameState   = false;
    } else if(textString == i18n("Obstaclegroup (lighted)")) {
      type        = BaseMapElement::LightObstacleGroup;
      elevState   = true;
      nameState   = false;
    } else if(textString == i18n("Compulsary reporting point")) {
      type        = BaseMapElement::CompPoint;
    } else if(textString == i18n("City (> 500.000)")) {
      pointList   = true;
      type        = BaseMapElement::HugeCity;
    } else if(textString == i18n("City (100.000 - 500.000)")) {
      pointList   = true;
      type        = BaseMapElement::BigCity;
    } else if(textString == i18n("City (50.000 - 100.000)")) {
      pointList   = true;
      type        = BaseMapElement::MidCity;
    } else if(textString == i18n("City (20.000 - 50.000)")) {
      pointList   = true;
      type        = BaseMapElement::SmallCity;
    } else if(textString == i18n("Village")) {
      type        = BaseMapElement::Village;
    } else if(textString == i18n("Oiltank")) {
      type        = BaseMapElement::Oiltank;
    } else if(textString == i18n("Factory")) {
      type        = BaseMapElement::Factory;
    } else if(textString == i18n("Ruin / Fort / Castle")) {
      type        = BaseMapElement::Castle;
    } else if(textString == i18n("Church / Monastry")) {
      type        = BaseMapElement::Church;
    } else if(textString == i18n("Tower")) {
      type        = BaseMapElement::Tower;
    } else if(textString == i18n("Highway")) {
      pointList   = true;
      type        = BaseMapElement::Highway;
    } else if(textString == i18n("Road (small)")) {
      pointList   = true;
      type        = BaseMapElement::SmallRoad;
    } else if(textString == i18n("Road (medium)")) {
      pointList   = true;
      type        = BaseMapElement::MidRoad;
    } else if(textString == i18n("Highway entry")) {
      type        = BaseMapElement::HighwayEntry;
    } else if(textString == i18n("Roadbridge")) {
      type        = BaseMapElement::RoadBridge;
    } else if(textString == i18n("Road tunnel")) {
      type        = BaseMapElement::RoadTunnel;
    } else if(textString == i18n("Railway")) {
      pointList   = true;
      type        = BaseMapElement::Railway;
    } else if(textString == i18n("Station")) {
      type        = BaseMapElement::Station;
    } else if(textString == i18n("Railwaybridge")) {
      type        = BaseMapElement::RailwayBridge;
    } else if(textString == i18n("Aerial Railway")) {
      pointList   = true;
      type        = BaseMapElement::AerialRailway;
    } else if(textString == i18n("Shore line")) {
      pointList   = true;
      type        = BaseMapElement::Coast;
    } else if(textString == i18n("Lake (small)")) {
      pointList   = true;
      type        = BaseMapElement::SmallLake;
    } else if(textString == i18n("Lake (medium)")) {
      pointList   = true;
      type        = BaseMapElement::MidLake;
    } else if(textString == i18n("Lake (big)")) {
      pointList   = true;
      type        = BaseMapElement::BigLake;
    } else if(textString == i18n("River / Canal (small)")) {
      pointList   = true;
      type        = BaseMapElement::SmallRiver;
    } else if(textString == i18n("River / Canal (medium)")) {
      pointList   = true;
      type        = BaseMapElement::MidRiver;
    } else if(textString == i18n("River / Canal (big)")) {
      pointList   = true;
      type        = BaseMapElement::BigRiver;
    } else if(textString == i18n("Damm / Barrage")) {
      type        = BaseMapElement::Dam;
    } else if(textString == i18n("Lock")) {
      type        = BaseMapElement::Lock;
    } else if(textString == i18n("Spot elevation")) {
      type        = BaseMapElement::Spot;
      elevState   = true;
      nameState   = false;
    } else if(textString == i18n("Mountain pass")) {
      type        = BaseMapElement::Pass;
      elevState   = true;
    } else if(textString == i18n("Glacier and ice caps")) {
      pointList   = true;
      type        = BaseMapElement::Glacier;
    }
    // mu noch erweitert werden !!!
  } else {
    typLabel->setText("");
  }

  abkE->setEnabled(aliasState);
  addB->setEnabled(runwayState);
  delB->setEnabled(runwayState);
  dirE->setEnabled(runwayState);
  elevE->setEnabled(elevState);
  lowerT->setEnabled(lowerTState);
  lowerE->setEnabled(lowerTState);
  freqE->setEnabled(freqState);
  gliderT->setEnabled(gliderState);
  nameE->setEnabled(nameState);
  runwE->setEnabled(runwayState);
  runwtE->setEnabled(runwayState);
  uppE->setEnabled(uppState);
  uppT->setEnabled(uppState);
  vdfE->setEnabled(vdfState);

  abkE->setText("");
  dirE->setText("");
  elevE->setText("");
  freqE->setText("");
  nameE->setText("");
  runwE->setText("");
  uppE->setText("");

  lowerT->setCurrentItem(0);
  uppT->setCurrentItem(0);
  gliderT->setCurrentItem(0);
  runwtE->setCurrentItem(0);

  runwList->clear();

  vdfE->setChecked(false);
}

bool KFLogView::checkInputFields()
{
  //
  // are all other data given?
  QString missingFields;
  bool error = false;
  // Name
  if(nameE->isEnabled() && !strlen(nameE->text())) {
    error = true;
    missingFields += "  \"";
    missingFields += i18n("Name");
    missingFields += "\"";
  }
  // Abbreviation
  if(abkE->isEnabled()  && !strlen(abkE->text())) {
    error = true;
    missingFields += "  \"";
    missingFields += i18n("Abbreviation");
    missingFields += "\"";
  }
  // Elevation
  if(elevE->isEnabled() && !strlen(elevE->text())) {
    error = true;
    missingFields += "  \"";
    missingFields += i18n("Elev.");
    missingFields += "\"";
  }
  // Frequency
  if(freqE->isEnabled() && !strlen(freqE->text())) {
    error = true;
    missingFields += "  \"";
    missingFields += i18n("Frequency");
    missingFields += "\"";
  }
  // Upper Limit
  if(uppE->isEnabled()  && !strlen(uppE->text())) {
    error = true;
    missingFields += "  \"";
    missingFields += i18n("Upper Limit");
    missingFields += "\"";
  }
  // Runways
  if(runwE->isEnabled() && !runwList->childCount()) {
    error = true;
    missingFields += "\nThere is no runway given!";
  }

  if(error) {
    QString errorMsg = i18n("The following fields are empty!\nPlease \
insert the missing data first.\n\n");
    errorMsg += missingFields;
    KMsgBox* errorBox = new KMsgBox(this,i18n("Missing Fields"),
        errorMsg,4,i18n("OK"));
    errorBox->exec();
    return false;
  }
  return true;
}

struct intrunway* KFLogView::createIntRunway()
{
  struct intrunway* irwData = new intrunway[runwList->childCount()];
  int loop = 0;

  for(QListViewItem* current = runwList->firstChild();
        current != 0; current = current->nextSibling()) {
    irwData[loop].length = ((QString)current->text(1)).toUInt();
    irwData[loop].direction = ((QString)current->text(2)).toUInt();
    if(current->text(3) == i18n("concrete")) {
      irwData[loop].surface = Airport::Concrete;
    } else if(current->text(3) == i18n("asphalt")) {
      irwData[loop].surface = Airport::Asphalt;
    } else {
      irwData[loop].surface = Airport::Grass;
    }
    irwData[loop].latitudeA = ((QString)current->text(4)).toLong();
    irwData[loop].longitudeA = ((QString)current->text(5)).toLong();
    irwData[loop].latitudeB = ((QString)current->text(6)).toLong();
    irwData[loop].longitudeB = ((QString)current->text(7)).toLong();
    loop++;
  }
  return irwData;
}

struct runway* KFLogView::createRunway()
{
  struct runway* rwData = new runway[runwList->childCount()];
  int loop = 0;

  for(QListViewItem* current = runwList->firstChild();
        current != 0; current = current->nextSibling()) {
    rwData[loop].length = ((QString)current->text(1)).toUInt();
    rwData[loop].direction = ((QString)current->text(2)).toUInt();
    if(current->text(3) == i18n("concrete")) {
      rwData[loop].surface = Airport::Concrete;
    } else if(current->text(3) == i18n("asphalt")) {
      rwData[loop].surface = Airport::Asphalt;
    } else {
      rwData[loop].surface = Airport::Grass;
    }
    loop++;
  }
  return rwData;
}

void KFLogView::createMapElement(long* latitude, long* longitude,
                             unsigned int length)
{
  const char* fileName = "/tmp/kflog.out";

  IntAirport* intAir;
  Airport* airport;
  GliderSite* glider;
  SinglePoint* singleP;
  ElevPoint* elevP;
  RadioPoint* radioP;
  Airspace* airspace;
  AreaElement* area;
  LineElement* lineE;

  switch(type) {
    case BaseMapElement::IntAirport:
      intAir = new IntAirport(nameE->text(), abkE->text(),
            length, latitude, longitude, ((QString) elevE->text()).toUInt(),
            freqE->text(), vdfE->isChecked(), createIntRunway(),
            runwList->childCount());
      intAir->writeElementAscii(fileName);
      mapContents->addElement(intAir);
      break;
    case BaseMapElement::Airport:
    case BaseMapElement::MilAirport:
    case BaseMapElement::CivMilAirport:
    case BaseMapElement::Airfield:
      airport = new Airport(nameE->text(), abkE->text(), type,
              latitude[0], longitude[0], ((QString) elevE->text()).toUInt(),
              freqE->text(), vdfE->isChecked(), createRunway(),
              runwList->childCount());
      airport->writeElementAscii(fileName);
      mapContents->addElement(airport);
      break;
    case BaseMapElement::ClosedAirfield:
      airport = new Airport(nameE->text(), 0, type,
              latitude[0], longitude[0], 0, 0, 0);
      airport->writeElementAscii(fileName);
      mapContents->addElement(airport);
        break;
    case BaseMapElement::CivHeliport:
    case BaseMapElement::MilHeliport:
    case BaseMapElement::AmbHeliport:
      airport = new Airport(nameE->text(), abkE->text(), type,
              latitude[0], longitude[0], ((QString) elevE->text()).toUInt(),
              freqE->text());
      airport->writeElementAscii(fileName);
      mapContents->addElement(airport);
      break;
    case BaseMapElement::Glidersite:
      glider = new GliderSite(nameE->text(), latitude[0], longitude[0],
            ((QString) elevE->text()).toUInt(), freqE->text(),
            gliderT->currentItem(), createRunway(),
            runwList->childCount());
      glider->writeElementAscii(fileName);
      mapContents->addElement(glider);
      break;
    case BaseMapElement::UltraLight:
    case BaseMapElement::HangGlider:
    case BaseMapElement::Parachute:
    case BaseMapElement::Ballon:
      singleP = new SinglePoint(nameE->text(), latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::AddSitesList,singleP);
      break;
    case BaseMapElement::Outlanding:
      elevP = new ElevPoint(nameE->text(), type, latitude[0], longitude[0],
            ((QString) elevE->text()).toUInt());
      elevP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::OutList,elevP);
      break;
    case BaseMapElement::VOR:
    case BaseMapElement::VORDME:
    case BaseMapElement::VORTAC:
    case BaseMapElement::NDB:
      radioP = new RadioPoint(nameE->text(), type,
            latitude[0], longitude[0], freqE->text(), abkE->text());
      radioP->writeElementAscii(fileName);
      mapContents->addElement(radioP);
      break;
    case BaseMapElement::AirC:
    case BaseMapElement::AirCtemp:
    case BaseMapElement::AirD:
    case BaseMapElement::AirDtemp:
      airspace = new Airspace(nameE->text(), type, length,
            latitude, longitude, ((QString) uppE->text()).toUInt(),
            uppT->currentItem(), ((QString) lowerE->text()).toUInt(),
            lowerT->currentItem());
      airspace->writeElementAscii(fileName);
      mapContents->addElement(airspace);
      break;
    case BaseMapElement::ControlD:
      airspace = new Airspace(nameE->text(), type, length,
            latitude, longitude, ((QString) uppE->text()).toUInt(),
            uppT->currentItem());
      airspace->writeElementAscii(fileName);
      mapContents->addElement(airspace);
      break;
    case BaseMapElement::AirElow:
    case BaseMapElement::AirEhigh:
      airspace = new Airspace(nameE->text(), type, length,
            latitude, longitude);
      airspace->writeElementAscii(fileName);
      mapContents->addElement(airspace);
      break;
    case BaseMapElement::AirF:
    case BaseMapElement::Restricted:
    case BaseMapElement::Danger:
      airspace = new Airspace(nameE->text(), type, length,
            latitude, longitude, ((QString) uppE->text()).toUInt(),
            uppT->currentItem(), ((QString) lowerE->text()).toUInt(),
            lowerT->currentItem());
      airspace->writeElementAscii(fileName);
      mapContents->addElement(airspace);
      break;
    case BaseMapElement::LowFlight:
      airspace = new Airspace(nameE->text(), type, length,
            latitude, longitude);
      airspace->writeElementAscii(fileName);
      mapContents->addElement(airspace);
      break;
    case BaseMapElement::Obstacle:
    case BaseMapElement::LightObstacle:
    case BaseMapElement::ObstacleGroup:
    case BaseMapElement::LightObstacleGroup:
      elevP = new ElevPoint(0, type, latitude[0], longitude[0],
            ((QString) elevE->text()).toUInt());
      elevP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::ObstacleList,elevP);
      break;
    case BaseMapElement::CompPoint:
      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::ReportList,singleP);
      break;
    case BaseMapElement::HugeCity:
    case BaseMapElement::BigCity:
    case BaseMapElement::MidCity:
    case BaseMapElement::SmallCity:
      area = new AreaElement(nameE->text(), type, length, latitude, longitude);
      area->writeElementAscii(fileName);
      mapContents->addElement(MapContents::CityList,area);
      break;
    case BaseMapElement::Village:
      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::VillageList,singleP);
      break;
    case BaseMapElement::Oiltank:
    case BaseMapElement::Factory:
    case BaseMapElement::Castle:
    case BaseMapElement::Church:
    case BaseMapElement::Tower:
      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::LandmarkList,singleP);
      break;
    case BaseMapElement::Highway:
      lineE = new LineElement(nameE->text(), type, length, latitude, longitude);
      lineE->writeElementAscii(fileName);
      mapContents->addElement(MapContents::HighwayList,lineE);
      break;
    case BaseMapElement::HighwayEntry:
      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::HighwayEntryList,singleP);
      break;
    case BaseMapElement::MidRoad:
    case BaseMapElement::SmallRoad:
      lineE = new LineElement(nameE->text(), type, length, latitude, longitude);
      lineE->writeElementAscii(fileName);
      mapContents->addElement(MapContents::RoadList,lineE);
      break;
    case BaseMapElement::RoadBridge:
    case BaseMapElement::RoadTunnel:
      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      // bislang falsche Liste ...
      mapContents->addElement(MapContents::RoadList,singleP);
      break;
    case BaseMapElement::Railway:
      lineE = new LineElement(nameE->text(), type, length, latitude, longitude);
      lineE->writeElementAscii(fileName);
      mapContents->addElement(MapContents::RailList,lineE);
      break;
    case BaseMapElement::RailwayBridge:
    case BaseMapElement::Station:
      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
      singleP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::StationList,singleP);
      break;
    case BaseMapElement::AerialRailway:
      lineE = new LineElement(nameE->text(), type, length, latitude, longitude);
      lineE->writeElementAscii(fileName);
      mapContents->addElement(MapContents::RailList,lineE);
      break;
    case BaseMapElement::Coast:
    case BaseMapElement::BigLake:
    case BaseMapElement::MidLake:
    case BaseMapElement::SmallLake:
    case BaseMapElement::BigRiver:
    case BaseMapElement::MidRiver:
    case BaseMapElement::SmallRiver:
      lineE = new LineElement(nameE->text(), type, length, latitude, longitude);
      lineE->writeElementAscii(fileName);
      mapContents->addElement(MapContents::HydroList,lineE);
      break;
    case BaseMapElement::Dam:
    case BaseMapElement::Lock:
	    // In welche Liste ??????
//      singleP = new SinglePoint(nameE->text(), type, latitude[0], longitude[0]);
//      singleP->writeElementAscii(fileName);
//      mapContents->addElement(MapContents::List,singleP);
        break;
    case BaseMapElement::Spot:
    case BaseMapElement::Pass:
      elevP = new ElevPoint(0, type, latitude[0], longitude[0],
                         ((QString) elevE->text()).toUInt());
      elevP->writeElementAscii(fileName);
      mapContents->addElement(MapContents::ObstacleList,elevP);
        break;
    case BaseMapElement::Glacier:
      area = new AreaElement(nameE->text(), type, length, latitude, longitude);
      area->writeElementAscii(fileName);
      mapContents->addElement(MapContents::TopoList,area);
      break;
  }
}

void KFLogView::startInput()
{
  if(type == BaseMapElement::NotSelected) return;

  if(!pointList) {
    if(!checkInputFields()) return;
  }

  TabletDialog* tab = new TabletDialog(0, !pointList);

  unsigned int length = tab->getListLength();
  if(!length) {
    delete tab;
    return;
  }

  latitude = new long[length];
  longitude = new long[length];

  tab->getPointList(latitude, longitude);
  if(!pointList) {
    // If we have an international airport, return. The runways are missing.
    if(type == BaseMapElement::IntAirport) return;
    createMapElement(latitude, longitude, 1);
  } else {
    int count = 0;
    long* latNew = new long[length];
    long* lonNew = new long[length];
    long oldPosLat, oldPosLon;

    // writing the first point:
    latNew[0] = latitude[0];
    lonNew[0] = longitude[0];

    // Reduzierung der Punkte in eigener Funktion !
    for(unsigned int loop = 1; loop < (length - 1); loop++) {
      // writing the other point only, if the distance is big enough!
      oldPosLat = latitude[loop];
      oldPosLon = longitude[loop];
      long latDis = oldPosLat - latNew[count];
      long lonDis = oldPosLon - lonNew[count];
      //Abstand als Voreinstellung ???
      if((abs(latDis) < 2000) && (abs(lonDis) < 2000)) {
        float distance = sqrt((latDis * latDis) + (lonDis * lonDis));
        if(distance > 2000) {
          count++;
          latNew[count] = latitude[loop];
          lonNew[count] = longitude[loop];
        }
      } else {
        count++;
        latNew[count] = latitude[loop];
        lonNew[count] = longitude[loop];
      }
    }
    count++;
    // writing the last point:
    latNew[count] = latitude[length - 1];
    lonNew[count] = longitude[length - 1];
    // Shrinking the point-arrays
    latNew = (long*) realloc(latNew,
          ((count + 1) * sizeof(long)));
    lonNew = (long*) realloc(lonNew,
          ((count + 1) * sizeof(long)));
    createMapElement(latNew, lonNew, (count + 1));
  }

  tab->~TabletDialog();
}

void KFLogView::slotAddRunway()
{
  if(!strlen(runwE->text())) {
    QString errorMsg = i18n("No runway-length given!\nPlease \
insert the missing data first.");
    KMsgBox* errorBox = new KMsgBox(parent,i18n("Missing Data"),
        errorMsg,4,i18n("OK"));
    errorBox->exec();
    return;
  }

  if(!strlen(dirE->text())) {
    QString errorMsg = i18n("No direction of the runway given!\nPlease \
insert the missing data first.");
    KMsgBox* errorBox = new KMsgBox(parent,i18n("Missing Data"),
        errorMsg,4,i18n("OK"));
    errorBox->exec();
    return;
  }

  QListViewItem* runway = new QListViewItem(runwList);
  QString temp;
  temp = runwE->text();
  runway->setText(1,temp);
  temp = dirE->text();
  runway->setText(2,temp);
  temp = runwtE->currentItem();
  switch(runwtE->currentItem()) {
    case 0:
      runway->setText(3,i18n("concrete"));
      break;
    case 1:
      runway->setText(3,i18n("asphalt"));
      break;
    case 2:
      runway->setText(3,i18n("grass"));
      break;
  }

  runwE->setText("");
  dirE->setText("");

  if(type == BaseMapElement::IntAirport) {
    pointList = true;
    long* runwayPointList = new long[4];
    startRunwayInput(runwayPointList);

    if((runwayPointList[0] == 0) && (runwayPointList[1] == 0) &&
       (runwayPointList[2] == 0) && (runwayPointList[3] == 0)) {
      delete runway;
      return;
    }
    QString temp;
    temp.sprintf("%d", runwayPointList[0]);
    runway->setText(4,temp);
    temp.sprintf("%d", runwayPointList[1]);
    runway->setText(5,temp);
    temp.sprintf("%d", runwayPointList[2]);
    runway->setText(6,temp);
    temp.sprintf("%d", runwayPointList[3]);
    runway->setText(7,temp);
  }
}

void KFLogView::startRunwayInput(long* runwayPointList)
{
  QString noteMsg = i18n("Now You can digitize the two points for \
the runway.\nThe digitize-dialog will appear twice.");
  KMsgBox* noteBox = new KMsgBox(parent,i18n("Runway-Data"),
      noteMsg,4,i18n("OK"));
  noteBox->exec();

  latitude = new long[1];
  longitude = new long[1];

  latitude[0] = 0;
  longitude[0] = 0;

  TabletDialog* tab = new TabletDialog(parent,true);

  tab->getPointList(latitude, longitude);

  runwayPointList[0] = latitude[0];
  runwayPointList[1] = longitude[0];

  latitude[0] = 0;
  longitude[0] = 0;

  tab = new TabletDialog(parent,true);

  tab->getPointList(latitude, longitude);

  runwayPointList[2] = latitude[0];
  runwayPointList[3] = longitude[0];
}

void KFLogView::slotDelRunway()
{
  QListViewItem* current = runwList->currentItem();
  current->~QListViewItem();

  runwE->setText("");
  dirE->setText("");
}

KFLogView::KFLogView(QWidget* parent, MapContents* molist)
  : QWidget(parent)
{
  mapView = new QFrame(this);
  QHBoxLayout* mapViewLayout = new QHBoxLayout(this);
  mapContents = molist;

  QFont tt("Courier");
  QFont bold;
  bold.setBold(true);

  // create a frame for the map
  mapFrame = new QFrame(mapView);

  mapFrame->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  mapFrame->setLineWidth(1);
  mapFrame->setMidLineWidth(1);

  // create frames for the additional data
  QFrame* dataTopFrame = new QFrame(mapView);
  dataTopFrame->setMinimumWidth(100);
  dataTopFrame->setMaximumWidth(300);
  dataTopFrame->setLineWidth(0);
  QFrame* dataBotFrame = new QFrame(mapView);
//  dataBotFrame->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  dataBotFrame->setLineWidth(1);

  // adding the contents of the data frames
  QLabel* titleL = new QLabel(i18n("Flight-data"), dataTopFrame);
  titleL->setMinimumWidth(titleL->sizeHint().width());
  titleL->setMaximumWidth(titleL->sizeHint().width());

  QLabel* label1 = new QLabel(i18n("Date:"), dataTopFrame);
  QLabel* label2 = new QLabel(i18n("Pilot:"), dataTopFrame);
  QLabel* label3 = new QLabel(i18n("Aircraft:"), dataTopFrame);
  QLabel* label4 = new QLabel(i18n("Route:"), dataTopFrame);
  wpList = new QListView(dataTopFrame);
  wpList->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  wpList->addColumn(i18n("Nr"), 35);
  wpList->addColumn(i18n("Name"), 150);
  wpList->addColumn(i18n("Time"), 80);
  wpList->addColumn(i18n("Position"), 200);
  wpList->addColumn(i18n("Distance"), 75);
  wpList->setColumnAlignment(4,2);
  wpList->setAllColumnsShowFocus(true);
  wpList->setSorting(5);

  text1 = new QLabel(dataTopFrame);
  text1->setFont(tt);
  text1->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  text2 = new QLabel(dataTopFrame);
  text2->setFont(tt);
  text2->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  text3 = new QLabel(dataTopFrame);
  text3->setFont(tt);
  text3->setFrameStyle(QFrame::Panel | QFrame::Sunken);

  QGridLayout* topLayout = new QGridLayout(dataTopFrame,26,3,5,0);
  topLayout->addWidget(titleL,0,0);
  topLayout->addWidget(label1,2,0);
  topLayout->addWidget(label2,4,0);
  topLayout->addWidget(label3,6,0);
  topLayout->addWidget(label4,8,0);
  topLayout->addMultiCellWidget(label4,8,8,0,1);
  //
  topLayout->addWidget(text1,2,2);
  topLayout->addWidget(text2,4,2);
  topLayout->addWidget(text3,6,2);
  topLayout->addMultiCellWidget(wpList,10,10,0,2);

  topLayout->addColSpacing(0,titleL->sizeHint().width());
  topLayout->addColSpacing(1,5);
  topLayout->setColStretch(2,2);

  topLayout->addRowSpacing(0,25);
  topLayout->addRowSpacing(1,3);
  topLayout->addRowSpacing(2,25);
  topLayout->addRowSpacing(3,3);
  topLayout->addRowSpacing(4,25);
  topLayout->addRowSpacing(5,3);
  topLayout->addRowSpacing(6,25);
  topLayout->addRowSpacing(7,7);
  topLayout->addRowSpacing(8,25);
  topLayout->addRowSpacing(10,155);

  // creating the layout
  QGridLayout* mapLayout = new QGridLayout(this,3,3);
  mapLayout->addMultiCellWidget(mapFrame,0,2,0,0);

  mapLayout->addWidget(dataTopFrame,0,2);
  mapLayout->addWidget(dataBotFrame,2,2);

  mapLayout->setColStretch(0,3);
  mapLayout->setColStretch(1,0);
  mapLayout->setColStretch(2,1);
  mapLayout->addColSpacing(1,8);

  mapLayout->setRowStretch(0,2);
  mapLayout->setRowStretch(1,0);
  mapLayout->setRowStretch(2,1);
  mapLayout->addRowSpacing(1,5);

  mapLayout->activate();

  mapViewLayout->addWidget(mapView);
  mapViewLayout->activate();

  //////////////////////////////////////////////////////////////////////////
  //
  // The widgets for digitizing
  //
  KConfig *config = kapp->getConfig();
  config->setGroup("Appearance");
  int mowidth = config->readNumEntry("ColWidth_1", 210);

  tabView = new QFrame(this);
  QHBoxLayout* tabViewLayout = new QHBoxLayout(this);

  // creating the layout
  QGridLayout* tabLayout = new QGridLayout(this,16,16,5,0);
        
  // creating the widgets for the entry
  digitList = new QListView(tabView);
  digitList->setMinimumWidth(215);
  digitList->addColumn("",40);
  digitList->addColumn(i18n("Map-Object"), mowidth);
  digitList->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  digitList->setAllColumnsShowFocus(true);
  digitList->setRootIsDecorated(true);
  tabLayout->addMultiCellWidget(digitList,0,15,0,0);

  typLabel = new QLabel(tabView);
  typLabel->setText("Typ des Karten-Objektes");
  typLabel->setFont(bold);
  typLabel->setMinimumHeight(40);
  tabLayout->addMultiCellWidget(typLabel,0,0,2,8);

  QLabel* nameL = new QLabel(tabView);
  nameL->setText(i18n("Name:"));
  nameE = new QLineEdit(tabView);
  nameE->setFont(tt);
  abkE = new KRestrictedLine(tabView);
  abkE->setValidChars("ABCDEFGHIJKLMNOPQRSTUVWXYZ");
  abkE->setFont(tt);
  abkE->setMinimumWidth(50);

  KButton* pointB = new KButton(tabView);
  pointB->setPixmap(Icon(KApplication::kde_datadir() +
            "/kflog/pics/tablet.xpm"));
  pointB->setMinimumWidth(60);
  pointB->setMinimumHeight(60);

  QLabel* elevL = new QLabel(tabView);
  elevL->setText(i18n("Elev."));
  elevE = new KIntegerLine(tabView);
  elevE->setFont(tt);
  elevE->setMinimumWidth(50);
  elevE->setMaximumHeight(elevE->sizeHint().height() + 15);
  elevE->setMinimumHeight(elevE->sizeHint().height() + 5);
  QLabel* freqL = new QLabel(tabView);
  freqL->setText(i18n("Frequency:"));
  freqL->setAlignment(AlignRight | AlignVCenter);
  freqE = new KRestrictedLine(tabView);
  freqE->setValidChars("1234567890,.");
  freqE->setFont(tt);
  freqE->setMinimumWidth(80);
  freqE->setMinimumHeight(freqE->sizeHint().height() + 5);
  freqE->setMaximumHeight(freqE->sizeHint().height() + 15);
  vdfE = new QCheckBox(i18n("VDF"),tabView);
  vdfE->setMinimumWidth(50);
  tabLayout->addWidget(elevL,4,2);
  tabLayout->addWidget(elevE,4,4);
  tabLayout->addMultiCellWidget(pointB,4,6,14,14);

  QLabel* gliderL = new QLabel(tabView);
  gliderL->setText(i18n("Glidersite:"));
  gliderL->setAlignment(AlignRight | AlignVCenter);
  gliderT = new KCombo(tabView);
  gliderT->insertItem(i18n("aero tow"));
  gliderT->insertItem(i18n("winch launching"));
  gliderT->setMaximumHeight(gliderT->sizeHint().height() + 15);
  gliderT->setMinimumHeight(gliderT->sizeHint().height() + 5);
  tabLayout->addMultiCellWidget(gliderL,4,4,6,8);
  tabLayout->addMultiCellWidget(gliderT,4,4,10,12);
  tabLayout->addWidget(freqL,6,8);
  tabLayout->addWidget(freqE,6,10);
  tabLayout->addWidget(vdfE,6,12);

  nameE->setMaximumHeight(gliderT->sizeHint().height() + 15);
  nameE->setMinimumHeight(gliderT->sizeHint().height() + 5);
  tabLayout->addWidget(nameL,2,2);
  tabLayout->addMultiCellWidget(nameE,2,2,4,12);
  tabLayout->addWidget(abkE,2,14);

  QLabel* lowerL = new QLabel(i18n("Lower Limit:"),tabView);
  lowerE = new KIntegerLine(tabView);
  lowerT = new KCombo(tabView);
  lowerT->insertItem(i18n("MSL"));
  lowerT->insertItem(i18n("GND"));
  lowerT->insertItem(i18n("FL"));
  lowerT->setMinimumWidth(60);
  lowerT->setMaximumHeight(lowerT->sizeHint().height() + 15);
  lowerT->setMinimumHeight(lowerT->sizeHint().height() + 5);
  QLabel* uppL = new QLabel(tabView);
  uppL->setText(i18n("Upper Limit:"));
  uppE = new KIntegerLine(tabView);
  uppE->setFont(tt);
  uppE->setMinimumWidth(70);
  uppT = new KCombo(tabView);
  uppT->insertItem(i18n("MSL"));
  uppT->insertItem(i18n("GND"));
  uppT->insertItem(i18n("FL"));
  uppT->setMinimumWidth(60);
  uppT->setMaximumHeight(uppT->sizeHint().height() + 15);
  uppT->setMinimumHeight(uppT->sizeHint().height() + 5);
  tabLayout->addWidget(lowerL,6,2);
  tabLayout->addWidget(lowerE,6,4);
  tabLayout->addWidget(lowerT,6,6);
  tabLayout->addWidget(uppL,8,2);
  tabLayout->addWidget(uppE,8,4);
  tabLayout->addWidget(uppT,8,6);

  runwList = new QListView(tabView);
  runwList->setMinimumWidth(270);
  runwList->addColumn(i18n("Nr."),30);
  runwList->addColumn(i18n("Length"),80);
  runwList->addColumn(i18n("Direction"),100);
  runwList->addColumn(i18n("Surface"),100);
  runwList->addColumn(i18n(""),80);
  runwList->addColumn(i18n(""),80);
  runwList->addColumn(i18n(""),80);
  runwList->addColumn(i18n(""),80);
  runwList->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  runwList->setAllColumnsShowFocus(true);
  runwList->setRootIsDecorated(false);
  tabLayout->addMultiCellWidget(runwList,10,10,2,14);

  QLabel* runwL = new QLabel(tabView);
  runwL->setText(i18n("Length:"));
  runwE = new KIntegerLine(tabView);
  runwE->setFont(tt);
  runwE->setMinimumHeight(25);
  runwE->setMaximumHeight(35);
  QLabel* dirL = new QLabel(tabView);
  dirL->setText(i18n("Direction:"));
  dirL->setAlignment(AlignRight | AlignVCenter);
  dirL->setMinimumWidth(70);
  dirE = new KIntegerLine(tabView);
  dirE->setFont(tt);
  dirE->setMinimumWidth(50);
  dirE->setMaximumHeight(35);
  QLabel* runwtL = new QLabel(tabView);
  runwtL->setText(i18n("Surface:"));
  runwtL->setAlignment(AlignRight | AlignVCenter);
  runwtL->setMinimumWidth(60);
  runwtE = new QComboBox(tabView);
  runwtE->insertItem(i18n("concrete"));
  runwtE->insertItem(i18n("asphalt"));
  runwtE->insertItem(i18n("grass"));
  runwtE->setMaximumHeight(runwtE->sizeHint().height() + 15);
  runwtE->setMinimumHeight(runwtE->sizeHint().height() + 5);
  tabLayout->addWidget(runwL,12,2);
  tabLayout->addWidget(runwE,12,4);
  tabLayout->addWidget(dirL,12,6);
  tabLayout->addWidget(dirE,12,8);
  tabLayout->addWidget(runwtL,12,10);
  tabLayout->addMultiCellWidget(runwtE,12,12,12,14);

  addB = new QPushButton(tabView);
  addB->setText(i18n("add runway"));
  addB->setMinimumHeight(addB->sizeHint().height() + 5);
  addB->setMaximumHeight(addB->sizeHint().height() + 15);
  delB = new QPushButton(tabView);
  delB->setText(i18n("delete runway"));
  tabLayout->addMultiCellWidget(addB,14,14,4,6);
  tabLayout->addMultiCellWidget(delB,14,14,8,10);

  tabLayout->setColStretch(0,4);
  tabLayout->setColStretch(4,2);
  tabLayout->setColStretch(8,2);
  tabLayout->setColStretch(14,1);
  tabLayout->addColSpacing(1,20);
  tabLayout->addColSpacing(2,lowerL->sizeHint().width());
  tabLayout->addColSpacing(3,5);
  tabLayout->addColSpacing(5,10);
  tabLayout->addColSpacing(7,10);
  tabLayout->addColSpacing(9,10);
  tabLayout->addColSpacing(11,10);
  tabLayout->addColSpacing(13,10);
  tabLayout->addColSpacing(15,4);

  tabLayout->setRowStretch(10,3);
  tabLayout->addRowSpacing(0,20);
  tabLayout->addRowSpacing(1,5);
  tabLayout->addRowSpacing(3,5);
  tabLayout->addRowSpacing(5,5);
  tabLayout->addRowSpacing(7,5);
  tabLayout->addRowSpacing(9,25);
  tabLayout->addRowSpacing(11,10);
  tabLayout->addRowSpacing(13,5);

  tabLayout->activate();

  tabViewLayout->addWidget(tabView);
  tabViewLayout->activate();
        
  QListViewItem* topo = new QListViewItem(digitList);
  topo->setPixmap(0,Icon("mini/folder.xpm"));
  topo->setText(1,i18n("Topography"));
  ENTRY(topo,"Glaciers and ice caps");
  ENTRY(topo,"Mountain pass");
  ENTRY(topo,"Spot elevation");
  ENTRY(topo,"Obstaclegroup (lighted)");
  ENTRY(topo,"Obstaclegroup");
  ENTRY(topo,"Obstacle (lighted)");
  ENTRY(topo,"Obstacle");

  QListViewItem* waters = new QListViewItem(digitList);
  waters->setPixmap(0,Icon("mini/folder.xpm"));
  waters->setText(1,i18n("Hydrography"));
  ENTRY(waters,"Lock");
  ENTRY(waters,"Dam / Barrage");
  ENTRY(waters,"River / Canal (small)");
  ENTRY(waters,"River / Canal (medium)");
  ENTRY(waters,"River / Canal (big)");
  ENTRY(waters,"Lake (small)");
  ENTRY(waters,"Lake (medium)");
  ENTRY(waters,"Lake (big)");
  ENTRY(waters,"Shore line");

  QListViewItem* roads = new QListViewItem(digitList);
  roads->setPixmap(0,Icon("mini/folder.xpm"));
  roads->setText(1,i18n("Roads, Railways"));
  ENTRY(roads,"Aerial Railway");
  ENTRY(roads,"Railwaybridge");
  ENTRY(roads,"Station");
  ENTRY(roads,"Railway");
  ENTRY(roads,"Roadbridge");
  ENTRY(roads,"Highway entry");
  ENTRY(roads,"Road (small)");
  ENTRY(roads,"Road (medium)");
  ENTRY(roads,"Highway");

  QListViewItem* landscape = new QListViewItem(digitList);
  landscape->setPixmap(0,Icon("mini/folder.xpm"));
  landscape->setText(1,i18n("Landmarks"));
  ENTRY(landscape,"Tower");
  ENTRY(landscape,"Church / Monastery");
  ENTRY(landscape,"Ruin / Fort / Castle");
  ENTRY(landscape,"Factory");
  ENTRY(landscape,"Oiltank");

  QListViewItem* town = new QListViewItem(digitList);
  town->setPixmap(0,Icon("mini/folder.xpm"));
  town->setText(1,i18n("Built-up areas"));
  ENTRY(town,"Village");
  ENTRY(town,"City (20.000 - 50.000)");
  ENTRY(town,"City (50.000 - 100.000)");
  ENTRY(town,"City (100.000 - 500.000)");
  ENTRY(town,"City (> 500.000)");

  QListViewItem* airspace = new QListViewItem(digitList);
  airspace->setPixmap(0,Icon("mini/folder.xpm"));
  airspace->setText(1,i18n("Airspace Structure"));
  ENTRY(airspace,"250ft low flight area");
  ENTRY(airspace,"Danger Area");
  ENTRY(airspace,"Restricted Area");
  ENTRY(airspace,"Airspace F (HX)");
  ENTRY(airspace,"Airspace E (1000ft)");
  ENTRY(airspace,"Airspace E (1700ft)");
  ENTRY(airspace,"Control Zone Airspace D");
  ENTRY(airspace,"Airspace D (temp)");
  ENTRY(airspace,"Airspace D");
  ENTRY(airspace,"Airspace C (temp)");
  ENTRY(airspace,"Airspace C");

  QListViewItem* navig = new QListViewItem(digitList);
  navig->setPixmap(0,Icon("mini/folder.xpm"));
  navig->setText(1,i18n("Radio Navigation Facilities"));
  ENTRY(navig,"Compulsary reporting point");
  ENTRY(navig,"NDB");
  ENTRY(navig,"VOR-TACAN");
  ENTRY(navig,"VOR / DME");
  ENTRY(navig,"VOR");

  QListViewItem* airports = new QListViewItem(digitList);
  airports->setPixmap(0,Icon("mini/folder.xpm"));
  airports->setText(1,i18n("Aerodromes"));
  ENTRY(airports,"Outlandingfields");
  ENTRY(airports,"Free ballon site");
  ENTRY(airports,"Parachute jumping site");
  ENTRY(airports,"Hang glider site");
  ENTRY(airports,"Ultra light flying");
  ENTRY(airports,"Glider site");
  ENTRY(airports,"amb. Heliport");
  ENTRY(airports,"mil. Heliport");
  ENTRY(airports,"civ. Heliport");
  ENTRY(airports,"Airfield closed");
  ENTRY(airports,"Airfield");
  ENTRY(airports,"Airfield (civil/military)");
  ENTRY(airports,"Military Aerodrome");
  ENTRY(airports,"Airport, Airfield IFR");
  ENTRY(airports,"International Airport");

  this->connect(digitList, SIGNAL(doubleClicked(QListViewItem*)),
                SLOT(selectMapElement(QListViewItem*)));
  this->connect(digitList, SIGNAL(returnPressed(QListViewItem*)),
                SLOT(selectMapElement(QListViewItem*)));
  this->connect(digitList, SIGNAL(currentChanged(QListViewItem*)),
                SLOT(selectMapElement(QListViewItem*)));
  this->connect(pointB, SIGNAL(clicked()), SLOT(startInput()));
  this->connect(addB, SIGNAL(clicked()), SLOT(slotAddRunway()));
  this->connect(delB, SIGNAL(clicked()), SLOT(slotDelRunway()));

  tabView->hide();
}

KFLogView::~KFLogView()
{
  KConfig *config = kapp->getConfig();
  config->setGroup("Appearance");
  config->writeEntry("ColWidth_1", digitList->columnWidth(1));
  config->setGroup(0);
}
