/***********************************************************************
**
**   mapcalc.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  1999 by Heiner Lamprecht ( heiner@kijumfo.de )
**   begin        :  Mon Jul  26 1999
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <cmath>
#include <mapcalc.h>
#include <qregexp.h>

/**
  * The earth's radius used for calculation, given in Meter.
  * NOTE: We use the earth as a sphere, not as a spheroid!
  */
#define RADIUS 6370289.509

long degreeToNum(const char* degree)
{
  long deg = 0, min = 0, sec = 0;
  long result;
  int count = 0;
  char direction = 'N';

  QRegExp number("^[0-9]+$");
  QRegExp deg1("[.]");
  QRegExp deg2(",");
  QRegExp deg3("'");
  if(number.match(degree) != -1) {
    result = degree[0] - '0';
    for(unsigned int loop = 1; loop < strlen(degree); loop++) {
      result = 10 * result + (degree[loop] - '0');
    }
    return result;
  }
  switch(deg1.match(degree)) {
    case 1:
      deg = degree[0] - '0';
      degree += 2;
      break;
    case 2:
      deg = 10 * (degree[0] - '0') + (degree[1] - '0');
      degree += 3;
      break;
    case 3:
      deg = 100 * (degree[0] - '0') + 10 * (degree[1] - '0')
            + (degree[2] - '0');
      degree += 4;
      break;
    default:
      if(deg1.match(degree) > 3) return 0;    // << degree is not correct!
      switch(strlen(degree)) {
        case 1:
          deg = degree[0] - '0';
          break;
        case 2:
          deg = 10 * (degree[0] - '0') + (degree[1] -'0');
          break;
        case 3:
          deg = 100 * (degree[0] - '0') + 10 * (degree[1] - '0')
                + (degree[2] - '0');
          break;
        default:
          return 0;                           // << degree is not correct!
      }
  }
  if(deg2.match(degree) != -1) {
    // Minuten mit Nachkommastellen!
    switch(deg2.match(degree)) {
      case 1:
        min = degree[0] - '0';
        for(unsigned int loop = 2; loop < strlen(degree); loop++) {
          if((degree[loop] >= '0') && (degree[loop] <= '9')) {
            sec = 10 * sec + (degree[loop] - '0');
            count++;
          }
        }
        break;
      case 2:
        min = 10 * (degree[0] - '0') + (degree[1] -'0');
        for(unsigned int loop = 3; loop < strlen(degree); loop++) {
          if((degree[loop] >= '0') && (degree[loop] <= '9')) {
            sec = (degree[loop] - '0') + (sec / 10);
            count++;
          }
        }
        break;
      default:
        if((deg2.match(degree) > 2) ||
           (deg2.match(degree) == 0))
          return 0;    // << degree is not correct!
    }
  } else if(deg3.match(degree) != -1) {
    // es folgen "echte" Sekunden
    switch(deg3.match(degree)) {
      case 1:
        min = degree[0] - '0';
        for(unsigned int loop = 2; loop < strlen(degree); loop++) {
          if((degree[loop] >= '0') && (degree[loop] <= '9')) {
            sec = sec * 10 + (degree[loop] - '0');
            count++;
          }
        }
        break;
      case 2:
        min = 10 * (degree[0] - '0') + (degree[1] -'0');
        for(unsigned int loop = 3; loop < strlen(degree); loop++) {
          if((degree[loop] >= '0') && (degree[loop] <= '9')) {
            sec = sec * 10 + (degree[loop] - '0');
            count++;
          }
        }
        break;
      default:
        if((deg2.match(degree) > 2) ||
           (deg2.match(degree) == 0))
          return 0;    // << degree is not correct!
    }
  }
  result = (long) ((600000 * deg) +
               (10000 * (min + (sec * pow(10,-count)))));
  if( direction == 'S' || direction == 'W' ) {
    return -result;
  }
  return result;
}

double numToRad(long internal)
{
  return ( PI * internal ) / 108000000.0;
}

long radToNum(double radial)
{
  return (long) (radial * 108000000.0 / PI );
}

double calc_Y_Lambert(double latitude, double longitude)
{
  double y1;
/*
  extern double _v1;
  extern double _v2;
  extern double _var1;
  extern double _var2;

  y1 = 2*(sqrt(cos(_v1)*cos(_v1)+(sin(_v1)-sin(latitude))
               *(sin(_v1)+sin(_v2)))/(sin(_v1)+sin(_v2)))
    *cos((sin(_v1)+sin(_v2))*longitude/2);
*/

  extern double _v1;
  extern double _var1;
  extern double _var2;

  y1 = 2 * ( sqrt( _var1 + ( sin(_v1) - sin(latitude) ) *_var2 )
             / _var2 )
         * cos( _var2 * longitude / 2 );

  return y1;
}

double calc_X_Lambert(double latitude, double longitude)
{
  double x1;
/*
  extern double _v1;
  extern double _v2;

  x1 = 2*(sqrt(cos(_v1)*cos(_v1)+(sin(_v1)-sin(latitude))
               *(sin(_v1)+sin(_v2)))/(sin(_v1)+sin(_v2)) )
    *sin((sin(_v1)+sin(_v2))*longitude/2);
*/

  extern double _v1;
  extern double _var1;
  extern double _var2;

  x1 = 2 * ( sqrt( _var1 + ( sin(_v1) - sin(latitude) ) * _var2 )
            / _var2 )
         * sin( _var2 * longitude / 2 );

  return x1;
}

void map2Lambert(double x, double y, struct point* loc)
{
  extern double _v1;
  extern double _v2;
  double lat, lon;

  lat = -asin(
              ( -4.0 * pow(cos(_v1), 2.0) - 4.0 * pow(sin(_v1), 2.0)
                -4.0 * sin(_v1) * sin(_v2)
                + y * y * pow(sin(_v1), 2.0) + pow(sin(_v1), 2.0)* x * x
                + 2.0 * y * y * sin(_v1) * sin(_v2) + 2.0 * sin(_v1)
                * sin(_v2) * x * x + y * y * pow(sin(_v2), 2.0)
                + pow(sin(_v2), 2.0) * x * x
                ) /
              ( sin(_v1) + sin(_v2) ) / 4 );

  lon = 2.0 * atan( y / x ) / ( sin(_v1) + sin(_v2) );

  loc->latitude = radToNum(lat);
  loc->longitude = radToNum(lon);
}

long map2LambertLat(double x, double y)
{
  extern double _v1;
  extern double _v2;
  double lat;
  lat = -asin(
              ( -4.0 * pow(cos(_v1), 2.0) - 4.0 * pow(sin(_v1), 2.0)
                -4.0 * sin(_v1) * sin(_v2)
                + y * y * pow(sin(_v1), 2.0) + pow(sin(_v1), 2.0)* x * x
                + 2.0 * y * y * sin(_v1) * sin(_v2) + 2.0 * sin(_v1)
                * sin(_v2) * x * x + y * y * pow(sin(_v2), 2.0)
                + pow(sin(_v2), 2.0) * x * x
                ) /
              ( sin(_v1) + sin(_v2) ) / 4 );
  return radToNum(lat);
}

long map2LambertLon(double x, double y)
{
  extern double _v1;
  extern double _v2;
  double lon;

  lon = 2.0 * atan( y / x ) / ( sin(_v1) + sin(_v2) );
  return radToNum(lon);

}

double dist(double lat1, double lon1, double lat2, double lon2)
{
  double pi_180 = PI / 108000000.0;

  double dlat = lat1 - lat2;
  double dlon = lon1 - lon2;

  double dist = RADIUS * sqrt(
      ((pi_180) * dlat * (pi_180) * dlat)
      + ((pi_180) * cos(pi_180 * lat1)* dlon
        * (pi_180) * cos(pi_180 * lat1)* dlon) );

  return dist / 1000.0;
}

QString printPos(long coord, bool isLat = true)
{
  QString pos, posDeg, posMin, posSec;

  int degree = coord / 600000;
  int min = (coord - (degree * 600000)) / 10000;
  int sec = (coord - (degree * 600000) - (min * 10000));
  sec = (sec * 60) / 10000;

  if(min < 10) {
    posMin.sprintf("0%d'", min);
  } else {
    posMin.sprintf("%d'", min);
  }

  if(sec < 10) {
    posSec.sprintf("0%d\"", sec);
  } else {
    posSec.sprintf("%d\"", sec);
  }

  if(isLat) {
    if(coord < 0) {
      posDeg.sprintf("%d", -degree);
      pos = posDeg + posMin + posSec + " S";
    } else {
      posDeg.sprintf("%d", degree);
      pos = posDeg + posMin + posSec + " N";
    }
  } else {
    if(coord < 0) {
      posDeg.sprintf("%d", -degree);
      pos = posDeg + posMin + posSec + " W";
    } else {
      posDeg.sprintf("%d", degree);
      pos = posDeg + posMin + posSec + " E";
    }
  }

  return pos;
}

QString printTime(int time)
{
  QString hour, min, sec;

  int hh = time / 3600;
  int mm = (time - (hh * 3600)) / 60;
  int ss = time - (hh * 3600) - mm * 60;

  hour.sprintf("%d", hh);
  if(mm < 10) {
    min.sprintf("0%d", mm);
  } else {
    min.sprintf("%d", mm);
  }
  if(ss < 10) {
    sec.sprintf("0%d", ss);
  } else {
    sec.sprintf("%d", ss);
  }

  return (hour + ":" + min + ":" + sec);
}
