/***********************************************************************
**
**   mapfile.cpp
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  2000 by Heiner Lamprecht
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#include <iostream>
#include <stdlib.h>

#include <mapfile.h>

#include <airport.h>
#include <elevpoint.h>
#include <glidersite.h>
#include <lineelement.h>
#include <mapcalc.h>
#include <radiopoint.h>
#include <singlepoint.h>

#include <qdatastream.h>
#include <qfile.h>
#include <qstring.h>
#include <qtextstream.h>

void ascii2bin()
{
cerr << "ascii2bin()\n";
  MapContents* mContents = new MapContents();

  if(readAsciiFile("/home/heiner/kflog_sites.out", mContents)) {
//    writeBinaryFile("/home/heiner/C++/kflog/kflog/konvert_sites.dat",
//                    mContents);
  }

  delete mContents;
  cerr << "Konvertieren fertig!\n";
}

void bin2ascii()
{
cerr << "bin2ascii()\n";
  MapContents* mContents = new MapContents();
/*
  if(readBinaryFile("/home/heiner/kflog_orig.out", mContents)) {
    writeAsciiFile("/home/heiner/C++/kflog/kflog/konvert_test.dat",
                    mContents);
  }
*/
  delete mContents;
  cerr << "Konvertieren fertig!\n";
}

bool readAsciiFile(const char* fileName, MapContents* mContents)
{
  QFile file(fileName);
  QTextStream stream(&file);
  QString line;

  unsigned int ulimit = 0, llimit = 0,
               ulimitType = 0, llimitType = 0;

  if(!file.open(IO_ReadOnly)) {
    // Fehler-Handling bislang nicht ausreichend ...
    cerr << "KFLog: No Mapfile found!" << endl;
    return false;
  }

  cerr << "KFLog: parsing mapfile " << fileName << endl;

  bool isObject = false;
  int objectCount = 0;
  int l = 0;
  int* runwayDir;
  int* runwayLength;
  int* runwayType;
  int* runway;
  long at_lat = 0, at_lon = 0, lat_temp = 0, lon_temp = 0;
  long* latitude;
  long* longitude;
  unsigned int posLength;
  unsigned int runLength;

  QString alias = 0;
  unsigned int elev = 0;
  QString frequency = 0;
  QString name;
  unsigned int type = 0;
  bool vdf, winch;

  while (!stream.eof()) {
    line = stream.readLine().simplifyWhiteSpace();
    // if it is a comment, ignore it!
    if(line.mid(0,1) == "#") continue;
    if(line.mid(0,11) == "[NEW]") {
      // a new object starts here
      isObject = true;
      objectCount++;

      l = 0;
      name = 0;
      type = 0;
      ulimit = 0;
      vdf = false;
      frequency = 0;
      alias = 0;
      elev = 0;
      at_lat = 0;
      at_lon = 0;
      winch = false;
      lat_temp = 0;
      lon_temp = 0;
      posLength = 0;
      runLength = 0;
      latitude = new long[1];
      longitude = new long[1];
      runwayDir = new int[1];
      runwayLength = new int[1];
      runwayType = new int[1];
    } else if(isObject) {
      if(line.mid(0,4) == "TYPE") {
        type = line.mid(5,2).toUInt();
      } else if(line.mid(0,5) == "ALIAS") {
        alias = line.mid(5,100);
      } else if(line.mid(0,2) == "AT") {
        unsigned int loop;
        for(loop = 3; loop < strlen(line); loop++) {
          if(line.mid(loop, 1) == " ") break;
        }
        at_lat = degreeToNum(line.mid(3,(loop - 3)));
        at_lon = degreeToNum(line.mid((loop + 1),100));
      } else if(line.mid(0,5) == "LTYPE") {
        llimitType = line.mid(6,1).toUInt();
      } else if(line.mid(0,5) == "UTYPE") {
        ulimitType = line.mid(6,1).toUInt();
      } else if(line.mid(0,5) == "LOWER") {
        llimit = line.mid(6,1).toUInt();
      } else if(line.mid(0,5) == "UPPER") {
        ulimit = line.mid(6,1).toUInt();
      } else if(line.mid(0,4) == "ELEV") {
        elev = line.mid(5,100).toUInt();
      } else if(line.mid(0,9) == "FREQUENCY") {
        frequency = line.mid(10,100);
      } else if(line.mid(0,4) == "NAME") {
        name = line.mid(5,1000);
      } else if(line.mid(0,6) == "RUNWAY") {
        runLength++;
        runwayDir = (int*) realloc(runwayDir, (runLength * sizeof(int)));
        runwayLength = (int*) realloc(runwayLength, (runLength * sizeof(int)));
        runwayType = (int*) realloc(runwayType, (runLength * sizeof(int)));

        unsigned int loop;
        unsigned int loop2;
        for(loop = 0; loop < strlen(line); loop++) {
          if(line.mid(loop, 1) == " ") break;
        }
        runwayDir[runLength - 1] = line.mid(7,(loop - 7)).toInt();
        loop2 = ++loop;
        for(loop = loop; loop < strlen(line); loop++) {
          if(line.mid(loop, 1) == " ") break;
        }
        runwayLength[runLength - 1] = line.mid(loop2, (loop - loop2)).toInt();
        runwayType[runLength - 1] = line.mid(++loop,1).toInt();
        if(type == BaseMapElement::IntAirport) {
          // we need four more points !!!
          runway = new int[4];
          loop++;
          loop2 = ++loop;
          for(loop = loop; loop < strlen(line); loop++) {
            if(line.mid(loop, 1) == " ") break;
          }
          runway[0] = degreeToNum(line.mid(loop2, (loop - loop2)));
          loop++;
          loop2 = loop;
          for(loop = loop; loop < strlen(line); loop++) {
            if(line.mid(loop, 1) == " ") break;
          }
          runway[1] = degreeToNum(line.mid(loop2, (loop - loop2)));
          loop++;
          loop2 = loop;
          for(loop = loop; loop < strlen(line); loop++) {
            if(line.mid(loop, 1) == " ") break;
          }
          runway[2] = degreeToNum(line.mid(loop2, (loop - loop2)));
          loop2 = ++loop;
          for(loop = loop; loop < strlen(line); loop++) {
            if(line.mid(loop, 1) == " ") break;
          }
          runway[3] = degreeToNum(line.mid(loop2, (loop - loop2)));
        }
      } else if((line.mid(0,1) >= "0") && (line.mid(0,1) <= "9")) {
        posLength++;
        latitude = (long*) realloc(latitude, (posLength * sizeof(long)));
        longitude = (long*) realloc(longitude, (posLength * sizeof(long)));

        unsigned int loop;
        for(loop = 0; loop < strlen(line); loop++) {
          if(line.mid(loop, 1) == " ") break;
        }
        lat_temp = degreeToNum(line.left(loop));
        lon_temp = degreeToNum(line.mid((loop + 1),100));
        latitude[posLength - 1] = lat_temp;
        longitude[posLength - 1] = lon_temp;
      } else if(line.mid(0,3) == "VDF") {
        vdf = line.mid(4,1).toUInt();
      } else if(line.mid(0,5) == "WINCH") {
        winch = line.mid(6,1).toUInt();
      } else if(line.mid(0,5) == "[END]") {
        switch (type) {
          case BaseMapElement::IntAirport:
            break;
          case BaseMapElement::Airport:
          case BaseMapElement::MilAirport:
          case BaseMapElement::CivMilAirport:
          case BaseMapElement::Airfield:
            mContents->addElement(
                new Airport(name, alias, type, at_lat, at_lon, elev,
                          frequency, vdf));
            break;
          case BaseMapElement::ClosedAirfield:
            mContents->addElement(
                new Airport(name, 0, type, at_lat, at_lon, 0, 0, 0));
            break;
          case BaseMapElement::CivHeliport:
          case BaseMapElement::MilHeliport:
          case BaseMapElement::AmbHeliport:
            mContents->addElement(
                new Airport(name, alias, type, at_lat, at_lon, elev,
                              frequency));
            break;
          case BaseMapElement::Glidersite:
            mContents->addElement(
                new GliderSite(name, at_lat, at_lon, elev, frequency, winch));
            break;
          case BaseMapElement::UltraLight:
          case BaseMapElement::HangGlider:
          case BaseMapElement::Parachute:
          case BaseMapElement::Ballon:
            mContents->addElement(MapContents::AddSitesList,
                new SinglePoint(name, at_lat, at_lon));
            break;
          case BaseMapElement::Outlanding:
            mContents->addElement(MapContents::OutList,
                new ElevPoint(name, type, at_lat, at_lon, elev));
            break;
          case BaseMapElement::VOR:
          case BaseMapElement::VORDME:
          case BaseMapElement::VORTAC:
          case BaseMapElement::NDB:
            mContents->addElement(
                new RadioPoint(name, type, at_lat, at_lon, frequency, alias));
            break;
          case BaseMapElement::AirC:
          case BaseMapElement::AirCtemp:
          case BaseMapElement::AirD:
          case BaseMapElement::AirDtemp:
            mContents->addElement(new Airspace(name, type, posLength,
              latitude, longitude, ulimit, ulimitType, llimit, llimitType));
            break;
          case BaseMapElement::ControlD:
            mContents->addElement(new Airspace(name, type, posLength,
              latitude, longitude, ulimit, ulimitType));
            break;
          case BaseMapElement::AirElow:
          case BaseMapElement::AirEhigh:
            mContents->addElement(new Airspace(name, type, posLength,
              latitude, longitude));
            break;
          case BaseMapElement::AirF:
          case BaseMapElement::Restricted:
          case BaseMapElement::Danger:
            mContents->addElement(new Airspace(name, type, posLength,
              latitude, longitude, ulimit, ulimitType, llimit, llimitType));
            break;
          case BaseMapElement::LowFlight:
            mContents->addElement(new Airspace(name, type, posLength,
              latitude, longitude));
            break;
          case BaseMapElement::Obstacle:
          case BaseMapElement::LightObstacle:
          case BaseMapElement::ObstacleGroup:
          case BaseMapElement::LightObstacleGroup:
            mContents->addElement(MapContents::ObstacleList,
                new ElevPoint(0, type, at_lat, at_lon, elev));
            break;
          case BaseMapElement::CompPoint:
            mContents->addElement(MapContents::ReportList,
                new SinglePoint(name, type, at_lat, at_lon));
            break;
          case BaseMapElement::HugeCity:
          case BaseMapElement::BigCity:
          case BaseMapElement::MidCity:
          case BaseMapElement::SmallCity:
            mContents->addElement(MapContents::CityList,
               new AreaElement(name, type, posLength, latitude, longitude));
            break;
          case BaseMapElement::Village:
            mContents->addElement(MapContents::VillageList,
                new SinglePoint(name, type, at_lat, at_lon));
            break;
          case BaseMapElement::Oiltank:
          case BaseMapElement::Factory:
          case BaseMapElement::Castle:
          case BaseMapElement::Church:
          case BaseMapElement::Tower:
            mContents->addElement(MapContents::LandmarkList,
                new SinglePoint(name, type, at_lat, at_lon));
            break;
          case BaseMapElement::Highway:
            mContents->addElement(MapContents::HighwayList,
               new LineElement(name, type, posLength, latitude, longitude));
            break;
          case BaseMapElement::HighwayEntry:
            mContents->addElement(MapContents::HighwayEntryList,
                new SinglePoint(name, type, at_lat, at_lon));
            break;
          case BaseMapElement::MidRoad:
          case BaseMapElement::SmallRoad:
            mContents->addElement(MapContents::RoadList,
               new LineElement(name, type, posLength, latitude, longitude));
            break;
          case BaseMapElement::RoadBridge:
          case BaseMapElement::RoadTunnel:
            // Bislang falsche Liste (Eintrag wird ignoriert) !!!
            mContents->addElement(MapContents::RoadList,
               new SinglePoint(name, type, at_lat, at_lon));
            break;
          case BaseMapElement::Railway:
            mContents->addElement(MapContents::RailList,
               new LineElement(name, type, posLength, latitude, longitude));
            break;
          case BaseMapElement::RailwayBridge:
          case BaseMapElement::Station:
            mContents->addElement(MapContents::StationList,
               new SinglePoint(0, type, at_lat, at_lon));
            break;
          case BaseMapElement::AerialRailway:
            mContents->addElement(MapContents::RailList,
               new LineElement(name, type, posLength, latitude, longitude));
            break;
          case BaseMapElement::Coast:
          case BaseMapElement::BigLake:
          case BaseMapElement::MidLake:
          case BaseMapElement::SmallLake:
          case BaseMapElement::BigRiver:
          case BaseMapElement::MidRiver:
          case BaseMapElement::SmallRiver:
            mContents->addElement(MapContents::HydroList,
               new LineElement(name, type, posLength, latitude, longitude));
            break;
          case BaseMapElement::Dam:
          case BaseMapElement::Lock:
          	// In welche Liste ??????
/*
            mContents->addElement(MapContents::List,
               new SinglePoint(0, type, at_lat, at_lon));                   */
            break;
          case BaseMapElement::Spot:
          case BaseMapElement::Pass:
            mContents->addElement(MapContents::ObstacleList,
               new ElevPoint(0, type, at_lat, at_lon, elev));
            break;
          case BaseMapElement::Glacier:
            mContents->addElement(MapContents::TopoList,
               new AreaElement(name, type, posLength, latitude, longitude));
            break;
        }
        isObject = false;
      } else {
        cout << "Unbekannt: " << line << endl;
      }
    }
  }

  file.close();

  cerr << "KFLog: " << objectCount << " mapobjects found" << endl;
  return true;
}

bool readBinaryFile(const char* fileName, MapContents* mContents)
{
  QFile eingabe(fileName);
  if(!eingabe.open(IO_ReadOnly)) {
    cerr << "KFLog: Can not open mapfile " << fileName << endl;
    return false;
  }

  QDataStream in(&eingabe);

  Q_UINT8 typeIn, llimitType, ulimitType, vdf, winch,
          rwNum, rwdir, rwsur, isW;
  Q_UINT16 elev, llimit, ulimit, length, rwlength;
  Q_UINT32 sumQ;

  char* name;
  char* header;
  char* frequency;

  QString alias;

  Q_INT32 at_lat, at_lon, lat_temp, lon_temp, latA, latB, lonA, lonB;

  unsigned int type = BaseMapElement::NotSelected,
             countObject = 0, sumInt = 0;

  long* latList;
  long* lonList;
  struct runway* rwData;
  struct intrunway * irwData;

  in >> header;
  in >> sumQ;
  sumInt = sumQ;

  while(!in.eof()) {
    name = 0;
    in >> typeIn;
    type = typeIn;
    length = 0;
    countObject++;
    winch = 0;

    switch (type) {
      case BaseMapElement::IntAirport:
        in >> name;
        in >> alias;
        in >> elev;
        in >> frequency;
        in >> vdf;
        in >> rwNum;
        irwData = new intrunway[rwNum];
        for(unsigned int i = 0; i < rwNum; i++) {
          in >> rwdir;
          in >> rwlength;
          in >> rwsur;
          in >> latA;
          in >> lonA;
          in >> latB;
          in >> lonB;

          irwData[i].direction = rwdir;
          irwData[i].length = rwlength;
          irwData[i].surface = rwsur;
          irwData[i].latitudeA = latA;
          irwData[i].longitudeA = lonA;
          irwData[i].latitudeB = latB;
          irwData[i].longitudeB = lonB;
        }
        break;
      case BaseMapElement::Airport:
      case BaseMapElement::MilAirport:
      case BaseMapElement::CivMilAirport:
      case BaseMapElement::Airfield:
        in >> name;
        in >> alias;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        in >> elev;
        in >> frequency;
        in >> vdf;
        in >> rwNum;
        rwData = new runway[rwNum];
        for(unsigned int i = 0; i < rwNum; i++) {
          in >> rwdir;
          in >> rwlength;
          in >> rwsur;

          rwData[i].direction = rwdir;
          rwData[i].length = rwlength;
          rwData[i].surface = rwsur;
        }
        mContents->addElement(
            new Airport(name, alias, type, at_lat, at_lon, elev,
                          frequency, vdf));
        break;
      case BaseMapElement::ClosedAirfield:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(
            new Airport(name, 0, type, at_lat, at_lon, 0, 0, 0));
        break;
      case BaseMapElement::CivHeliport:
      case BaseMapElement::MilHeliport:
      case BaseMapElement::AmbHeliport:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        in >> elev;
        in >> frequency;
        mContents->addElement(
            new Airport(name, alias, type, at_lat, at_lon, elev,
                          frequency));
        break;
      case BaseMapElement::Glidersite:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> elev;
        in >> frequency;
        in >> winch;
        in >> isW;
        in >> rwNum;
        rwData = new runway[rwNum];
        for(unsigned int i = 0; i < rwNum; i++) {
          in >> rwdir;
          in >> rwlength;
          in >> rwsur;

          rwData[i].direction = rwdir;
          rwData[i].length = rwlength;
          rwData[i].surface = rwsur;
        }
        mContents->addElement(
            new GliderSite(name, at_lat, at_lon, elev, frequency, winch));
        break;
      case BaseMapElement::UltraLight:
      case BaseMapElement::HangGlider:
      case BaseMapElement::Parachute:
      case BaseMapElement::Ballon:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(MapContents::AddSitesList,
            new SinglePoint(name, at_lat, at_lon));
        break;
      case BaseMapElement::Outlanding:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        in >> elev;
        mContents->addElement(MapContents::OutList,
            new ElevPoint(name, type, at_lat, at_lon, elev));
        break;
      case BaseMapElement::VOR:
      case BaseMapElement::VORDME:
      case BaseMapElement::VORTAC:
      case BaseMapElement::NDB:
        in >> name;
        in >> alias;
        in >> at_lat;
        in >> at_lon;
        in >> frequency;
        mContents->addElement(
            new RadioPoint(name, type, at_lat, at_lon, frequency, alias));
        break;
      case BaseMapElement::AirC:
      case BaseMapElement::AirCtemp:
      case BaseMapElement::AirD:
      case BaseMapElement::AirDtemp:
        in >> name;
        in >> llimit;
        in >> llimitType;
        in >> ulimit;
        in >> ulimitType;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(new Airspace(name, type, length, latList, lonList,
                          ulimit, ulimitType, llimit, llimitType));
        break;
      case BaseMapElement::ControlD:
        in >> name;
        in >> ulimit;
        in >> ulimitType;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(new Airspace(name, type, length, latList, lonList,
              ulimit, ulimitType));
        break;
      case BaseMapElement::AirElow:
      case BaseMapElement::AirEhigh:
        in >> name;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(new Airspace(name, type, length,
                          latList, lonList));
        break;
      case BaseMapElement::AirF:
      case BaseMapElement::Restricted:
      case BaseMapElement::Danger:
        in >> name;
        in >> llimit;
        in >> llimitType;
        in >> ulimit;
        in >> ulimitType;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(new Airspace(name, type, length, latList, lonList,
                          ulimit, ulimitType, llimit, llimitType));
        break;
      case BaseMapElement::LowFlight:
        in >> name;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(new Airspace(name, type, length,
                          latList, lonList));
        break;
      case BaseMapElement::Obstacle:
      case BaseMapElement::LightObstacle:
      case BaseMapElement::ObstacleGroup:
      case BaseMapElement::LightObstacleGroup:
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        in >> elev;
        mContents->addElement(MapContents::ObstacleList,
            new ElevPoint(0, type, at_lat, at_lon, elev));
        break;
      case BaseMapElement::CompPoint:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(MapContents::ReportList,
            new SinglePoint(name, type, at_lat, at_lon));
        break;
      case BaseMapElement::HugeCity:
      case BaseMapElement::BigCity:
      case BaseMapElement::MidCity:
      case BaseMapElement::SmallCity:
        in >> name;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::CityList,
            new AreaElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::Village:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(MapContents::VillageList,
            new SinglePoint(name, type, at_lat, at_lon));
        break;
      case BaseMapElement::Oiltank:
      case BaseMapElement::Factory:
      case BaseMapElement::Castle:
      case BaseMapElement::Church:
      case BaseMapElement::Tower:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(MapContents::LandmarkList,
            new SinglePoint(name, type, at_lat, at_lon));
        break;
      case BaseMapElement::Highway:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::HighwayList,
            new LineElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::HighwayEntry:
        in >> name;
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(MapContents::HighwayEntryList,
            new SinglePoint(name, type, at_lat, at_lon));
        break;
      case BaseMapElement::MidRoad:
      case BaseMapElement::SmallRoad:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::RoadList,
            new LineElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::RoadBridge:
      case BaseMapElement::RoadTunnel:
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        // Bislang falsche Liste (Eintrag wird ignoriert) !!!
        mContents->addElement(MapContents::RoadList,
            new SinglePoint(0, type, at_lat, at_lon));
        break;
      case BaseMapElement::Railway:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::RailList,
            new LineElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::RailwayBridge:
      case BaseMapElement::Station:
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        mContents->addElement(MapContents::StationList,
            new SinglePoint(0, type, at_lat, at_lon));
        break;
      case BaseMapElement::AerialRailway:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::RailList,
            new LineElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::Coast:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::HydroList,
            new LineElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::BigLake:
      case BaseMapElement::MidLake:
      case BaseMapElement::SmallLake:
        in >> name;
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::HydroList,
            new LineElement(name, type, length, latList, lonList));
        break;
      case BaseMapElement::BigRiver:
      case BaseMapElement::MidRiver:
      case BaseMapElement::SmallRiver:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::HydroList,
            new LineElement(0, type, length, latList, lonList));
        break;
      case BaseMapElement::Dam:
      case BaseMapElement::Lock:
        in >> at_lat;
        in >> at_lon;
        in >> isW;
	// In welche Liste ??????
	/*
        mContents->addElement(MapContents::List,
            new SinglePoint(0, type, at_lat, at_lon));           	         */
        break;
      case BaseMapElement::Spot:
      case BaseMapElement::Pass:
        in >> at_lat;
        in >> at_lon;
        in >> isW;
        in >> elev;
        mContents->addElement(MapContents::ObstacleList,
            new ElevPoint(0, type, at_lat, at_lon, elev));
        break;
      case BaseMapElement::Glacier:
        in >> length;
        latList = new long[length];
        lonList = new long[length];
        for(unsigned int i = 0; i < length; i++) {
          in >> lat_temp;
          in >> lon_temp;
          latList[i] = lat_temp;
          lonList[i] = lon_temp;
        }
        mContents->addElement(MapContents::TopoList,
            new AreaElement(name, type, length, latList, lonList));
        break;
    }
  }
  eingabe.close();

  return true;
}

bool writeAsciiFile(const char* fileName, MapContents* mContents)
{
//  QFile mapFile(fileName);
  QFile mapFile("/tmp/kflog.out");
  if(!mapFile.open(IO_ReadWrite)) {
    return false;
  }
  mapFile.at(mapFile.size());
  QTextStream mapOut(&mapFile);

  if(!mapFile.size()) {
    cerr << "KFLog: MapFile " << fileName << " is empty. Creating it" << endl;
    mapOut << "#KFLog-Mapfile Version: 0.3 (c) 2000 The KFLog-Team\n";
  }

  for(int i = 1; i < 20; i++) {
    for(unsigned int loop = 0; loop < mContents->getListLength(i); loop++) {
      mContents->getElement(i, loop)->writeElementAscii(fileName);
    }
  }
  mapFile.close();

  return true;
}

bool writeBinaryFile(const char* fileName, MapContents* mContents)
{
  QFile outputFile(fileName);
  if(!outputFile.open(IO_WriteOnly)) {
    return false;
  }
  QDataStream out(&outputFile);

  // First, we will count all objects and write the sum into the file
  unsigned int sum = 0;
  for(unsigned int i = 1; i < 20; i++) {
    sum += mContents->getListLength(i);
  }

  // Header used for identifying the format-version.
  out << "#KFLog-Mapfile Version: 0.3 (c) 2000 The KFLog-Team";
  out << (Q_UINT32) sum;
  outputFile.close();

  for(int i = 1; i < 20; i++) {
    for(unsigned int loop = 0; loop < mContents->getListLength(i); loop++) {
      mContents->getElement(i, loop)->writeElementBin(fileName);
    }
  }
  return true;
}
