/***********************************************************************
**
**   tablet.h
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  1999 by Heiner Lamprecht ( heiner@kijumfo.de )
**   begin        :  Fri Jul  9 1999
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef TABLET_H
#define TABLET_H

#include <wp.h>

#include <krestrictedline.h>
#include <qdialog.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qwidget.h>

#include <X11/Xlib.h>
#include <X11/extensions/XInput.h>
#include <X11/Xutil.h>

#define INVALID_EVENT_TYPE -1

/**
  * This class provides basic functions for accessing a tablet.
  * The class is based on the program "xinput" by Frederic Lepied.
  * Use this class on Your own risk! It is a realy dirty hack, but it
  * works fine with my Genius-HighSketch. I can't say, if it will work
  * with any other hardware.
  *
  * The error handling is not implemented yet. So if there is any Problem,
  * it is hard to determine, what will happen ...
  *
  * @author Heiner Lamprecht (heiner@kijumfo.de)
  * @version 0.1
  */
class Tablet : public QWidget
{
 public:
  /**
    * Creates a tablet-object and enables the input from the tablet.
    */
  Tablet(QWidget* parent);
  /**
    * Destroys the tablet. (Okay, only the "Tablet"-object,
    * not Your hardware, i hope.)
    */
  ~Tablet();
  /**
    * Returns true, if there is a tablet present.
    * NOTE: The error-handling is NOT correctly implemented yet. If there
    * is any problem with the xinput-modul, the reaction of the program
    * is not known ;-(
    */
  bool isPresent();
  /**
    * Reads the points given with the tablet. If "onlyOne" is true,
    * the function returns after reading only one point, otherwise the
    * function will return, if the 4. button is pressed on the tablet.
    */
  unsigned int getPoints(bool onlyOne, const char* name);
  /**
    * Fills the points read from the tablet into the two given arrays.
    */
  void getList(long* lat, long* lon);

 private:
  /** The number of points read from the tablet. */
  unsigned int length;
  /** The array used to store the x-coordinates. */
  long* latitude;
  /** The array used to store the y-coordinates. */
  long* longitude;
  /** Returns the motion type. */
  int motionType;
  /** The id of the pressed button. */
  int buttonPressType;
  /** The id of the released button. */
  int buttonReleaseType;
  /** The type of the key-press event. */
  int keyPressType;
  /** The type of the key-release event. */
  int keyReleaseType;
  /**
    * Returns an event, if the puck of the tablet enters proximity.
    */
  int proximityInType;
  /**
    * Returns an event, if the puck of the tablet leaves proximity.
    */
  int proximityOutType;
  /** Contains a pointer to the X-Display. */
  Display* display;
  /** Contains a pointer to the parent widget. */
  QWidget* parent;
  /**
    * Returns the number of points read from the tablet.
    */
  int registerEvents(XDeviceInfo* info, bool handleProximity);
  /**
    * Reads the events given from the tablet. if "onlyOne", the function
    * will return after reading only one point, otherwise it will return
    * after the 4. button is pressed. Returns the number of points read.
    */
  unsigned int readEvents(bool onlyOne);
  /**
    * Pointer to the XDeviceInfo about the tablet.
    */
  XDeviceInfo* findInfo(const char* name, bool only_ext);
};

/**
  * This class provides a dialog-window for digitizing. With the window,
  * You can calibrate Your tablet and/or start digitizing the map.
  * @author Heiner Lamprecht (heiner@kijumfo.de)
  * @version 0.2
  */
class TabletDialog : public QWidget
{
  Q_OBJECT

 public:
  /**
    * Creates a TabletDialog-Window.
    */
  TabletDialog(QWidget* parent, bool onlyone);
  /**
    * Destroys the the Tablet-object created (@see #tab)
    * and then destroys the TabletDialog-Window.
    */
  ~TabletDialog();
  /**
    * Stores the given arrays "latitude" and "longitude" into the
    * private arrays "latList" and "lonList".
    * @see #latList
    * @see #lonList
    */
  void getPointList(long* latitude, long* longitude);
  /**
    * Returns the length of the arrays "latList" and "lonList".
    * @see #latList
    * @see #lonList
    */
  int getListLength();

 public slots:
  /**
    * Starts digitizing and stores the points read from "tab" into the two
    * arrays "latList" and "lonList".
    * @see #tab
    * @see #latList
    * @see #lonList
    */
  void startDigitizing();
  /**
    * Closes the dialog-window.
    */
  void stopDigitizing();
  /**
    * Enables calibrating the tablet. The function sets the scale, the
    * two vectors vectorMapX and vectorMapY and the angle.
    * @see #vectorMapX
    * @see #vectorMapY
    * @see #scale
    * @see #angle
    */
  void calibrateTablet();

 private:
  /**
    * Returns true, if calibrating is done.
    * @see #calibrateTablet
    */
  bool done;
  /**
    * Used to read only one point from the tablet. With this, it is very
    * simple to digitize a single point, because the digitizing-function
    * will return immediately after reading the first point. if "onlyOne"
    * is false, digitizing will occure until the 4. button is pressed on
    * the puck.
    * @see Tablet#getPoints
    */
  bool onlyOne;
  /**
    * The scale used to transform the points read from the tablet into
    * geographic coordinates.
    * @see #startDigitizing
    */
  double scale;
  /**
    * The angle between the map on the tablet an the tablet itself.
    * Used to transform the points read from the tablet into
    * Cartesian-coordinates.
    * @see #startDigitizing
    */
  double angle;
  /**
    * The name of the tablet. It must be the same as given in
    * "/etc/XF86Config"
    */
  const char* tabName;
  /**
    * Pointer to the dialog-window.
    */
  QDialog* dW;
  /**
    * Pointer to the layout-object used in the dialog-window.
    * @see#dW
    */
  QGridLayout* dWLayout;
  QPushButton* cal;
  QLabel* dWL1;
  QLabel* calL;
  QLabel* point1L;
  KRestrictedLine* point1E1;
  KRestrictedLine* point1E2;
  QPushButton* point1B;
  QLabel* point2L;
  KRestrictedLine* point2E1;
  KRestrictedLine* point2E2;
  QPushButton* point2B;
  QLabel* point3L;
  KRestrictedLine* point3E1;
  KRestrictedLine* point3E2;
  QPushButton* point3B;
  /**
    * Pointer to the Tablet-object used for accessing the tablet.
    * @see Tablet
    */
  Tablet* tab;
  /**
    * The first digitized point used for calibrating.
    * @see #startDigitizing
    */
  struct point point1D;
  /**
    * The second digitized point used for calibrating.
    * @see #startDigitizing
    */
  struct point point2D;
  /**
    * The third digitized point used for calibrating.
    * @see #startDigitizing
    */
  struct point point3D;
  /**
    * vectorMapX and vectorMapY are used to calibrate the tablet.
    */
  double vectorMapX;
  double vectorMapY;
  /**
    * The coordinates for the object, given as an array in the
    * internal format. The size of the array is given during
    * construction of the object.
    *
    * This one is for the latitude.
    */
  long* latList;
  /**
    * And this for the longitude.
    */
  long* lonList;
  /**
    * The length of the two arrays "latList" and "lonList".
    * @see #latList
    * @see #lonList
    */
  int listLength;

 private slots:
  /**
    * Starts digitizing the first point for calibrating the tablet.
    * After reading one point, the function stores the point in point1D.
    * @see #startDigitizing
    * @see #point1D
    */
  void inputPoint1();
  /**
    * Starts digitizing the second point for calibrating the tablet.
    * After reading one point, the function stores the point in point2D.
    * @see #startDigitizing
    * @see #point2D
    */
  void inputPoint2();
  /**
    * Starts digitizing the third point for calibrating the tablet.
    * After reading one point, the function stores the point in point3D.
    * @see #startDigitizing
    * @see #point3D
    */
  void inputPoint3();
};

#endif
