#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <termio.h>
#include <sys/termios.h>

#include "gps_parse.h"

#include <kgps.h>
#include <kgpswidget.h>
#include <kgpswidget.moc>

#include <kconfig.h>
#include <kmsgbox.h>

#include <qlayout.h>
#include <qdialog.h>
#include <qpushbutton.h>
#include <qbuttongroup.h>
#include <qsize.h>

#include <xpm/background2.xpm>
#include <xpm/toolbox.xpm>

#define WHITE_COLOR	QColor(255,255,255)

#define BUFFERSIZE	4096
static char gps_buffer[BUFFERSIZE];

static int port = -1;
static struct termio ttyset;
static char* toolboxstr = "Properties";
static int MIN_WIDTH = 240;
static int MAX_WIDTH = 240;

const char* chWarning = "Warning!";
const char* chPortError = "Can't open Port!";
const char* chAbort = "Abort";

extern KConfig* config;

KGPSWidget::KGPSWidget(QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	
	KGPS* pApp = (KGPS*) parent;

	QBoxLayout* layout = new QVBoxLayout( this);
	QBoxLayout* hbox = new QHBoxLayout();
	
	layout->addLayout(hbox);
	
	sky = new KSky(this);
	bar = new KBar(this);
	
	hbox->addWidget(bar, 0, AlignLeft);
	hbox->addWidget(sky, 0, AlignRight);
	
	bar->setBackgroundPixmap(QPixmap((const char **)background2_xpm));
	bar->setMinimumWidth(MIN_WIDTH);	bar->setMaximumWidth(MAX_WIDTH);
	
	sky->setBackgroundColor(WHITE_COLOR);


	toolbar = new KToolBar(pApp);
	pApp->addToolBar(toolbar);

	toolbutton = new QToolButton(toolbar);
	toolbutton->setPixmap(QPixmap((const char **)toolbox_xpm));
	toolbutton->setTextLabel(toolboxstr);
	toolbar->insertWidget(25,25, toolbutton);

	connect(toolbutton, SIGNAL(clicked()), this, SLOT(properties()));
	open();
}

KGPSWidget::~KGPSWidget()
{
	config->sync();
}

void KGPSWidget::open()
{
	QString receiver = config->readEntry("GPS Receiver", "Garmin");
	QString initstr = QString("\r\n") + config->readEntry(receiver, "$PGRMO,,2\r\n$PGRMO,GPRMC,1\r\n$PGRMO,GPGSV,1\r\n$PGRMO,GPGSA,1")
							+ QString("\r\n");

	int portnr = config->readNumEntry("Port", 1);
	int baudval = config->readNumEntry("Baudrate", 4800);
	baudval == 4800 ? baudval = B4800 : baudval = B9600;
	/*****************************************************/    	

	if ((port = open_serialport(portnr, baudval)) < 0)
	{
		KMsgBox::message(this,  chWarning, chPortError,
				KMsgBox::EXCLAMATION, chAbort);
	}
	else
	{
		int r;
		if((r = write(port, (const char*)initstr, initstr.length())) < 0)	printf("write 3\n");
		gps_init_parser();
		startTimer(500);
	}
	/*****************************************************/    	
}

int KGPSWidget::open_serialport(int port, int baudval)
{
	char driver[50];
	int p;
	
	sprintf((char*)&driver, "/dev/ttyS%d", port);
	p = ::open((char*)&driver, O_RDWR | O_NDELAY);/*O_NONBLOCK);*/
	if (p > -1 && isatty(p))
	{
        if (ioctl(p, TCGETA, &ttyset) < 0)	printf("ioctl 1\n");

		ttyset.c_cflag = CBAUD & baudval;
		ttyset.c_cflag &= ~(PARENB | CRTSCTS);
		ttyset.c_cflag |= (CSIZE & CS8) | CREAD | CLOCAL;
		ttyset.c_iflag = ttyset.c_oflag = ttyset.c_lflag = (ushort) 0;
		ttyset.c_oflag = (ONLRET);

		if (ioctl(p, TCSETAF, &ttyset) < 0)	printf("ioctl 2\n");
	}
	return p;
}

int KGPSWidget::close_serialport(int port)
{
	if (port > -1)
	{
		if (isatty(port))
		{
			ttyset.c_cflag = CBAUD & B0;
			ioctl(port, TCSETAF, &ttyset);
		}
		::close(port);
	}
	return -1;
}

void KGPSWidget::close()
{
	killTimers();
	port = close_serialport(port);
}

void KGPSWidget::properties()
{
	KPropertiesDlg dlg(this, "Properties");
	dlg.exec();
}

extern RMCData rmcData;
extern GSVData gsvData;

void KGPSWidget::timerEvent(QTimerEvent*)
{
	int count;
	count = read(port, &gps_buffer[0], BUFFERSIZE);
	if(count > 0)
	{
		gps_parse(&gps_buffer[0], count);
	}
	if(rmcData.newdata)
	{
		sky->updateDisplay();
		bar->updateData();
		gsvData.newdata = 0;
	}
}

/***********************************************************************/
/*                                                                     */
/*                    KPropertiesDlg                                   */
/*                                                                     */
/***********************************************************************/

#define KPROPERTIESDLG_SIZE					QSize(400,250)
#define KPROPERTIESDLG_OKBTN_RECT			QRect(175,215,50,30)

#define KPROPERTIESDLG_BAUDRATEGROUP_RECT	QRect(10,10, 185, 90)
#define KPROPERTIESDLG_GPSGROUP_RECT		QRect(10,110, 185, 90)
#define KPROPERTIESDLG_COMGROUP_RECT		QRect(205,10, 185, 190)

#define KPROPERTIESDLG_BAUD4800BTN_RECT		QRect(20,25,140,25)
#define KPROPERTIESDLG_BAUD9600BTN_RECT		QRect(20,55,140,25)
#define KPROPERTIESDLG_GARMINBTN_RECT		QRect(20,25,140,25)
#define KPROPERTIESDLG_ROCKWELLBTN_RECT		QRect(20,55,140,25)
#define KPROPERTIESDLG_COM1BTN_RECT			QRect(40,35,140,25)
#define KPROPERTIESDLG_COM2BTN_RECT			QRect(40,70,140,25)
#define KPROPERTIESDLG_COM3BTN_RECT			QRect(40,105,140,25)
#define KPROPERTIESDLG_COM4BTN_RECT			QRect(40,140,140,25)

static char* changegarminbaudrate_str = "\r\n$PGRMC,,,,,,,,,,%d,,\r\n";

KPropertiesDlg::KPropertiesDlg(QWidget *parent, const char *name)
	: QDialog(parent, name, TRUE)
{
	QString receiver;
	int port, baudval;
	
	setMinimumSize(KPROPERTIESDLG_SIZE);	setMaximumSize(KPROPERTIESDLG_SIZE);

	QPushButton* okbutton = new QPushButton("Ok", this);
	okbutton->setGeometry(KPROPERTIESDLG_OKBTN_RECT);
	connect(okbutton, SIGNAL(clicked()), SLOT(accept()));

	/*********** groups ******************************************/
	QButtonGroup* baudrategroup = new QButtonGroup("Baudrate", this);
	baudrategroup->setGeometry(KPROPERTIESDLG_BAUDRATEGROUP_RECT);
	
	QButtonGroup* gpsgroup = new QButtonGroup("GPS Receiver", this);
	gpsgroup->setGeometry(KPROPERTIESDLG_GPSGROUP_RECT);
	
	QButtonGroup* comgroup = new QButtonGroup("COM Port", this);
	comgroup->setGeometry(KPROPERTIESDLG_COMGROUP_RECT);

	/*********** buttons ******************************************/	
	baud4800button = new QRadioButton("4800 Baud", baudrategroup);
	baud4800button->setGeometry(KPROPERTIESDLG_BAUD4800BTN_RECT);
	baudrategroup->insert(baud4800button, 4800);
	
	baud9600button = new QRadioButton("9600 Baud", baudrategroup);
	baud9600button->setGeometry(KPROPERTIESDLG_BAUD9600BTN_RECT);
	baudrategroup->insert(baud9600button, 9600);
	
	garminbutton = new QRadioButton("Garmin", gpsgroup);
	garminbutton->setGeometry(KPROPERTIESDLG_GARMINBTN_RECT);
	
	rockwellbutton = new QRadioButton("Rockwell", gpsgroup);
	rockwellbutton->setGeometry(KPROPERTIESDLG_ROCKWELLBTN_RECT);
	
	com1button = new QRadioButton("COM1", comgroup);
	com1button->setGeometry(KPROPERTIESDLG_COM1BTN_RECT);
	
	com2button = new QRadioButton("COM2", comgroup);
	com2button->setGeometry(KPROPERTIESDLG_COM2BTN_RECT);
	
	com3button = new QRadioButton("COM3", comgroup);
	com3button->setGeometry(KPROPERTIESDLG_COM3BTN_RECT);
	
	com4button = new QRadioButton("COM4", comgroup);
	com4button->setGeometry(KPROPERTIESDLG_COM4BTN_RECT);

	/**************************************************************/	
	baudval = config->readNumEntry("Baudrate", 4800);
	baudrategroup->setButton(baudval);
	receiver = config->readEntry("GPS Receiver", "Garmin");
	strcmp(receiver, "Garmin") ? gpsgroup->setButton(1) : gpsgroup->setButton(0);

	port = config->readNumEntry("Port", 1);
	comgroup->setButton(port);

	connect(baudrategroup, SIGNAL(clicked(int)), this, SLOT(baudrate(int)));
	connect(gpsgroup, SIGNAL(clicked(int)), this, SLOT(gpsreceiver(int)));
	connect(comgroup, SIGNAL(clicked(int)), this, SLOT(comport(int)));
}

void KPropertiesDlg::baudrate(int id)
{
	char msg[80];
	int r;
	
	QString receiver = config->readEntry("GPS Receiver", "Garmin");

	config->writeEntry("Baudrate", id);
	if(id == 4800)
	{
		if(strcmp(receiver, "Garmin") == 0)
		{
			sprintf((char*)&msg, changegarminbaudrate_str, 3);
		}
		else
		{
		}
	}
	else if(id == 9600)
	{
		if(strcmp(receiver, "Garmin") == 0)
		{
			sprintf((char*)&msg, changegarminbaudrate_str, 4);
		}
		else
		{
		}
	}
	if(port > -1) r = write(port, (char*)&msg, strlen(msg));
	((KGPSWidget*)parent())->close();
	((KGPSWidget*)parent())->open();
}

void KPropertiesDlg::gpsreceiver(int id)
{
	id ? config->writeEntry("GPS Receiver", "Rockwell") :  config->writeEntry("GPS Receiver", "Garmin");
	((KGPSWidget*)parent())->close();
	((KGPSWidget*)parent())->open();
}

void KPropertiesDlg::comport(int id)
{
	config->writeEntry("Port", id);
	((KGPSWidget*)parent())->close();
	((KGPSWidget*)parent())->open();
}
