//////////////////////////////////////////////////////////////////////////////
// This software is distributed under the terms of the General Public License.
//
// Program : kless
// Author  : Norbert Drees & Ralf Haferkamp
// E-Mail  : norbert@et-inf.fho-emden.de
//           hafer@et-inf.fho-emden.de
//////////////////////////////////////////////////////////////////////////////

#include "main.h"

extern KApplication *app;
extern QString ConfigWidth;
extern QString	ConfigHeight;
extern QString	ConfigEditor;
extern QString	ConfigPrinter;
extern QString	ConfigFont;
extern QString	ConfigTab;

KLessWindow::KLessWindow( QWidget *, const char *name )
	: KTopLevelWidget( name ) {

	windowList.setAutoDelete( FALSE );
	
	setMinimumSize( MIN_WINDOW_WIDTH, MIN_WINDOW_HEIGHT);

	multiedit = new TextView( this );
	multiedit->setReadOnly( TRUE );
	multiedit->setAutoUpdate( TRUE );
	multiedit->setFocus();

	KConfig *KLessConfig=kapp->getConfig();
	KLessConfig->setGroup( "Options" );
	QColor bgcolor = KLessConfig->readColorEntry( "BackGround", &multiedit->backgroundColor() );
	QColor fgcolor = KLessConfig->readColorEntry( "ForeGround", &multiedit->foregroundColor() );
	QFont font = KLessConfig->readFontEntry( "Font" );
	
	multiedit->setFont( font );
	QColorGroup cgrp = ( (multiedit->palette() ).copy() ).normal();
	QColorGroup newgrp(
		fgcolor,
		cgrp.background(),
		cgrp.light(),
		cgrp.dark(),
		cgrp.mid(),
		fgcolor,bgcolor
	);
	QPalette newpalette;
	newpalette.setNormal( newgrp );
	newpalette.setDisabled( newgrp );
	newpalette.setActive( newgrp );
	multiedit->setPalette( newpalette );
	
	filemenu = new QPopupMenu();
	CHECK_PTR( filemenu );
	filemenu->insertItem( klocale->translate( "Load" ), this, SLOT( Load() ), CTRL+Key_L );
	menuItem[0] = filemenu->insertItem( klocale->translate( "Reload" ), this, SLOT( Reload() ), CTRL+Key_R );
	menuItem[1] = filemenu->insertItem( klocale->translate( "Save As" ), this, SLOT( SaveAs() ), CTRL+Key_A );
	filemenu->insertSeparator();
	menuItem[2] = filemenu->insertItem( klocale->translate( "Edit" ), this, SLOT( Edit() ), CTRL+Key_E );
	menuItem[3] = filemenu->insertItem( klocale->translate( "Print" ), this, SLOT( Print() ), CTRL+Key_P );
	filemenu->insertSeparator();
	filemenu->insertItem( klocale->translate( "New Window" ), this, SLOT( NewWindow() ), CTRL+Key_W );
	filemenu->insertSeparator();
	filemenu->insertItem( klocale->translate( "Close" ), this, SLOT( Close() ), CTRL+Key_C );
	filemenu->insertItem( klocale->translate( "Quit" ), this, SLOT( Quit() ), CTRL+Key_Q );
 
	filemenu->setItemEnabled( menuItem[0], false ); 
	filemenu->setItemEnabled( menuItem[1], false ); 
	filemenu->setItemEnabled( menuItem[2], false ); 
	filemenu->setItemEnabled( menuItem[3], false ); 

	searchmenu = new QPopupMenu();
	CHECK_PTR( searchmenu );
	menuItem[4] = searchmenu->insertItem( klocale->translate( "Search" ), this, SLOT( Search() ), CTRL+Key_S );
	menuItem[5] = searchmenu->insertItem( klocale->translate( "Search next" ), this, SLOT( SearchNext() ), CTRL+Key_N );

	searchmenu->setItemEnabled( menuItem[4], false ); 
	searchmenu->setItemEnabled( menuItem[5], false ); 

	colors = new QPopupMenu();
	CHECK_PTR( colors );
	colors->insertItem (klocale->translate( "&Background" ), this, SLOT(bgColor() ) );
	colors->insertItem (klocale->translate( "&Foreground" ), this, SLOT(fgColor() ) );
	
	optionsmenu = new QPopupMenu();
	CHECK_PTR( optionsmenu );
	optionsmenu->insertItem( klocale->translate( "&Font" ), this, SLOT( Font() ) );
	optionsmenu->insertItem( klocale->translate( "&Colors"), colors);
	optionsmenu->insertSeparator();
	optionsmenu->insertItem( klocale->translate( "&Tab stops" ), this, SLOT( Tab() ) );
	optionsmenu->insertItem( klocale->translate( "&Editor" ), this, SLOT( Editor() ) );
	optionsmenu->insertItem( klocale->translate( "&Printer" ), this, SLOT( Printer() ) );
	optionsmenu->insertSeparator();
	optionsmenu->insertItem( klocale->translate( "&Save Options" ), this, SLOT( saveOptions() ) );

	helpmenu = new QPopupMenu();
	CHECK_PTR( helpmenu );
	helpmenu->insertItem( klocale->translate( "&About" ), this, SLOT( About() ) );
	helpmenu->insertItem( klocale->translate( "&Help" ), this, SLOT( Help() ) );
    
	menubar = new KMenuBar( this );
	CHECK_PTR( menubar );
	menubar->setFrameStyle( QFrame::NoFrame );
	menubar->insertItem( klocale->translate( klocale->translate( "&File" ) ), filemenu );
	menubar->insertItem( klocale->translate( klocale->translate( "&Search" ) ), searchmenu );
	menubar->insertItem( klocale->translate( klocale->translate( "&Options" ) ), optionsmenu );
	menubar->insertSeparator();
	menubar->insertItem( klocale->translate( klocale->translate( "&Help" ) ), helpmenu );
	connect( menubar, SIGNAL ( moved( menuPosition ) ), SLOT ( menubarMoved() ) );  

	statusbar = new KStatusBar( this );
	statusbar->setInsertOrder( KStatusBar::RightToLeft );
	statusbar->insertItem( klocale->translate( "Size: 0 Bytes                   "), 0 );
	statusbar->insertItem( klocale->translate( "Name: unnamed" ), 1 );
	 
	KDNDDropZone *dropZone = new KDNDDropZone( this, DndURL );
	connect( dropZone, SIGNAL( dropAction( KDNDDropZone * ) ),
				this, SLOT( dropEvent(  KDNDDropZone * ) ) );
	kfmConnection = NULL;

	// at the first start these values are NULL -> 0 for width and height
	resize( ConfigWidth.toInt(), ConfigHeight.toInt() );
	
	QDir h;
	HomeDir = h.homeDirPath();
	name = NULL;
	searchText = NULL;
	sense = false;
	searchdlg_open = false;
	data_path.sprintf( "%s/kless", app->kde_datadir().data() );
	about_picture.setStr( data_path + "/kless.gif" );
	// at the first start this value is NULL -> 0 for tab width.
	// the range for TabWindow is set to 1-8 -> min value 1
	tab_width = ConfigTab.toInt();

	if( !strcmp( ConfigEditor, "" ) ) { ConfigEditor.setStr( DEFAULT_EDITOR ); }
	if( !strcmp( ConfigPrinter, "" ) ) { ConfigPrinter.setStr( DEFAULT_PRINTER ); }

	#ifdef KLESSDEBUG
		cout << "KLessWindow::KLessWindow" << endl;
		cout << "========================" << endl;
		cout << "* ConfigHeight : " << ConfigHeight << endl;
		cout << "* ConfigWidth : " << ConfigWidth << endl;
		cout << "* ConfigEditor : " << ConfigEditor << endl;
		cout << "* ConfigPrinter : " << ConfigPrinter << endl;
		cout << "* HomeDir : " << HomeDir << endl;
		cout << "* name : " << name << endl;
		cout << "* kfmConnection : " << kfmConnection << endl;
		cout << "* about_picture : " << about_picture << endl; 
		cout << "* tab_width : " << tab_width << endl; 
	#endif
}

void KLessWindow::updateMenu() {
	
	filemenu->setItemEnabled( menuItem[0], true ); 
	filemenu->setItemEnabled( menuItem[1], true ); 
	filemenu->setItemEnabled( menuItem[2], true ); 
	filemenu->setItemEnabled( menuItem[3], true ); 

	searchmenu->setItemEnabled( menuItem[4], true ); 
	searchmenu->setItemEnabled( menuItem[5], true ); 

	menubar->repaint();
}

void KLessWindow::getFile() {

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::getFile()" << endl;
		cout << "===========================" << endl;
	#endif

	KURL u( name );
	// absolute path and filename
	const char *url = u.path();

	QFileInfo info( url );

	#ifdef KLESSDEBUG
		cout << "* u.protocol() : " << u.protocol() << endl;
		cout << "* u.directory() : " << u.directory() << endl; // crash while using a protocol string like "http::"
		cout << "* u.filename() : " << u.filename() << endl;
		cout << "* url : " << url << endl;
	#endif

	if( info.isDir() ) {
		QMessageBox::information(
			this,
			"KLess",
			klocale->translate( "This is a directory !" ),
			klocale->translate( "Ok" )
		 );
		 return;
	}

	if( strcmp( u.protocol(), "file" ) == 0 ) {
		readFile( url );
	} else {	
 		CopyToHomeDir = HomeDir + "/" + u.filename();
 		
 		#ifdef KLESSDEBUG
 			cout << "* CopyToHomeDir : " << CopyToHomeDir << endl;
		#endif
		
		KFM *kfmConnection = new KFM;
	  	connect( kfmConnection, SIGNAL( finished() ), this, SLOT( slotKFMJobDone() ) );
  		kfmConnection->copy( name, "file:" + CopyToHomeDir );	
	}
}

void KLessWindow::readSTDIN() {

	QString buffer;
	QTextStream cin( stdin,  IO_ReadOnly );
	int size = 0;

	// output the text in the multiline edit 
	multiedit->clear();	
	multiedit->deselect();
	multiedit->setAutoUpdate( FALSE );
	while( ! cin.eof() ) {
      buffer = cin.readLine();
      size += (buffer.length() + 1);  
		multiedit->insertLine( buffer );
	}
	multiedit->setAutoUpdate( TRUE );
	multiedit->repaint();

	// update the statusbar
	QString s;
	s.sprintf( "Name: %s", klocale->translate( "unknown" ) );
	statusbar->changeItem( s, 1 );
	s.sprintf( klocale->translate( "Size: %d Bytes" ), size );
	statusbar->changeItem( s, 0 );

	if( size == 0 ) {								// there was no input via STDIN
		return;
	}

	name.setStr( "READ_DATA_VIA_STDIN" );	// See slots.cpp Reload()
	updateMenu();
	filemenu->setItemEnabled( menuItem[0], false ); 
	filemenu->setItemEnabled( menuItem[2], false ); 
	filemenu->setItemEnabled( menuItem[3], false ); 
}

void KLessWindow::readFile( QString file ) {

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::readFile( QString file )" << endl;
		cout << "==========================================" << endl;
		cout << "* file : " << file << endl;
	#endif
			
	QFile f( file );
	
	if ( f.open( IO_ReadOnly ) ) {

		QFileInfo info( f );
		QTextStream t( &f );
		KURL u( file );
		QString status;
		QString line, display_line, tab;
		int len, i;

		// replace tabs with this fill-string
		tab.fill( ' ', tab_width );

		// update the statusbar
		status.sprintf( "Name: %s", file.data() );
		statusbar->changeItem( status, 1 );
		status.sprintf( klocale->translate( "Size: %d Bytes" ), info.size() );
		statusbar->changeItem( status, 0 );

		// show the filename in the titlebar
		setCaption( u.filename() );

		// setup the multiline edit 
		multiedit->clear();	
		multiedit->deselect();
		multiedit->setAutoUpdate( FALSE );

		while( ! t.eof() ) {
			line = t.readLine();
			display_line = "";
			// This is a little work-around, because i can't find a 
			// method for setting up the tab width for a qmultiline edit.
			// So i do the following:
			// 1) Check every character in the line for tab
			// 2) If there is a tab, replace the tab with the fill-string
			len = line.length();
			for( i = 0; i != len; i++ ) {
				if( line[i] == 9 ) display_line += tab;
				else               display_line += line[i];
			}
			// Insert the modified line into the multiline edit
			multiedit->insertLine( display_line );
		}
		// Redraw the multiline edit
		multiedit->setAutoUpdate( TRUE );
		multiedit->repaint();

		updateMenu();

	} else {

		#ifdef KLESSDEBUG
			cout << "- can't open file" << endl;
		#endif

		KMsgBox *msg = new KMsgBox(
			this,
			klocale->translate( "KLess message" ),
			klocale->translate( "can't open file" ),
			1,
			"Ok"
		);
		msg->exec();
		delete msg; 		
	}
}

#ifdef KLESSDEBUG
#include <iostream.h>
#endif
void KLessWindow::argLoad( const char *file ){

	#ifdef KLESSDEBUG
		cout << "void KLessWindow::argLoad(const char* file)" << endl;
		cout << "===========================================" << endl;
		cout << "* file: " << file << endl;
	#endif

	KURL u( file );

	if( strcmp( u.protocol(), "" ) == 0 ) {
		 
		#ifdef KLESSDEBUG
			cout << "- local file without protocol" << endl;
		#endif
		
		QFileInfo info( file );
		if( !info.exists() ) {
			cout << "KLess error: Can't load file '" << file << "'" << endl;
			exit( -1 );
		}

		QDir dir;
		name = "file:" + dir.absFilePath( name.setStr( file ) );
		
		#ifdef KLESSDEBUG
			cout << "* name : " << name << endl;
		#endif

	} else {

		#ifdef KLESSDEBUG
			cout << "* u.protocol() : " << u.protocol() << endl;
		#endif
		name.setStr( file );

		// program run with protocol, but there is no protocol at the
		// beginning of the filename. is this an error ?
		if(	name.find( "ftp:", 0, false)  && 
				name.find( "http:", 0, false ) && 
				name.find( "file:", 0, false ) ) {
			if( name.find( "/", 0, false ) ) {
				name = "file:/" + name;
			} else {
				name = "file:" + name;
			}
		}

		#ifdef KLESSDEBUG
			cout << "* name : " << name << endl;
			// to use output in multiedit comment out the last lines in this method
			multiedit->insertLine( name );
		#endif

		// ok, at this moment i can look for a the string ":/"	
		if( name.find( ":/", 0, false ) == -1 ) {
			int pos = name.find( ':', 0, false );
			name.insert( pos + 1, "/" );
		}
		#ifdef KLESSDEBUG
			cout << "* name : " << name << endl;
			// to use output in multiedit command out the last lines in this method
			multiedit->insertLine( name );
		#endif
	}

	if( name != NULL ) {
		getFile();
	}
}

bool KLessWindow::hasSelectedText() {

	if( multiedit->hasSelectedText() == false ) {
		return false;	
	}

	QMessageBox yesno(
		"KLess",
		klocale->translate( "Work with selected text region ?" ),
		QMessageBox::NoIcon,
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No,
		0
	);
	yesno.setButtonText( QMessageBox::Yes, klocale->translate( "Yes" ) );
	yesno.setButtonText( QMessageBox::No, klocale->translate( "No" ) );

	if ( yesno.exec() == QMessageBox::No ) {
		return false;	
	}

	return true;	
}

void KLessWindow::Save( bool selected ) {

	bool write_file = false;

	QString file;
	file.setStr( KFileDialog::getSaveFileName() );
	
	if( file.isNull() ) {								// canceled
		return;
	}

	QFileInfo info( file );									// now save the file

	if( info.isDir() ) {
		QMessageBox::information(
			this,
			"KLess",
			klocale->translate( "This is a directory !" ),
			klocale->translate( "Ok" )
		 );
		 return;
	}

	if( info.exists() && !info.isWritable() ) {
		QMessageBox::information(
			this,
			"KLess",
			klocale->translate( "Permission denied !" ),
			klocale->translate( "Ok" )
		);
		return;
	}

	if( info.exists() && info.isWritable() ) {

		QString s; 
		s.sprintf( "%s '%s'", klocale->translate( "Override existing file ?" ), info.fileName().data() ); 
		QMessageBox yesno(
			"KLess",
  			s,
			QMessageBox::NoIcon,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::No,
			0
		);
		yesno.setButtonText( QMessageBox::Yes, klocale->translate( "Yes" ) );
		yesno.setButtonText( QMessageBox::No, klocale->translate( "No" ) );

		if ( yesno.exec() == QMessageBox::Yes ) {
			write_file = true;
		}

	} else {
		write_file = true;
	}

	if( write_file == true ) {

		name.setStr( file );

		QString text;
		if( selected == true ) {
			text = multiedit->getSelectedText();
		} else {
			text = multiedit->text();
		}

		ofstream os;
		os.open( name, ios::out );
		os << text << "\n";
		os.close();

		Reload();
	}
}

void KLessWindow::runCommand( QString prg, QString file ) {

	if( ( prg == " " ) || ( prg == "" ) ) return;

	#ifdef KLESSDEBUG
		cout << "KLessWindow::runCommand( QString prg, QString file )" << endl;
		cout << "====================================================" << endl;
	#endif

	QString ifs( " " );							// seperator for arguments
	uint nr_ifs;									// number of seperators

	QString arg;									// one argument
	uint i = 0;
	uint len;

	QString str1( prg + ifs + file );			// the command line string
	QString str = str1.simplifyWhiteSpace();
	len = str.length();
	nr_ifs = str.contains( ' ' ) + 2;		// + 2 : number of arguments + 1
														//			end of arguments (will be initalized with 0)
	
	char **args = new char * [ nr_ifs ];	// memory for arguments

	arg = strtok( str.data(), ifs );			// get first argument (program name)

	while( arg ) {									// get all arguments
		len = arg.length();
		args[i] = new char [len];				// memory for each argument
		strcpy( args[i++], arg );
		arg = strtok( 0, ifs );
	}

	args[i] = 0;									// mark end of arguments

	#ifdef KLESSDEBUG
		for( i = 0; i != nr_ifs; i++ ) {
			cout << "args[" << i << "] : " << args[i] << endl;
		}
	#endif

	if( fork() == 0 ) {							// run program
		execvp( args[0], args );
		::exit( 1 );
	}

	for( i = 0; i != nr_ifs; i++ ) {			// clean memory
		delete args[i];
	}
	delete [] args;
}

void KLessWindow::resizeEvent ( QResizeEvent * ){

	int menubar_width 	=	menubar->width();
	int menubar_height	=	menubar->height();

	int searchdlg_height =	40;

	if( menubar->menuBarPos() == KMenuBar::Floating ) {
		menubar_width		=	0;
		menubar_height 	=	0;
	}

	if( searchdlg_open == true ) {
		searchdlg->move( 0, menubar_height );
		searchdlg->setFixedSize( width(), searchdlg_height );
		searchdlg->setActiveWindow();
	}	else {
		searchdlg_height = 0;
		multiedit->setFocus();
	}

	multiedit->move(
		0,
		menubar_height + searchdlg_height
	);
	multiedit->resize(
		width(),
		height() - menubar_height - statusbar->height() - searchdlg_height
	);
			 	
	statusbar->move(
		0,
		menubar_height + searchdlg_height + multiedit->height()
	);
	statusbar->resize(
		width(), 
		statusbar->height()
	);
	
	menubar->resize(
		width(),
		menubar_height
	);
}

void KLessWindow::closeEvent( QCloseEvent * )	{

	Close();
}
