/*
	This file is part of `klp', a KDE Line Printer queue manager

	Copyright (C) 1998
	Frans van Dorsselaer
	<dorssel@MolPhys.LeidenUniv.nl>

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "config.h"

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include <qpopupmenu.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <qstrlist.h>
#include <qstring.h>
#include <qlabel.h>
#include <qkeycode.h>
#include <qsocketnotifier.h>

#include <kapp.h>
#include <kurl.h>
#include <kconfig.h>

#include "mainw.h"
#include "dock.h"
#include "inter.h"
#include "select.h"
#include "update.h"
#include "queuebox.h"
#include "options.h"

#include "pixmaps/status_play.xpm"
#include "pixmaps/status_stop.xpm"
#include "pixmaps/status_pause.xpm"
#include "pixmaps/status_fault.xpm"
#include "pixmaps/status_busy.xpm"
#include "pixmaps/status_ready.xpm"

MainWindow *mainWindow;


// Menu item id's
#define M_DOCK_IN_PANEL		1
#define M_CLOSE			2
#define M_SHOW_STATUSBAR	3
#define M_SHOW_TOOLBAR		4
#define M_REMOVE_JOBS		5
#define M_UPDATE_QUEUE		6
#define M_STOP			7
#define M_SELECT_PRINTER	8

// Toolbar item id's
#define T_UPDATE_QUEUE	1
#define T_REMOVE_JOBS	2
#define T_STOP		3
#define T_PRINTERS	4

// Statusbar item id's
#define S_PRINTER_STATE	1
#define S_BUSY		2
#define S_TEXT		3


MainWindow::MainWindow(struct options *opts) : KTMainWindow()
		, dock_window(0) , current_command(0), expect_abort(false)
		, should_show(true)
{
	queue_box = new QueueBox(this);
	setView(queue_box);

	printer_status = new QLabel(statusBar());
	statusBar()->insertWidget(printer_status, 10, S_PRINTER_STATE);

	busy_status = new QLabel(statusBar());
	statusBar()->insertWidget(busy_status, 10, S_BUSY);

	statusBar()->insertItem("", S_TEXT);

	QPopupMenu *file = new QPopupMenu();
	QPopupMenu *options = new QPopupMenu();
	file->insertItem(i18n("Select &printer..."), this, SLOT(selectPrinter()), 0, M_SELECT_PRINTER);
	file->insertSeparator();
	file->insertItem(i18n("&Update queue"), this, SLOT(updateQueue()), CTRL + Key_U, M_UPDATE_QUEUE);
	file->insertItem(i18n("&Delete job(s)"), this, SLOT(removeJobs()), Key_Delete, M_REMOVE_JOBS);
	file->insertItem(i18n("&Stop"), this, SLOT(abortAction()), Key_Escape, M_STOP);
	file->insertSeparator();
	file->insertItem(i18n("&Close"), this, SLOT(hide()), CTRL + Key_W, M_CLOSE);
	file->setItemEnabled(M_CLOSE, false);
	file->insertItem(i18n("E&xit"), this, SLOT(exit()), CTRL + Key_Q, -2);
	menuBar()->insertItem(i18n("&File"), file, 0, -2);
	options->insertItem(i18n("&Dock in panel"), this, SLOT(toggleDock()), 0, M_DOCK_IN_PANEL);
	options->insertItem(i18n("Show &statusbar"), this, SLOT(toggleStatusBar()), 0, M_SHOW_STATUSBAR);
	options->insertItem(i18n("Show &toolbar"), this, SLOT(toggleToolBar()), 0, M_SHOW_TOOLBAR);
	options->insertSeparator();
	options->insertItem(i18n("&Auto update..."), this, SLOT(autoUpdate()), 0, -2);
	options->insertSeparator();
	options->insertItem(i18n("Save &options"), this, SLOT(saveOptions()), 0, -2);
	menuBar()->insertItem(i18n("&Options"), options, 0, -2);

	char s[1000];
	sprintf(s, i18n("KDE Line Printer queue manager\n"
		"Version %s\n\n"
		"Copyright (c) 1998\n"
		"Frans van Dorsselaer\n"
		"<dorssel@MolPhys.LeidenUniv.nl>\n\n"
		"This program is licenced under the\n"
		"GNU General Public Licence (GPL)."), VERSION);
	menuBar()->insertItem(i18n("&Help"), kapp->getHelpMenu(true, s), 0, -2);

	toolBar()->insertButton(QPixmap(kapp->kde_toolbardir() + "/reload.xpm"), T_UPDATE_QUEUE, SIGNAL(clicked()), this, SLOT(updateQueue()), true, i18n("Update queue"));
	toolBar()->insertButton(QPixmap(kapp->kde_toolbardir() + "/filedel.xpm"), T_REMOVE_JOBS, SIGNAL(clicked()), this, SLOT(removeJobs()), true, i18n("Delete job(s)"));
	toolBar()->insertButton(QPixmap(kapp->kde_toolbardir() + "/stop.xpm"), T_STOP, SIGNAL(clicked()), this, SLOT(abortAction()), true, i18n("Stop"));
	toolBar()->insertCombo("", T_PRINTERS, false, SIGNAL(activated(int)), this, SLOT(setActivePrinter(int)), false, i18n("Select printer"));
	toolBar()->setItemAutoSized(T_PRINTERS);

	KDNDDropZone *drop_zone = new KDNDDropZone(this, DndURL);
	connect(drop_zone, SIGNAL(dropAction(KDNDDropZone *)), SLOT(dropAction(KDNDDropZone *)));

	connect(queue_box, SIGNAL(highlighted(int, int)), SLOT(highlighted(int, int)));
	connect(queue_box, SIGNAL(popupMenu(int, int)), SLOT(popupMenu(int, int)));

	socket_notifier = new QSocketNotifier(pipe_r, QSocketNotifier::Read, this);
	connect(socket_notifier, SIGNAL(activated(int)), SLOT(readServer()));

	update_timer = new QTimer(this);
	connect(update_timer, SIGNAL(timeout()), SLOT(updateQueue()));

	busy = true;
	printer_state = printer_ready;
	setPrinterState(printer_fault);
	setBusy(false);

	printers.setAutoDelete(true);

	if (kapp->getConfigState() != KApplication::APPCONFIG_NONE) {
		kapp->getConfig()->setGroup(0);
		readPrinters(kapp->getConfig());
	}
	updateCombo();
	if (!kapp->isRestored() && (kapp->getConfigState() != KApplication::APPCONFIG_NONE)) {
		kapp->getConfig()->setGroup(0);
		readProperties(kapp->getConfig(), opts);
	}

	saneGUI();
}


void MainWindow::readServer()
{
	char c;

	if (!iread_char(&c)) {
		fprintf(stderr, "klp: Error: server closed pipe\n");
		delete socket_notifier;
		socket_notifier = NULL;
		QMessageBox::critical(this, i18n("klp: Fatal error"),
			i18n("klp's child process has exited for unknown\n"
			"reasons.  Maybe you explicitely killed that\n"
			"process.  If not, please report this as a bug.\n"
			"\n"
			"klp will now exit."));
		exit();
		return;
	}
	switch (c) {
		case 'q':
			if (expect_abort)
				break;

			if (current_command != 'q')
				goto fail;

			PrinterState new_state;
			int no_print;
			int no_queue;
			int item_count;

			iread_int(&no_print);
			iread_int(&no_queue);
			iread_int(&item_count);

			int i;
			QueueItemList *new_list;
			new_list = new QueueItemList;
			for (i = 0 ; i < item_count ; ++i) {
				QueueItem *item = new QueueItem;
				char *s;
				iread_int(&item->rank);
				iread_string(&s);
				item->job_name = s;
				free(s);
				iread_string(&s);
				item->user = s;
				free(s);
				iread_string(&s);
				item->job_id = s;
				free(s);
				iread_int(&item->size);
				new_list->append(item);
			}
			queue_box->setQueueItemList(new_list);
			if (no_queue)
				new_state = printer_stop;
			else if (no_print)
				new_state = printer_pause;
			else if (!strcmp(username, "root"))
				new_state = queue_box->count() ? printer_play : printer_ready;
			else
				new_state = queue_box->haveOwnedJob() ? printer_play : printer_ready;
			setPrinterState(new_state);
			current_command = 0;
			setBusy(false);
			statusBar()->changeItem("", S_TEXT);
			if (auto_update)
				update_timer->start(update_interval * 1000, true);
			saneGUI();
			break;
		case 's':
			if (expect_abort)
				break;
			
			if (current_command != 's')
				goto fail;
			
			current_command = 0;
			setPrinterState(printer_ready);
			updateQueue();
			break;
		case 'f':
			if (expect_abort)
				break;

			char *s;
			iread_string(&s);
			statusBar()->changeItem(i18n(s), S_TEXT);
			free(s);
			current_command = 0;
			setPrinterState(printer_fault);
			setBusy(false);
			saneGUI();
			break;
		case 'p':
		case 'r':
			if (expect_abort)
				break;

			current_command = 0;
			updateQueue();
			break;
		case 'a':
			if (!expect_abort)
				goto fail;
			
			expect_abort = false;
			if (!current_command || (current_command == 'q'))
				setBusy(false);
			saneGUI();
			break;
		default:
			goto fail;
	}
	return;

fail:
	delete socket_notifier;
	socket_notifier = NULL;
	QMessageBox::critical(this, i18n("klp: Fatal error"),
		i18n("klp appears to have lost synchronisation with\n"
		"its child process.  Please report this as a bug.\n"
		"\n"
		"klp will now exit."));
	exit();
}


void MainWindow::dropAction(KDNDDropZone *drop_zone)
{
	if ((printer_state == printer_fault) || (printer_state == printer_stop)) {
		QMessageBox::warning(this, i18n("klp Error"),
			i18n("There is a problem with the printer.  Either your\n"
			"setup is incorrect or the printer is currently not\n"
			"accepting print jobs."));
		return;
	}
	// We are _not_ reentrant
	if (busy) {
		QMessageBox::warning(this, i18n("klp Error"),
			i18n("klp is still processing your previous request.\n"
			"Please try again after processing has finished."));
		return;
	}
	if (drop_zone->getDataType() != DndURL) {
		QMessageBox::warning(this, i18n("klp Error"),
			i18n("The object that you just tried to print has no URL.\n"
			"klp can only print documents referred to by an URL\n"
			"(e.g. by dragging files from kfm)."));
		return;
	}
	QStrList &url_list = drop_zone->getURLList();
	char *s;
	KURL url;
	QStrList files(true);
	for (s = url_list.first() ; s ; s = url_list.next()) {
		url.parse(s);
		if (url.isMalformed()) {
			QString msg = i18n("The object that you just tried to print\n"
                                "has an invalid URL:\n\n");
			msg += s;
			QMessageBox::warning(this, i18n("klp Error"), msg);
			continue;
		}
		if (!url.isLocalFile()) {
			QMessageBox::warning(this, i18n("klp Error"),
				i18n("Printing of remote documents is not supported."));
			continue;
		}
/*
		QString tmp;
		setBusy(true);
		KFM::download(s, tmp);
		fprintf(stderr, "klp: tmp: %s\n", (const char *)tmp);
		// TODO: write(frontend_server, "printjob !!");
*/
		files.append(url.path());
/*
		KFM::removeTempFile(tmp);
*/
	}
	sendJob(files);
}


void MainWindow::sendJob(QStrList &files)
{
	if (!files.count())
		return;
	if (current_command && (current_command != 'q'))
		return;
	abortAction();
	iwrite_char('p');
	iwrite_int(files.count());
	char *file;
	for (file = files.first() ; file ; file = files.next())
		iwrite_string(file);
	current_command = 'p';
	setBusy(true);
	statusBar()->changeItem("Sending job ...", S_TEXT);
	saneGUI();
}	


void MainWindow::setPrinterState(PrinterState new_printer_state)
{
	const char **s;
	if (new_printer_state == printer_state)
		return;
	printer_state = new_printer_state;
	emit printerStateChanged(printer_state);
	switch(printer_state) {
		case printer_ready:
			s = (const char **)status_ready;
			break;
		case printer_pause:
			s = (const char **)status_pause;
			break;
		case printer_stop:
			s = (const char **)status_stop;
			break;
		case printer_play:
			s = (const char **)status_play;
			break;
		case printer_fault:
		default:
			printer_state = printer_fault;
			s = (const char **)status_fault;
	}
	printer_status->setPixmap(QPixmap(s));
	if (printer_state == printer_fault) {
		queue_box->setQueueItemList(NULL);
		update_timer->stop();
	}
}


void MainWindow::setBusy(bool new_busy)
{
	if (new_busy == busy)
		return;
	busy = new_busy;
	emit busyChanged(busy);
	busy_status->setPixmap(QPixmap((const char **)(busy ? status_busy : status_ready)));
}


void MainWindow::selectPrinter()
{
	if (busy)
		return;
	SelectDialog sd(printers, active_printer, this);
	sd.exec();
	if (sd.printersChanged()) {
		if (kapp->getConfigState() != KApplication::APPCONFIG_READWRITE)
			QMessageBox::warning(this, i18n("klp Error"),
				i18n("Could not open the configuration file for writing.\n"));
		else
			writePrinters(kapp->getConfig());
		updateCombo();
		emit printersChanged();
	}
	if (sd.activeChanged()) {
		setActivePrinter(sd.activePrinter());
	}
}


void MainWindow::setActivePrinter(int i)
{
	if (current_command && (current_command != 'q'))
		return;
	if ((i > (int)printers.count()) || (i < -1))
		i = -1;
	abortAction();
	active_printer = i;
	if (active_printer >= 0) {
		iwrite_char('s');
		iwrite_int(printers.at(active_printer)->type);
		iwrite_string(printers.at(active_printer)->name);
		iwrite_int(printers.at(active_printer)->local ? 1 : 0);
		if (!printers.at(active_printer)->local)
			iwrite_string(printers.at(active_printer)->host);
		current_command = 's';
		setPrinterState(printer_ready);
		setBusy(true);
		statusBar()->changeItem(i18n("Connecting to printer ..."), S_TEXT);
		setCaption(QString("klp - ") + printers.at(active_printer)->description);
	} else {
		statusBar()->changeItem(i18n("No printer selected"), S_TEXT);
		setCaption(QString("klp - ") + i18n("(No printer selected)"));
	}
	toolBar()->setCurrentComboItem(T_PRINTERS, i);
	saneGUI();
	emit activePrinterChanged(i);
}


void MainWindow::updateCombo()
{
	toolBar()->clearCombo(T_PRINTERS);
	for (Printer *p = printers.first() ; p ; p = printers.next())
		toolBar()->insertComboItem(T_PRINTERS, p->description, -1);
}


void MainWindow::removeJobs()
{
	if (current_command && (current_command != 'q'))
		return;
	int i;
	int t = 0;
	for ( i = queue_box->count() - 1 ; i >= 0 ; --i)
		if (queue_box->isMarked(i))
			if (!strcmp(username, "root") || (username == queue_box->userAt(i)))
				++t;
	if (!t)
		return;
	abortAction();
	iwrite_char('r');
	iwrite_int(t);
	for ( i = queue_box->count() - 1 ; i >= 0 ; --i)
		if (queue_box->isMarked(i))
			if (!strcmp(username, "root") || (username == queue_box->userAt(i)))
				iwrite_string(queue_box->jobIdAt(i));
	current_command = 'r';
	setBusy(true);
	statusBar()->changeItem(i18n("Deleting job(s) ..."), S_TEXT);
	saneGUI();
}


void MainWindow::updateQueue()
{
	update_timer->stop();

	if (current_command)
		return;

	if (printer_state == printer_fault)
		setActivePrinter(active_printer);
	else {
		iwrite_char('q');
		current_command = 'q';
		saneGUI();
	}
}


void MainWindow::abortAction()
{
	if (!current_command)
		return;
	iwrite_char('a');
	current_command = 0;
	statusBar()->changeItem("", S_TEXT);
	expect_abort = true;
	saneGUI();
}


void MainWindow::autoUpdate()
{
	UpdateDialog ud(auto_update, update_interval, this);
	if (ud.exec()) {
		update_interval = ud.update_interval;
		if (auto_update != ud.auto_update) {
			auto_update = ud.auto_update;
			if (auto_update) {
				if (!current_command && (printer_state != printer_fault))
					updateQueue();
			} else
				update_timer->stop();
		}
	}
}


/*****************************************************************
 *
 *	These should all be OK
 *
 */


MainWindow::~MainWindow()
{
	if (dock_window)
		delete dock_window;
}


void MainWindow::readPrinters(KConfig *config)
{
	int c;
	c = config->readUnsignedNumEntry("PrinterCount");
	printers.clear();
	Printer *p;
	QString s;
	for ( int i = 1 ; i <= c ; ++i ) {
		s.setNum(i);
		s = "Printer" + s;
		config->setGroup(s);
		p = new Printer;
		p->description = config->readEntry("Description");
		p->local = config->readBoolEntry("Local");
		/* This to correctly read klp-0.1 config files */
		p->type = config->readUnsignedNumEntry("ServerType", p->local ? 0 : 1);
		p->name = config->readEntry("Name");
		p->host = config->readEntry("Host");
		printers.append(p);
	}
	config->setGroup(0);
}


void MainWindow::writePrinters(KConfig *config)
{
	config->writeEntry("PrinterCount", printers.count());
	QString s;
	int i;
	Printer *p;
	for (i = 1 , p = printers.first() ; p ; ++i , p = printers.next()) {
		s.setNum(i);
		s = "Printer" + s;
		config->setGroup(s);
		config->writeEntry("Description", p->description);
		config->writeEntry("ServerType", p->type);
		config->writeEntry("Name", p->name);
		config->writeEntry("Local", p->local);
		config->writeEntry("Host", p->host);
	}
	config->setGroup(0);		
	config->sync();
}


void MainWindow::saveOptions()
{
	if (kapp->getConfigState() != KApplication::APPCONFIG_READWRITE) {
		QMessageBox::warning(this, i18n("klp Error"),
			i18n("Could not open the configuration file for writing.\n"));
		return;
	}
	KConfig *config = kapp->getConfig();
	saveProperties(config);
	config->sync();
}


void MainWindow::toggleDock()
{
	if (dock_window) {
		delete dock_window;
		dock_window = 0;
		menuBar()->setItemChecked(M_DOCK_IN_PANEL, false);
		menuBar()->setItemEnabled(M_CLOSE, false);
	} else {
		dock_window = new DockWindow(this);
		dock_window->show();
		menuBar()->setItemChecked(M_DOCK_IN_PANEL, true);
		menuBar()->setItemEnabled(M_CLOSE, true);
	}
}


void MainWindow::toggleStatusBar()
{
	enableStatusBar();
	menuBar()->setItemChecked(M_SHOW_STATUSBAR, statusBar()->isVisible());
}


void MainWindow::toggleToolBar()
{
	enableToolBar();
	menuBar()->setItemChecked(M_SHOW_TOOLBAR, toolBar()->isVisible());
}


bool MainWindow::close(bool forceKill)
{
	if (!forceKill && dock_window) {
		hide();
		return false;
	}
	return KTMainWindow::close(forceKill);
}


void MainWindow::exit()
{
	if (dock_window)
		toggleDock();
	close();
}


void MainWindow::highlighted(int, int)
{
	saneRemoveJobs();
}


void MainWindow::popupMenu(int, int)
{
	QPopupMenu menu;

	menu.insertItem(i18n("&Delete job(s)"), this, SLOT(removeJobs()), 0, 1);
	menu.setItemEnabled(1, toolBar()->getButton(T_REMOVE_JOBS)->isEnabled());
	menu.exec(QCursor::pos());
}


void MainWindow::saneUpdateQueue()
{
	bool b = false;
	if ((active_printer >= 0)
	    && (!current_command || (current_command == 'q')))
		b = true;
	toolBar()->setItemEnabled(T_UPDATE_QUEUE, b);
	menuBar()->setItemEnabled(M_UPDATE_QUEUE, b);
}


void MainWindow::saneRemoveJobs()
{
	bool b = false;
	int i;
	if (!current_command || (current_command == 'q'))
		for ( i = queue_box->count() - 1 ; i >= 0 ; --i)
			if (queue_box->isMarked(i))
				if (!strcmp(username, "root") || (username == queue_box->userAt(i))) {
					b = true;
					break;
				}
	menuBar()->setItemEnabled(M_REMOVE_JOBS, b);
	toolBar()->setItemEnabled(T_REMOVE_JOBS, b);
}	


void MainWindow::saneStop()
{
	bool b = false;
	if (current_command && (current_command != 'q'))
		b = true;
	menuBar()->setItemEnabled(M_STOP, b);
	toolBar()->setItemEnabled(T_STOP, b);
}


void MainWindow::saneSelectPrinter()
{
	bool b = false;
	bool b2 = false;
	if (!current_command || (current_command == 'q')) {
		b = true;
		if (printers.count())
			b2 = true;
	}
	menuBar()->setItemEnabled(M_SELECT_PRINTER, b);
	toolBar()->setItemEnabled(T_PRINTERS, b2);
}


void MainWindow::saneGUI()
{
	saneUpdateQueue();
	saneRemoveJobs();
	saneStop();
	saneSelectPrinter();
}


void MainWindow::saveProperties(KConfig* config)
{
	config->writeEntry("ActivePrinter", active_printer + 1);

	config->writeEntry("DockInPanel", dock_window != NULL);
	config->writeEntry("ShowToolBar", toolBar()->isVisible());
	config->writeEntry("ShowStatusBar", statusBar()->isVisible());
	config->writeEntry("AutoUpdate", auto_update);
	config->writeEntry("UpdateInterval", update_interval);

	config->writeEntry("WindowHidden", !isVisible());
	// TODO: Window size / queue_box configuration
}


void MainWindow::readProperties(KConfig* config, struct options *opts)
{
	int active;

	if (opts && opts->set.opt.active_printer)
		active = opts->active_printer - 1;
	else
		active = (int)config->readUnsignedNumEntry("ActivePrinter") - 1;
	setActivePrinter(active);

	bool dock;
	if (opts && opts->set.opt.dock)
		dock = opts->dock;
	else
		dock = config->readBoolEntry("DockInPanel", false);
		
	if (dock) {
		if (!dock_window)
			toggleDock();
		bool hide;
		if (opts && opts->set.opt.hide)
			hide = opts->hide;
		else
			hide = config->readBoolEntry("WindowHidden", false);
		should_show = !hide;
		if (hide)
			close(false);
		menuBar()->setItemChecked(M_DOCK_IN_PANEL, true);
	} else {
		if (dock_window)
			toggleDock();
		menuBar()->setItemChecked(M_DOCK_IN_PANEL, false);
	}
	if (config->readBoolEntry("ShowToolBar", true)) {
		toolBar()->enable(KToolBar::Show);
		menuBar()->setItemChecked(M_SHOW_TOOLBAR, true);
	} else {
		toolBar()->enable(KToolBar::Hide);
		menuBar()->setItemChecked(M_SHOW_TOOLBAR, false);
	}
	if (config->readBoolEntry("ShowStatusBar", true)) {
		statusBar()->enable(KStatusBar::Show);
		menuBar()->setItemChecked(M_SHOW_STATUSBAR, true);
	} else {
		statusBar()->enable(KStatusBar::Hide);
		menuBar()->setItemChecked(M_SHOW_STATUSBAR, false);
	}
	auto_update = config->readBoolEntry("AutoUpdate", false);
	update_interval = config->readUnsignedNumEntry("UpdateInterval", 10);
	if ((update_interval < 3) || (update_interval > 99999))
		update_interval = 10;
}


void MainWindow::readProperties(KConfig* config)
{
	readProperties(config, NULL);
}


#include "mainw.moc"
