/*
    KMLOCfg
    
    A utility to configure the ELSA MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz (o.gantz@tu-bs.de)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "loaddlg.h"
#include "loaddlg.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <qglobal.h>
#include <qwidget.h>
#include <qprogressdialog.h>
#include <qdatetime.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "modem.h"
#include "global.h"


#define BOOL_TO_INT(x) ((x) ? 1 : 0)



LoadDlg::LoadDlg( QWidget * parent, const char * name, bool modal, WFlags f ) : QProgressDialog( parent, name, modal, f )
{
	modem = new Modem( this, "modem" );
	CHECK_PTR( modem );

	modem->setData( 8 );
	modem->setParity( 'N' );
	modem->setStop( 1 );

	init();

	setMinimumDuration( 0 );

	setDefaultConfig();

	connect( modem, SIGNAL(timeout()), SLOT(fetchTimeout()) );
	connect( this, SIGNAL(cancelled()), SLOT(cancel()) );

	setCaption( i18n("Download") );
}


LoadDlg::~LoadDlg()
{
	reset();
}


void LoadDlg::setDefaultConfig()
{
	c.firmware[0] = 0;
	c.country = 0x04;
	c.mem_free = -1;
	c.fax_takeover = TRUE;		/* AT$JCFGF	*/
	c.call_accept = TRUE;

	c.announce_system = TRUE;	/* AT$JCFGM	*/
	c.announce_number = TRUE;
	c.announce_time = TRUE;
	c.outgoing_message = 0;

	c.fax_operation = TRUE;		/* AT$JCFGT	*/
	c.voice_operation = TRUE;
	c.voice_recording = TRUE;
	c.hook_config = FALSE;
	c.keyboard_config = FALSE;
	c.operation_control = FALSE;

	c.max_rec_time = 0;		/* AT$JCFGV	*/
	c.min_rec_time = 0;
	strcpy(c.rec_quality, "ADPCM4-7");
	c.rec_speaker = TRUE;
	c.rec_monitor = TRUE;

	c.fax_id[0] = 0;		/* AT$JFLI	*/

	c.remote_query = FALSE;		/* AT$JPWD	*/
	c.remote_config = FALSE;
	strcpy(c.remote_pw, "1111");

	c.ring_number = 4;		/* AT$JRING	*/
	c.ring_signal = TRUE;
	c.ring_suppress_ri = FALSE;
	c.ring_fixed = TRUE;

	c.voice_rec = 128;		/* AT$JVGR	*/
	c.voice_play = 128;		/* AT$JVGT	*/
	c.voice_micro = 255;		/* AT$JVGM	*/
	c.voice_speaker = 255;		/* AT$JVGS	*/

	c.year = 0;			/* AT$JDATE	*/
	c.month = 0;
	c.day = 0;

	c.hour = 0;			/* AT$JTIME	*/
	c.minute = 0;
	c.second = 0;

	c.sync_timer = FALSE;
}


mlo_config_t * LoadDlg::getConfig()
{
	return &c;
}


void LoadDlg::setConfig( mlo_config_t * conf )
{
	c = *conf;
}


bool LoadDlg::startDownload( const char * device, int speed )
{
	static int speeds[] = { 300, 2400, 9600, 19200, 38400, 57600, 115200, 230400 };

	if (status)
		return FALSE;

	setDefaultConfig();

	modem->setDevice( device );
	modem->setSpeed( speeds[speed] );

	if (!modem->open()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Cannot open modem device."), KMsgBox::EXCLAMATION);
		return FALSE;
	}
	if (!modem->dsrOn()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem is off."), KMsgBox::EXCLAMATION);
		modem->close();
		return FALSE;
	}
	if (!modem->ctsOn()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem is busy."), KMsgBox::EXCLAMATION);
		modem->close();
		return FALSE;
	}

	disconnect( modem, SIGNAL(gotLine(const char *)), this, SLOT(fetchModemLineUp(const char *)) );
	connect( modem, SIGNAL(gotLine(const char *)), SLOT(fetchModemLineDown(const char *)) );

	modem->writeLine( "" );
	usleep( 250000 );
	modem->flush();
	modem->writeLine( "AT&F" );
//	modem->timerStart( 5000 );

	setTotalSteps( 34 );
	setProgress( 0 );

	status = 1;

	return TRUE;
}


bool LoadDlg::startUpload( const char * device, int speed )
{
	static int speeds[] = { 300, 2400, 9600, 19200, 38400, 57600, 115200, 230400 };

	if (status)
		return FALSE;

	modem->setDevice( device );
	modem->setSpeed( speeds[speed] );

	if (!modem->open()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Cannot open modem device."), KMsgBox::EXCLAMATION);
		return FALSE;
	}
	if (!modem->dsrOn()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem is off."), KMsgBox::EXCLAMATION);
		modem->close();
		return FALSE;
	}
	if (!modem->ctsOn()) {
		KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem is busy."), KMsgBox::EXCLAMATION);
		modem->close();
		return FALSE;
	}

	disconnect( modem, SIGNAL(gotLine(const char *)), this, SLOT(fetchModemLineDown(const char *)) );
	connect( modem, SIGNAL(gotLine(const char *)), SLOT(fetchModemLineUp(const char *)) );

	modem->writeLine( "" );
	usleep( 250000 );
	modem->flush();
	modem->writeLine( "AT&F" );
//	modem->timerStart( 5000 );

	setTotalSteps( 15 );
	setProgress( 0 );

	status = 1;

	return TRUE;
}


void LoadDlg::cancel()
{
	QProgressDialog::cancel();

	reset();
}


void LoadDlg::fetchModemLineDown( const char * line )
{
	int d1, d2, d3, d4, d5, d6;

	if (!strcmp(line,"ERROR")) {
		emit downDone( FALSE );
		reset();
		return;
	}
	
	setProgress( status );

	switch (status) {
		case 1:
			if (!strcmp(line, "OK")) {
				modem->writeLine("ATI3");
				status++;
			}
			break;
		case 2:
			if (!strncmp(line, "Version", 7)) {
				strncpy(c.firmware, line, 40);
				c.firmware[40] = 0;
				status++;
			}
			break;
		case 3:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT+GCI?");
				status++;
			}
			break;
		case 4:
			if (sscanf(line, "+GCI: %x", &c.country) == 1) {
				status++;
			}
			break;
			
			strncpy(c.firmware, line, 40);
			c.firmware[40] = 0;
			status++;
			break;
		case 5:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JMEM=Free");
				status++;
			}
			break;
		case 6:
			if (sscanf(line, "$JMEM: %d", &c.mem_free) == 1) {
				status++;
			}
			break;
		case 7:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JCFGF?");
				status++;
			}
			break;
		case 8:
			if (sscanf(line, "$JCFGF: %d,%d", &d1, &d2) > 0) {
				c.fax_takeover = d1 != 0;
				c.call_accept = d2 != 0;
				status++;
			}
			break;
		case 9:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JCFGM?");
				status++;
			}
			break;
		case 10:
			if (sscanf(line, "$JCFGM: %d,%d,%d,%d", &d1, &d2, &d3, &c.outgoing_message) > 0) {
				c.announce_system = d1 != 0;
				c.announce_number = d2 != 0;
				c.announce_time = d3 != 0;
				status++;
			}
			break;
		case 11:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JCFGT?");
				status++;
			}
			break;
		case 12:
			if (sscanf(line, "$JCFGT: %d,%d,%d,%d,%d,%d", &d1, &d2, &d3, &d4, &d5, &d6) > 0) {
				c.fax_operation = d1 != 0;
				c.voice_operation = d2 != 0;
				c.voice_recording = d3 != 0;
				c.hook_config = d4 != 0;
				c.keyboard_config = d5 != 0;
				c.operation_control = d6 != 0;
				status++;
			}
			break;
		case 13:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JCFGV?");
				status++;
			}
			break;
		case 14:
			if (sscanf(line, "$JCFGV: %d,%d,%*[^,],%[^,],%d,%d", &c.max_rec_time, &c.min_rec_time, c.rec_quality, &d1, &d2) > 0) {
				c.rec_speaker = d1 != 0;
				c.rec_monitor = d2 != 0;
				status++;
			}
			break;
		case 15:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JFLI?");
				c.fax_id[0] = 0;
				status++;
			}
			break;
		case 16:
			if (!strncmp(line, "$JFLI: \"", 8)) {
				c.fax_id[0] = 0;
				sscanf(line, "$JFLI: \"%[^\"]\"", c.fax_id);
				status++;
			}
			break;
		case 17:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JPWD?");
				status++;
			}
			break;
		case 18:
			if (sscanf(line, "$JPWD: %d,%d,\"%[^\"]\"", &d1, &d2, c.remote_pw) > 0) {
				c.remote_query = d1 != 0;
				c.remote_config = d2 != 0;
				status++;
			}
			break;
		case 19:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JRING?");
				status++;
			}
			break;
		case 20:
			if (sscanf(line, "$JRING: %d,%d,%d,%d", &c.ring_number, &d1, &d2, &d3) > 0) {
				c.ring_signal = d1 != 0;
				c.ring_suppress_ri = d2 != 0;
				c.ring_fixed = d3 != 0;
				status++;
			}
			break;
		case 21:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JVGR?");
				status++;
			}
			break;
		case 22:
			if (sscanf(line, "$JVGR: %d", &c.voice_rec) > 0) {
				status++;
			}
			break;
		case 23:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JVGT?");
				status++;
			}
			break;
		case 24:
			if (sscanf(line, "$JVGT: %d", &c.voice_play) > 0) {
				status++;
			}
			break;
		case 25:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JVGM?");
				status++;
			}
			break;
		case 26:
			if (sscanf(line, "$JVGM: %d", &c.voice_micro) > 0) {
				status++;
			}
			break;
		case 27:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JVGS?");
				status++;
			}
			break;
		case 28:
			if (sscanf(line, "$JVGS: %d", &c.voice_speaker) > 0) {
				status++;
			}
			break;
		case 29:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JDATE?");
				status++;
			}
			break;
		case 30:
			if (sscanf(line, "$JDATE: %d,%d,%d", &c.year, &c.month, &c.day) == 3) {
				status++;
			}
			break;
		case 31:
			if (!strcmp(line, "OK")) {
				modem->writeLine("AT$JTIME?");
				status++;
			}
			break;
		case 32:
			if (sscanf(line, "$JTIME: %d,%d,%d", &c.hour, &c.minute, &c.second) == 3) {
				status++;
			}
			break;
		case 33:
			if (!strcmp(line, "OK")) {
				setProgress( 34 );
				emit downDone( TRUE );
				reset();
			}
			break;
	}
}


void LoadDlg::fetchModemLineUp( const char * line )
{
	char buf[40];

	if (!strcmp(line,"ERROR")) {
		emit upDone( FALSE );
		reset();
		return;
	}
	
	setProgress( status );

	switch (status) {
		case 1:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JCFGF=%d,%d", BOOL_TO_INT(c.fax_takeover), BOOL_TO_INT(c.call_accept));
				modem->writeLine(buf);
				status++;
			}
			break;
		case 2:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JCFGM=%d,%d,%d,%d", BOOL_TO_INT(c.announce_system), BOOL_TO_INT(c.announce_number), BOOL_TO_INT(c.announce_time), c.outgoing_message);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 3:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JCFGT=%d,%d,%d,%d,%d,%d", BOOL_TO_INT(c.fax_operation), BOOL_TO_INT(c.voice_operation), BOOL_TO_INT(c.voice_recording), BOOL_TO_INT(c.hook_config), BOOL_TO_INT(c.keyboard_config), BOOL_TO_INT(c.operation_control));
				modem->writeLine(buf);
				status++;
			}
			break;
		case 4:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JCFGV=%d,%d,,%s,%d,%d", c.max_rec_time, c.min_rec_time, c.rec_quality, BOOL_TO_INT(c.rec_speaker), BOOL_TO_INT(c.rec_monitor));
				modem->writeLine(buf);
				status++;
			}
			break;
		case 5:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JFLI=\"%s\"", c.fax_id);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 6:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JPWD=%d,%d,\"%s\"", BOOL_TO_INT(c.remote_query), BOOL_TO_INT(c.remote_config), c.remote_pw);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 7:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JRING=%d,%d,%d,%d", c.ring_number, BOOL_TO_INT(c.ring_signal), BOOL_TO_INT(c.ring_suppress_ri), BOOL_TO_INT(c.ring_fixed));
				modem->writeLine(buf);
				status++;
			}
			break;
		case 8:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JVGR=%d", c.voice_rec);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 9:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JVGT=%d", c.voice_play);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 10:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JVGM=%d", c.voice_micro);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 11:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JVGS=%d", c.voice_speaker);
				modem->writeLine(buf);
				if (c.sync_timer)
					status++;
				else
					status = 14;
			}
			break;
		case 12:
			if (!strcmp(line, "OK")) {
				c.year = QDate::currentDate().year();
				c.month = QDate::currentDate().month();
				c.day = QDate::currentDate().day();
				c.hour = QTime::currentTime().hour();
				c.minute = QTime::currentTime().minute();
				c.second = QTime::currentTime().second();
				sprintf(buf, "AT$JDATE=%d,%d,%d", c.year, c.month, c.day);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 13:
			if (!strcmp(line, "OK")) {
				sprintf(buf, "AT$JTIME=%d,%d,%d", c.hour, c.minute, c.second);
				modem->writeLine(buf);
				status++;
			}
			break;
		case 14:
			if (!strcmp(line, "OK")) {
				setProgress( 15 );
				emit upDone( TRUE );
				reset();
			}
			break;
	}
}


void LoadDlg::fetchTimeout()
{
	KMsgBox::message( 0, i18n("Modem Error"), i18n("Modem response timeout."), KMsgBox::EXCLAMATION);

	reset();
}


void LoadDlg::init()
{
	status = 0;
}


void LoadDlg::reset()
{
	QProgressDialog::reset();

	disconnect( modem, SIGNAL(gotLine(const char *)), this, SLOT(fetchModemLineDown(const char *)) );
	disconnect( modem, SIGNAL(gotLine(const char *)), this, SLOT(fetchModemLineUp(const char *)) );

	modem->close();

	init();
}
