/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "faxview.h"
#include "faxview.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <glob.h>

#include <qglobal.h>
#include <qlistview.h>
#include <qpixmap.h>
#include <qwidget.h>
#include <qpopupmenu.h>
#include <qevent.h>

#include <kapp.h>
#include <kiconloader.h>

#include "global.h"
#include "mlofile.h"
#include "senderaliases.h"



FaxViewItem::FaxViewItem( QListView * parent = 0, const char * name = 0 ) : QListViewItem( parent, name )
{
	new_flag = FALSE;
	sid[0] = 0;
	pg = 0;
	siz = 0;
	file_flag = FALSE;
}


FaxViewItem::~FaxViewItem()
{
}


void FaxViewItem::setNew( bool n, QPixmap & yes, QPixmap & no )
{
	new_flag = n;
	setPixmap( 1, n ? yes : no );
}

void FaxViewItem::setSender( const char * s )
{
	strncpy( sid, s, 20 );
	sid[20] = 0;
}

void FaxViewItem::setPages( int p )
{
	char buff[16];

	pg = p;
	sprintf(buff, "%3d", pg);
	setText( 4, buff );
}

void FaxViewItem::setSize( int s )
{
	char buff[16];

	siz = s;
	sprintf(buff, "%7d", s);
	setText( 5, buff );
}

void FaxViewItem::setFile( bool f, QPixmap & yes, QPixmap & no )
{
	file_flag = f;
	setPixmap( 6, f ? yes : no );
}




FaxView::FaxView( QWidget * parent, const char * name ) : QListView( parent, name )
{
	KIconLoader * kil = mykapp->getIconLoader();

	got_tooltips = FALSE;

	aliases = new SenderAliases( 0, "aliases" );
	CHECK_PTR( aliases );

	yes_pix = kil->loadIcon( "yes.xpm" );
	no_pix = kil->loadIcon( "no.xpm" );

	selectedItem = 0;

	addColumn( i18n("Name") );
	addColumn( i18n("New") );
	addColumn( i18n("Date") );
	addColumn( i18n("Sender") );
	addColumn( i18n("Pages") );
	addColumn( i18n("Size") );
	addColumn( i18n("File") );
	setColumnWidth( 0, 140 );
	setColumnWidth( 1, 40 );
	setColumnWidth( 2, 130 );
	setColumnWidth( 3, 160 );
	setColumnWidth( 4, 50 );
	setColumnWidth( 5, 60 );
	setColumnWidth( 6, 40 );
	setColumnAlignment( 1, AlignVCenter | AlignRight );
	setColumnAlignment( 4, AlignVCenter | AlignHCenter );
	setColumnAlignment( 5, AlignVCenter | AlignRight );
	setMinimumSize( sizeHint() );
	setSorting( 0, FALSE );
	setAllColumnsShowFocus( TRUE );

	context_menu = new QPopupMenu( 0, "contextmenu" );
	CHECK_PTR( context_menu );
	context_menu->insertItem( i18n("&View..."), CONTEXT_ID_VIEW );
	context_menu->insertItem( i18n("&Print..."), CONTEXT_ID_PRINT );
	context_menu->insertItem( i18n("&Mail..."), CONTEXT_ID_MAIL );
	context_menu->insertItem( i18n("De&lete"), CONTEXT_ID_DELETE );
	context_menu->insertItem( i18n("&Sender Alias..."), CONTEXT_ID_ALIAS );

	connect( this, SIGNAL(selectionChanged(QListViewItem *)), SLOT(selectionChangedSlot(QListViewItem *)) );
	connect( this, SIGNAL(doubleClicked(QListViewItem *)), SLOT(doubleClickedSlot(QListViewItem *)) );
	connect( this, SIGNAL(returnPressed(QListViewItem *)), SLOT(doubleClickedSlot(QListViewItem *)) );
	connect( aliases, SIGNAL(applyButtonPressed()), SLOT(refreshAliases()) );
	connect( context_menu, SIGNAL(activated(int)), SLOT(contextMenuSlot(int)) );
}


FaxView::~FaxView()
{
	delete context_menu;
	delete aliases;
}


void FaxView::loadDir()
{
	char buff[80], * p, * name, * time, * sender;
	bool isnew;
	int pages, size;
	FILE * f;
	FaxViewItem * item;

	clear();

	if ((f = fopen(expand_path(KMLOFAX_FAXDIR), "r"))) {
		while (fgets(buff, 80, f))
			if ((buff[0] != '#') && (buff[0] != 0)) {
				if (!(name = strchr(buff, '"')))
					continue;
				name++;
				if (!(p = strchr(name, '"')))
					continue;
				*p = 0;
				do
					p++;
				while (*p == ' ');
				if (*p != ',')
					continue;
				do
					p++;
				while (*p == ' ');
				if ((*p != 'N') && (*p != 'Y'))
					continue;
				isnew = (*p == 'Y');
				do
					p++;
				while (*p == ' ');
				if (*p != ',')
					continue;
				do
					p++;
				while (*p == ' ');
				if (!(time = strchr(p, '"')))
					continue;
				time++;
				if (!(p = strchr(time, '"')))
					continue;
				*p = 0;
				do
					p++;
				while (*p == ' ');
				if (*p != ',')
					continue;
				do
					p++;
				while (*p == ' ');
				if (!(sender = strchr(p, '"')))
					continue;
				sender++;
				if (!(p = strchr(sender, '"')))
					continue;
				*p = 0;
				p++;
				if (sscanf(p, ", %d, %d", &pages, &size) != 2)
					continue;
				item = new FaxViewItem( this );
				CHECK_PTR( item );
				setItem( item, name, isnew, time, sender, pages, size);
			}
		fclose(f);
	}
}


void FaxView::saveDir()
{
	FILE * f;
	FaxViewItem * item;

	if ((f = fopen(expand_path(KMLOFAX_FAXDIR), "w"))) {
		fprintf(f, "# KMLOFax Version %s\n# This file is generated. Do not edit!\n#\n", VERSION);
		item = (FaxViewItem *)firstChild();
		while (item) {
			
			fprintf(f, "\"%s\", %c, \"%s\", \"%s\", %d, %d\n", faxItemName( item ), faxItemNew( item ) ? 'Y' : 'N', item->text(2), item->sender(), atoi(item->text(4)), atoi(item->text(5)));
			item = (FaxViewItem *)item->nextSibling();
		}
		fclose(f);
	}
}


void FaxView::purgeDir()
{
	FaxViewItem * item, * item_buff;

	item = (FaxViewItem *)firstChild();
	while (item) {
		if (!(item->hasFile())) {
			item_buff = item;
			item = (FaxViewItem *)item->nextSibling();
			delete item_buff;
		} else
			item = (FaxViewItem *)item->nextSibling();
	}
}


void FaxView::rebuildDir()
{
	MLOFile f;
	glob_t globbuf;
	FaxViewItem * item;
	int i;
	char time[18], *p;

	clear();

	glob(expand_path("*.FAX"), 0, 0, &globbuf);
	for (i=0; i < (int)globbuf.gl_pathc; i++) {
		f.setName(globbuf.gl_pathv[i]);
		if (!(f.open()))
			continue;
		if (!(p = strrchr(globbuf.gl_pathv[i], '/'))) {
			f.close();
			continue;
		}
		p++;
		if (strlen(p) != 18) {
			f.close();
			continue;
		}
		item = new FaxViewItem( this );
		strncpy(time, p, 4);
		time[4] = '-';
		time[5] = p[4];
		time[6] = p[5];
		time[7] = '-';
		time[8] = p[6];
		time[9] = p[7];
		time[10] = ' ';
		time[11] = ' ';
		time[12] = p[8];
		time[13] = p[9];
		time[14] = ':';
		time[15] = p[10];
		time[16] = p[11];
		time[17] = 0;
		
		setItem( item, p, FALSE, time, f.sender(), f.pages(), f.size() );
		
		f.close();
	}

	globfree(&globbuf);
}


void FaxView::addFaxItem( const char * name, const char * time, int fsize )
{
	FaxViewItem * item;
	char sender[21];
	int pages;
	MLOFile file;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (!strcmp(faxItemName( item ), name))
			break;
	
	if (!item)
		item = new FaxViewItem( this );

	sender[0] = 0;
	pages = 0;

	file.setName( expand_path(name) );
	if (file.open()) {
		strncpy(sender, file.sender(), 20);
		sender[20] = 0;
		pages = file.pages();
		file.close();
	}

	setItem( item, name, TRUE, time, sender, pages, fsize );
}


void FaxView::setFaxItemNew( bool isnew )
{
	setFaxItemNew( (FaxViewItem *)selectedItem, isnew );
}


void FaxView::setFaxItemNew( FaxViewItem * item, bool isnew )
{
	if (item)
		item->setNew( isnew, yes_pix, no_pix );
}


bool FaxView::faxItemNew()
{
	return faxItemNew( (FaxViewItem *)selectedItem );
}


bool FaxView::faxItemNew( FaxViewItem * item )
{
	if (item)
		return item->isNew();

	return FALSE;
}


const char * FaxView::faxItemName()
{
	return faxItemName( (FaxViewItem *)selectedItem );
}


const char * FaxView::faxItemName( FaxViewItem * item )
{
	if (item)
		return item->text(0);

	return 0;
}


const char * FaxView::faxItemSender()
{
	return faxItemSender( (FaxViewItem *)selectedItem );
}


const char * FaxView::faxItemSender( FaxViewItem * item )
{
	if (item)
		return item->sender();

	return 0;
}


const char * FaxView::faxItemAlias()
{
	return faxItemAlias( (FaxViewItem *)selectedItem );
}


const char * FaxView::faxItemAlias( FaxViewItem * item )
{
	if (item)
		return item->text(3);

	return 0;
}


void FaxView::setFaxItemFile( bool fexists )
{
	setFaxItemFile( (FaxViewItem *)selectedItem, fexists );
}


void FaxView::setFaxItemFile( FaxViewItem * item, bool fexists )
{
	if (item)
		item->setFile( fexists, yes_pix, no_pix );
}


bool FaxView::faxItemFile()
{
	return faxItemFile( (FaxViewItem *)selectedItem );
}


bool FaxView::faxItemFile( FaxViewItem * item )
{
	if (item)
		return item->hasFile();

	return FALSE;
}


void FaxView::mousePressEvent( QMouseEvent * e )
{
	FaxViewItem * item;

	if ((e->button() == RightButton) && (e->pos().y() >= 0)) {
		if ((item = (FaxViewItem *)itemAt(e->pos()))) {
			setSelected( item, TRUE );
			context_menu->popup(viewport()->mapToGlobal(e->pos()));
		}
	} else
		QListView::mousePressEvent( e );
}


void FaxView::selectionChangedSlot( QListViewItem * item )
{
	emit hasSelection( (selectedItem = item) != 0 );
}


void FaxView::doubleClickedSlot( QListViewItem * item )
{
	if (faxItemFile( (FaxViewItem *)item ))
		emit viewRequested( (FaxViewItem *)item );
}


void FaxView::refreshAliases()
{
	FaxViewItem * item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		item->setText( 3, aliases->getAlias(item->sender()) );
}


void FaxView::contextMenuSlot( int id )
{
	switch (id) {
		case CONTEXT_ID_VIEW:
			emit contextView();
			break;
		case CONTEXT_ID_PRINT:
			emit contextPrint();
			break;
		case CONTEXT_ID_MAIL:
			emit contextMail();
			break;
		case CONTEXT_ID_DELETE:
			emit contextDelete();
			break;
		case CONTEXT_ID_ALIAS:
			emit contextAlias();
			break;
		default:
			break;
	}
}


void FaxView::setItem( FaxViewItem * item, const char * name, bool isnew, const char * time, const char * sender, int pages, int fsize )
{
	char buff[21], *p;

	item->setText( 0, name );
	setFaxItemNew( item ,isnew );
	item->setText( 2, time );
	
	strcpy(buff, sender);
	p = strtrim(buff);

	item->setSender( p );
	item->setText( 3, aliases->getAlias(p) );
	item->setPages( pages );
	item->setSize( fsize );
	setFaxItemFile( item, !access(expand_path(name), R_OK) );
}
