/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <o.gantz@tu-bs.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "senderaliases.h"
#include "senderaliases.moc"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <stdio.h>
#include <string.h>

#include <qglobal.h>
#include <qdialog.h>
#include <qwidget.h>
#include <qlistview.h>
#include <qframe.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qsize.h>
#include <qtooltip.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "global.h"



SenderAliases::SenderAliases( QWidget * parent, const char * name ) : QDialog ( parent, name )
{
	QVBoxLayout * vbox;
	QHBoxLayout * hbox;
	QGroupBox * gb;
	QGridLayout * grid;
	QLabel * label;
	QSize size;

	gotToolTips = FALSE;

	vbox = new QVBoxLayout( this, 8 );

	gb = new QGroupBox( this );

	grid = new QGridLayout( gb, 3, 2, 8 );

	label = new QLabel( i18n("Sender ID"), gb );
	label->setMinimumSize( label->sizeHint() );
	grid->addWidget( label, 0, 0 );

	sender_id = new QLineEdit( gb );
	sender_id->setEnabled( FALSE );
	sender_id->setMaxLength( 20 );
	sender_id->setFixedHeight( sender_id->sizeHint().height() );
	grid->addWidget( sender_id, 1, 0 );

	label = new QLabel( i18n("Alias"), gb );
	label->setMinimumSize( label->sizeHint() );

	grid->addWidget( label, 0, 1 );

	sender_name = new QLineEdit( gb );
	sender_name->setEnabled( FALSE );
	sender_name->setMaxLength( 40 );
	sender_name->setFixedHeight( sender_name->sizeHint().height() );
	grid->addWidget( sender_name, 1, 1 );

	bapply = new QPushButton( i18n("Apply"), gb );
	bapply->setEnabled( FALSE );
	bapply->setFixedSize( bapply->sizeHint() );

	grid->addWidget( bapply, 2, 1, AlignRight );

	grid->setColStretch( 0, 1 );
	grid->setColStretch( 1, 2 );

	grid->activate();

	vbox->addWidget( gb );
	
	list = new QListView( this, "aliaslist" );
	CHECK_PTR( list );
	list->addColumn( i18n("Sender ID") );
	list->addColumn( i18n("Alias" ) );
	list->setColumnWidth( 0, 200 );
	list->setColumnWidth( 1, 280 );
	list->setFrameStyle( QFrame::Panel | QFrame::Sunken );
	list->setAllColumnsShowFocus( TRUE );

	vbox->addWidget( list, 1 );

	hbox = new QHBoxLayout( 8 );
	vbox->addLayout( hbox );

	badd = new QPushButton( i18n("Add"), this );
	bdel = new QPushButton( i18n("Delete"), this );
	bdel->setEnabled( FALSE );
	bclose = new QPushButton( i18n("Close"), this );

	size = badd->sizeHint();
	size = size.expandedTo( bdel->sizeHint() );
	size = size.expandedTo( bclose->sizeHint() );

	badd->setFixedSize( size );
	bdel->setFixedSize( size );
	bclose->setFixedSize( size );

	hbox->addWidget( badd );
	hbox->addStretch( 1 );
	hbox->addWidget( bdel );
	hbox->addStretch( 1 );
	hbox->addWidget( bclose );

	vbox->activate();

	connect( bapply, SIGNAL(clicked()), SLOT(applyItem()) );
	connect( bdel, SIGNAL(clicked()), SLOT(deleteItem()) );
	connect( list, SIGNAL(currentChanged(QListViewItem *)), SLOT(currentChangedSlot(QListViewItem *)) );
	connect( badd, SIGNAL(clicked()), SLOT(addItem()) );
	connect( bclose, SIGNAL(clicked()), SLOT(hide()) );

	setCaption( i18n("KMLOFax Sender Aliases") );

	loadDir();
}


SenderAliases::~SenderAliases()
{
}


void SenderAliases::loadDir()
{
	char buff[80], * p, * sender, * alias;
	FILE * f;
	QListViewItem * item;

	list->clear();

	if ((f = fopen(expand_path(KMLOFAX_ALIASLIST), "r"))) {
		while (fgets(buff, 80, f))
			if ((buff[0] != '#') && (buff[0] != 0)) {
				if (!(sender = strchr(buff, '"')))
					continue;
				sender++;
				if (!(p = strchr(sender, '"')))
					continue;
				*p = 0;
				p++;

				if (!(alias = strchr(p, '"')))
					continue;
				alias++;
				if (!(p = strchr(alias, '"')))
					continue;
				*p = 0;
				item = new QListViewItem( list );
				item->setText( 0, sender );
				item->setText( 1, alias );
			}
		fclose(f);
	}
}


void SenderAliases::saveDir()
{
	FILE * f;
	QListViewItem * item;

	if ((f = fopen(expand_path(KMLOFAX_ALIASLIST), "w"))) {
		fprintf(f, "# KMLOFax Version %s\n# This file is generated. Do not edit!\n#\n", VERSION);
		item = list->firstChild();
		while (item) {
			fprintf(f, "\"%s\", \"%s\"\n", item->text(0), item->text(1));
			item = item->nextSibling();
		}
		fclose(f);
	}
}


void SenderAliases::addToolTips( bool on )
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add( sender_id, i18n("ID of the sender") );
		QToolTip::add( sender_name, i18n("Alias for the sender") );
		QToolTip::add( bapply, i18n("Apply changes to the list") );
		QToolTip::add( badd, i18n("Add a blank entry to the list") );
		QToolTip::add( bdel, i18n("Delete entry from the list") );
		QToolTip::add( bclose, i18n("Close this dialog") );
	}
	else {
		QToolTip::remove( sender_id );
		QToolTip::remove( sender_name );
		QToolTip::remove( bapply );
		QToolTip::remove( badd );
		QToolTip::remove( bdel );
		QToolTip::remove( bclose );
	}
}	


const char * SenderAliases::getAlias( const char * id )
{
	QListViewItem * item;

	item = list->firstChild();
	while (item) {
		if (!strcmp(item->text(0), id))
			return item->text(1);
		item = item->nextSibling();
	}

	return id;
}


void SenderAliases::setAlias( const char * id )
{
	QListViewItem * item;

	item = list->firstChild();
	while (item) {
		if (!strcmp(item->text(0), id))
			break;
		item = item->nextSibling();
	}

	if (!item) {
		item = new QListViewItem( list );
		item->setText( 0, id );
	}

	list->setSelected( item, TRUE );

	if (isVisible())
		raise();
	else
		show();
}


void SenderAliases::addItem()
{
	QListViewItem * item;

	item = new QListViewItem( list );

	list->setSelected( item, TRUE );
}


void SenderAliases::applyItem()
{
	QListViewItem * item;
	char buff[41];

	if ((item = list->currentItem())) {
		strcpy(buff, sender_id->text());
		item->setText( 0, replace_quotes(buff) );
		strcpy(buff, sender_name->text());
		item->setText( 1, replace_quotes(buff) );
		emit applyButtonPressed();
		saveDir();
	}
}


void SenderAliases::deleteItem()
{
	QListViewItem * item;

	if ((item = list->currentItem())) {
		if (KMsgBox::yesNo( 0, i18n("Question"), i18n("Really delete entry?"), KMsgBox::QUESTION ) == 1) {
			delete item;
			sender_id->clear();
			sender_name->clear();
			sender_id->setEnabled( FALSE );
			sender_name->setEnabled( FALSE );
			bapply->setEnabled( FALSE );
			bdel->setEnabled( FALSE );
			emit applyButtonPressed();
			saveDir();
		}
		
	}
}


void SenderAliases::currentChangedSlot( QListViewItem * item )
{
	if (item) {
		sender_id->setText( item->text(0) );
		sender_name->setText( item->text(1) );
		entry_changed = FALSE;
		sender_id->setEnabled( TRUE );
		sender_name->setEnabled( TRUE );
		bapply->setEnabled( TRUE );
		bdel->setEnabled( TRUE );
	}
}
