/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdio.h>
#include <unistd.h>

#include <qglobal.h>
#include <qkeycode.h>
#include <qpixmap.h>
#include <qsize.h>

#include <kiconloader.h>
#include <kmsgbox.h>

#include "kmlofax.h"
#include "preferences.h"
#include "faxview.h"
#include "downloaddlg.h"
#include "preview.h"
#include "printdlg.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "senderaliases.h"
#include "global.h"



KMLOFaxApp::KMLOFaxApp(int& argc, char **argv, const QString& rAppName) : KApplication(argc, argv, rAppName)
{
}


KMLOFaxApp::~KMLOFaxApp()
{
}


bool KMLOFaxApp::x11EventFilter(XEvent *ev)
{
	if (KApplication::x11EventFilter(ev))
		return true;
		
	if (fw)
		fw->x11EventFilter(ev);

	return false;
}



KMLOFax::KMLOFax(const char *name) : KTMainWindow(name)
{
	QString s;
	
	config = kapp->getConfig();

	initMenuBar();
	initToolBar();
	initStatusBar();
	initKeyAccel();
	
	prefs = new Preferences(0, "preferences");
	CHECK_PTR(prefs);
	prefs->reset();

	senderaliases = new SenderAliases(0, "senderaliases");
	CHECK_PTR(senderaliases);

	view = new FaxView(this, "faxview");
	CHECK_PTR(view);
	setView(view);
	
	downloaddlg = new DownloadDlg(0, "downloaddlg");
	CHECK_PTR(downloaddlg);
	preview = new Preview("preview");
	CHECK_PTR(preview);
	printdlg = new PrintDlg(0, "printdlg");
	CHECK_PTR(printdlg);
	exportdlg = new ExportDlg(0, "exportdlg");
	CHECK_PTR(exportdlg);
	maildlg = new MailDlg(0, "maildlg");
	CHECK_PTR(maildlg);

	readOptions();

	newPrefs();

	gotSelection(false);

	connect(prefs, SIGNAL(apply_clicked()), SLOT(newPrefs()));

	connect(downloaddlg, SIGNAL(message(const QString&)), SLOT(slotStatusHelpMsg(const QString&)));
	connect(downloaddlg, SIGNAL(gotFile(const QString&)), SLOT(fetchFile(const QString&)));
	connect(downloaddlg, SIGNAL(done()), SLOT(fetchDone()));

	connect(view, SIGNAL(hasSelection(bool)), SLOT(gotSelection(bool)));
	connect(view, SIGNAL(command(int)), SLOT(commandCallback(int)));

	show();

	s.sprintf(i18n("%d File(s) in directory."), view->loadDir());
	slotStatusHelpMsg(s);
}


KMLOFax::~KMLOFax()
{
	delete maildlg;
	delete exportdlg;
	delete printdlg;
	delete preview;
	delete downloaddlg;
	delete view;
	delete senderaliases;
	delete prefs;	
}


void KMLOFax::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void KMLOFax::readOptions()
{
	config->setGroup("MainWindow Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	optionsMenu->setItemChecked(ID_OPTIONS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		enableToolBar(KToolBar::Hide);
	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	optionsMenu->setItemChecked(ID_OPTIONS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		enableStatusBar(KStatusBar::Hide);
	menuBar()->setMenuBarPos((KMenuBar::menuPosition)config->readNumEntry("MenuBarPos", (int)KMenuBar::Top));
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	QSize geoSize(650, 400);
	resize(config->readSizeEntry("Geometry", &geoSize));

	view->readOptions();
}


void KMLOFax::saveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("MenuBarPos", (int)menuBar()->menuBarPos());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());

	view->saveOptions();
}


bool KMLOFax::queryClose()
{
	view->saveDir();
	saveOptions();

	return true;
}


void KMLOFax::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "mainfilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(Icon("filednl.xpm"), i18n("&Download"), ID_FILE_DOWNLOAD);
	fileMenu->insertSeparator();
	fileMenu->insertItem(Icon("print_preview.xpm"), i18n("&View..."), ID_FILE_VIEW);
	fileMenu->insertItem(Icon("fileprint.xpm"), i18n("&Print..."), ID_FILE_PRINT);
	fileMenu->insertItem(Icon("filefloppy.xpm"), i18n("&Export..."), ID_FILE_EXPORT);
	fileMenu->insertItem(Icon("filemail.xpm"), i18n("&Mail..."), ID_FILE_MAIL);
	fileMenu->insertItem(Icon("filedel.xpm"), i18n("De&lete"), ID_FILE_DELETE);
	fileMenu->insertItem(i18n("Sender &Alias..."), ID_FILE_SENDER_ALIAS);
	fileMenu->insertSeparator();
	fileMenu->insertItem(i18n("P&urge Directory"), ID_FILE_PURGE_DIRECTORY);
	fileMenu->insertItem(i18n("&Rebuild Directory"), ID_FILE_REBUILD_DIRECTORY);
	fileMenu->insertSeparator();
	fileMenu->insertItem(Icon("exit.xpm"), i18n("&Quit"), ID_FILE_QUIT);

	optionsMenu = new QPopupMenu(0, "mainoptionsmenu");
	CHECK_PTR(optionsMenu);
	optionsMenu->setCheckable(true);
	optionsMenu->insertItem(Icon("tick.xpm"), i18n("&Preferences..."), ID_OPTIONS_PREFERENCES);
	optionsMenu->insertItem(i18n("Sender &Aliases..."), ID_OPTIONS_SENDER_ALIASES);
	optionsMenu->insertSeparator();
	optionsMenu->insertItem(i18n("Show &Toolbar"), ID_OPTIONS_TOOLBAR);
	optionsMenu->setItemChecked(ID_OPTIONS_TOOLBAR, true);
	optionsMenu->insertItem(i18n("Show &Statusbar"), ID_OPTIONS_STATUSBAR);
	optionsMenu->setItemChecked(ID_OPTIONS_STATUSBAR, true);

	QString aboutstr = QString("KMLOFax ") + VERSION + QString(i18n(
		"\nCopyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>\n\n" \
		"A utility to process facsimile received with the\n" \
		"ELSA MicroLink(tm) Office modem.\n\n" \
		"This program is free software and is distributed under the terms\n" \
		"of the GNU General Public License.\n\n" \
		"ELSA and MicroLink are trademarks of ELSA AG, Aachen."));
	
	helpMenu = kapp->getHelpMenu(true, aboutstr);
	
	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Options"), optionsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu);

	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	
	connect(optionsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void KMLOFax::initToolBar()
{
	toolBar()->insertButton(Icon("filednl.xpm"), ID_FILE_DOWNLOAD, true, i18n("Download Facsimile"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(Icon("print_preview.xpm"), ID_FILE_VIEW, true, i18n("View Facsimile"));
	toolBar()->insertButton(Icon("fileprint.xpm"), ID_FILE_PRINT, true, i18n("Print Facsimile"));
	toolBar()->insertButton(Icon("filefloppy.xpm"), ID_FILE_EXPORT, true, i18n("Export Facsimile"));
	toolBar()->insertButton(Icon("filemail.xpm"), ID_FILE_MAIL, true, i18n("Mail Facsimile"));
	toolBar()->insertButton(Icon("delete.xpm"), ID_FILE_DELETE, true, i18n("Delete Facsimile"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(Icon("tick.xpm"), ID_OPTIONS_PREFERENCES, true, i18n("Preferences"));
	toolBar()->insertButton(Icon("help.xpm"), ID_HELP_CONTENTS, SIGNAL(clicked()), kapp, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->insertButton(Icon("exit.xpm"), ID_FILE_QUIT, true, i18n("Quit"));
	toolBar()->alignItemRight(ID_FILE_QUIT, true);

	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void KMLOFax::initStatusBar()
{
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG);
}


void KMLOFax::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Download Facsimile"), "DownloadFiles", CTRL+Key_D);
	
	keyAccel->insertItem(i18n("View Facsimile"), "ViewFile", CTRL+Key_V);
	keyAccel->insertItem(i18n("Export Facsimile"), "ExportFile", CTRL+Key_E);
	keyAccel->insertItem(i18n("Mail Facsimile"), "MailFile", CTRL+Key_M);
	keyAccel->insertItem(i18n("Delete Facsimile"), "DeleteFile", Key_Delete);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);

	keyAccel->readSettings();
	
	keyAccel->connectItem("DownloadFiles", this, SLOT(slotFileDownload()));
	keyAccel->connectItem("ViewFile", this, SLOT(slotFileView()));
	keyAccel->connectItem(KAccel::Print, this, SLOT(slotFilePrint()));
	keyAccel->connectItem("ExportFile", this, SLOT(slotFileExport()));
	keyAccel->connectItem("MailFile", this, SLOT(slotFileMail()));
	keyAccel->connectItem("DeleteFile", this, SLOT(slotFileDelete()));
	keyAccel->connectItem(KAccel::Quit, this, SLOT(slotFileQuit()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotOptionsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotOptionsShowStatusbar()));
	
	keyAccel->connectItem(KAccel::Help, kapp, SLOT(appHelpActivated()));

	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DOWNLOAD, "DownloadFiles");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_VIEW, "ViewFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_PRINT, KAccel::Print);
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_EXPORT, "ExportFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_MAIL, "MailFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_DELETE, "DeleteFile");
	keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KAccel::Quit);

	keyAccel->changeMenuAccel(optionsMenu, ID_OPTIONS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(optionsMenu, ID_OPTIONS_STATUSBAR, "ShowStatusbar");
}


void KMLOFax::commandCallback(int id_)
{
	switch (id_) {
		case ID_FILE_DOWNLOAD:
			slotFileDownload();
			break;
		case ID_FILE_VIEW:
			slotFileView();
			break;
		case ID_FILE_PRINT:
			slotFilePrint();
			break;
		case ID_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_FILE_MAIL:
			slotFileMail();
			break;
		case ID_FILE_DELETE:
			slotFileDelete();
			break;
		case ID_FILE_SENDER_ALIAS:
			slotFileSenderAlias();
			break;
		case ID_FILE_PURGE_DIRECTORY:
			slotFilePurgeDir();
			break;
		case ID_FILE_REBUILD_DIRECTORY:
			slotFileRebuildDir();
			break;
		case ID_FILE_QUIT:
			slotFileQuit();
			break;
		case ID_OPTIONS_PREFERENCES:
			slotOptionsPreferences();
			break;
		case ID_OPTIONS_SENDER_ALIASES:
			slotOptionsSenderAliases();
			break;
		case ID_OPTIONS_TOOLBAR:
			slotOptionsShowToolbar();
			break;
		case ID_OPTIONS_STATUSBAR:
			slotOptionsShowStatusbar();
			break;
		default:
			break;
	}
}

		
void KMLOFax::slotFileDownload()
{
	if (downloaddlg->startDownload()) {
		enableCommand(ID_FILE_DOWNLOAD, false);
		slotStatusHelpMsg(i18n("Looking for new facsimile..."));
	}
}


void KMLOFax::slotFileView()
{
	if (view->selectedFaxFile()) {
		preview->showFax(view->selectedFaxName(), view->selectedFaxAlias());
		view->setSelectedFaxNew(false);
	}
}


void KMLOFax::slotFilePrint()
{
	if (view->selectedFaxFile()) {
		printdlg->printFax(view->selectedFaxName(), view->selectedFaxAlias());
		view->setSelectedFaxNew(false);
	}
}


void KMLOFax::slotFileExport()
{
	if (view->selectedFaxFile()) {
		exportdlg->exportFax(view->selectedFaxName(), view->selectedFaxAlias());
	}
}


void KMLOFax::slotFileMail()
{
	if (view->selectedFaxFile()) {
		maildlg->mailFax(view->selectedFaxName(), view->selectedFaxAlias());
	}
}


void KMLOFax::slotFileDelete()
{
	if (view->selectedFaxFile()) {
		if (view->selectedFaxNew()) {
			KMsgBox::message(0, i18n("File Error"), i18n("Cannot delete new facsimile file.\nPlease view or print facsimile first."), KMsgBox::EXCLAMATION);
			return;
		}

		if (KMsgBox::yesNo(0, i18n("Question"), i18n("Really delete file?"), KMsgBox::QUESTION) == 1) {
			unlink(expandPath(view->selectedFaxName()));
			view->setSelectedFaxFile(false);
			gotSelection(true);
		}
	}
}


void KMLOFax::slotFileSenderAlias()
{
	if (view->selectedFaxSender())
		senderaliases->setAlias(view->selectedFaxSender());
}


void KMLOFax::slotFilePurgeDir()
{
	if (KMsgBox::yesNo(0, i18n("Question"), i18n("Really remove entries with deleted files?"), KMsgBox::QUESTION) == 1)
		view->purgeDir();
}


void KMLOFax::slotFileRebuildDir()
{
	if (KMsgBox::yesNo(0, i18n("Question"), i18n("Really discard current directory and scan for existing facsimile files?"), KMsgBox::QUESTION) == 1) {
		QString s;
		s.sprintf(i18n("%d File(s) found."), view->rebuildDir());
		slotStatusHelpMsg(s);
	}
}


void KMLOFax::slotFileQuit()
{
	close();
}


void KMLOFax::slotOptionsPreferences()
{
	if (prefs->isVisible())
		prefs->raise();
	else
		prefs->show();
}


void KMLOFax::slotOptionsSenderAliases()
{
	if (senderaliases->isVisible())
		senderaliases->raise();
	else
		senderaliases->show();
}


void KMLOFax::slotOptionsShowToolbar()
{
	bool toolBarOn = !optionsMenu->isItemChecked(ID_OPTIONS_TOOLBAR);

	menuBar()->setItemChecked(ID_OPTIONS_TOOLBAR, toolBarOn);
	enableToolBar(toolBarOn ? KToolBar::Show : KToolBar::Hide);
}


void KMLOFax::slotOptionsShowStatusbar()
{
	bool statusBarOn = !optionsMenu->isItemChecked(ID_OPTIONS_STATUSBAR);

	menuBar()->setItemChecked(ID_OPTIONS_STATUSBAR, statusBarOn);
	enableStatusBar(statusBarOn ? KStatusBar::Show : KStatusBar::Hide);
}


void KMLOFax::slotStatusMsg(const QString& text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KMLOFax::slotStatusHelpMsg(const QString& text)
{
	statusBar()->message(text, 2000);
}



void KMLOFax::newPrefs()
{
	bool tooltips;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	tooltips = config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS);

	prefs->addToolTips(tooltips);
	senderaliases->addToolTips(tooltips);
	printdlg->addToolTips(tooltips);
	exportdlg->addToolTips(tooltips);
	maildlg->addToolTips(tooltips);
}


void KMLOFax::gotSelection(bool on)
{
	view->gotSelection(on);
	
	enableCommand(ID_FILE_SENDER_ALIAS, on);

	if (on)
		on = view->selectedFaxFile();

	enableCommand(ID_FILE_VIEW, on);
	enableCommand(ID_FILE_PRINT, on);
	enableCommand(ID_FILE_EXPORT, on);
	enableCommand(ID_FILE_MAIL, on);
	enableCommand(ID_FILE_DELETE, on);
}


void KMLOFax::fetchFile(const QString& name)
{
	QString s;
	
	slotStatusHelpMsg(s.sprintf(i18n("Received file %s."), (const char *)name));
	view->addFaxItem(name);	
}


void KMLOFax::fetchDone()
{
	enableCommand(ID_FILE_DOWNLOAD, true);
	view->saveDir();
}
