/***************************************************************************
                          cddb_fill.cpp  -  CDDB client
                             -------------------                                         

    version              :                                   
    begin                : Sun Jan 10 1999                                           
    copyright            : (C) 1999 by Denis Oliver Kropp                         
    email                : dok@fischlustig.de                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

// alter mach kaffe

#include "cddb_fill.h"
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include "../config.h"

#ifdef HAVE_LINUX_CDROM_H
	#include <linux/cdrom.h>
#endif

#ifdef HAVE_LINUX_UCDROM_H
	#include <linux/ucdrom.h>
#endif

#include <kmsgbox.h>

#define READ_WRITE	0x01
#define READ_ONLY	0x02

int code = 0;		/* Return value for sending data to the server */
char cddb_msg[255];	/* Return message of server info */
int sock = 0;		/* Descriptor for our socket */
int sock_mode = 0;	/* Server read/write status */
FILE *sk;		/* Stream descriptor for our socket */

Track_Info::Track_Info( int _track, int _min, int _sec, int _frame )
{
	track = _track;
	min = _min;
	sec = _sec;
	length = min * 60 + sec;
	start = length * 75 + _frame;
}

CD_Info::CD_Info()
{
	trk.setAutoDelete(true);
}

CDDB_Fill::CDDB_Fill( KoverFile* _kover_file ) : QObject() 
{
	kover_file = _kover_file;
	cd_fd = -1;
}

CDDB_Fill::~CDDB_Fill()
{
}

bool CDDB_Fill::execute()
{
	if (!openCD())
	{
		return false;
	}
	if (!readTOC())
	{
		closeCD();
		return false;
	}

	
      if (cddb_connect(CDDB_HOST, CDDB_PORT))
	{
	  if (cddb_login())
	    {
	      cddb_query();
	      cddb_disconnect();
	    }
 	}
	
	
	QString tracks, contents;
	kover_file->setTitle( cdinfo.artist + "\n" + cdinfo.cdname );
	for (int i=0; i<cdinfo.ntracks; i++)
	{
		tracks.sprintf( "%d. ", i+1 );
		tracks.append( cdinfo.trk.at(i)->songname );
		tracks.append( "\n" );
		contents.append( tracks );
	}
	kover_file->setContents( contents );
	
	closeCD();
	
	return true;
}

void CDDB_Fill::cdInfo()
{
	QString str;
	
	str.sprintf( "CD contains %d tracks, total time is %d:%02d, the magic number is 0x%x", cdinfo.ntracks, cdinfo.length/60, cdinfo.length%60, cdinfo.cddb_id );
	emit statusText( str );
}

int CDDB_Fill::openCD()
{
	int ds;
	
	if (cd_fd != -1)
	{
		emit statusText( "Internal error: Filedescriptor is not -1, already opened?" );
		return false;
	}
	
	if ((cd_fd = open(CD_DEVICE, O_RDONLY | O_NONBLOCK)) < 0)
	{
		if (errno == EACCES)
		{
			emit statusText( "You dont have permission to read from /dev/cdrom!" );
		} else
		if (errno == ENOMEDIUM)
		{
			emit statusText( "Theres no medium in /dev/cdrom!" );
		} else
		if (errno == EBUSY)
		{
			emit statusText( "/dev/cdrom is busy!" );
		} else
		{
			emit statusText( "Unknown error while opening /dev/cdrom!" );
		}
		return false;
	}

	ds = ioctl(cd_fd, CDROM_DISC_STATUS);
	if (ds != CDS_AUDIO)
	{
		emit statusText( "Theres no audio cd in /dev/cdrom! (ignoring)" );
	}
	
	return true;
}

void CDDB_Fill::closeCD()
{
	if (cd_fd != -1)
	{
		close(cd_fd);
		cd_fd = -1;
	}
}

bool CDDB_Fill::readTOC()
{
	cdrom_tochdr	hdr;
	cdrom_tocentry	entry;
	int			i, pos;

	if (cd_fd < 0)
	{
		emit statusText( "Internal error: Filedescriptor is -1, not opened?" );
		return false;
	}

	emit statusText( "Reading table of contents..." );
	
	if (ioctl(cd_fd, CDROMREADTOCHDR, &hdr))
	{
		emit statusText( "Error while reading table of contents!" );
		return false;
	}

	cdinfo.artist.setStr("Artist");
	cdinfo.cdname.setStr("Title");
	cdinfo.length = 0;
	cdinfo.ntracks = hdr.cdth_trk1;

	cdinfo.trk.clear();
	
	for (i = 0; i <= cdinfo.ntracks; i++)
	{
		if (i == cdinfo.ntracks)
			entry.cdte_track = CDROM_LEADOUT;
		else
			entry.cdte_track = i + 1;
		entry.cdte_format = CDROM_MSF;
		if (ioctl(cd_fd, CDROMREADTOCENTRY, &entry))
		{
			emit statusText( "Error while reading TOC entry!" );
			return false;
		}

		cdinfo.trk.append( new Track_Info(i+1, entry.cdte_addr.msf.minute, entry.cdte_addr.msf.second, entry.cdte_addr.msf.frame) );
	}

	pos = cdinfo.trk.first()->length;

	for (i = 0; i < cdinfo.ntracks; i++)
	{
		cdinfo.trk.at(i)->length = cdinfo.trk.at(i+1)->length - pos;
		pos = cdinfo.trk.at(i+1)->length;
	}

	cdinfo.length = cdinfo.trk.last()->length;
        
	cdinfo.cddb_id = calcID();
	
	emit statusText( "Table of contents successfully read" );
	return true;
}


int CDDB_Fill::cddb_sum(int n)
{
   char    buf[12],
           *p;
   int     ret = 0;

   sprintf(buf, "%lu", (unsigned long) n);
   for (p = buf; *p != '\0'; p++)
      ret += (*p - '0');
                        
   return (ret);
}

unsigned long CDDB_Fill::calcID()
{
   int     i,
           t = 0,
           n = 0;
           
   for (i = 0; i < cdinfo.ntracks; i++)
   {
      n += cddb_sum((cdinfo.trk.at(i)->min * 60) + cdinfo.trk.at(i)->sec);
   }
                        
   t = ((cdinfo.trk.last()->min * 60) + cdinfo.trk.last()->sec) -
       ((cdinfo.trk.first()->min * 60) + cdinfo.trk.first()->sec);
                                             
   return ((n % 0xff) << 24 | t << 8 | cdinfo.ntracks);
}


/*      
      if (cddb_connect(CDDB_HOST, CDDB_PORT))
	{
	  if (cddb_login())
	    {
	      cddb_query();
	      cddb_disconnect();
	    }
	  kover_file->setTitle( QString(thiscd.artist) + QString(" - ") + QString(thiscd.cdname) );
	  kover_file->setContents( QString("") );
	  for (i=0; i<thiscd.ntracks; i++)
	    {
	      tracks.sprintf( "%d. %s", i+1, thiscd.trk[i].songname );
	      contents_edit->append( tracks );
	    }
 	}
*/





///////////////////////////////////////////////

void CDDB_Fill::parse_trails(char *ss)
{
   unsigned int i;
   
   for (i=0; i<strlen(ss); i++)
   {
      if (ss[i] == '\r' || ss[i] == '\n')
        ss[i] = 0;
   }
}

/**** START GENERAL NETWORK CODE *****************************************/

void CDDB_Fill::cddb_code()
{
   char s[255];


   code = 0;

   if (fgets(s, 255, sk) == NULL)
     return;
   
   s[3] = 0;
   
   code = atoi(s);
   strcpy(cddb_msg, &s[4]);
}  

int CDDB_Fill::cddb_connect(char *host, int port)
{
   sockaddr_in sin;
   hostent *h;


   emit statusText(QString("Connecting to ") + QString(host) + "..." );

   emit statusText( "Resolving host..." );

   if ((h = gethostbyname(host)) == NULL)
     return(0);
 
   bcopy(h->h_addr, (char *) &sin.sin_addr, h->h_length);
                    
   sin.sin_family = h->h_addrtype;
   sin.sin_port   = htons(port);

   emit statusText( "Connecting to CDDB server..." );
                            
   if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
     return(0);
                                               
   if (::connect(sock, (sockaddr*)&sin, sizeof(sin)) < 0)
     return(0);
   
   sk = fdopen(sock, "r+");
   if (sk == NULL)
   {
     close(sock);
     return(0);
   }
    
   cddb_code();
   if (code == 200)
     sock_mode = READ_WRITE;
   else {
     if (code == 201)
       sock_mode = READ_ONLY;
     else {
       emit statusText("Connection refused after connect: " + QString(code));
       fflush(stdout);
       getchar();
       return(0);
     }
   }
   
   return(1);   
}

void CDDB_Fill::cddb_disconnect()
{
   SEND("quit");
   close(sock);
   fclose(sk);
}

int CDDB_Fill::cddb_login()
{
   char *hostname, *logname, s[255];
   
    
   emit statusText( "Logging in..." );
     
   hostname = getenv("HOSTNAME");
   logname  = getenv("LOGNAME");
   if (hostname == NULL || logname == NULL)
   {
     return(0);
   }
   
     
   sprintf(s,"cddb hello %s %s %s %s", logname, hostname, "Kover", "0.31");
   SEND(s);
   
   cddb_code();
   if (code == 200 || code == 402)
     return(1);
   else {
//     emit statusText( "CDDB access denied (code %d).\n", code );
     getchar();
   }
   
   return(0);
}

/**** END GENERAL NETWORK CODE *******************************************/

/**** START CDDB INTENSIVE ROUTINES **************************************/

/* Sends query to server -- this is the first thing to be done */
void CDDB_Fill::cddb_query()
{
   char s[255], s1[81], *ss;
   int i;
  
   emit statusText( "Querying database..." );
   
   sprintf(s,"cddb query %08x %d ",(unsigned int) cdinfo.cddb_id, cdinfo.ntracks);
   for (i=0; i<cdinfo.ntracks; i++)
   {
      sprintf(s1,"%d ", cdinfo.trk.at(i)->start);
      strcat(s, s1);
   }
   
   sprintf(s1,"%d", cdinfo.length);
   strcat(s, s1);
     
   SEND(s);
   cddb_code();
   
   switch(code)
   {
     case 200:  /* Success, get the catagory ID */
       ss = strchr(cddb_msg, 32);
       *ss = 0;
       cdinfo.catagory = cddb_msg;
      
       break;
     default:
       
       cddb_msg[strlen(cddb_msg)-1] = 0;
       
       printf( "(%02d): %s\n", code, cddb_msg );
       
       return;
   }
  
   cddb_readcdinfo(sk);
}
 
void CDDB_Fill::cddb_readcdinfo(FILE *desc)
{
   char s[255], *ss;
   int t;
   
   if (code != 69)
   {
     if (code == 200)  // cddb_query was a success, request info
     {
       emit statusText( "Downloading CD info..." );
       
       sprintf(s,"cddb read %s %08x",(const char*)cdinfo.catagory, (unsigned int) cdinfo.cddb_id);
       SEND(s);   
       cddb_code();
 
       if (code != 210) 
       {   
         printf("(%d): %s",code,cddb_msg);
         return;
       }
     }
   }
   
   s[0] = 0;
   while (strncmp(s,".", 1)!=0)
   {
     if (!fgets(s, 255, desc))
       break;
     
     if ((ss=strstr(s, "DTITLE")) != NULL)
     { 
       ss += 7;
       ss[strlen(ss)-2] = 0;
       cdinfo.cdnames = ss;
              
       ss = strchr(s,'/');
       *ss = 0;
       cdinfo.artist = s+7;
       cdinfo.cdname = ss+2;
     }
     
       
     
     if ((ss=strstr(s, "TTITLE")) == NULL)
       continue;
     
     /* Yeah, yeah. Cheap hack, but it's guarenteed to work! :)
      * The following just hacks the returned track name into the variable 
      */
     
     ss += 6;
     t = atoi(ss);
     if (t < 100)
       ss += t < 10 ? 2 : 3;
     else
       ss += 4;    
       
     parse_trails(ss);
     
     if (cdinfo.trk.at(t)->songname.length())
     {       
       cdinfo.trk.at(t)->songname += ss;
     } else {
       cdinfo.trk.at(t)->songname = ss;
     }
   }
   
}

