    /*

    gda.C  for  ktsp-0.1.0

    Copyright (C) 1999 Uwe Thiem
                       uwe@uwix.alt.na

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */


#include <iostream.h>
#include <kapp.h>

#include "gda.h"



// Constructor and destructor do nothing. We could as well have left them
// out.
Gda::Gda()
  {
  stopped = false;
  _confFactor = 1.0;
  }


Gda::~Gda()
  {
  }


void Gda::startGda( DBL scaleFactor, DBL qual, int loopLength )
  {
  DBL diff;

  // There's no theoretical reason for these two values. They are
  // completely heuristic.
  _rainFactor = ( 1.0 / 500.0 ) * _confFactor;
  _rainMin = ( 0.01 * scaleFactor / 500.0 ) * _confFactor;

  _quality = qual;
  _loopLength = loopLength;
  _waterLevel = 1.2 * _quality;
  _actions = _accepted = _better = _rejected = 0;
  _finished = false;

  // We call afterLoop() one time after having initialized all relevant
  // internal variables in order to allow the GUI to display the initial
  // information.
  afterLoop();

  // _finished will be set to "true" as soon as one whole loop has been
  // performed without the occurance of one single accepted change. In that
  // case we stop GDA and declare the actual run as finished.
  while ( !_finished )
    {
    // Before each loop we set the internal state to "unchanged". If at least
    // one change is accepted _changed is set to "true".
    _changed = false;
    for ( int i = 0; i < _loopLength; i++ )
      {
      // nextAction() returns a quality difference "old - new"
      diff = nextAction();
      evaluateAction( _quality - diff );
      if ( stopped ) return;
      }
    loopComplete();
    }
  }


void Gda::stop()
  {
  stopped = true;
  }


void Gda::loopComplete()
  {
  // During one loop we deal only with quality differences. To allow GDA
  // to adjust all the little inaccuracies we calculate the real quality
  // once after each loop.
  _quality = calcQuality();
  // Let's allow KApplication to process some events in order to keep the
  // GUI "responsive".
  kapp->processEvents();
  // We allow the GUI to catch up with the internal state.
  afterLoop();
  // If no changes have been accepted during the last loop we declare
  // GDA as finished.
  if ( !_changed )
    {
    _finished = true;
    }
  }




void Gda::afterLoop()
  {
  // Emit signals with the internal state of GDA to allow the GUI to
  // catch up.
  emit quality( _quality );
  emit actions( _actions );
  emit accepted( _accepted );
  emit better( _better );
  emit rejected( _rejected );
  emit worseAccepted( _accepted - _rejected );
  }



void Gda::evaluateAction( DBL qual )
  {
  // Increase number of performaed actions.
  _actions++;
  if ( qual >= _waterLevel )
    {
    // In case we reject this change we just increment the number of
    // rejected actions. Nothing else needs to be done.
    _rejected++;
    }
  else
    {
    if ( qual < _quality )
      {
      // Here we got a change that really improves the quality.
      _better++;
      }
    // Whether the accepted change was really better doesn't matter here.
    // We just increment the number of accepted changes.
    _accepted++;
    // We accept the new quality.
    _quality = qual;
    // Now it has to rain.
    adjustWaterLevel();
    // We accepted the change. Therefore we have to change the internal
    // state to "changed".
    _changed = true;
    // Here we tell the derived class that we accepted the change. The
    // derived class must decide what to do with this information.
    acceptAction();
    }
  }



void Gda::adjustWaterLevel()
  {
  DBL diff = ( _waterLevel - _quality ) * _rainFactor;
  // The rain speed depends on the difference betwwen quality and
  // water level. If this difference becomes too small we fall back
  // to a minimum rain speed.

  // Despite the fact that we talk about "rain speed" and rising water
  // in fact we are looking for a minimum. We don't really look for the
  // highest mountain in our "space of solutions" but for the deepest
  // valley.
  if ( diff > _rainMin )
    {
    _waterLevel -= diff;
    }
  else
    {
    _waterLevel -= _rainMin;
    }
  }



int Gda::confFactor()
  {
  return (int) ( _confFactor * 50.0 );
  }


void Gda::setConfFactor( int factor )
  {
  _confFactor = factor / 50.0;
  }
