    /*

    ktsp.C  for  ktsp-0.1.0

    Copyright (C) 1999 Uwe Thiem
                       uwe@kde.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream.h>
#include <qstring.h>
#include <qfile.h>
#include <qlayout.h>
#include <kiconloader.h>
#include <kconfig.h>
#include <kfiledialog.h>
#include <kmsgbox.h>
#include <kfm.h>

#include "ktsp.h"
#include "tspgda.h"
#include "tspdialog.h"
#include "graphicdialog.h"


KTsp::KTsp( const char *fileName )
       : KTMainWindow()
  {
  // create the main widget of Ktsp
  w = new QWidget( this, "Main Widget" );
  // check whether the main widget could be created
  CHECK_PTR( w );
  setMinimumSize( size() );

  // KTMainWindow::setView() tells KTMW that w is our main widget
  setView( w );

  // Make sure dispay points to nothing.
  display = 0;

  // set up a menubar
  setupMenubar();

  // set up a toolbar
  setupToolbar();

  // set up a statusbar
  setupStatusbar();

  // set up TSP
  setupTsp();

  // Read configuration
  readConfig();

  // set up view
  setupView();

  // set up drag and drop
  setupDND();

  dirty = false;
  finished = false;
  cannotOpen = false;
  noClearDisplay = false;
  _numberOfPoints = 0;
  performedRuns = 0;
  connect( tsp, SIGNAL( bestQuality( double ) ),
           this, SLOT( slotBestQuality( double ) ) );
  if ( fileName )
    {
    loadFileName = fileName;
    timer = new QTimer( this, "File Timer" );
    CHECK_PTR( timer );
    connect( timer, SIGNAL( timeout() ),
             this, SLOT( slotForcedLoad() ) );
    timer->start( 500, true );
    }
  }


KTsp::~KTsp()
  {
  }


void KTsp::setupMenubar()
  {
  // Create an KAccel instance
  accel = new KAccel( this );

  // Each entry in a menubar is of the type QPopupMenu. So we need at least
  // one of them.
  QPopupMenu *p = new QPopupMenu;
  CHECK_PTR( p );
  fileMenu = p;

  // Create a file menu, connect the accelerator key to our slot, and
  // tell KAccel to show the key in the menu.
  new_id = p->insertItem( i18n( "&New TSP" ), this, SLOT( slotStart() ) );
  accel->connectItem( KAccel::New, this, SLOT( slotStart() ) );
  accel->changeMenuAccel( p, new_id, KAccel::New );
  open_id = p->insertItem( i18n( "&Open..." ), this, SLOT( slotOpen() ) );
  accel->connectItem( KAccel::Open, this, SLOT( slotOpen() ) );
  accel->changeMenuAccel( p, open_id, KAccel::Open );
  p->insertSeparator();
  save_id = p->insertItem( i18n( "&Save as..." ), this,
                           SLOT( slotSaveAs() ) );
  accel->connectItem( KAccel::Save, this, SLOT( slotSaveAs() ) );
  accel->changeMenuAccel( p, save_id, KAccel::Save );
  p->setItemEnabled( save_id, false );
  p->insertSeparator();
  int id = p->insertItem( i18n( "&Quit" ), this, SLOT( slotQuit() ) );
  accel->connectItem( KAccel::Quit, this, SLOT( slotQuit() ) );
  accel->changeMenuAccel( p, id, KAccel::Quit );

  // The file menu p goes into the menubar.
  menuBar()->insertItem( i18n( "&File" ), p );

  // The TSP menu
//  tspMenu = new QPopupMenu;
//  CHECK_PTR( tspMenu );

  // Create a TSP menu
//  _numberOfPoints = 10;
//  new_id = tspMenu->insertItem( i18n( "&New TSP" ), this, SLOT( slotStart() ) );
//  accel->connectItem( KAccel::New, this, SLOT( slotStart() ) );
//  accel->changeMenuAccel( tspMenu, new_id, KAccel::New );

  // The TSP menu p goes into the menubar.
//  menuBar()->insertItem( i18n( "&TSP" ), tspMenu );

  // The View menu
  // We need to access this menu and its entries later.
  // Therefore we store the menu as well as the entry ids.
  viewMenu = new QPopupMenu;
  CHECK_PTR( viewMenu );

  // Create a View menu
  // and disable the entries because there isn't anything to
  // view yet.
  best_id = viewMenu->insertItem( i18n( "Show &Best TSP" ),
                                  this, SLOT( slotViewBestTsp() ) );
  viewMenu->setItemEnabled( best_id, false );
  orig_id = viewMenu->insertItem( i18n( "Show &Original TSP" ),
                                  this, SLOT( slotViewOrigTsp() ) );
  viewMenu->setItemEnabled( orig_id, false );
  current_id = viewMenu->insertItem( i18n( "Show &Current TSP" ),
                                     this, SLOT( slotViewCurrentTsp() ) );
  viewMenu->setItemEnabled( current_id, false );

  // The View menu goes into the menubar.
  menuBar()->insertItem( i18n( "&View" ), viewMenu );

  // The config menu.
  p = new QPopupMenu();
  CHECK_PTR( p );

  // Create the config menu.
  id = p->insertItem( i18n( "&TSP Preferences..." ),
                      this, SLOT( slotConfigTsp() ) );
  id = p->insertItem( i18n( "&Graphic Preferences..." ),
                      this, SLOT( slotConfigGraphic() ) );

  // The config menu goes into the menubar
  menuBar()->insertItem( i18n( "&Options" ), p );

  // KDE will generate a appropriate help menu for us. We just have to
  // provide the 'About Ktsp' entry.
  p = kapp->getHelpMenu( false,
			 i18n( "KTsp - The TSP Optimizer\n"
                               "Version 0.1.0\n\n"
			       "(c) 1999 Uwe Thiem\n"
                               "uwe@kde.org\n\n"
                               "The algorithm GDA has\n"
                               "been invented by\n"
                               "Gunter Dueck." ) );
  menuBar()->insertSeparator();
  menuBar()->insertItem( i18n( "&Help" ), p );
  }


void KTsp::setupToolbar()
  {
  // Some buttons for the toolbar. The pics are standard, we don't have to
  // provide them.
  toolBar()->insertButton( Icon( "exit.xpm" ), TOOLBAR_QUIT,
                           true, i18n( "Exit" ) );
  toolBar()->insertSeparator();
  toolBar()->insertButton( Icon( "fileopen.xpm" ), TOOLBAR_OPEN,
                           true, i18n( "Open TSP" ) );
  toolBar()->insertButton( Icon( "filefloppy.xpm" ), TOOLBAR_SAVE,
                           false, i18n( "Save TSP" ) );
  toolBar()->insertSeparator();
  toolBar()->insertButton( Icon( "run.xpm" ), TOOLBAR_RUN,
                           true, i18n( "Generate and run a TSP" ) );
  toolBar()->insertSeparator();
  toolBar()->insertButton( Icon( "help.xpm" ), TOOLBAR_HELP,
                           true, i18n( "Help" ) );

  // and connect the whole toolbar to a slot
  connect( toolBar(), SIGNAL( clicked( int ) ), SLOT( slotToolbarClicked( int ) ) );
  }



void KTsp::setupStatusbar()
  {
  // A statusbar with two entries.
  statusBar()->insertItem( "A very long text which will be replaced asap." , 0 );
  statusBar()->changeItem( i18n( "Waiting for user" ), 0 );
  statusBar()->insertItem( "                    ", 1 );
  }


void KTsp::setupTsp()
  {
  // Just create an instance of TspGda
  tsp = new TspGda();
  CHECK_PTR( tsp );
  connect( tsp, SIGNAL( statusChanged( int ) ),
           this, SLOT( slotStatusChanged( int ) ) );
  connect( tsp, SIGNAL( progressNeighbours( int ) ),
           this, SLOT( slotNeighbours( int ) ) );
  connect( tsp, SIGNAL( pointsNumber( int ) ),
           this, SLOT( slotPointsMax( int ) ) );
  connect( tsp, SIGNAL( neighboursNumber( int ) ),
           this, SLOT( slotNeighboursMax( int ) ) );
  connect( tsp, SIGNAL( coursesReady() ),
           this, SLOT( slotCoursesReady() ) );
  connect( tsp, SIGNAL( noCoursesReady() ),
           this, SLOT( slotNoCoursesReady() ) );
  }


void KTsp::setupView()
  {             	
  // The toplevel vertical layout	
  QVBoxLayout *layout = new QVBoxLayout( w );
  CHECK_PTR( layout );

  // a horizontal layout for both displays
  QHBoxLayout *hlayout = new QHBoxLayout();
  CHECK_PTR( hlayout );

  // a QLabel for the headline
  QLabel *label = new QLabel( i18n( "The TSP Optimizer" ),
                              w, "Headline" );
  QFont f = label->font();
  f.setPointSize( f.pointSize() + 2 );
  label->setFont( f );
  label->setMinimumSize( label->sizeHint() );
  label->setAlignment( AlignCenter );

  // into the toplevel layout with them
  layout->addSpacing( 10 );
  layout->addWidget( label, 0 );
  layout->addSpacing( 5 );
  layout->addLayout( hlayout, 30 );
  layout->addSpacing( 10 );

  // construct the display for course length over actions
  aDisplay = new ActionsDisplay( w, "Actions Graph" );
  CHECK_PTR( aDisplay );
  connect( tsp, SIGNAL( quality( double ) ),
           aDisplay, SLOT( slotNextValue( double ) ) );
  connect( tsp, SIGNAL( newRun() ),
           aDisplay, SLOT( slotClearDisplay() ) );

  // construct the display for course length over runs
  rDisplay = new RunsDisplay( w, "Runs Graph" );
  CHECK_PTR( rDisplay );
  connect( tsp, SIGNAL( currentQuality( double ) ),
           rDisplay, SLOT( slotNextValue( double ) ) );

  // into the horizontal layout with them
  hlayout->addSpacing( 10 );
  hlayout->addWidget( aDisplay );
  hlayout->addSpacing( 20 );
  hlayout->addWidget( rDisplay );
  hlayout->addSpacing( 10 );

  // a new horizontal layout for the progress bar
  // that goes into the toplevel layout
  hlayout = new QHBoxLayout();
  CHECK_PTR( hlayout );
  layout->addLayout( hlayout, 1 );
  layout->addSpacing( 20 );

  // a QLabel informing what the progess bar is for
  label = new QLabel( i18n( "Runs" ), w, "Runs Progress" );
  CHECK_PTR( label );
  label->setMinimumSize( label->sizeHint() );

  // the progress bar itself
  progressBar = new QProgressBar( w, "Runs Pregess Bar" );
  CHECK_PTR( progressBar );
  connect( tsp, SIGNAL( currentRun( int ) ),
           progressBar, SLOT( setProgress( int ) ) );

  // into the horizontal layout with them
  hlayout->addSpacing( 10 );
  hlayout->addWidget( label, 0 );
  hlayout->addSpacing( 10 );
  hlayout->addWidget( progressBar, 1 );
  hlayout->addSpacing( 10 );

  // finally, we activate layout management
  layout->activate();
  }


void KTsp::setupDND()
  {
  // create a drop zone over the whole main window and connect
  // it to slotDrop()
  KDNDDropZone *zone = new KDNDDropZone( this, DndURL );
  CHECK_PTR( zone );
  connect( zone, SIGNAL( dropAction( KDNDDropZone *) ),
           this, SLOT( slotDrop( KDNDDropZone * ) ) );
  }


void KTsp::slotToolbarClicked( int item )
  {
  // Handle all buttons in toolbar.
  switch ( item )
    {
    case TOOLBAR_QUIT:
      slotQuit();
      break;
    case TOOLBAR_HELP:
      kapp->invokeHTMLHelp( "", "" );
      break;
    case TOOLBAR_RUN:
      slotStart();
      break;
    case TOOLBAR_OPEN:
      slotOpen();
      break;
    case TOOLBAR_SAVE:
      slotSaveAs();
      break;
    default:
      // OOps! This shouldn't happen.
      break;
    }
  }


void KTsp::slotStatusChanged( int id )
  {
  // Change statusbar according to internal state
  switch( id )
    {
    case STATUSBAR_READY:
      statusBar()->changeItem( i18n( "Ready" ), 0 );
      break;
    case STATUSBAR_WATING:
      statusBar()->changeItem( i18n( "Waiting for user" ), 0 );
      break;
    case STATUSBAR_GENERATING:
      statusBar()->changeItem( i18n( "Generating TSP" ), 0 );
      break;
    case STATUSBAR_NEIGHBOURS:
      progress = new QProgressDialog(
                   i18n( "Preparing TSP for optimization..." ),
                   0,
                   _numberOfPoints,
                   this, "Neighbours Progress" );
      CHECK_PTR( progress );
      progress->setMinimumSize( progress->sizeHint() );
      progress->setMinimumWidth( progress->sizeHint().width() + 10 );
      connect( tsp, SIGNAL( progressNeighbours( int ) ),
               progress, SLOT( setProgress( int ) ) );
      statusBar()->changeItem( i18n( "Generating neighbours" ), 0 );
      break;
    case STATUSBAR_OPTIMIZING:
      delete progress;
      if ( !noClearDisplay )
        {
        rDisplay->slotClearDisplay();
        rDisplay->slotNumberOfRuns( _runs );
        }
      noClearDisplay = false;
      aDisplay->slotClearDisplay();
      statusBar()->changeItem( i18n( "Optimizing" ), 0 );
      break;
    case STATUSBAR_LOADING:
      statusBar()->changeItem( i18n( "Loading..." ), 0 );
      break;
    case STATUSBAR_SAVING:
      statusBar()->changeItem( i18n( "Saving..." ), 0 );
      break;
    case STATUSBAR_LOAD_ERROR:
      statusBar()->changeItem( i18n( "Loading error" ), 0 );
      showFileError( i18n( "Unable to read the requested file.\n" ) );
      statusBar()->changeItem( i18n( "Ready" ), 0 );
      break;
    case STATUSBAR_SAVE_ERROR:
      statusBar()->changeItem( i18n( "Saving error" ), 0 );
      showFileError( i18n( "Unable to write file.\n" ) );
      statusBar()->changeItem( i18n( "Ready" ), 0 );
      break;
    case STATUSBAR_RESUMING:
      statusBar()->changeItem( i18n( "Resuming TSP..." ), 0 );
      break;
    case STATUSBAR_RESUME_ERROR:
      statusBar()->changeItem( i18n( "Resuming error" ), 0 );
      showFileError( i18n( "Unable to resume requested TSP.\n" ) );
      statusBar()->changeItem( i18n( "Ready" ), 0 );
      break;
    default:
      statusBar()->changeItem( i18n( "Something is wrong" ), 0 );
      break;
    }
  }


void KTsp::slotNeighbours( int number )
  {
  char s[100];
  if ( number >= _numberOfPoints )
    {
    sprintf( s, "                         " );
    }
  else
    {
    sprintf( s, i18n( "For point: %d" ), number );
    }
  statusBar()->changeItem( s, 1 );
  }


void KTsp::slotPointsMax( int number )
  {
  _numberOfPoints = number;
//  QString s;
//  s.sprintf( "Points: %d", number );
//  headline2->setText( s );
  }


void KTsp::slotNeighboursMax( int number )
  {
  _numberOfNeighbours = number;
//  QString s;
//  s.sprintf( "Neighbours: %d", number );
//  headline3->setText( s );
  }


void KTsp::slotQuit()
  {
  // Stop all calculations, save configuration and quit.
  tsp->stop();
  kapp->processEvents();
  while ( dirty ) queryUnsaved();
  saveConfig();
  kapp->quit();
  }


void KTsp::slotStart()
  {
  tsp->stop();
  while( dirty ) queryUnsaved();
  disableOpenNew();
  tsp->startTsp( _runs );
  enableOpenNew();
  }


void KTsp::slotViewOrigTsp()
  {
  viewTsp( ORIG_TSP );
  }


void KTsp::slotViewCurrentTsp()
  {
  viewTsp( CURRENT_TSP );
  }


void KTsp::slotViewBestTsp()
  {
  viewTsp( BEST_TSP );
  }


void KTsp::slotCoursesReady()
  {
  // All three internal TSPs are initialized. We can enable the
  // correnponding menu entries now.
  viewMenu->setItemEnabled( best_id, true );
  viewMenu->setItemEnabled( orig_id, true );
  viewMenu->setItemEnabled( current_id, true );
  fileMenu->setItemEnabled( save_id, true );
  toolBar()->setItemEnabled( TOOLBAR_SAVE, true );
  dirty = true;
  }


void KTsp::slotNoCoursesReady()
  {
  // TSPs aren't available temporarely. Disable the
  // corresponding menu entries.
  viewMenu->setItemEnabled( best_id, false );
  viewMenu->setItemEnabled( orig_id, false );
  viewMenu->setItemEnabled( current_id, false );
  fileMenu->setItemEnabled( save_id, false );
  toolBar()->setItemEnabled( TOOLBAR_SAVE, false );
  dirty = false;
  progressBar->setTotalSteps( _runs );
  progressBar->reset();
  }



void KTsp::viewTsp( int id )
  {
  // This looks like a memory leakage but isn't because View
  // is a new KTMW.
  display = new View( _pointsShape, _pointsColour, _pointsSize,
                      _linesStyle, _linesColour, _linesWidth );
  CHECK_PTR( display );
  display->show();
  display->drawTsp( tsp->getPoints( id ) );
  }


void KTsp::readConfig()
  {
  KConfig *config = kapp->getConfig();
  config->setGroup( "Graphic" );
  _pointsShape = config->readNumEntry( "PointShape",
                                       GraphicDialog::Circle );
  _pointsColour = config->readNumEntry( "PointColour",
                                        GraphicDialog::Blue );
  _pointsSize = config->readNumEntry( "PointSize",
                                      GraphicDialog::Small );
  _linesStyle = config->readNumEntry( "LineStyle",
                                      GraphicDialog::Solid );
  _linesColour = config->readNumEntry( "LineColour",
                                       GraphicDialog::Red );
  _linesWidth = config->readNumEntry( "LineWidth",
                                      GraphicDialog::Small );
  config->setGroup( "Optimization" );
  tsp->setNumberOfPoints( config->readNumEntry( "Points", 300 ) );
  tsp->setNumberOfNeighbours( config->readNumEntry( "Neighbours", 10 ) );
  tsp->setConfFactor( config->readNumEntry( "Rainspeed", 50 ) );
  _runs = config->readNumEntry( "Runs", 50 );
  }


void KTsp::saveConfig()
  {
  KConfig *config = kapp->getConfig();
  config->setGroup( "Graphic" );
  config->writeEntry( "PointShape", _pointsShape );
  config->writeEntry( "PointColour", _pointsColour );
  config->writeEntry( "PointSize", _pointsSize );
  config->writeEntry( "LineStyle", _linesStyle );
  config->writeEntry( "LineColour", _linesColour );
  config->writeEntry( "LineWidth", _linesWidth );
  config->setGroup( "Optimization" );
  config->writeEntry( "Points", tsp->numberOfPoints() );
  config->writeEntry( "Neighbours", tsp->numberOfNeighbours() );
  config->writeEntry( "Runs", _runs );
  config->writeEntry( "Rainspeed", tsp->confFactor() );
  }


bool KTsp::queryClose()
  {
  tsp->stop();
  finished = true;
  while ( dirty ) queryUnsaved();
  saveConfig();
  return true;
  }



void KTsp::slotConfigTsp()
  {
  TspDialog dlg( tsp->numberOfPoints(), 300,
                 _runs, 50,
                 tsp->numberOfNeighbours(), 10,
                 tsp->confFactor(), 50,
                 this, "TSP Dialog" );
  connect( &dlg, SIGNAL( pointsChanged( int ) ),
           tsp, SLOT( setNumberOfPoints( int ) ) );
  connect( &dlg, SIGNAL( runsChanged( int ) ),
           this, SLOT( slotRunsChanged( int ) ) );
  connect( &dlg, SIGNAL( neighboursChanged( int ) ),
           tsp, SLOT( setNumberOfNeighbours( int ) ) );
  connect( &dlg, SIGNAL( rainspeedChanged( int ) ),
           tsp, SLOT( setConfFactor( int ) ) );
  dlg.exec();
  }


void KTsp::slotConfigGraphic()
  {
  GraphicDialog dlg( _pointsShape, GraphicDialog::Circle,
                     _pointsColour, GraphicDialog::Blue,
                     _pointsSize, GraphicDialog::Small,
                     _linesStyle, GraphicDialog::Solid,
                     _linesColour, GraphicDialog::Red,
                     _linesWidth, GraphicDialog::Small,
                     this, "Graphic Dialog" );
  connect( &dlg, SIGNAL( pointsShapeChanged( int ) ),
           this, SLOT( slotPointsShapeChanged( int ) ) );
  connect( &dlg, SIGNAL( pointsColourChanged( int ) ),
           this, SLOT( slotPointsColourChanged( int ) ) );
  connect( &dlg, SIGNAL( pointsSizeChanged( int ) ),
           this, SLOT( slotPointsSizeChanged( int ) ) );
  connect( &dlg, SIGNAL( linesStyleChanged( int ) ),
           this, SLOT( slotLinesStyleChanged( int ) ) );
  connect( &dlg, SIGNAL( linesColourChanged( int ) ),
           this, SLOT( slotLinesColourChanged( int ) ) );
  connect( &dlg, SIGNAL( linesWidthChanged( int ) ),
           this, SLOT( slotLinesWidthChanged( int ) ) );
  dlg.exec();
  }


void KTsp::slotRunsChanged( int number )
  {
  _runs = number;
  }


void KTsp::slotPointsShapeChanged( int value )
  {
  _pointsShape = value;
  }


void KTsp::slotPointsColourChanged( int value )
  {
  _pointsColour = value;
  }


void KTsp::slotPointsSizeChanged( int value )
  {
  _pointsSize = value;
  }


void KTsp::slotLinesStyleChanged( int value )
  {
  _linesStyle = value;
  }


void KTsp::slotLinesColourChanged( int value )
  {
  _linesColour = value;
  }


void KTsp::slotLinesWidthChanged( int value )
  {
  _linesWidth = value;
  }


void KTsp::slotBestQuality( double quality )
  {
  bestQuality = quality;
  }


void KTsp::slotOpen()
  {
  while ( dirty ) queryUnsaved();
  QString url = KFileDialog::getOpenFileURL( 0, "*.tsp",
                                             this, "Open Dialog" );
  loadFile( url );
  }


void KTsp::slotDrop( KDNDDropZone *drop )
  {
  if ( cannotOpen )
    {
    // we don't accept another TSP while an optimization
    // is running
    KMsgBox box( this,
                 i18n( "Drop error" ),
                 i18n( "Busy optimizing.\n"
                       "Cannot accept another TSP" ),
                 KMsgBox::EXCLAMATION,
                 i18n( "Dismiss" ) );
    box.exec();
    return;
    }
  while( dirty ) queryUnsaved();
  // we can't handle more than one TSP at a time
  // therefor we just take the first one
  QString url = drop->getURLList().first();
  loadFile( url );
  }


void KTsp::slotForcedLoad()
  {
  delete timer;
  loadFile( loadFileName );
  }


void KTsp::slotForcedResume()
  {
  disableOpenNew();
  noClearDisplay = true;
  tsp->resumeTsp( resumedOrigTsp, resumedBestTsp,
                  _runs, performedRuns,
                  resumedBestQuality );
  enableOpenNew();
  }


void KTsp::loadFile( QString url )
  {
  QString name = "";
  if ( url.isEmpty() ) return;
  if ( !KFM::download( url, name ) )
    {
    return;
    }
  QFile file( name );
  if ( !file.exists() )
    {
    showFileError( i18n( "The requested file does not exist!" ) );
    return;
    }
  disableOpenNew();
  tsp->startTsp( name, _runs );
  KFM::removeTempFile( name );
  enableOpenNew();
  }


void KTsp::showFileError( const char *message )
  {
  KMsgBox box( this,
               i18n( "File error" ),
               message,
               KMsgBox::EXCLAMATION,
               i18n( "Dismiss" ) );
  box.exec();
  }


void KTsp::slotSaveAs()
  {
  QString name = KFileDialog::getSaveFileName( 0, "*.tsp",
                                               this, "Save Dialog" );
  if ( name.isEmpty() ) return;
  QFile file( name );
  if ( file.exists() )
    {
    int rc = KMsgBox::yesNo( this, i18n( "File exists" ),
                             i18n( "The file already exists.\n"
                                   "Overwrite it?" ),
                             KMsgBox::QUESTION | KMsgBox::DB_SECOND );
    if ( rc == 2 ) return;
    }
  if ( tsp->saveTsp( name, BEST_TSP ) ) dirty = false;
  }


void KTsp::queryUnsaved()
  {
  int rc = KMsgBox::yesNo( this, i18n( "unsaved data" ),
                           i18n( "The optimized TSP isn't saved yet.\n"
                                 "Would you like to save it first?" ),
                           KMsgBox::QUESTION | KMsgBox::DB_FIRST );
  if ( rc == 2 )
    {
    dirty = false;
    return;
    }
  slotSaveAs();
  }


void KTsp::enableOpenNew()
  {
  if ( !finished )
    {
    toolBar()->setItemEnabled( TOOLBAR_RUN, true );
    toolBar()->setItemEnabled( TOOLBAR_OPEN, true );
    fileMenu->setItemEnabled( new_id, true );
    fileMenu->setItemEnabled( open_id, true );
    }
  cannotOpen = false;
  }


void KTsp::disableOpenNew()
  {
  toolBar()->setItemEnabled( TOOLBAR_RUN, false );
  toolBar()->setItemEnabled( TOOLBAR_OPEN, false );
  fileMenu->setItemEnabled( new_id, false );
  fileMenu->setItemEnabled( open_id, false );
  cannotOpen = true;
  }


void KTsp::saveProperties( KConfig *config )
  {
  if ( _numberOfPoints != 0 )
    {
    QString s1 = tmpnam( 0 );
    QString s2 = tmpnam( 0 );
    bool orig = tsp->saveTsp( s1, ORIG_TSP );
    bool best = tsp->saveTsp( s2, BEST_TSP );
    if( orig && best )
      {
      config->writeEntry( "Points", _numberOfPoints ),
      config->writeEntry( "Neighbours", tsp->numberOfNeighbours() );
      config->writeEntry( "Runs", _runs );
      config->writeEntry( "CurrentRun", progressBar->progress() );
      QString s;
      s.setNum( bestQuality, 'g', 60 );
      config->writeEntry( "BestQuality", s );
      config->writeEntry( "OrigName", s1 );
      config->writeEntry( "BestName", s2 );
      s = "";
      int n = rDisplay->numberOfValues();
      if ( n > 0 )
        {
        s.setNum( rDisplay->firstValue(), 'g', 60 );
        s += "\n";
        QString tmp;
        for ( int i = 1; i < n; i++ )
          {
          tmp.setNum( rDisplay->nextValue(), 'g', 60 );
          s += tmp + "\n";
          }
        config->writeEntry( "ReceivedValues", s );
        }
      }
    else
      {
      QFile f( s1 );
      f.remove();
      f.setName( s2 );
      f.remove();
      }
    }
  }


void KTsp::readProperties( KConfig *config )
  {
  int points;
  int neighbours;
  int runs;
  points = config->readNumEntry( "Points", 0 );
  neighbours = config->readNumEntry( "Neighbours", 0 );
  runs = config->readNumEntry( "Runs", 0 );
  if ( runs == 0 ) runs = _runs;
  QString s1 = config->readEntry( "OrigName" );
  QString s2 = config->readEntry( "BestName" );
  if ( ( points >= 10 ) &&
       ( ( neighbours >= 3 ) && ( neighbours < points ) ) &&
       ( !s1.isEmpty() ) &&
       ( !s2.isEmpty() ) )
    {
    tsp->setNumberOfNeighbours( neighbours );
    _runs = runs;
    resumedOrigTsp = s1;
    resumedBestTsp = s2;
    QString s = config->readEntry( "ReceivedValues" );
    rDisplay->slotClearDisplay();
    rDisplay->slotNumberOfRuns( runs );
    resumedBestQuality = config->readDoubleNumEntry( "BestQuality" );
    performedRuns = config->readNumEntry( "CurrentRun", 0 );
    if ( !s.isEmpty() )
      {
      QTextStream t( s, IO_ReadOnly );
      QString tmp;
      while ( !t.eof() )
        {
        tmp = t.readLine();
        rDisplay->slotNextValue( tmp.toDouble() );
        }
      }
    timer = new QTimer( this, "Resume Timer" );
    CHECK_PTR( timer );
    connect( timer, SIGNAL( timeout() ),
             this, SLOT( slotForcedResume() ) );
    timer->start( 500, true );
    }
  else
    {
    // data inconsistent
    // let's try to remove the temporary files
    QFile f( s1 );
    f.remove();
    f.setName( s2 );
    f.remove();
    }
  }


