/***************************************************************************
                          kaspasql.cpp  -  description
                             -------------------                                         
    begin                : Tue Sep 7 1999                                           
    copyright            : (C) 1999 by Jan Mueller                         
    email                : janmueller7@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include "kaspasql.h"

//////////////////////////////////////////////
// Global:

/**  escapes the "'" characters to "\'". */
char *esc4sql(const char *s){
	int i=0;
	int j=0;
	while(s[i]) {
		if(s[i]=='\'') j++;
		i++;
	}
	char *t=new char[strlen(s)+j+1];

	i=0;
	j=0;
	while(s[i]) {
		if(s[i]=='\'') {
			t[j]='\\';
			j++;
			t[j]=s[i];
		}
		else
			t[j]=s[i];
			
		i++;
		j++;
	}
	t[j]=0;
	return t;
}


char *oid2str(Oid o) {
  return uint2str(o);
}

void oid2str(Oid o, Str *S) {
  char *c=oid2str(o);
  *S+=c;
  free(c);
}

Oid str2oid(const char *c) {
  return strtoul(c, 0, 0);
}


/////////////////////////////////////////////
////// Other Exception Classes
  SqlErr::SqlErr(const char *s=0): KaspaErr(s) {};
  LoIOErr::LoIOErr(const char *s=0): SqlErr(s) {};
  LoHandleErr::LoHandleErr(const char *s=0): SqlErr(s) {};
  SqlConnectionErr::SqlConnectionErr(const char *s=0): SqlErr(s){};
  LoOidErr::LoOidErr(const char *s=0): SqlErr(s) {};
  SqlOutOfRange::SqlOutOfRange(const char *s=0): SqlErr(s) {};
  SqlExecErr::SqlExecErr(const char *s=0): SqlErr(s) {};
  SqlExecCanceled::SqlExecCanceled(const char *s=0): SqlErr(s) {};



//////////////////////////////////////////////////////
///   Sql

Sql::Sql(const Sql& s) throw (SqlConnectionErr): working(false), pgConn(0L), pgResult(0L), pgCloseConnection(false) {
	pgConn=s.getConn();
	checkErr();
}

Sql::Sql(const char *name) throw (SqlConnectionErr): working(false), pgConn(0L), pgResult(0L), pgCloseConnection(true) {
  pgConn = PQconnectdb(name);
	checkErr();
}

Sql::~Sql() throw() {
	/* if(working) cancel(); */
	if(pgCloseConnection) {
	  if(pgResult) PQclear(pgResult);
  	if(pgConn) PQfinish(pgConn);
	}
}



PGconn *Sql::getConn() const throw() {
	return pgConn;
}

bool Sql::isWorking() throw() {
	return working;
}

void Sql::checkErr() throw (SqlConnectionErr) {
  if(connectionBad()) throw SqlConnectionErr(errorMessage());
}

const char* Sql::errorMessage() const throw() {
	return (const char *)PQerrorMessage(pgConn);
}

void Sql::traceOn(FILE *f) throw() {
	PQtrace(pgConn, f);
}

int Sql::lOpen(Oid id, int mode) throw(LoHandleErr, SqlConnectionErr) {
  int h;
  checkErr();
  if((h=lo_open(pgConn, id, mode))==-1)
    throw LoHandleErr(errorMessage());
  return h;
}

void Sql::lClose(int fd) throw(LoHandleErr, SqlConnectionErr){
  checkErr();
  if(lo_close(pgConn, fd)==-1)
    throw LoHandleErr(errorMessage());
}

Oid Sql::lCreate(int mode) throw(LoOidErr, SqlConnectionErr) {
  Oid o;
  checkErr();
  if((o=lo_creat(pgConn, mode))==InvalidOid)
    throw LoOidErr(errorMessage());
  return o;
}

int Sql::lSeek(int fd, int offs, int mode) throw(LoIOErr, SqlConnectionErr) {
  int r;
  checkErr();
  if((r=lo_lseek(pgConn, fd, offs, mode))==-1)
    throw LoIOErr(errorMessage());
  return r;
}

int Sql::lTell(int fd) throw(LoIOErr, SqlConnectionErr) {
  int r;
  checkErr();
  if((r=lo_tell(pgConn, fd))==-1)
    throw LoIOErr(errorMessage());
  return r;
}

Oid Sql::lImport(const char *name) throw(LoOidErr, SqlConnectionErr) {
  Oid o;
  checkErr();
	char *s=mystrdup(name); /* lo_import(pgConn, char *) -> const-warning with 'name' */
  if((o=lo_import(pgConn, s))==InvalidOid)
    throw LoOidErr(errorMessage());
	delete s;
  return o;
}

void Sql::lExport(Oid id, const char *name) throw(LoOidErr, SqlConnectionErr) {
  checkErr();
	char *s=mystrdup(name); /* lo_export(pgConn, Oid, char *) -> const-warning with 'name' */
  if(lo_export(pgConn, id, s)==-1)
    throw LoOidErr(errorMessage());
}

int Sql::lWrite(int fd, const char *buf, int len) throw(LoIOErr, SqlConnectionErr) {
  int r;
  checkErr();
  if((r=lo_write(pgConn, fd, const_cast<char*>(buf), len))==-1)
    throw LoIOErr(errorMessage());
  return r;
}

int Sql::lRead(int fd, char *buf, int len) throw (LoIOErr, SqlConnectionErr){
  int r;
  checkErr();
  if((r=lo_read(pgConn, fd, buf, len))==-1)
    throw LoIOErr(errorMessage());
  return r;
}

void Sql::lUnlink(Oid id) throw(LoOidErr, SqlConnectionErr) {
  checkErr();
  if(lo_unlink(pgConn, id)==-1)
    throw LoOidErr(errorMessage());
}

/*
void Sql:: cancel() {
  PGresult *res=0;

  if(!working) throw SqlExecErr("Sql::cancel(): No query to cancel!");
	PQrequestCancel(pgConn);
 	if(res) { PQclear(res); res=0; }
	while(PQisBusy(pgConn)) { PQconsumeInput(pgConn);  }
  while((res=PQgetResult(pgConn))) {
 	  if (pgResult) { PQclear(pgResult); pgResult=0; }
    pgResult=res;
  }
 	working=false;
	exec("rollback");
}
*/

void Sql::exec(const char *q, bool block) throw(SqlExecCanceled, SqlExecErr, SqlConnectionErr) {
  if(working) throw SqlExecErr("Sql::exec(char*) is not reentrant!!!");
  working=TRUE;
  PGresult *res=0;

  try {
	    checkErr();

  	  struct timeval timeout;

    	/* Initialize the file descriptor set. */
	    int fd=PQsocket(pgConn);
  	  if(fd<0) throw SqlExecErr("Sql::Exec(): Kann Socket nicht ermitteln!");

	    /* Initialize the timeout data structure. */
  	  timeout.tv_sec = 0;
    	timeout.tv_usec = 0;

	    if(!PQsendQuery(pgConn, q))
  	    throw SqlExecErr(errorMessage());
    	
	    /* select  returns 0 if timeout, 1 if input available, -1 if error. */
  	  do {
    	  fd_set set;
      	FD_ZERO (&set);
	      FD_SET (fd, &set);
	      int ret;
  	    while( ((ret=select(FD_SETSIZE, &set, 0L, 0L, &timeout))==-1) && (errno==EINTR) );
//				printf("\nSql::exec - ret=%i", ret);
    	  if(ret==-1)
					throw SqlExecErr("Sql::Exec(): select()-IO-Fehler!");

	      if(ret==1)
					if(!PQconsumeInput(pgConn))
					  throw SqlExecErr(errorMessage());

				/* blocking or idle-processing and cancel? */
	      if(ret==0 && !block && !idle())
					if(!PQrequestCancel(pgConn))
					  throw SqlExecErr(errorMessage());
					else
					  throw SqlExecCanceled("Query canceled!");	

    	  if(!PQisBusy(pgConn)) break;
  	  }	while(1);
	
	    while((res=PQgetResult(pgConn))) {
  	    if (pgResult) { PQclear(pgResult); pgResult=0; }
    	  if(PQresultStatus(res)==PGRES_NONFATAL_ERROR ||
					 PQresultStatus(res)==PGRES_FATAL_ERROR ||
					 PQresultStatus(res)==PGRES_BAD_RESPONSE)
					throw SqlExecErr(errorMessage());
  	    pgResult=res;
    	}
	  }
		catch (...) {
			/* cleaning up... */
    	if(res) { PQclear(res); res=0; }
	    while(PQisBusy(pgConn)) { PQconsumeInput(pgConn); idle(); }
  	  while((res=PQgetResult(pgConn))) {
    	  if (pgResult) { PQclear(pgResult); pgResult=0; }
	      pgResult=res;
  	  }
    	working=FALSE;
	    throw;
  	}
  working=FALSE;	
}

bool Sql::idle() throw() {
	/* waiting... */
  struct timeval timeout;
  timeout.tv_sec = 0;
  timeout.tv_usec = 100;
	select (0L, 0L, 0L, 0L, &timeout);

	return TRUE;
};

int Sql::tuples() throw() { return PQntuples(pgResult); };

int Sql::fields() throw() { return PQnfields(pgResult); };

const Oid Sql::type(int num) {
	return PQftype(pgResult, num);
}

const bool Sql::fieldIsText(int num) {
	Oid t=type(num);
	if(t==18   || t==19   || t==25   || t==1002 || t==1003 || t==1009 ||
		 t==1014 || t==1015 || t==1026 || t==1042 || t==1043 || t==1082 ||
		 t==1083 || t==1182 || t==1183 || t==1184 || t==1185 || t==1266 ||
		 t==1270 || t==1625)
		return true;
	else
		return false;
}

const char *Sql::fieldName(int num) throw(SqlOutOfRange) {
  const char *n;
  if((n=PQfname(pgResult, num)))
    return n;
  else
    throw SqlOutOfRange(errorMessage());
}

const char *Sql::getValue(int t, int f) throw(SqlOutOfRange) {
  const char *n;
  if((n=PQgetvalue(pgResult, t, f)))
    return n;
  else
    throw SqlOutOfRange(errorMessage());
}

const char *Sql::getValue(int t, const char *f) throw(SqlOutOfRange) {
  const char *n;
  if((n=PQgetvalue(pgResult, t, PQfnumber(pgResult, f))))
    return n;
  else
    throw SqlOutOfRange(errorMessage());
}


bool Sql::table(Str name) {
	return check4rel("pg_class", "relname", name);
}

bool Sql::index(Str name) {
	return check4rel("pg_indexes", "indexname", name);
}

bool Sql::check4rel(const Str rel, const Str field, const Str name) {
	Str s("select ");
	s+=field+" from ";
	s+=rel+" where ";
	s+=field+" = '";
	s+=name+"'";
	exec(s);
	return (tuples()>0);

}
