/***************************************************************************
                          funitem.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Aug 29 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <dlfcn.h>
#include <qfileinfo.h>
#include <qurl.h>
#include <klocale.h>
#include <ksimpleconfig.h>
#include "funitem.h"
#include "funcdlg.h"
#include "kpldoc.h"
#include "kplgraph.h"
#include "kplchecklistitem.h"
#include "utils.h"

FunItem::FunItem()
{
  init();
}

FunItem::FunItem(const FunItem& f) :
 ScaledItem(f), logxo(f.logxo), tmin(f.tmin), tmax(f.tmax), dt(f.dt),
 tmino(f.tmino), tmaxo(f.tmaxo), dto(f.dto), fkty(f.fkty), fktyo(f.fktyo),
 tv(f.tv), yv(f.yv), namey(f.namey), pathy(f.pathy), hmody(f.hmody)
{
  memcpy(py, f.py, sizeof(py));
  memcpy(pyo, f.pyo, sizeof(pyo));
  tv.detach();
  yv.detach();
  if (f.hmody)
    Utils::getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

FunItem::FunItem(Kpl::AutoStruct* aut) : ScaledItem(aut)
{
  init();
}

FunItem::FunItem(KSimpleConfig* plo, Kpl::AutoStruct* aut, const KURL& uPlo) :
 ScaledItem(plo, aut)
{
  init();
  tmin = plo->readDoubleNumEntry("xfmin");
  tmax = plo->readDoubleNumEntry("xfmax", tmin + 1.0);
  dt = plo->readDoubleNumEntry("dx");
  char s1[256];
  strcpy(s1, plo->readEntry("p", "0"));
  QString s2;
  py[0] = ((s2 = strtok(s1, " "))) ? s2.toDouble() : 0.0;
  for (int j = 1; j < KPL_NPMAX; j++)
    py[j] = ((s2 = strtok(0, " "))) ? s2.toDouble() : 0.0;
  namey = plo->readEntry("name", "");
  s2 = plo->readEntry("path", "");
  pathy = QUrl::isRelativeUrl(s2) ? (uPlo.directory(false) + s2) : s2;
  Utils::getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

FunItem::FunItem(bool act, int sym, const QString& col,
                 double xn, double yn, double xmin, double xmax, double dx,
                 const QString& name, const KURL& path)
 : ScaledItem(act, sym, col, xn, yn), logxo(false), tmin(xmin), tmax(xmax),
 dt(dx), namey(name), pathy(path)
{
  fkty = 0;
  memset(py, 0, sizeof(py));
  hmody = 0;
  Utils::getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

FunItem::~FunItem()
{
  if (hmody)
    dlclose(hmody);
}

FunItem& FunItem::operator=(const FunItem& f)
{
  if (this != &f) {
    *(ScaledItem*)this = f;
    logxo = f.logxo;
    tmin = f.tmin;
    tmax = f.tmax;
    dt = f.dt;
    tmino = f.tmino;
    tmaxo = f.tmaxo;
    dto = f.dto;
    fkty = f.fkty;
    fktyo = f.fktyo;
    memcpy(py, f.py, sizeof(py));
    memcpy(pyo, f.pyo, sizeof(pyo));
    tv = f.tv;
    tv.detach();
    yv = f.yv;
    yv.detach();
    namey = f.namey;
    pathy = f.pathy;
    if (hmody)
      dlclose(hmody);
    hmody = 0;
    if (f.hmody)
      Utils::getFuncAddr(pathy.path(), namey, &hmody, &fkty);
  }
  return *this;
}

KplItem::ItemTypes FunItem::iType() const
{
  return Function;
}

void FunItem::init()
{
  tmin = tmax = dt = 0.0;
  logxo = false;
  fkty = 0;
  memset(py, 0, sizeof(py));
  hmody = 0;
}

int FunItem::calcTable(bool logx) const
{
  bool newv = tv.isNull();
  double d;
  int n;
  if (dt) {
    n = int(fabs((tmax - tmin) / dt)) + 1;
    d = logx ? (log10(tmax / tmin) / (n - 1)) : dt;
  } else {
    n = 101;
    d = 0.01 * (logx ? log10(tmax / tmin) : (tmax - tmin));
  }
  int i;
  if (newv || (tmin != tmino) || (tmax != tmaxo) || (d != dto) ||
      (logx != logxo)) {
    tv.detach();
    tv.resize(n);
    for (i = 0; i < n; i++)
      tv[i] = logx ? pow(10.0, log10(tmin) + i * d) : (tmin + i * d);
  }
  bool pchanged = false;
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fkty != fktyo) || (logx != logxo)) {
    yv.detach();
    yv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fkty != fktyo) || (logx != logxo))
    for (i = 0; i < n; i++)
      yv[i] = ((*fkty)(tv[i], py));
  if (pchanged)
    for (i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
  if (fkty != fktyo)
    fktyo = fkty;
  if (tmin != tmino)
    tmino = tmin;
  if (tmax != tmaxo)
    tmaxo = tmax;
  if (d != dto)
    dto = d;
  if (logx != logxo)
    logxo = logx;
  return n;
}

void FunItem::draw(KplGraph* g) const
{
  if (fkty && active) {
    setProperties(g);
    int n = calcTable(g->logx);
    g->plArray(tv.data(), yv.data(), fx, fy, n);
  }
}

void FunItem::exportTable(FILE* f, KplDoc* m) const
{
  int n = calcTable(logxo);
  QCString s1, s2;
  for (int i = 0; i < n; i++){
    s1.setNum(fx * tv[i], m->aut.format, m->aut.prec);
    s2.setNum(fy * yv[i], m->aut.format, m->aut.prec);
    fprintf(f, "%s %s\n", s1.data(), s2.data());
  }
  fclose(f);
}

void FunItem::writePlo(FILE* f, const KURL& url, bool abs) const
{
  fprintf(f, "Type=FUNITEM\n");
  ScaledItem::writePlo(f, url, abs);
  fprintf(f, "xfmin=%g\nxfmax=%g\ndx=%g\np=", tmin, tmax, dt);
  for (int i = 0; i < 19; i++)
    fprintf(f, "%g ", py[i]);
  fprintf(f, "%g\nname=%s\npath=%s\n", py[19], namey.latin1(),
          Utils::relPath(url, pathy, abs).latin1());
}

void FunItem::setText(KplCheckListItem* it, bool*, bool* funcs) const
{
  *funcs = true;
  it->setText(1, i18n("Function"));
  QFileInfo fi(pathy.path());
  it->setText(2, QString("y = ") + fi.baseName() + "." + namey + "(x)");
}

int FunItem::editItem(QWidget* parent, KplDoc* m, int)
{
  FuncDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* FunItem::copy() const
{
  return new FunItem(*this);
}

void FunItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (fkty) {
    Utils::expo(QMAX(fabs(tmin), fabs(tmax)), iext, fxt);
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void FunItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (fkty) {
    *xmi = tmin;
    *xma = tmax;
    int n = calcTable(logxo);
    Utils::minMaxFile(ymi, yma, yv, n);
  }
}

void FunItem::setPar(int i, double value, bool yFun)
{
  if (yFun)
    py[i] = value;
}

ParFunItem::ParFunItem()
{
  init();
}

ParFunItem::ParFunItem(const ParFunItem& f) :
 FunItem(f), fktx(f.fktx), fktxo(f.fktxo), xv(f.xv),
 namex(f.namex), pathx(f.pathx), hmodx(f.hmodx)
{
  memcpy(px, f.px, sizeof(px));
  memcpy(pxo, f.pxo, sizeof(pxo));
  xv.detach();
  if (f.hmodx)
    Utils::getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
}

ParFunItem::ParFunItem(Kpl::AutoStruct* aut) : FunItem(aut)
{
  init();
}

ParFunItem::ParFunItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                       const KURL& uPlo)
{
  fktx = 0;
  hmodx = 0;
  active = plo->readBoolEntry("active", true);
  ScaledItem::readPlo(plo, aut);
  tmin = plo->readDoubleNumEntry("tmin");
  tmax = plo->readDoubleNumEntry("tmax", tmin + 1.0);
  dt = plo->readDoubleNumEntry("dt");
  char s1[256];
  strcpy(s1, plo->readEntry("px", "0"));
  QString s2;
  px[0] = ((s2 = strtok(s1, " "))) ? s2.toDouble() : 0.0;
  for (int j = 1; j < KPL_NPMAX; j++)
    px[j] = ((s2 = strtok(0, " "))) ? s2.toDouble() : 0.0;
  strcpy(s1, plo->readEntry("py", "0"));
  py[0] = ((s2 = strtok(s1, " "))) ? s2.toDouble() : 0.0;
  for (int j = 1; j < KPL_NPMAX; j++)
    py[j] = ((s2 = strtok(0, " "))) ? s2.toDouble() : 0.0;
  namex = plo->readEntry("namex", "");
  namey = plo->readEntry("namey", "");
  s2 = plo->readEntry("pathx", "");
  pathx = QUrl::isRelativeUrl(s2) ? (uPlo.directory(false) + s2) : s2;
  s2 = plo->readEntry("pathy", "");
  pathy = QUrl::isRelativeUrl(s2) ? (uPlo.directory(false) + s2) : s2;
  Utils::getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
  Utils::getFuncAddr(pathy.path(), namey, &hmody, &fkty);
}

ParFunItem::ParFunItem(bool act, int sym, const QString& col,
                       double xn, double yn, double tmi, double tma, double d,
                       const QString& name_x, const KURL& path_x,
                       const QString& name_y, const KURL& path_y) :
 FunItem(act, sym, col, xn, yn, tmi, tma, d, name_y, path_y),
 namex(name_x), pathx(path_x)
{
  memset(px, 0, sizeof(px));
  Utils::getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
}

ParFunItem::~ParFunItem()
{
  if (hmodx)
    dlclose(hmodx);
}

ParFunItem& ParFunItem::operator=(const ParFunItem& f)
{
  if (this != &f) {
    *(FunItem*)this = f;
    fktx = f.fktx;
    fktxo = f.fktxo;
    memcpy(px, f.px, sizeof(px));
    memcpy(pxo, f.pxo, sizeof(pxo));
    xv = f.xv;
    xv.detach();
    namex = f.namex;
    pathx = f.pathx;
    if (hmodx)
      dlclose(hmodx);
    hmodx = 0;
    if (f.hmodx)
      Utils::getFuncAddr(pathx.path(), namex, &hmodx, &fktx);
  }
  return *this;
}

KplItem::ItemTypes ParFunItem::iType() const
{
  return ParFunction;
}

void ParFunItem::init()
{
  fktx = 0;
  memset(px, 0, sizeof(px));
  hmodx = 0;
}

int ParFunItem::calcTable() const
{
  bool newv = tv.isNull();
  double d;
  int n;
  if (dt) {
    n = int(fabs((tmax - tmin) / dt)) + 1;
    d = dt;
  } else {
    n = 101;
    d = 0.01 * (tmax - tmin);
  }
  int i;
  if (newv || (tmin != tmino) || (tmax != tmaxo) || (d != dto)) {
    tv.detach();
    tv.resize(n);
    for (i = 0; i < n; i++)
      tv[i] = tmin + i * d;
  }
  bool pchanged = false;
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fkty != fktyo)) {
    yv.detach();
    yv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fkty != fktyo))
    for (i = 0; i < n; i++)
      yv[i] = ((*fkty)(tv[i], py));
  if (pchanged) {
    for (i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
    pchanged = false;
  }
  for (i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (px[i] != pxo[i])))
      break;
  if (pchanged || newv || (tmin != tmino) || (tmax != tmaxo) ||
      (d != dto) || (fktx != fktxo)) {
    xv.detach();
    xv.resize(n);
  }
  if (pchanged || newv || (tmin != tmino) || (tmax > tmaxo) ||
      (d != dto) || (fktx != fktxo))
    for (i = 0; i < n; i++)
      xv[i] = ((*fktx)(tv[i], px));
  if (pchanged)
    for (i = 0; i < KPL_NPMAX; i++)
      pxo[i] = px[i];
  if (fktx != fktxo)
    fktxo = fktx;
  if (fkty != fktyo)
    fktyo = fkty;
  if (tmin != tmino)
    tmino = tmin;
  if (tmax != tmaxo)
    tmaxo = tmax;
  if (d != dto)
    dto = d;
  return n;
}

void ParFunItem::draw(KplGraph* g) const
{
  if (fkty && active) {
    setProperties(g);
    int n = calcTable();
    g->plArray(xv.data(), yv.data(), fx, fy, n);
  }
}

void ParFunItem::exportTable(FILE* f, KplDoc* m) const
{
  int n = calcTable();
  QCString s1, s2;
  for (int i = 0; i < n; i++){
    s1.setNum(fx * xv[i], m->aut.format, m->aut.prec);
    s2.setNum(fy * yv[i], m->aut.format, m->aut.prec);
    fprintf(f, "%s %s\n", s1.data(), s2.data());
  }
  fclose(f);
}

void ParFunItem::writePlo(FILE* f, const KURL& url, bool abs) const
{
  fprintf(f, "Type=PARFUNITEM\n");
  ScaledItem::writePlo(f, url, abs);
  fprintf(f, "tmin=%g\ntmax=%g\ndt=%g\npx=", tmin, tmax, dt);
  int i;
  for (i = 0; i < 19; i++)
    fprintf(f, "%g ", px[i]);
  fprintf(f, "%g\npy=", px[19]);
  for (i = 0; i < 19; i++)
    fprintf(f, "%g ", py[i]);
  fprintf(f, "%g\nnamex=%s\nnamey=%s\n",
          py[19], namex.latin1(), namey.latin1());
  fprintf(f, "pathx=%s", Utils::relPath(url, pathx, abs).latin1());
  fprintf(f, "pathy=%s", Utils::relPath(url, pathy, abs).latin1());
}

void ParFunItem::setText(KplCheckListItem* it, bool*, bool*) const
{
  it->setText(1, i18n("Par. function"));
  QFileInfo fi1(pathx.path());
  QFileInfo fi2(pathy.path());
  it->setText(2, QString("x = ") + fi1.baseName() + "." + namex
              + "(t), y = " + fi2.baseName() + "." + namey + "(t)");
}

int ParFunItem::editItem(QWidget* parent, KplDoc* m, int)
{
  ParFuncDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* ParFunItem::copy() const
{
  return new ParFunItem(*this);
}

void ParFunItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (fktx && fkty) {
    double xmin, xmax, ymin, ymax;
    minMax(&xmin, &xmax, &ymin, &ymax);
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), ieyt, fyt);
  }
}

void ParFunItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (fktx && fkty) {
    int n = calcTable();
    Utils::minMaxFile(xmi, xma, xv, n);
    Utils::minMaxFile(ymi, yma, yv, n);
  }
}

void ParFunItem::setPar(int i, double value, bool yFun)
{
  if (yFun)
    py[i] = value;
  else
    px[i] = value;
}
