/***************************************************************************
                          kpldoc.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPLDOC_H
#define KPLDOC_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qobject.h>
#include <qlist.h>
#include <qdatetime.h>
#include <kio/job.h>
#include <kurl.h>
#include "kplitem.h"
#include "kplnamespace.h"

class ArrayItem;
class FrameItem;
class FunItem;
class KplItem;

/**
  * Kpl model class.
  * Provides methods for reading, writing and scaling of plot data.
  * @author Werner Stille
  */
class KplDoc : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructor.
   * @param parent pointer to parent QObject.
   */
  KplDoc(QObject* parent);
  /** Destructor. */
  ~KplDoc();
  /** Types of files */
  enum FileTypes{Unknown, Data, Plot};
	/** Initializes the document generally. */
	void newDocument();
  /**
   * Saves the document under URL.
   * @param url URL of file.
   * @param abs save absolute URL when true.
   * @return true for success.
   */
  bool saveDocument(const KURL& url, bool abs = true);
  /**
   * Sets the modified flag for the document after a modifying action on the
   * view connected to the document.
   * @param modified true for document modified after last save action.
   */
  void setModified(bool modified = true);
  /**
   * Returns if the document is modified or not. Use this to determine if your
   * document needs saving by the user on closing.
   * @return true for document modified after last save action.
   */
  bool isModified() const;
  /**
   * Sets the current directory path.
   * @param url current path.
   */
  void setCurrentDir(const KURL& url);
  /**
   * Returns the current path.
   * @return current directory path.
   */
  const QString& currentDir() const;
	/**
	 * Sets the URL of the document.
   * @param url URL.
	 */
	void setURL(const KURL& url);
	/**
	 * Returns the URL of the document.
   * @return URL.
	 */
	const KURL& URL() const;
	/**
	 * Returns the modification time of the URL of the document.
   * @return modification date and time.
	 */
  const QDateTime& URLTime() const;
  /**
   * Appends new item to list and initializes structure.
   * @param ityp type of item. One of the following:
   * Frame, Array, Function, ParFunction, Legend, Arrow.
   */
  void newItem(KplItem::ItemTypes ityp);
  /**
   * Reads file and initializes plot items.
   * @param url URL of the file.
   * @param type one of the following: Unknown, Data, Plot.
   * @return true for success.
   */
  bool openDocument(const KURL& url, FileTypes type = Unknown);
  /** Connects IO signal finished() to slot slotIOfinished(). */
  void connectIOfinished();
  /**
   * Restores items for a undo or redo action.
   * @param undo true for undo.
   * @return true for success.
   */
  bool restoreItems(bool undo = true);
  /**
    * Checks if undo action is possible.
    * @return true if possible.
    */
  bool undoAllowed() const;
  /**
    * Checks if redo action is possible.
    * @return true if possible.
    */
  bool redoAllowed() const;
  /**
   * Backup items for undo or redo actions if possible.
   * @param list true when item list should be uddated.
   */
  void backupItems(bool list = true);
  /**
   * Deletes plot item.
   * @param i item index.
   */
  void deleteItem(int i);
  /**
   * Moves plot item.
   * @param is source item index.
   * @param id destination item index.
   */
  void moveItem(int is, int id);
  /**
   * Calculates scaling parameters for all child items of a fram item.
   * @param autonorm true for automatic normalization.
   * @param item index of frame item.
   * @param iex logarithm of normalization factor for x axis.
   * @param iey logarithm of normalization factor for y axis.
   * @param fx normalization factor for x axis.
   * @param fy normalization factor for y axis.
   * @param xmi rounded minimum x value.
   * @param xma rounded maximum x value.
   * @param ymi rounded minimum y value.
   * @param yma rounded maximum y value.
   * @param xtic major x tic distance.
   * @param ytic major y tic distance.
   * @param mticx intervals between major x tics.
   * @param mticy intervals between major y tics.
   * @param ndigx number of places after decimal point for lettering of
   * the x axis.
   * @param ndigy number of places after decimal point for lettering of
   * the y axis.
   * @param c scaling constant, desired value for tic / (xma - xmi).
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   */
  void autoScaleFrame(bool autonorm, int item, int* iex, int* iey,
                      double*fx, double* fy, double* xmi, double* xma,
                      double* ymi, double* yma, double* xtic, double* ytic,
                      int* mticx, int* mticy, int* ndigx, int* ndigy,
                      double c, bool logx, bool logy);

public slots:
  /** Deletes KFM object and temporary file for upload. */
  void slotIOfinished();

signals:
  /**
   * Emitted when changes may require updating the menus and repainting
   * the page view.
   * @param plot true when plot should be uddated.
   * @param list true when item list should be uddated.
   */
  void modelChanged(bool plot, bool list);

protected:
  /** Deallocates storage of all plot items. */
  void freeAllItems();
  /**
   * Calculates scaling parameters for an array.
   * @param xmi pointer to double for rounded minimum value.
   * @param xma pointer to double for rounded maximum value.
   * @param tic pointer to double for major tic distance.
   * @param mtic pointer to int for intervals between major tics.
   * @param ndig pointer to int for number of places after decimal point
   * for lettering.
   * @param ie pointer to int for logarithm of normalization factor.
   * @param fn pointer to double for normalization factor.
   * @param x double array.
   * @param n size of array.
   * @param c scaling constant, desired value for tic / (xma - xmi).
   * @param log true for logarithmic axis.
   * @return true for error
   */
  bool autoFile(double* xmi, double* xma, double* tic, int* mtic,
                int* ndig, int* ie, double* fn, const double* x, int n,
                double c, bool log);
  /**
   * Sets array and frame item elements by autoscaling.
   * @param fd pointer to frame item.
   * @param ad pointer to array item.
   * @return true for error
   */
  bool autoScale(FrameItem* fd, ArrayItem* ad);
  /**
   * Sets function item elements for autofit.
   * @param ad pointer to array item.
   * @param fd pointer to function item.
   */
  void autoFit(ArrayItem* ad, FunItem* fd);
 	/** The modified flag of the current document. */
  bool modified;
  FileTypes fType;
  KURL docURL;
  QString m_dir;
  QDateTime m_time;
  QList<KplItem> itd;
  QList<QList<KplItem> > itb;
  Kpl::AutoStruct aut;
  KIO::FileCopyJob* job;
  double pLast[KPL_NPMAX], pErr[KPL_NPMAX];
  double chisq;

  friend FunItem;
  friend class AutoDlg;
  friend class AutoFitDlg;
  friend class EditFitParameterDlg;
  friend class EditParameterDlg;
  friend class FrameDlg;
  friend class FitDlg;
  friend class FuncDlg;
  friend class ItemDlg;
  friend class KplApp;
  friend class KplView;
  friend class ParFunItem;
  friend class ParFuncDlg;
  friend class Utils;
};

#endif // KPLDOC_H
