/*
     File: khronodatedisplay.cpp
     Khrono : Watch, timer and countdown utility for KDE 2
     Copyright (C) 1999,2000,2001 Axel von Arnim <vonarnim@club-internet.fr>

     This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either
     version 2 of the License, or (at your option) any later
     version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "khronodatedisplay.h"
#include "khronotimer.h"
#include <qpainter.h>
#include <stdio.h>

// *** Constructeurs *** //

KhronoDateDisplay::KhronoDateDisplay( QWidget *parent, const char *name )
    : QWidget( parent, name )
{
  date=new QDate();
  days=0;
  modeState=TIME;
  pointSize=10;
}

// *** Evenements *** //

void KhronoDateDisplay::paintEvent( QPaintEvent * )
{
  QPainter p;
  p.begin( this );
  switch( modeState ) {
    case TIME: {
      paintDate( p );
      break;
    }
    case CHRONO:
    case COUNTDOWN: {
      paintDays( p );
      break;
    }
  }
  QColorGroup colors=palette().normal();
  p.setPen(colors.foreground());
  p.drawRect(0, 0, width(), height());
  p.setPen(colors.background());
  p.drawLine(1, 0, width()-2, 0);
  p.end();
}

void KhronoDateDisplay::resizeEvent( QResizeEvent * )
{
  recomputePointSize();
}

void KhronoDateDisplay::recomputePointSize()
{
  QPainter p;
  p.begin( this );
  int border=5;
  QFont font=p.font();
  p.end();
  pointSize=height()/2;
  if( pointSize<=0 ) pointSize=1;
  font.setPointSize(pointSize);
  QFontMetrics fm(font);
  const char* s="Mar 05 2000";
  int w=fm.width(s);
  int maxw=width()-2*border;
  if( w>maxw ) {
    pointSize=(pointSize*maxw)/w;
    if( pointSize<=0 ) pointSize=1;
  }
}

// *** Slots publics *** //

void KhronoDateDisplay::setMode(int mode)
{
  if( modeState==mode ) return;
  modeState=mode;
  repaint();
}

void KhronoDateDisplay::setDate(QDate &d)
{
  if( *date==d ) return;
  date->setYMD(d.year(), d.month(), d.day());
  repaint();
}

void KhronoDateDisplay::setDays(int d)
{
  if( days==d ) return;
  days=d;
  repaint();
}

// *** Methodes privees *** //

void KhronoDateDisplay::paintDate( QPainter &p )
{
  int lettersSize=paintDaysLetters(p);
  if( !date->isValid() ) return;
  int border=5;
  QColorGroup colors=palette().normal();
  QFont font=p.font();
  int offset=border;
  
  p.setPen(colors.foreground());
  char* s=new char[24];
  sprintf(s, "%s %02d %d", (const char*)(date->monthName(date->month())), date->day(), date->year());
  font.setPointSize(pointSize);
  QFontMetrics fm(font);
  int w=fm.width(s);
  p.setFont(font);
  int h=fm.ascent();
  offset=(width()-w)/2;
  p.drawText( offset, (height()-lettersSize)/2+h/2, s );
  delete s;
}

int KhronoDateDisplay::paintDaysLetters( QPainter &p )
{
  int border=5;
  QColorGroup colors=palette().normal();
  QFont font=p.font();
  int ps=pointSize/2;
  if( ps<=0 ) ps=1;
  font.setPointSize(ps);
  QFontMetrics fm(font);
  int offset=border;
  int day=date->dayOfWeek();
  char *s=new char[15];
  sprintf(s, "M T W T F S S");
  p.setFont(font);
  p.setPen(colors.dark());
  p.drawText( offset, height()-border, s );
  if( modeState==TIME ) {
    p.setPen(colors.foreground());
    char tmp=s[2*(day-1)];
    s[2*(day-1)]=0;
    int w=fm.width(s);
    s[2*(day-1)]=tmp;
    s[2*(day-1)+1]=0;
    p.drawText( offset+w, height()-border, s+2*(day-1) );
  }
  delete s;
  return fm.ascent()+border;
}

void KhronoDateDisplay::paintDays( QPainter &p )
{
  int lettersSize=paintDaysLetters(p);
  QColorGroup colors=palette().normal();
  QFont font=p.font();
  font.setPointSize(pointSize);
  QFontMetrics fm(font);
  int h=fm.ascent();
  p.setFont(font);
  char *s=new char[10];
  p.setPen(colors.foreground());
  if( days<2 )
    sprintf(s, "%d day", days);
  else
    sprintf(s, "%d days", days);
  int w=fm.width(s);
  int offset=(width()-w)/2;
  p.drawText( offset, (height()-lettersSize)/2+h/2, s );
  delete s;
}
