/***************************************************************************
 *
 * knetworkmanager-devicestore.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig     <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson <wstephenson@suse.de>, <wstephenson@kde.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager.h"
#include "knetworkmanager-devicestore.h"
#include "knetworkmanager-storage.h"

extern "C"
{
#include <netlink/route/link.h>
}


void
DeviceStore::addDialUp (DialUp* dialup)
{
	_dialUpList.append (dialup);
}

void
DeviceStore::removeDialUp (DialUp* dialup)
{
	_dialUpList.remove (dialup);
}

void
DeviceStore::activateDialUp (DialUp* dialup)
{
	DeviceStoreDBus::activateDialUp (dialup);
	dialup->setActive (true);
}

void
DeviceStore::deactivateDialUp (DialUp* dialup)
{
	DeviceStoreDBus::deactivateDialUp (dialup);
	dialup->setActive (false);
}

void
DeviceStore::removeNetwork (const QString & obj_path, const QString & net_path)
{
	Device*  dev = NULL;
	Network* net = NULL;

	dev = getDevice (obj_path);
	if (!dev) {
		goto out;
	}

	net = dev->getNetwork (net_path);
	if (!net) {
		goto out;
	}
	// this works as long as we only remove networks from the store with removeNetwork when they disappear.
	emit networkDisappeared( net );
	dev->removeNetwork (net);
out:
	return;
}

void
DeviceStore::updateNetworkStrength (const QString & obj_path, const QString & net_path, int strength)
{
	Device*  dev = NULL;
	Network* net = NULL;

	dev = getDevice (obj_path);
	if (!dev) {
		goto out;
	}

	net = dev->getNetwork (net_path);
	if (!net) {
		/* network strength for an not existing network. happens if
		 * NM is running, KNM is started and reports hidden networks,
		 * NM then finds the network and tells signal strength about it
		 */
		//TODO: Do we really need it - hidden networks are in DeviceStore on par with others?
		DeviceStoreDBus::updateNetwork (obj_path.ascii(), net_path.ascii(), NULL);
		//net = dev->getNetwork (net_path);
		goto out;
	}

	net->setStrength (strength);
out:
	return;
}

void
DeviceStore::updateActivationStage (const QString & obj_path, NMActStage act_stage)
{
	Device* dev = NULL;

	dev = getDevice (obj_path);
	if (!dev) {
		goto out;
	}

	dev->setActivationStage (act_stage);

out:
	return;
}

void
DeviceStore::addDevice (Device* dev)
{
	_deviceList.append (dev);
}

void
DeviceStore::removeDevice (Device* dev)
{
	_deviceList.remove (dev);
}

void
DeviceStore::removeDevice (QString obj_path)
{
	DeviceList::iterator i;

	if (_deviceList.empty ()) {
		goto out;
	}
	for (i = _deviceList.begin (); i != _deviceList.end (); ++i) {
		if ((*i)->getObjectPath () == obj_path) {
			removeDevice (*i);
			goto out;
		}
	}

out:
	return;
}

Network*
DeviceStore::getActiveNetwork (const Device* dev)
{
	Network* net = 0;
	NetworkList networkList;
	
	if (!dev)
		goto out;
	
	networkList = dev->getNetworkList ();
	if (networkList.empty ()) {
		goto out;
	}

	for (NetworkList::iterator i = networkList.begin (); i != networkList.end (); ++i) {
		if ((*i)->isActive ()) {
			net = *i;
			goto out;
		}
	}
out:
	return net;
}

Device*
DeviceStore::getActiveDevice ()
{
	Device* dev = NULL;
	DeviceList::iterator i;

	if (_deviceList.empty ()) {
		goto out;
	}
	for (i = _deviceList.begin (); i != _deviceList.end (); ++i) {
		if ((*i)->isActive ()) {
			dev = *i;
			goto out;
		}
	}

out:
	return dev;
}

Device*
DeviceStore::getDevice (Network* net)
{
	Device* dev = NULL;
	DeviceList::iterator i;

	if (_deviceList.empty ()) {
		goto out;
	}
	for (i = _deviceList.begin (); i != _deviceList.end (); ++i) {
		if ((*i)->isWireless ()) {
			NetworkList networkList = (*i)->getNetworkList();
			for (NetworkList::iterator j = networkList.begin (); j != networkList.end (); ++j) {
				if (*j == net){
					dev = *i;
					goto out;
				}
			}
		}
	}

out:
	return dev;
}

Device*
DeviceStore::getDevice (QString obj_path)
{
	DeviceList::iterator i;
	Device* dev = NULL;

	if (_deviceList.empty ()) {
		goto out;
	}
	for (i = _deviceList.begin (); i != _deviceList.end (); ++i) {
		if ((*i)->getObjectPath () == obj_path) {
			dev = *i;
			goto out;
		}
	}

out:
	return dev;
}

DeviceList
DeviceStore::getDeviceList ()
{
	return _deviceList;
}

void
DeviceStore::activateDevice (Device* dev)
{
	DeviceStoreDBus::activateDevice (dev);
}

void
DeviceStore::activateNetwork (Network* net, Device* dev)
{
	DeviceStoreDBus::activateNetwork (net, dev);
}

s_status*
DeviceStore::getStatus ()
{
	return &_status;
}

void
DeviceStore::invalidateActiveDevices ()
{
	DeviceList::iterator i;

	if (_deviceList.empty () == false) {
		for (i = _deviceList.begin (); i != _deviceList.end (); ++i) {
			(*i)->setActive (false);
		}
	}
}


void
DeviceStore::commitUpdate ()
{
	_status.wiredDevices      = 0;
	_status.wirelessDevices   = 0;
	_status.wirelessNetworks  = 0;
	_status.dialUpConnections = 0;

	if (_deviceList.empty () == false) {
		for (DeviceList::iterator i = _deviceList.begin (); i != _deviceList.end (); ++i) {
			if ((*i)->isWired ()) {
				_status.wiredDevices ++;
			} else if ((*i)->isWireless ()) {
				_status.wirelessDevices ++;
				NetworkList networkList = (*i)->getNetworkList ();
				_status.wirelessNetworks += networkList.count();
			}
		}
	}

	if (_dialUpList.empty () == false) {
		for (DialUpList::iterator i = _dialUpList.begin (); i != _dialUpList.end (); ++i) {
			_status.dialUpConnections++;
		}
	}

	emit deviceStoreChanged (this);
	return;
}

void
DeviceStore::clear ()
{
	_dialUpList.clear ();
	_deviceList.clear ();
	return;
}

void
DeviceStore::push (KNetworkManager* ctx)
{
	DeviceStoreDBus::push (ctx);
	DeviceStoreDBus::populateStore ();
}

DeviceStore::DeviceStore ( QObject * parent, const char * name ) : QObject( parent, name )
{
	_status.wiredDevices      = 0;
	_status.wirelessDevices   = 0;
	_status.wirelessNetworks  = 0;
	_status.dialUpConnections = 0;
}

DeviceStore::~DeviceStore ()
{
	for (DeviceList::iterator i = _deviceList.begin (); i != _deviceList.end (); ++i)
		delete *i;
	for (DialUpList::iterator i = _dialUpList.begin (); i != _dialUpList.end (); ++i)
		delete *i;
}

void DeviceStore::emitStrengthChange( Device * dev )
{
	emit strengthChange( dev );
}

void DeviceStore:: emitCarrierOn( Device * dev )
{
	emit carrierOn( dev );
}
void DeviceStore:: emitCarrierOff( Device * dev )
{
	emit carrierOff( dev );
}
void DeviceStore:: emitAdded( Device * dev )
{
	emit added( dev );
}
void DeviceStore:: emitRemoved( Device * dev )
{
	emit removed( dev );
}
void DeviceStore:: emitNoLongerActive( Device * dev )
{
	emit noLongerActive( dev );
}
void DeviceStore:: emitActive( Device * dev )
{
	emit active( dev );
}
void DeviceStore:: emitActivating( Device * dev )
{
	emit activating( dev );
}
void DeviceStore::emitNetworkFound( Network * net )
{
	emit networkFound( net );
}

void DeviceStore::setStats(struct rtnl_link *obj)
{
	char* iface = rtnl_link_get_name(obj);

	for (DeviceList::Iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
	{
		if (QString(iface) == (*it)->getInterface())
		{
			uint64_t st;
			int delta_msecs = 1;
			if ( (*it)->getLastStatRefresh().isValid() )
			{
				delta_msecs = (*it)->getLastStatRefresh().time().msecsTo(QTime::currentTime());
				if (delta_msecs == 0)
					delta_msecs = 1;
			}

			st = rtnl_link_get_stat(obj, RTNL_LINK_RX_PACKETS);
			(*it)->setRxPackets(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_RX_BYTES);
			if ((*it)->getRxBytes() > 0)
				(*it)->setRxBytesPerSec((uint32_t) ((st - (*it)->getRxBytes()) / ((float)delta_msecs / 1000.0)) );
			(*it)->setRxBytes(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_RX_DROPPED);
			(*it)->setRxDropped(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_RX_ERRORS);
			(*it)->setRxErrors(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_TX_PACKETS);
			(*it)->setTxPackets(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_TX_BYTES);
			if ((*it)->getTxBytes() > 0)
				(*it)->setTxBytesPerSec((uint32_t) ((st - (*it)->getTxBytes()) / ((float)delta_msecs / 1000.0)) );
			(*it)->setTxBytes(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_TX_DROPPED);
			(*it)->setTxDropped(st);

			st = rtnl_link_get_stat(obj, RTNL_LINK_TX_ERRORS);
			(*it)->setTxErrors(st);

			
			(*it)->setLastStatRefresh(QDateTime::currentDateTime());
		}
	}
}

static void setStatsCb(struct nl_object *obj, void *arg)
{
	DeviceStore* store = (DeviceStore*) arg;
	store->setStats((struct rtnl_link*) obj);
}

void DeviceStore::getInterfaceStat()
{
    struct nl_handle* nlh = nl_handle_alloc ();
    if (!nlh)
        return;

    if (!nl_connect (nlh, NETLINK_ROUTE))
    {
        struct nl_cache* link_cache = rtnl_link_alloc_cache (nlh);
        if (link_cache)
        {
            // collect the statistcs
            nl_cache_foreach (link_cache, setStatsCb, (void*)this);

            nl_cache_free (link_cache);
        }

        nl_close (nlh);
    }

    nl_handle_destroy (nlh);

    // new statistics
    commitUpdate();
}
#include "knetworkmanager-devicestore.moc"
